
/* SUDMOVZ: $$Revision:$ ; $Date:$		*/

/*----------------------------------------------------------------------
 * Copyright (c) Colorado School of Mines, 1992.
 * All rights reserved.
 *
 * This code is part of SU.  SU stands for Seismic Unix, a processing line
 * developed at the Colorado School of Mines, partially based on Stanford
 * Exploration Project (SEP) software.  Inquiries should be addressed to:
 *
 *  Jack K. Cohen, Center for Wave Phenomena, Colorado School of Mines,
 *  Golden, CO 80401  (jkc@dix.mines.colorado.edu)
 *----------------------------------------------------------------------
 */

#include <localsys.h>
#include <stdio.h>
#include <usp_headers.h>
#include <size_defs.h>
#include <cu_defs.h>
#include <io_defs.h>
#include <ut_defs.h>
#include <save_defs.h>
#include <math.h>
#include <cwp.h>

#define NPTAB 401
#define NITER 10
#define TOL 1.0e-4
#ifndef NULL
#define NULL 0
#endif

/*********************** self documentation **********************/
/* - comment out - jmw
char *sdoc[] = {
"									",
" SUDMOVZ - DMO for V(Z) media for common-offset gathers		",
"									",
" sudmovz <stdin >stdout cdpmin= cdpmax= dxcdp= noffmix= [...]		",
"									",
" Required Parameters:							",
" cdpmin         minimum cdp (integer number) for which to apply DMO	",
" cdpmax         maximum cdp (integer number) for which to apply DMO	",
" dxcdp          distance between adjacent cdp bins (m)			",
" noffmix        number of offsets to mix (see notes)			",
"									",
" Optional Parameters:							",
" vfile=         binary (non-ascii) file containing interval velocities (m/s)",
" tdmo=0.0       times corresponding to interval velocities in vdmo (s)	",
" vdmo=1500.0    interval velocities corresponding to times in tdmo (m/s)",
" fmax=0.5/dt    maximum frequency in input traces (Hz)			",
" smute=1.5      stretch mute used for NMO correction			",
" speed=1.0      twist this knob for speed (and aliasing)		",
" verbose=0      =1 for diagnostic print				",
"									",
" Notes:								",
" Input traces should be sorted into common-offset gathers.  One common-",
" offset gather ends and another begins when the offset field of the trace",
" headers changes.							",
"									",
" The cdp field of the input trace headers must be the cdp bin NUMBER, NOT",
" the cdp location expressed in units of meters or feet.		",
"									",
" The number of offsets to mix (noffmix) should typically equal the ratio of",
" the shotpoint spacing to the cdp spacing.  This choice ensures that every",
" cdp will be represented in each offset mix.  Traces in each mix will	",
" contribute through DMO to other traces in adjacent cdps within that mix.",
"									",
" vfile should contain the regularly sampled interval velocities as a	",
" function of time.  If vfile is not supplied, the interval velocity	",
" function is defined by linear interpolation of the values in the tdmo	",
" and vdmo arrays.  The times in tdmo must be monotonically increasing.	",
"									",
" For each offset, the minimum time to process is determined using the	",
" smute parameter.  The DMO correction is not computed for samples that	",
" have experienced greater stretch during NMO.				",
"									",
" Trace header fields accessed:  nt, dt, delrt, offset, cdp.		",
NULL};
- comment out - jmw */
/**************** end self doc ********************************/

/* Credits:
 *	CWP: Craig
 *
 * Technical Reference:
 *	Artley, C., 1992,
 *	Dip-moveout processing for depth-variable velocity,
 *	M.S. Thesis, Colorado School of Mines;
 *	also published at Center for Wave Phenomena report CWP-115.
 */

/* function prototypes for functions defined below */
void dmooff (float *x, float *a, float *tau, int nttab, float dt, float ft,
	int nptab, float dptab, float fp, float lp,
	int nx, float dx, float offset, int nt,
	float *vrms, float fmax, float smute, float speed, int verbose,
	float *q);
void dmomap (float *x, float *a, float *tau, int nt, float dt, float ft,
	int nptab, float dptab, int np, float dp2, float fp, float lp,
	int itn, float tn, float h, float *vrms,
	int verbose, float *t0table, int ntn);
void dmoapply (int nx, float dx, int nt, float dt, int np, float dp2,
	float h, float *tntable, int *it0min, int *nt0,
	float tmin, float fmax, float *qx);
void jakub1k (float k, float h, int np, float dp2, int nt, float dt, float ft,
	float *tntable, int *it0min, int *nt0, float fmax, complex *qq);
float tmute(float *vrms, int nt, float dt, float ft, float h, float smute);
void getX (float tsg, float h, float p0, float v, float *vars);
void getFJ (float *x, float *a, float *tau,
	int nt, int np, float dt, float dp, float tsg, float h, float p0,
	float *vars, float *eqns, float *jacob);
void intder (float *f, int nx, float dx, float fx,
	int ny, float dy, float fy, float x, float y,
	float *fxy, float *dfdx, float *dfdy);
void rayvt (float p, int nt, float dt,
	float v[], float tau[], float x[], float a[]); 


char l_head[12000];
char *tr,*tro;
char *argv0;			/* use this in error messages */

int main(int argc, char **argv)
{
	int nt;		/* number of time samples per trace */
	int nsi;	/* smaple interval (whole number */
	float unitsc;	/* time sampling scaler */
	float dt;	/* time sampling interval */
	float ft;	/* time of first sample */
	int it;		/* time sample index */
	int cdpmin;	/* minimum cdp to process */
	int cdpmax;	/* maximum cdp to process */
	float dx;	/* cdp sampling interval */
	int nx;	        /* number of cdps to process */
	int nxfft;	/* number of cdps after zero padding for fft */
	int nxpad;	/* minimum number of cdps for zero padding */
	int ix;		/* cdp index, starting with ix=0 */
	int noffmix;	/* number of offsets to mix */
	float *tdmo;	/* times at which rms velocities are specified */
	float *vdmo;	/* rms velocities at times specified in tdmo */
	int ntdmo;	/* number tnmo values specified */
	int itdmo;	/* index into tnmo array */
	int nvdmo;	/* number vnmo values specified */
	float fmax;	/* maximum frequency */
	float *vt;	/* uniformly sampled v(t) */
	float *vrms;	/* uniformly sampled vrms(t) */
	float *q;	/* traces for one offset - common-offset gather */
	float *qmix;	/* DMO-corrected and mixed traces to be output */
	float offset;	/* source-receiver offset of current trace */
	int ioffset;	/* source-receiver integer offset of current trace */
	int noff;	/* number of offsets processed in current mix */
	int offave;	/* average of mixed offsets */
	int cdpcur;	/* depth index of current trace */
	int first;	/* flag */
	int stacor;	/* static word value */
	int numrec;	/* number records in data set */
	int numtrc;	/* number traces in data set */
	int current_record;	/* current record number */
	int recnum;	/* record number */
	int trcnum;	/* trace number */
	int cdpind;	/* CDP number */
	int ntrace;	/* number of traces processed in current mix */
	int live;	/* number of live traces processed in current mix */
	int itrace;	/* trace index */
	int done;	/* non-zero if done */
	int verbose;	/* =1 for diagnostic print */
	int help;
	char vfile[256];	/* name of interval velocity file */
	FILE *hfp;	/* file pointer for temporary header file */
	int nttab;	/* number of times in ray tables */
	int nptab=NPTAB;/* number of ray parameters in ray tables */
	int ip;		/* ray parameter counter */
	int j;
	int k;
	float smute;	/* NMO stretch mute */
	float t;	/* time */
	float sum;	/* sum accumulator */
	float lt;	/* last time */
	float lttab;	/* last time in ray tables */
	float p;	/* ray parameter */
	float fp;	/* first ray parameter */
	float lp;	/* last ray parameter */
	float dptab;	/* ray parameter sampling interval in ray tables */
	float speed;	/* speed knob --- twist for speed and aliasing */
	float *x;	/* table [nptab][nttab] of horizontal distance ray */
	float *tau;	/* table [nptab][nttab] of time depth of ray */
	float *a;	/* table [nptab][nttab] of propagation angle of ray */
	float *sample; /* jmw */
	int luin;	/* logical unit for input */
	int luout;	/* logical unit for output */
	char ntap[256],otap[256];
	int nbytes,hdrbytes,lbytes;
	char *tdmo_par,*vdmo_par;
	void help_sub();
	FILE *tattle = NULL;
	char *ikp_pid,*tattle_file;
	int hostnamelen = 64;
	char hostname[64];

	save_cmd_args(argc, argv); 

	/* hook up getpar */

	argv0 = argv[0];

	help = C_ARGIS("-h",&argc,argv);
	help += C_ARGIS("-?",&argc,argv);
	if (help) {
	  help_sub();
	  exit(0);
	  }

/* added this section so stderr would go to a print file rather than the 
   tty. Running multiple copies in parallel created mass output which was
   intermingled.
					- joe m. wade 6/29/94 
*/
	tattle_file = (char *) malloc(32*sizeof(char));
	if (tattle_file != NULL) {
	    if (((ikp_pid = (char *)getenv("IKP_PID")) != NULL) &&
	      (gethostname(hostname,hostnamelen) == 0 ))
	        sprintf(tattle_file,"DMOVZ.%06d.%s",getpid(),&hostname[0]);
	    else
	      sprintf(tattle_file,"DMOVZ.%06d",getpid());
	    freopen(tattle_file,"w",stderr);
	    }
	free(tattle_file);

	/* get parameters */

	C_ARGSTR( "-N",ntap," "," " ,&argc,argv);

	C_ARGSTR( "-O",otap," "," " ,&argc,argv);
	
	C_ARGI4("-cdpmin",&cdpmin,0,0,&argc,argv);
	if (cdpmin == 0) {
	  fprintf(stderr,"%s: must specify cdpmin\n",argv0);
	  exit(100);
	  }
	C_ARGI4("-cdpmax",&cdpmax,0,0,&argc,argv);
	if (cdpmax == 0) {
	  fprintf(stderr,"%s: must specify cdpmax\n",argv0);
	  exit(100);
	  }
	if (cdpmin>cdpmax) {
		fprintf(stderr,"%s: cdpmin must not be greater than cdpmax\n",
			argv0);
		exit(100);
		}
	C_ARGR4("-dxcdp",&dx,0.,0.,&argc,argv);
	if (dx == 0.) {
	  fprintf(stderr,"%s: must specify dxcdp\n",argv0);
	  exit(100);
	  }
	C_ARGI4("-noffmix",&noffmix,1,1,&argc,argv);
	  /* exit(100); */

	C_ARGR4("-fmax",&fmax,0.,0.,&argc,argv);
	verbose = C_ARGIS("-V",&argc,argv);
	C_ARGR4("-speed",&speed,1.0,1.0,&argc,argv);
	C_ARGR4("-smute",&smute,1.5,1.5,&argc,argv);

	/* open i/o units */

	if (strcmp(ntap," ") == 0)
	  luin = 0;
	else
	  C_LBOPEN(&luin,ntap,"r");

	if (strcmp(otap," ") == 0)
	  luout = 1;
	else
	  C_LBOPEN(&luout,otap,"w");

	/* get information from the first header */
	C_RTAPE(luin,l_head,&nbytes);
	if (nbytes == 0) {
		fprintf(stderr,"%s: can't get first trace\n",argv0);
		exit(100);
		}
	nt = ((usp_line_header *)l_head)->NumSmp;	/* tr.ns;	*/
	nsi = ((usp_line_header *)l_head)->SmpInt;
	C_SAVER(l_head,"UnitSc",&unitsc,LINEHEADER);
	if ( unitsc == 0.0 ) unitsc = .001;
	numrec = ((usp_line_header *)l_head)->NumRec;
	numtrc = ((usp_line_header *)l_head)->NumTrc;
	dt = ((usp_line_header *)l_head)->SmpInt/1000.;	/* tr.dt/1000000.0; */
	if (fmax == 0.) {
		fmax = .5/dt;
	}
	dt = dt * unitsc;
	ft = ((usp_line_header *)l_head)->TmMsFS; /* tr.delrt/1000.0;	*/
	if (ft!=0.0)
	  fprintf(stderr,"%s: ft=%f --- nonzero ft not yet implemented!\n",
		argv0,ft);

	tr  = (char *) malloc(SZTRHD + (nt * SZSMPD));
	tro = (char *) malloc(SZTRHD + (nt * SZSMPD));
	if ((tr == NULL) || (tro == NULL)) {
	  fprintf(stderr,"%s: unable to allocate %d ",argv0,SZTRHD+(nt*SZSMPD));
	  fprintf(stderr,"bytes for trace area\n");
	  exit(100);
	  }

	if (verbose) {
	  fprintf(stderr,"\n");
	  fprintf(stderr,"number records in line = %d\n",numrec);
	  fprintf(stderr,"number traces in line  = %d\n",numtrc);
	  fprintf(stderr,"sample interval (sec)  = %f\n",dt);
	  fprintf(stderr,"sample interval (ms,us)= %d\n",nsi);
	  fprintf(stderr,"\n");
	  fprintf(stderr,"minimum cdp number          = %d\n",cdpmin);
	  fprintf(stderr,"maximum cdp number          = %d\n",cdpmax);
	  fprintf(stderr,"cdp bin spacing (ft,m)      = %f\n",dx);
	  fprintf(stderr,"number offsets to mix       = %d\n",noffmix);
	  fprintf(stderr,"maximum frequency to process= %f\n",fmax);
	  fprintf(stderr,"speed parameter             = %f\n",speed);
	  fprintf(stderr,"muting parameter            = %f\n",smute);
	  fprintf(stderr,"\n");
	  show_hlh(l_head);
	}

	/* adjust line header & output */
	numrec = numrec / noffmix;
	((usp_line_header *)l_head)->NumRec = numrec;
	C_SAVEW(l_head,"UnitSc",unitsc,LINEHEADER);
	lbytes = nbytes;
	save_hlh (l_head, lbytes, &nbytes);
	C_WRTAPE(luout,l_head,nbytes);


	/* allocate space for velocity arrays */
	nttab = 2*nt-1;
        vt = (float *) malloc(nttab * sizeof(float));
	if (vt == 0) {
	  fprintf(stderr,"%s: unable to allocate %d bytes for vt\n",
		argv0,nttab * sizeof(float));
	  exit(100);
	  }
	vrms = (float *) malloc(nttab * sizeof(float));
	if (vrms == 0) {
	  fprintf(stderr,"%s: unable to allocate %d bytes for vrms\n",
		argv0, nttab * sizeof(float));
	  exit(100);
	  }

        /* determine velocity function v(t) */
	C_ARGSTR("-vfile",vfile," "," ",&argc,argv);
	
        if (strcmp(vfile," ") == 0) {
	  tdmo_par = (char *)malloc(1024 * sizeof(char));
	  if (tdmo_par == NULL) {
	    fprintf(stderr,"%s: unable to allocate space for tdmo_par\n",
		argv0);
	    exit(1);
	    }
	  C_ARGSTR("-tdmo",tdmo_par," "," ",&argc,argv);
	  ntdmo = count_fppar(tdmo_par);
          if (ntdmo==0) ntdmo = 1;
          tdmo = (float *) malloc(ntdmo * sizeof(float));
	  if (tdmo == 0) {
	    fprintf(stderr,"%s: unable to allocate %d bytes for tdmo\n",
		argv0, ntdmo * sizeof(float));
	    exit(100);
	    }
	  if (get_fppar(tdmo_par,tdmo) == 0) tdmo[0] = 0.0;

        /* convert ms to seconds */
	for (it=0; it<ntdmo; it++) {
		tdmo[it] = tdmo[it] / 1000.;
	}


	  vdmo_par = (char *)malloc(1024 * sizeof(char));
	  if (vdmo_par == NULL) {
	    fprintf(stderr,"%s: unable to allocate space for vdmo_par\n",
		argv0 );
	    exit(1);
	    }
	  C_ARGSTR("-vdmo",vdmo_par," "," ",&argc,argv);
	  nvdmo = count_fppar(vdmo_par);
          if (nvdmo==0) nvdmo = 1;
          if (nvdmo!=ntdmo)
	    fprintf(stderr,"number of tdmo and vdmo must be equal\n");
          vdmo = (float *) malloc(nvdmo * sizeof(float));
	  if (vdmo == 0) {
	    fprintf(stderr,"%s: unable to allocate %d bytes for vdmo\n",
	  	argv0, nvdmo * sizeof(float));
	    exit(100);
	    }

	  if (get_fppar(vdmo_par,vdmo) == 0) vdmo[0] = 1500.0;
          for (itdmo=1; itdmo<ntdmo; ++itdmo)
            if (tdmo[itdmo]<=tdmo[itdmo-1])
               fprintf(stderr,"tdmo must increase monotonically\n");
          for (it=0,t=0.0; it<nt; ++it,t+=dt)
            intlin(ntdmo,tdmo,vdmo,vdmo[0],vdmo[ntdmo-1],1,&t,&vt[it]);
          } else {
                if (fread(vt,sizeof(float),nt,fopen(vfile,"r"))!=nt)
                  fprintf(stderr,"cannot read %d velocities from file %s\n",
			  nt,vfile);
        }
	if (verbose) {
	  fprintf(stderr,"\n");
	  for (it=0; it<ntdmo; it++)
		fprintf(stderr,"it= %d, time = %f, velocity = %f\n",it,tdmo[it],vdmo[it]);
	  fprintf(stderr,"\n");
	}

	/* constant extrapolation of interval velocities */
	for (it=nt; it<nttab; ++it)
		vt[it] = vt[nt-1];

        /* compute vrms */
	vrms[0] = vt[0];
	for (it=1,sum=0.0; it<nttab; ++it) {
		sum += vt[it-1]*vt[it-1];
			vrms[it] = sqrt(sum/it);
	}

        /* compute half-velocities */
        for (it=0; it<nttab; ++it)
                vt[it] *= 0.5;

	/* compute extreme values */
	ft = 0.0;
	lt = ft+(nt-1)*dt;
	lttab = ft+(nttab-1)*dt;
	fp = 0.0;
	lp = 1.0/vt[0];

	/* compute p sampling for ray tables */
	dptab = lp/(nptab-1);

	/* allocate space for ray tables */
	tau = (float *) malloc(nttab * nptab * sizeof(float));
	if (tau == 0) {
	  fprintf(stderr,"%s: unable to allocate %d bytes for tau\n",
		argv0, nttab * nptab * sizeof(float));
	  exit(100);
	  }
	x = (float *) malloc(nttab * nptab * sizeof(float));
	if (x == 0) {
	  fprintf(stderr,"%s: unable to allocate %d bytes for x\n",
		argv0, nttab * nptab * sizeof(float));
	  exit(100);
	  }
	a = (float *) malloc(nttab * nptab * sizeof(float));
	if (a == 0) {
	  fprintf(stderr,"%s: unable to allocate %d bytes for a\n",
		argv0, nttab * nptab * sizeof(float));
	  exit(100);
	  }

	/* compute ray tables tau(p,t), x(p,t), a(p,t) */
	if (verbose) fprintf(stderr,"Computing ray tables...\n");
	for (ip=0,p=fp; ip<nptab; ++ip,p+=dptab)
		rayvt(p,nttab,dt,vt,tau+(ip*nttab),x+(ip*nttab),a+(ip*nttab));
	if (verbose) fprintf(stderr,"Ray tables ready...\n");
	free(vt);

	/* determine number of cdps to process */
	nx = cdpmax - cdpmin + 1;
	
	/* allocate and zero offset mix accumulator qmix(t,x) */
	qmix = (float *) malloc(nt * nx * sizeof(float));
	if (qmix == 0) {
	  fprintf(stderr,"%s: unable to allocate %d bytes for qmix\n",
		argv0, nt * nx * sizeof(float));
	  exit(100);
	  }
	
	for (ix=0; ix<nx; ++ix)
	  for (it=0; it<nt; ++it)
	    qmix[(ix*nt)+it] = 0.0;
		
	/* open temporary file for headers */
	hfp = tmpfile();
	if (hfp == NULL) {
	  fprintf(stderr,"%s: error opening temp file for headers\n",argv0);
	  exit(100);
	  }
	
	/* initialize */
	done = 0;
	ntrace = 0;
	noff = 0;
	first = 0;
	offave = 0;

	/* loop over traces */

	for ( j=0; j < numrec; j++ ) {	

		for ( k=0; k < numtrc; k++ ) {


	        C_RTAPE(luin, tr, &nbytes);
		hdrbytes = nbytes;	/* jmw added */
	        if (nbytes == 0) break;

		stacor =  abs((float)((usp_trace_header *)tr)->StaCor);
		if ( stacor != 30000 ) {
		recnum =  abs((float)((usp_trace_header *)tr)->RecNum);
		trcnum =  abs((float)((usp_trace_header *)tr)->TrcNum);
		offset =      (float)((usp_trace_header *)tr)->DstSgn ;
		cdpcur =  abs((float)((usp_trace_header *)tr)->DphInd);
		ioffset = offset;
		offset  = fabs (offset);
		}

/*
fprintf(stderr,"j %d k %d cdpcur %d offset %f\n",j,k,cdpcur,offset);
*/
		if ( cdpcur >= cdpmin && cdpcur <= cdpmax ) {

		   if (fwrite(tr,SZTRHD,1,hfp) == 0) {
		      fprintf(stderr,"%s: error writing headers ",argv0);
		      fprintf(stderr,"to temp file\n");
		      exit(100);
		   }
	           ++ntrace;

          /* allocate space for new offset */
		   if ( first == 0 ) {
		        offave = offave + ioffset;
		        nxpad = 0.5*ABS(offset/dx);
		        nxfft = npfar(nx+nxpad);
		        q = (float *) malloc(nt * (nxfft+2) * sizeof(float));
		        if (q == 0) {
		           fprintf(stderr,
			           "%s: unable to allocate %d bytes for q\n",
		           argv0, nt * (nxfft+2) * sizeof(float));
		           exit(100);
		        }
		        for (ix=0; ix<nxfft; ++ix)
			        for (it=0; it<nt; ++it)
				        q[(ix*nt)+it] = 0.0;
		        first = 1;

	   	   }

/* I did away with the TrcDat definition in the usp_headers file...

		   bcopy(&((usp_trace_header *)tr)->TrcDat[0],
			   q+((cdpcur-cdpmin)*nt),
		 	   nt*sizeof(float));
*/
		   memcpy(q+((cdpcur-cdpmin)*nt),
			   ( char *)tr + sizeof(usp_trace_header),
		 	   nt*sizeof(float));
		}
	}
	first = 0;

/* fprintf(stderr,"dmo for offset %d\n",ioffset); */

          /* do dmo for current common-offset gather */

	if ( offset != 0.0 )
	dmooff(x,a,tau,nttab,dt,ft,nptab,dptab,fp,lp,
		nx,dx,offset,nt,vrms,fmax,smute,speed,
		verbose,q);

          /* add dmo-corrected traces to mix */
	for (ix=0; ix<nx; ++ix)
		for (it=0; it<nt; ++it)
			qmix[(ix*nt)+it] += q[(ix*nt)+it];

          /* count offsets in mix */
	++noff;
 
          /* free space for old common-offset gather */
	free(q);

          /* if a mix of offsets is complete */
	if (noff==noffmix) {
 
		offave = offave / noff;
                /* rewind trace header file */
		fseek(hfp,0L,SEEK_SET);
 
                /* loop over all output traces */
		for (itrace=0; itrace<numtrc; ++itrace) {
 
			if ( itrace < ntrace ) {

			if (fread(tro,SZTRHD,1,hfp) == 0) {
			fprintf(stderr,"%s: error reading headers ",argv0);
			fprintf(stderr,"from temp file\n");
			exit(100);
			}
 
                        /* get dmo-corrected data */

/* I did away with the TrcDat definition in the usp_headers file...

			bcopy(
			&qmix[(((usp_trace_header *)tro)->DphInd-cdpmin)*nt],
			&((usp_trace_header *)tro)->TrcDat[0],
			nt*sizeof(float));
*/
			memcpy(
			(char *)tro + sizeof(usp_trace_header),
			&qmix[(((usp_trace_header *)tro)->DphInd-cdpmin)*nt],
			nt*sizeof(float));

			}
			else {

			current_record = ((usp_trace_header *)tro)->RecNum;
			memset(tro,0,SZTRHD+(nt*SZSMPD));
			((usp_trace_header *)tro)->RecNum = current_record;
			((usp_trace_header *)tro)->TrcNum = itrace+1;
			((usp_trace_header *)tro)->StaCor = 30000;

			}

			((usp_trace_header *)tro)->DstSgn = offave;
			C_WRTAPE(luout,tro,hdrbytes);
		}
 
                /* report */
		if (verbose) {
		fprintf(stderr,"\tCompleted mix of %d offsets with ",noff);
		fprintf(stderr,"%d traces\n",ntrace);
		}

		noff = 0;
		live = 0;
		ntrace = 0;
		offave = 0;
		fseek(hfp,0L,SEEK_SET);
	        for (ix=0; ix<nx; ++ix)
	    	          for (it=0; it<nt; ++it)
	  	 	          qmix[(ix*nt)+it] = 0.0;

		}
	}

	fflush(stderr);
	C_LBCLOS(luout);
	return EXIT_SUCCESS;
}

void dmooff (float *x, float *a, float *tau, int nttab, float dt, float ft,
	int nptab, float dptab, float fp, float lp,
	int nx, float dx, float offset, int nt,
	float *vrms, float fmax, float smute, float speed, int verbose,
	float *q)
/***************************************************************************
Compute and apply DMO correction to one common offset section.
****************************************************************************
Input:
x	array[nptab][nttab] of horizontal position of ray 
a	array[nptab][nttab] of propagation angle along ray
tau	array[nptab][nttab] of time depth of ray
nttab	number of times in ray tables
dt	time sampling interval
ft	first time (must be zero for now)
nptab	number of ray parameters in tables
dptab	ray parameter sampling interval in tables
fp	first ray parameter
lp	last ray parameter
nx	number of traces in common offset section
dx	trace sampling interval
offset	source-receiver full offset
nt	number of time samples
vrms	array[nttab] of rms velocity as function of two-way vertical time
fmax	maximum frequency to process
smute	NMO stretch mute.  Samples experiencing greater NMO stretch are muted
speed	knob to increase speed and aliasing by coarsening the slope sampling
verbose	flag for diagnostic info
q	array[nx][nt] of input common offset section

Output:
q	array[nx][nt] of output common offset section (DMO corrected)
****************************************************************************
Author: Craig Artley, Colorado School of Mines, 09/28/91
****************************************************************************/
{
	int it,ntn,itn,itnmin,itnmax,np,npdmo,ip,it0max,*it0min,*nt0;
	float h,lt,tn,ftn,dtn,dp2,tmin,*t0table,*tntable;

	/* compute minimum time based on stretch mute */
	tmin = tmute(vrms,nt,dt,ft,0.5*offset,smute);
	
	/* compute p sampling required to avoid aliasing */
	lt = (nt-1)*dt;
	if (tmin>=lt)
	 fprintf(stderr,"tmin=%f >= lt=%f:  no data survive the mute!\n",
		tmin,lt);

	dtn = 10.0*dt;
	ftn = tmin;
	ntn = NINT((lt-ftn)/dtn)+1;
	dp2 = tmin/(fmax*0.25*offset*offset);
	dp2 *= speed;
	np = NINT(lp*lp/dp2)+1;

	/* allocate space for dmo mapping tables */

	it0min = (int *) malloc(np * sizeof(int));
	if (it0min == 0) {
	  fprintf(stderr,"%s: unable to allocate %d bytes for it0min\n",
		argv0, np * sizeof(int));
	  exit(100);
	  }

	nt0 = (int *) malloc(np * sizeof(int));
	if (nt0 == 0) {
	  fprintf(stderr,"%s: unable to allocate %d bytes for nt0\n",
		argv0, np * sizeof(int));
	  exit(100);
	  }

	t0table = (float *) malloc(ntn * np * sizeof(float));
	if (t0table == 0) {
	  fprintf(stderr,"%s: unable to allocate %d bytes for t0table\n",
		argv0, ntn * np * sizeof(float));
	  exit(100);
	  }

	tntable = (float *) malloc(nt * np * sizeof(float));
	if (tntable == 0) {
	  fprintf(stderr,"%s: unable to allocate %d bytes for tntable\n",
		argv0, nt * np * sizeof(float));
	  exit(100);
	  }

	/* zero the tables */
	for (ip=0; ip<np; ++ip)
		for (itn=0; itn<ntn; ++itn)
			t0table[(ip*ntn)+itn] = 0.0;
	for (ip=0; ip<np; ++ip)
		for (it=0; it<nt; ++it)
			tntable[(ip*nt)+it] = 0.0;
	
	/* compute (unsigned) half-offset */
	h = ABS(offset/2.0);

	/* loop over nmo times tn */
	for (itn=0,tn=ftn; itn<ntn; ++itn,tn+=dtn) {

		/* compute dmo mapping t0(p) for this tn */
		dmomap(x,a,tau,nttab,dt,ft,nptab,dptab,np,dp2,fp,lp,
			itn,tn,h,vrms,verbose,t0table,ntn);
	}

	/* inverse interpolation to get tn(t0,p) */
	for (ip=0; ip<np; ++ip) {
		for (itn=0; itn<ntn; ++itn)
			if (t0table[(ip*ntn)+itn] > 0.0) break;
		itnmin = itn;
		
		it0min[ip] = NINT(t0table[(ip*ntn)+itnmin]/dt);
		if (it0min[ip]>=nt) it0min[ip] = nt-1;

		for (itn=ntn-1; itn>=0; --itn)
			if (t0table[(ip*ntn)+itn] > 0.0) break;
		itnmax = itn;
		if (itnmax <= itnmin) break;

		it0max = NINT(t0table[(ip*ntn)+itnmax]/dt);
		if (it0max>=nt) it0max = nt-1;

		nt0[ip] = it0max-it0min[ip]+1;

		/* check that table is monotonically increasing */
		for (itn=itnmin+1; itn<itnmax; ++itn)
			if (t0table[(ip*ntn)+itn] <= t0table[(ip*ntn)+itn-1]
			  && verbose)
				fprintf(stderr,
					"t0table not monotonically "
					"increasing---extrapolating\n");

		yxtoxy(itnmax-itnmin+1,dtn,ftn+itnmin*dtn,
			&t0table[(ip*ntn)+itnmin],
			nt0[ip],dt,it0min[ip]*dt,
			ftn+itnmin*dtn,ftn+itnmax*dtn,&tntable[ip*nt]);

		/* extend tntable by linear extrapolation */
		for (it=nt0[ip]; it<nt; ++it) {
			tntable[(ip*nt)+it] = 2.0*tntable[(ip*nt)+it-1]
				-tntable[(ip*nt)+it-2];
		}
		nt0[ip] = nt-it0min[ip];
	}

	/* reset the number of ray parameters for dmo */
	npdmo = MIN(ip,np);

	/* free space */
	free(t0table);

	/* dmo by dip decomposition */
	dmoapply(nx,dx,nt,dt,npdmo,dp2,h,tntable,it0min,nt0,tmin,fmax,q);

	/* free space */
	free(it0min);
	free(nt0);
	free(tntable);

}

void dmomap (float *x, float *a, float *tau, int nt, float dt, float ft,
	int nptab, float dptab, int np, float dp2, float fp, float lp,
	int itn, float tn, float h, float *vrms,
	int verbose, float *t0table, int ntn)
/***************************************************************************
Compute DMO correction for one NMO time, tn, using Newton's method to solve
system of equations.
****************************************************************************
Input:
x	array[nptab][nt] of horizontal position of ray 
a	array[nptab][nt] of propagation angle along ray
tau	array[nptab][nt] of time depth of ray
nt	number of times in ray tables
dt	time sampling interval
ft	first time (must be zero for now)
nptab	number of ray parameters in ray tables
dptab	ray parameter sampling interval in ray tables
np	number of ray parameters for which to find DMO mapping
dp2	sampling interval in (ray parameter)^2 in DMO mapping table
fp	first ray parameter
lp	last ray parameter
itn	index of NMO time for which to compute DMO mapping
tn	NMO time for which to compute DMO mapping
h	source-receiver half offset
vrms	array[nt] of rms velocity as function of two-way vertical time
verbose	flag for diagnostic info (does nothing here)
ntn

Output:
t0table	array[np][ntn] containing DMO mapping t0(tn,p0)
	One call to this function fills in the itn'th row of this table.
****************************************************************************
Author: Craig Artley, Colorado School of Mines, 09/28/91
****************************************************************************/
{
	int ip,i,neqns=5,niter=NITER,iter,*ipvt;
	float p2,vtn,tsg,p0,tautp,dtaudt,dtaudp,error,rcond,
		lt,tol=TOL,*vars,*eqns,*work,t0,x0,*jacob;

	/* compute last time in ray tables */
	lt = ft+(nt-1)*dt;

	/* allocate space for system of equations */

	vars = (float *) malloc(neqns * sizeof(float));	 /* variables X */
	if (vars == 0) {
	  fprintf(stderr,"%s: unable to allocate %d bytes for vars\n",
		argv0, neqns * sizeof(float));
	  exit(100);
	  }

	eqns = (float *) malloc(neqns * sizeof(float));	 /* F(X) */
	if (eqns == 0) {
	  fprintf(stderr,"%s: unable to allocate %d bytes for eqns\n",
		argv0, neqns * sizeof(float));
	  exit(100);
	  }

		/* Jacobian dF_i/dx_j */
	jacob = (float *) malloc(neqns * neqns * sizeof(float));
	if (jacob == 0) {
	  fprintf(stderr,"%s: unable to allocate %d bytes for jacob\n",
		argv0, neqns * neqns * sizeof(float));
	  exit(100);
	  }

		/* work array for sgeco() */
	work = (float *) malloc(neqns * sizeof(float));
	if (work == 0) {
	  fprintf(stderr,"%s: unable to allocate %d bytes for work\n",
		argv0, neqns * sizeof(float));
	  exit(100);
	  }

		/* pivot index array for sgeco() */
	ipvt = (int *) malloc(neqns * sizeof(float));
	if (ipvt == 0) {
	  fprintf(stderr,"%s: unable to allocate %d bytes for ipvt\n",
		argv0, neqns * sizeof(float));
	  exit(100);
	  }


	/* calculate tsg from tn via Dix approximation */
	vtn = vrms[(int)(tn/dt)];
	tsg = sqrt(tn*tn+4*h*h/(vtn*vtn));

	/* get initial values for parameters */
	getX(tsg,h,fp,vtn,vars);

	for (ip=0,p2=fp; ip<np; ++ip,p2+=dp2) {

		/* compute p0 */
		p0 = sqrt(p2);

		/* use Newton's method to improve the guess */
		for (iter=0; iter<niter; ++iter) {
	
			/* calculate F(X) and J(X) */
			getFJ(x,a,tau,nt,nptab,dt,dptab,tsg,h,p0,
				vars,eqns,jacob);

			/* compute total error */
			for (i=0,error=0.0; i<neqns; ++i)
				error += ABS(eqns[i]);

			/* if iteration has converged, break */
			if (error<=tol) break;

			/* solve system */
			sgeco(jacob,neqns,ipvt,&rcond,work);
			sgesl(jacob,neqns,ipvt,eqns,0);

			/* update X */
			if (rcond!=0.0) {
				for (i=0; i<neqns; ++i)
					vars[i] -= eqns[i];
			} else {
				/* system is ill-conditioned, break */
				break;
			}

			/* clip ps and pg values to pmax if necessary */
			if (ABS(vars[1])>lp) {
				vars[1] = SGN(vars[1])*lp;
			}
			if (ABS(vars[2])>lp) {
				vars[2] = SGN(vars[2])*lp;
			}

			/* clip tg to 0<tg<tsg */
			if (vars[3]<0.0) {
				vars[3] = 0.0;
			}
			if (vars[3]>tsg) {
				vars[3] = tsg;
			}

			/* clip t0 values to 0<t0<lt */
			if (vars[4]<0.0) {
				vars[4] = 0.0;
			}
			if (vars[4]>lt) {
				vars[4] = lt;
			}

		}

		/* vars[] now contains the solution to the system */

		/* compute depth of reflection point */
		intder(tau,nt,dt,ft,nptab,dptab,fp,vars[4],ABS(p0),
			&tautp,&dtaudt,&dtaudp);

		/* if reflector is at surface, break */
		if (tautp<dt) {
			break;
		}

		/* if system is ill-conditioned, break */
		if (rcond==0.0) {
			break;
		}

		/* now have t0(x0) for this tsg, p0, h -- store it */
		x0 = vars[0];
		t0 = vars[4];
		t0table[(ip*ntn)+itn] = t0+p0*x0;

	}

	/* free space */
	free(vars);
	free(eqns);
	free(jacob);
	free(work);
	free(ipvt);
}

void dmoapply (int nx, float dx, int nt, float dt, int np, float dp2,
	float h, float *tntable, int *it0min, int *nt0,
	float tmin, float fmax, float *qx)
/*****************************************************************************
Apply DMO correction to common offset section using previously computed
mapping tables.
******************************************************************************
Input:
nx		number of traces in common offset section
dx		trace sampling interval
nt		number of time samples per trace
dt		time sampling interval
np		number of ray parameters in DMO mapping tables
dp2		ray parameter (squared) sampling interval
h		source-receiver half-offset
tntable		array[np][nt0] of DMO mapping table tn(t0,p0)
it0min		array[np] of index of first entry in DMO mapping table 
nt0		array[np] of number of t0's in DMO mapping table
fmax		maximum frequency of data (frequencies above fmax are muted)
qx		array[nx][nt] of input NMO corrected data

Output:
qx		array[nx][nt] of output DMO corrected data
******************************************************************************
Author: Craig Artley, Colorado School of Mines, 08/29/91
******************************************************************************/
{
	int ix,nxfft,it,itmin,ik,nk;
	float k,dk,fk,ft,scale;
	complex **qk;

	/* determine wavenumber sampling (for real to complex FFT) */
	nxfft = npfar(nx);
	nk = nxfft/2+1;
	dk = 2.0*PI/(nxfft*dx);
	fk = 0.0;

	ft = 0.0;

	/* pointers to complex q(t,k) */
	qk = (complex**) malloc(nk * sizeof(complex *));
	if (qk == 0) {
	  fprintf(stderr,"%s: unable to allocate %d bytes for qk\n",
		argv0,nk * sizeof(complex *));
	  exit(100);
	  }

	for (ik=0; ik<nk; ++ik)
		qk[ik] = (complex *)&qx[0]+ik*nt;
	
	/* Fourier transform x to k */
	pfa2rc(-1,2,nt,nxfft,qx,qk[0]);

	/* loop over wavenumber */
	for (ik=0,k=fk; ik<nk; ++ik,k+=dk) {

		/* apply DMO correction */
		jakub1k(k,h,np,dp2,nt,dt,ft,tntable,
			it0min,nt0,fmax,qk[ik]);
	}

	/* Fourier transform k to x (including FFT scaling and mute) */
	pfa2cr(1,2,nt,nxfft,qk[0],qx);
	scale = 1.0/nxfft;
	itmin = MIN(nt,NINT(tmin/dt));
	for (ix=0; ix<nx; ++ix) {
		for (it=0; it<itmin; ++it)
			qx[(ix*nt)+it] = 0.0;
		for (it=itmin; it<nt; ++it)
			qx[(ix*nt)+it] *= scale;
	}

	/* free space */
	free(qk);
}

void jakub1k (float k, float h, int np, float dp2, int nt, float dt, float ft,
	float *tntable, int *it0min, int *nt0, float fmax, complex *qq)
/*****************************************************************************
Jakubowicz's dip decomposition DMO for one wavenumber
******************************************************************************
Input:
k		wavenumber
h		half-offset
np		number of ray paramaters
dp2		sampling interval in (ray parameter)^2
nt		number of time samples
dt		time sampling interval
ft		first time sample
qq		complex array[nt] of input NMO corrected data 

Output:
qq		complex array[nt] of output DMO corrected data
******************************************************************************
Author: Craig Artley, Colorado School of Mines, 08/29/91
******************************************************************************/
{
	int ntfft,nw,it,iw,iwlm,iwhm,iwlp,iwhp,ip;
	float p,ph,pl,ph2,dw,fw,wl,wh,scale;
	complex czero=cmplx(0.0,0.0),*qn,*q0;

	/* determine frequency sampling */
	ntfft = npfa(nt);
	nw = ntfft;
	dw = 2.0*PI/(ntfft*dt);
	fw = -PI/dt;
	
	/* allocate workspace */
	qn = (complex *) malloc(ntfft * sizeof(complex));
	if (qn == 0) {
	  fprintf(stderr,"%s: unable to allocate %d bytes for qn\n",
		argv0,ntfft * sizeof(complex));
	  exit(100);
	  }

	q0 = (complex *) malloc(ntfft * sizeof(complex));
	if (q0 == 0) {
	  fprintf(stderr,"%s: unable to allocate %d bytes for q0\n",
		argv0,ntfft * sizeof(complex));
	  exit(100);
	  }

	/* zero w0 domain accumulator */
	for (iw=0; iw<nw; ++iw) {
		qn[iw] = czero;
		q0[iw] = czero;
	}

	/* loop over zero-offset ray parameter */
	for (ip=0,p=pl=0.0,ph2=dp2; ip<np; ++ip,ph2+=dp2) {

		/* update ray parameter */
		ph = sqrt(ph2);

		/* determine limits of w0 for this wavenumber and slope */
		wl = k/ph;
		if (pl==0.0) {
			wh = 2.0*PI*fmax;
		} else {
			wh = MIN(k/pl,2.0*PI*fmax);
		}

		/* compute limits of sum for -p */
		iwlm = (int) ((-wl-fw)/dw);
		iwhm = (int) ((-wh-fw)/dw);
		if (iwhm<0) iwhm = 0;
		++iwhm;

		/* compute limits of sum for +p */
		iwlp = (int) ((wl-fw)/dw);
		iwhp = (int) ((wh-fw)/dw);
		if (iwhp>=nw) iwhp = nw-1;
		++iwlp;

		/* if this range of frequencies contributes to the sum */
		if ((iwhm<=iwlm) || (iwlp<=iwhp)) {
		
			/* interpolate, padding with zeros */
			for (it=0; it<it0min[ip]; ++it)
				qn[it] = czero;
			ints8c(nt,dt,ft,qq,czero,czero,nt0[ip],tntable+(ip*nt),
				qn+it0min[ip]);
			for (it=it0min[ip]+nt0[ip]; it<ntfft; ++it)
				qn[it] = czero;

			/* Fourier transform t to w, with w centered */
			for (it=1; it<ntfft; it+=2) {
				qn[it].r = -qn[it].r;
				qn[it].i = -qn[it].i;
			}
			pfacc(1,ntfft,qn);

			/* accumulate into q0 for slopes near -p */
			for (iw=iwhm; iw<=iwlm; ++iw) {
				q0[iw].r += qn[iw].r;
				q0[iw].i += qn[iw].i;
			}

			/* accumulate into q0 for slopes near +p */
			for (iw=iwlp; iw<=iwhp; ++iw) {
				q0[iw].r += qn[iw].r;
				q0[iw].i += qn[iw].i;
			}
		}

		/* roll ray parameters */
		pl = p;
		p = ph;
	}

	/* Fourier transform w to t, with w centered */
	/* (including FFT scaling and division by 2 for plane filling) */
	pfacc(-1,ntfft,q0);
	scale = 0.5/ntfft;
	for (it=0; it<nt; it+=2) {
		qq[it].r = q0[it].r*scale;
		qq[it].i = q0[it].i*scale;
	}
	for (it=1; it<nt; it+=2) {
		qq[it].r = -q0[it].r*scale;
		qq[it].i = -q0[it].i*scale;
	}

	/* free workspace */
	free(qn);	
	free(q0);	
}
	
float tmute (float *vrms, int nt, float dt, float ft, float h, float smute)
/***************************************************************************
Returns MUTE Time for given rms velocity function and half-offset.
****************************************************************************
Input:
vrms	array[nt] of rms velocity as a function of NMO time
nt	number of times
dt	time sampling interval
ft	first time
h	source-receiver half-offset
smute	maximum allowable NMO stretch t/tn [(100+stretch_percentage)/100]

Return:
	gretest time that is subject to given NMO stretch mute
****************************************************************************
Author: Craig Artley, Colorado School of Mines, 09/28/91
****************************************************************************/
{
	int it;
	float tn,*t;

	/* allocate space */
	t = (float *) malloc(nt * sizeof(float));
	if (t == 0) {
	  fprintf(stderr,"%s: unable to allocate %d bytes for t\n",
		argv0,nt * sizeof(float));
	  exit(100);
	  }


	for (it=0,tn=ft; it<nt; ++it,tn+=dt)
		t[it] = sqrt(tn*tn+4.0*h*h/(vrms[it]*vrms[it]));

	/* find greatest nmo time that is subject to stretch mute */
	for (it=nt-1; it>0; --it)
		if (dt/(t[it]-t[it-1])>smute) break;

	/* free space */
	free(t);

	return ft+it*dt;
}

void getX (float tsg, float h, float p0, float v, float *vars)
/***************************************************************************
GET trial solution X to system of equations assuming constant velocity.
****************************************************************************
Input:
tsg	recording time (from source to receiver)
h	source-receiver half-offset
p0	ray parameter of zero-offset ray
v	constant velocity

Output:
vars	array[5] of variables that satisfies the system for const. v
****************************************************************************
Author: Craig Artley, Colorado School of Mines, 09/21/91
****************************************************************************/
{
	float x,z,a2,b2,h2,x0,ps,pg,tg,t0,sine,tangent;

	/* compute constants that determine prestack migration ellipse */
	h2 = h*h;
	a2 = v*v*tsg*tsg/4.0;
	b2 = MAX(a2-h2,0.0);

	/* compute sine and tangent of propagation angle */
	sine = 0.5*p0*v;
	tangent = sine/sqrt(MAX(1.0-sine*sine,0.0));
	
	/* find reflection point (x,z) on prestack migration ellipse */
	x = tangent*sqrt(a2/(b2+tangent*tangent));
	z = sqrt(b2*MAX(1.0-x*x/a2,0.0));
	
	/* find zero-offset location */
	x0 = h2/a2*x;

	/* find times along receiver and zero-offset rays */
	tg = sqrt((h-x)*(h-x)+z*z)/v;
	t0 = sqrt((x0-x)*(x0-x)+z*z)*2.0/v;

	/* find ray parameters for source and receiver rays */
	ps = (x+h)/(v*v*(tsg-tg));
	pg = (x-h)/(v*v*tg);

	/* load trial solution into the variables array */
	vars[0] = x0;
	vars[1] = ps;
	vars[2] = pg;	
	vars[3] = tg;
	vars[4] = t0;
}

void getFJ (float *x, float *a, float *tau, int nt, int np,
	float dt, float dp, float tsg, float h, float p0,
	float *vars, float *eqns, float *jacob)
/***************************************************************************
Compute equation array and Jacobian matrix for system of equations for
a given set of parameters.
****************************************************************************
Input:
x	array[np][nt] of horizontal location of tabled rays
a	array[np][nt] of propagation angle of tabled rays
tau	array[np][nt] of travel-time depth of tabled rays
nt	number of times in ray tables
np	number of ray parameters in ray tables
dt	time sampling interval in tables
dp	ray parameter sampling interval in tables
tsg	recording time (from source to receiver)
h	source-receiver half offset
p0	zero-offset ray parameter
vars	array[5] of variables of system of equations

Output:
eqns	array[5] of right-hand-sides of equations
jacob	pointer to array[5][5] of Jacobian partial derivatives
****************************************************************************
Author: Craig Artley, Colorado School of Mines, 09/21/91
****************************************************************************/
{
	int i,j,neqns=5;
	float x0,ps,pg,ts,tg,t0,ft=0.0,fp=0.0,ftp,dfdt,dfdp,sign;

	/* extract parameters from variables array */
	x0 = vars[0];
	ps = vars[1];
	pg = vars[2];
	tg = vars[3];
	t0 = vars[4];
	ts = MAX(tsg-tg,0.0);

	/* zero out F and J */
	for (i=0; i<neqns; ++i)
		eqns[i] = 0.0;
	for (j=0; j<neqns; ++j)
		for (i=0; i<neqns; ++i)
			jacob[(j*neqns)+i] = 0.0;

	/* insert constant terms into equations */
	eqns[0] += 2.0*h;
	eqns[1] += h-x0;
	jacob[1] -= 1.0;

	/* get x(ps,2*ts) and partial derivatives, load into equations */
	intder(x,nt,dt,ft,np,dp,fp,2.0*ts,ABS(ps),&ftp,&dfdt,&dfdp);
	sign = SGN(ps);
	eqns[0] -= sign*ftp;
	jacob[neqns] -= dfdp;
	jacob[3*neqns] += 2.0*sign*dfdt;

	/* get x(pg,2*tg) and partial derivatives, load into equations  */
	intder(x,nt,dt,ft,np,dp,fp,2.0*tg,ABS(pg),&ftp,&dfdt,&dfdp);
	sign = SGN(pg);
	eqns[0] += sign*ftp;
	eqns[1] += sign*ftp;
	jacob[2*neqns] += dfdp;
	jacob[(2*neqns)+1] += dfdp;
	jacob[3*neqns] += 2.0*sign*dfdt;
	jacob[(3*neqns)+1] += 2.0*sign*dfdt;

	/* get x(p0,t0) and partial derivatives, load into equations */
	intder(x,nt,dt,ft,np,dp,fp,t0,ABS(p0),&ftp,&dfdt,&dfdp);
	sign = SGN(p0);
	eqns[1] -= sign*ftp;
	jacob[(4*neqns)+1] -= sign*dfdt;

	/* get tau(ps,2*ts) and partial derivatives, load into equations */
	intder(tau,nt,dt,ft,np,dp,fp,2.0*ts,ABS(ps),&ftp,&dfdt,&dfdp);
	sign = SGN(ps);
	eqns[2] -= ftp;
	jacob[(1*neqns)+2] -= sign*dfdp;
	jacob[(3*neqns)+2] += 2.0*dfdt;

	/* get tau(pg,2*tg) and partial derivatives, load into equations */
	intder(tau,nt,dt,ft,np,dp,fp,2.0*tg,ABS(pg),&ftp,&dfdt,&dfdp);
	sign = SGN(pg);
	eqns[2] += ftp;
	eqns[3] += ftp;
	jacob[(2*neqns)+2] += sign*dfdp;
	jacob[(2*neqns)+3] += sign*dfdp;
	jacob[(3*neqns)+2] += 2.0*dfdt;
	jacob[(3*neqns)+3] += 2.0*dfdt;

	/* get tau(p0,t0) and partial derivatives, load into equations */
	intder(tau,nt,dt,ft,np,dp,fp,t0,ABS(p0),&ftp,&dfdt,&dfdp);
	sign = SGN(p0);
	eqns[3] -= ftp;
	jacob[(4*neqns)+3] -= dfdt;

	/* get a(ps,2*ts) and partial derivatives, load into equations */
	intder(a,nt,dt,ft,np,dp,fp,2.0*ts,ABS(ps),&ftp,&dfdt,&dfdp);
	sign = SGN(ps);
	eqns[4] += sign*ftp;
	jacob[(1*neqns)+4] += dfdp;
	jacob[(3*neqns)+4] -= 2.0*sign*dfdt;

	/* get a(pg,2*tg) and partial derivatives, load into equations */
	intder(a,nt,dt,ft,np,dp,fp,2.0*tg,ABS(pg),&ftp,&dfdt,&dfdp);
	sign = SGN(pg);
	eqns[4] += sign*ftp;
	jacob[(2*neqns)+4] += dfdp;
	jacob[(3*neqns)+4] += 2.0*sign*dfdt;

	/* get a(p0,t0) and partial derivatives, load into equations */
	intder(a,nt,dt,ft,np,dp,fp,t0,ABS(p0),&ftp,&dfdt,&dfdp);
	sign = SGN(p0);
	eqns[4] -= 2.0*sign*ftp;
	jacob[(4*neqns)+4] -= 2.0*sign*dfdt;
}

void intder (float *f, int nx, float dx, float fx,
	int ny, float dy, float fy, float x, float y,
	float *fxy, float *dfdx, float *dfdy)
/***************************************************************************
Finds f(x,y), df/dx, df/dy given function values on a mesh f[iy][ix],
via linear interpolation and finite differencing.
****************************************************************************
Input:
f	array[ny][nx] of sampled function values
nx	number of samples in x direction (fast dimension)
dx	x sampling interval
fx	first sample in x direction
ny	number of samples in y direction (slow dimension)
dy	y sampling interval
fy	first sample in y direction
x	x value at which to evaluate functions
y	y value at which to evaluate functions

Output:
fxy	interpolated function value f(x,y)
dfdx	partial derivative df/dx evaluated at (x,y)
dfdy	partial derivative df/dy evaluated at (x,y)
****************************************************************************
Note: Constant extrapolation of function and zero derivatives for points
	outside the table.  This is desirable, but somewhat awkward.
****************************************************************************
Author: Craig Artley, Colorado School of Mines, 08/31/91
****************************************************************************/
{
	int	ix0,ix1,iy0,iy1;
	float	x0i,y0i,fracx,fracy,lx,ly,f00,f10,f01,f11;

	/* compute x and y extent of table */
	lx = fx+(nx-1)*dx;
	ly = fy+(ny-1)*dy;

	if (x<=fx) {
		ix0 = ix1 = 0;
		fracx = 0.0;
	} else if (x>=lx) {
		ix0 = ix1 = nx-1;
		fracx = 1.0;
	} else {
		x0i = (x-fx)/dx;
		ix0 = x0i;
		ix1 = ix0+1;
		fracx = x0i-ix0;
	}

	if (y<=fy) {
		iy0 = iy1 = 0;
		fracy = 0;
	} else if (y>=ly) {
		iy0 = iy1 = ny-1;
		fracy = 1.0;
	} else {
		y0i = (y-fy)/dy;
		iy0 = y0i;
		iy1 = iy0+1;
		fracy = y0i-iy0;
	}

	/* get tabled function values nearest (x,y) */
	f00 = f[(iy0*nx)+ix0];	f01 = f[(iy1*nx)+ix0];
	f10 = f[(iy0*nx)+ix1];	f11 = f[(iy1*nx)+ix1];

	/* linear interpolation to find f(x,y) */
	*fxy = (1-fracx)*(1-fracy)*f00 + (1-fracx)*fracy*f01
		+ fracx*(1-fracy)*f10 + fracx*fracy*f11;

	/* interpolate and finite difference to find partial derivatives */
	*dfdx = ((1-fracy)*(f10-f00) + fracy*(f11-f01))/dx;
	*dfdy = ((1-fracx)*(f01-f00) + fracx*(f11-f10))/dy;
}

void rayvt (float p, int nt, float dt,
	float v[], float tau[], float x[], float a[]) 
/*****************************************************************************
trace ray for v = v(tau), tau=ONE-way vertical time
******************************************************************************
Input:
p		ray parameter
nt		number of times (including t=0.0)
dt		time sampling interval
v		array[nt] of velocities (v = v(tau))

Output:
tau		array[nt] of vertical times tau(t)
x		array[nt] of horizontal distances x(t)
a		array[nt] of propagation angles a(t)
******************************************************************************
Notes:
Ray tracing begins at tau=x=t=0.0.  If the ray turns and reaches tau=0.0
at some time index it, then tau[it+1:nt-1] = tau[it], x[it+1:nt-1] = x[it],
and a[it+1:nt-1] = a[it].  In other words, constant extrapolation is used
to fill the remaining values in the output arrays.
******************************************************************************
Author:  Dave Hale, Colorado School of Mines, 08/30/90
Modified:  Craig Artley, Colorado School of Mines, 09/25/90
	   Fixed bug in calculating propagation angles.
	   Added linear interpolation between velocity samples.
******************************************************************************/
{
	int it,jt,itau;
	float vt,dvt,at,taut,xt,cosat,taui,frac;

	/* initialize ray tracing parameters */
	vt = v[0];
	dvt = v[1]-v[0];
	at = asin(MIN(1.0,p*vt));
	taut = xt = 0.0;
	tau[0] = taut;
	x[0] = xt;
	a[0] = at;

	/* loop over times t */
	for (it=1; it<nt; ++it) {

		/* compute cosine of propagation angle */
		cosat = cos(at);

		/* update vertical time */
		tau[it] = tau[it-1]+dt*cosat;

		/* if ray has returned to surface, break */
		if (tau[it]<=0.0) break;

		/* update horizontal distance */
		x[it] = x[it-1]+dt*p*vt*vt;

		/* update angle */
		at += p*dvt;
		a[it] = at;

		/* update velocity and derivative */
		if (it<nt-1) {
			taui = tau[it]/dt;
			itau = taui;
			frac = taui-itau;
			vt = (1.0-frac)*v[itau] + frac*v[itau+1];
			dvt = v[itau+1]-v[itau];
		}
	}

	/* if ray returned to surface, extrapolate with surface values */
	for (jt=it; jt<nt; ++jt) {
		tau[jt] = tau[jt-1];
		x[jt] = x[jt-1];
		a[jt] = a[jt-1];
	}
}
void help_sub()
{

fprintf(stderr,"\n DMOVZ - DMO for V(Z) media for common-offset gathers\n\n");
fprintf(stderr," Required Parameters:\n");
fprintf(stderr," -cdpmin   (no default) ");
fprintf(stderr,"minimum cdp (integer number) for which to apply DMO\n");
fprintf(stderr," -cdpmax   (no default) ");
fprintf(stderr,"maximum cdp (integer number) for which to apply DMO\n");
fprintf(stderr," -dxcdp    (no default) ");
fprintf(stderr,"distance between adjacent cdp bins (m)\n");
fprintf(stderr," -noffmix  (no default) ");
fprintf(stderr,"number of offsets to mix (see notes)\n\n");
fprintf(stderr," Optional Parameters:\n");
fprintf(stderr," -vfile    (no default)     binary");
fprintf(stderr,"(non-ascii) file containing interval \n");
fprintf(stderr,"                            velocities (m/s)\n");
fprintf(stderr," -tdmo     (default=0.)     ");
fprintf(stderr,"times corresponding to interval velocities  \n");
fprintf(stderr,"                            in vdmo (s)\n");
fprintf(stderr," -vdmo     (default=1500.)  ");
fprintf(stderr,"interval velocities corresponding to times  \n");
fprintf(stderr,"                            in tdmo (m/s)\n");
fprintf(stderr," -fmax     (default=0.5/dt) ");
fprintf(stderr,"maximum frequency in input traces (Hz)\n");
fprintf(stderr," -smute    (default=1.5)    ");
fprintf(stderr,"stretch mute used for NMO correction\n");
fprintf(stderr," -speed    (default=1.0)    ");
fprintf(stderr,"twist this knob for speed (and aliasing)\n");
fprintf(stderr," -verbose  (default = off)  diagnostic print\n");
}
