C***********************************************************************
C                 copyright 2001, Amoco Production Company             *
C                             All Rights Reserved                      *
C                     an affiliate of BP America Inc.                  *
C***********************************************************************
c***********************************************************************
c    NAME: fxpgeom
c PURPOSE: read the geometry file produced by PREPMIG
c          write the shorter geometry file read by FXSMIG
c***********************************************************************
c
      subroutine fxpgeom( LER, LUGIN, LUGOUT, gitape, gotape, ntrc, nrec
     &        , maxsta, nlive, live, soptnm, isleft, isright, istalv
     &        , zsta, irec1, irec2, increc, ntrc2, ierr )
c
      implicit none
c
c input parameters:
c
      integer   ioversion       ! version no. of prepmig (geometry) file
      integer   LER             ! logical unit number, error file
      integer   LUGIN           ! logical unit number, prepmig geometry file
      integer   LUGOUT          ! logical unit number, output geometry file
      integer   mintopo         ! topography written from mintopo to maxtopo
      integer   maxtopo         ! topography written from mintopo to maxtopo
      integer   mtopo           ! subscript for reading past topo lines
      character gitape*(*)      ! name of geometry file from PREPMIG
      character gotape*(*)      ! name of output geometry file for FXSMIG
      integer   maxsta          ! maximum number of stations
      integer   nrec            ! number of records in input dataset
      integer   ntrc            ! number of traces per record in input
      integer   ntrc2           ! number of traces per record in output file
      integer   ntrc_tmp        ! number of traces per record in output file
      integer   increc          ! increment between input records
      integer   irec1           ! first input record to use
      integer   irec2           ! last  input record to use
      logical   vsp             ! indicates a vsp dataset  
c
c output parameters:
c
      integer   istalv(ntrc,nrec)! station index (of live stations)
      integer   ierr            ! error flag
      integer   isleft(nrec)    ! station number, first live trace
      integer   isright(nrec)   ! station number, last live trace
      logical   live(ntrc,nrec) ! station live or dead?
      integer   nlive(nrec)     ! number of live stations in shot
      integer   soptnm(nrec)    ! station number of shot
      real      zsta(maxsta)    ! depth at receiving stations
c
c local variables:
c
      real      dstation        ! distance between stations
      integer   is              ! shot (record) index
      integer   istst           ! shot (record) index
      integer   ista            ! station index
      integer   istachk         ! station index for checking
      integer   itrc            ! trace index
      integer   itrctst
      integer   krec            ! index of current output record
      integer   maxrecind       ! index of last station of all shots
      integer   minrecind       ! index of first station of all shots
      integer   nlivetst
      integer   nlive_chk
      integer   nshift          ! offset number between station numbers
                                ! and velocity model
      integer   nrec2           ! number of records in output dataset
      integer   nshot           ! number of shots (records) in geo file
      integer   nsta            ! total number of stations in data
      integer   ntrace          ! number of traces per shot in geo file
c
c scratch variables:
c
      character cscr*1          ! dummy variable used to skip line of input
      integer   iscr1
      real      rscr1, rscr2, rscr3, rscr4, rscr5, xr, yr, zr
c
c***********************************************************************
c
      open( unit=LUGIN, file=gitape, status='old', iostat=ierr )
      if( ierr .ne. 0 ) then
         write( LER, * ) ' ***** ERROR OPENING INPUT GEOMETRY FILE: '
         go to 800
      endif
c
c===  top parameter information from prepmig output
c
c===  next two lines contain headings and
c     nsamp, dtmsec, ntrace, ioversion
c
      read( LUGIN , '(a)' ) cscr
      read( LUGIN , *) iscr1, rscr1, ntrace, ioversion
c
      if(ioversion.eq.1)then
c===    next two lines contain headings and
c       nshot, minrecind, maxrecind, dstation, vsp
c
        read( LUGIN , '(a)' ) cscr
        read( LUGIN , * ) nshot, minrecind, maxrecind, dstation, vsp
c
c===    next two lines contain headings and
c       xdatamin, xdatamax, gdistmin, gdistmax, xsbackup
c
        read( LUGIN , '(a)' ) cscr
        read( LUGIN , '(a)' ) cscr
c
c===    next two lines contain headings and
c       topomin, topomax, xbegin, ybegin, xbeginprime, azim
c
        read( LUGIN ,  '(a)' ) cscr
        read( LUGIN ,  '(a)' ) cscr

      elseif(ioversion.eq.2)then

c===    next two lines contain headings and
c       nshot, minrecind, maxrecind, dstation, vsp
c
        read( LUGIN , '(a)' ) cscr
        read( LUGIN , * ) nshot, minrecind, maxrecind, dstation, vsp
c
c===    next two lines contain headings and
c       xdatamin, xdatamax, gdistmin, gdistmax, xsbackup
c
        read( LUGIN , '(a)' ) cscr
        read( LUGIN , '(a)' ) cscr
c
c===    next two lines contain headings and
c       topomin, topomax, xbegin, ybegin, xbeginprime, azim, theta
c
        read( LUGIN ,  '(a)' ) cscr
        read( LUGIN ,  '(a)' ) cscr

      elseif(ioversion.eq.3)then

c===    next two lines contain headings and
c       nshot, minrecind, maxrecind, mintopo, maxtopo, dstation, vsp
c
        read( LUGIN , '(a)' ) cscr
        read( LUGIN , * ) nshot, minrecind, maxrecind, mintopo,
     &                    maxtopo, dstation, vsp
c
c===    next two lines contain headings and
c       xdatamin, xdatamax, gdistmin, gdistmax, xsbackup
c
        read( LUGIN , '(a)' ) cscr
        read( LUGIN , '(a)' ) cscr
c
c===    next two lines contain headings and
c       topomin, topomax, xbegin, ybegin, xbeginprime, azim, theta
c
        read( LUGIN ,  '(a)' ) cscr
        read( LUGIN ,  '(a)' ) cscr

      else

        write(ler, *)' error--different version of prepmig geometry'
     &                 //' file detected'
        write(ler,*)' ioversion = ',ioversion
        ierr = 1
      endif
c
c===  get offset for station indices and total number of stations
c
      nshift = minrecind - 1
      nsta   = maxrecind - nshift
c
c***********************************************************************
c
c===  consistency checks with data file
c
      ierr = 0
c
      if( ntrace .ne. ntrc ) then
         write( LER, * ) ' ***** ERROR IN GEOMETRY FILE: '
     &                  , 'ntrace must equal ntrc *****'
     &                  , ' ntrace, ntrc = ', ntrace, ntrc
         ierr = 1
      endif
c
      if( nshot .ne. nrec ) then
         write( LER, * ) ' ***** ERROR IN GEOMETRY FILE: '
     &                  , 'nshot must equal nrec *****'
     &                  , ' nshot, nrec = ', nshot, nrec
         ierr = 1
      endif
c

      if( nsta .gt. maxsta ) then
         write( LER, * ) ' ***** ERROR IN GEOMETRY FILE: '
     &                  , 'nsta > maxsta *****'
     &                  , ' nsta, maxsta = ', nsta, maxsta
         ierr = 1
      endif

      if( ierr .ne. 0 ) go to 800
c
c***********************************************************************
c
c===  shots listing
c
c===  skip heading line
      read( LUGIN , '(a)' ) cscr
c
c===  get number of live stations and shot station index for ALL shots
c
      do is = 1, nshot
         read( LUGIN , * ) rscr1, rscr2, rscr3, rscr4, rscr5
     &           , nlive(is), soptnm(is)
      enddo

      if(ioversion.eq.3)then
         read(LUGIN,*)
         do mtopo=mintopo,maxtopo
            read(LUGIN,*)iscr1,rscr2,rscr3,rscr4
         enddo
         read(LUGIN,*)
         do ista = minrecind, maxrecind
            read(LUGIN,*) istachk, rscr1, rscr2, zsta(ista-nshift)
     &                   ,rscr3,rscr4
            if( ista .ne . istachk) then
               write( LER, * )' ***** INPUT FILE DATA ERROR: '
     &              ,' ista, istachk = ', ista, istachk
               ierr = 1
               go to 800
            endif
         enddo
      else
c
c===     get depth at receiving stations
c
c===     skip heading line
         read( LUGIN , '(a)' ) cscr
c
         do ista = minrecind, maxrecind
            read( LUGIN , * ) istachk, rscr1, rscr2, zsta(ista-nshift)
     &                         , rscr3
            if( ista .ne . istachk) then
               write( LER, * )' ***** INPUT FILE DATA ERROR: '
     &              ,' ista, istachk = ', ista, istachk
               ierr = 1
               go to 800
            endif
         enddo
      endif
c***********************************************************************
c***********************************************************************
c
c===  read geometry file produced by prepmig:
c          find station numbers of live stations for this shot
c
      ierr = 0
      ntrc2 = 0
c
c===  begin loop over shots
      do is = 1, irec2
c
c======  skip heading line
         read( LUGIN , '(a)' ) cscr
c
         read( LUGIN , * ) istst, nlive_chk
         if( istst .ne. is .or. nlive_chk .ne. nlive(is) ) then
            write( LER, * )' ***** INPUT FILE DATA ERROR: '
     &           ,' is, istst = ',  is, istst
     &           ,' nlive_chk, nlive = ', nlive_chk, nlive(is)
            ierr = 1
            go to 800
         endif
c
c======  read trace data
c
c======  skip heading line
         read( LUGIN , '(a)' ) cscr
c
         nlivetst = 0
         isleft(is) = 0
         isright(is) = 0
c
         do itrc = 1, ntrc
            read( LUGIN,*) itrctst, istalv(itrc,is), xr, yr, zr
     &                , live(itrc,is)
            if( live(itrc,is) ) then
               nlivetst = nlivetst + 1
               if( nlivetst .eq. 1 ) isleft(is) = istalv(itrc,is)
               if( nlivetst .eq. nlive(is) ) 
     &                               isright(is) = istalv(itrc,is)
            endif
c
         enddo
c
c=========  missing live stations?
c
         if( nlivetst .ne. nlive(is) ) then
            write( LER, * )' ***** INPUT FILE DATA ERROR: '
     &              ,' there appear to be missing stations'
     &              ,' nlivetst, nlive = ', nlivetst, nlive(is)
            ierr = 1
            go to 800
         endif
c
c=========  get largest number of traces in output file
c
         ntrc2 = max0( ntrc2, isright(is) - isleft(is)+1 )
c
      enddo
c===  end loop over shots
c
c***********************************************************************
c
c     write output file for fxsmig
c
      open( unit=LUGOUT, file=gotape, status='unknown', iostat=ierr)
      if( ierr .ne. 0 ) then
         write( LER, * ) ' ***** ERROR OPENING OUTPUT GEOMETRY FILE: '
         go to 800
      endif
c
      nrec2 = ( irec2 - irec1 ) / increc + 1
      ntrc_tmp = ntrc2
      ntrc_tmp = max0( ntrc_tmp, ntrc + 1 )
      if( mod( ntrc_tmp , 2 ) .eq. 0 ) ntrc_tmp = ntrc_tmp + 1
cdan  write( LUGOUT, *) ntrace+1, nrec2, nshift, nsta, dstation
      write( LUGOUT, *) ntrc_tmp, nrec2, nshift, nsta, dstation
c
      do is = 1, nsta
         write( LUGOUT , * ) is, zsta(is)
      enddo
c
c===  write only the shots to be put in the output file
      krec = 0
      do is = irec1, irec2, increc
         krec = krec + 1
         write( LUGOUT, * ) krec, nlive(is), soptnm(is), isleft(is)
     &   , isright(is)
      enddo
c
c***********************************************************************
800   continue
      close( LUGIN )
      close( LUGOUT )
      return
      end
c***********************************************************************
