C***********************************************************************
C                 copyright 2001, Amoco Production Company             *
C                             All Rights Reserved                      *
C                     an affiliate of BP America Inc.                  *
C***********************************************************************
c***********************************************************************
c    NAME: fxsmigr
c PURPOSE: performs an f(x), pre-stack, shot migration.
c
c***********************************************************************
c
      subroutine fxsmigr( LUP, nw, nxc, nxl, nxr, nz, nap, nawov, awov0
     &                 , dawov, nabs, base, omega, vshot, jz, jzmin
     &                 , jzmax, ctab, psi, coeffs, abndry, pinc, pref
     &                 , wrkinc, wrkref, image, strfac, nap_use, xlam )
c
      implicit none
c
c parameters:
c
      real pi                   ! pi
      integer napbig            ! largest possible nap (for by nkx=1024)
      parameter ( pi = 3.14159265358979, napbig = 512 )
c
c input parameters:
c
      integer   LUP             ! logical unit number, print file
      integer   nap             ! number of apertures
      integer   nawov           ! number of awov's
      integer   nw              ! number frequencies (omega) kept
      integer   nxc             ! max number of x's on each side of shot
      real      awov0           ! awov reference value
      real      base            ! controls sharpness of absorp bdry
      complex   ctab(0:nap-1,nawov)! fft coefficient table
      real      dawov           ! delta awov
      integer   jz(-nxc:nxc)    ! centered depth index
      integer   jzmax           ! maximum depth index for a shot
      integer   jzmin           ! minimum depth index for a shot
      integer   nabs            ! number of absorbing zones
      integer   nap_use         ! number of apertures to use
      integer   nxl             ! number of x's to left of shot
      integer   nxr             ! number of x's to right of shot
      integer   nz              ! number of z's (in migration)
      real      omega(nw)       ! angular frequency vector
      complex   psi(-nxc:nxc,nw) ! complex wave field (x,w)
      real      strfac          ! 2 pi * vmax / dx, used for migration
      real      vshot(-nxc:nxc,nz)! centered velocity model
      real      xlam            ! number of wavelengths, half aperture
c
c local variables:
c
      real      ab               ! used in calculating absorption
      real      alpha            ! interpolation parameter
      real      awov             ! omega/v for interpolation
      real      beta             ! interpolation parameter
      real      eps              ! parameter to prevent division by 0
      integer   ia               ! interpolation parameter
      integer   iap              ! aperture index
      integer   iap_stride       ! aperture stride
      integer   iap_stride_p     ! aperture stride, previous frequency
      integer   iw               ! frequency index
      integer   ix               ! trace, station index
      integer   iz               ! depth index
      integer   nap_top          ! highest aperture to use
      logical   newvel           ! new interpolation needed?
      real      pinc_i           ! imag part of incident field
      real      pinc_r           ! real part of incident field
      real      pref_i           ! imag part of reflected field
      real      pref_r           ! real part of reflected field
      real      rap_stride       ! real( iap_stride )
cmat  real      t0               ! scratch time variable
cmat  real      t1               ! scratch time variable
cmat  real      t2               ! scratch time variable
cmat  real      t3               ! scratch time variable
cmat  real      tcoef            ! CPU Time spend calculating coeffs.
cmat  real      tcont            ! CPU time spent on continuation
cmat  real      timag            ! CPU time spent imaging
      real      w                ! frequency
      real      wind(0:napbig-1) ! windowing function correction
c
c scratch space parameters:
c
      real      abndry(-nxc:nxc)       ! boundary condition vector
      complex   coeffs(-nxc:nxc,0:nap-1) ! fft interpolated coefficients
      complex   pinc(-nxc-nap+1:nxc+nap-1) ! incident field
      complex   pref(-nxc-nap+1:nxc+nap-1) ! reflected field
      complex   wrkinc(-nxc:nxc)       ! work space for incident field
      complex   wrkref(-nxc:nxc)       ! work space for reflected field
c
c output parameters:
c
      real      image(-nxc:nxc,0:nz) ! output image
c
c functions:
c
cmat  real      second          ! returns elapsed CPU time
c
c data initialization:
c
      data eps / 0.3 /
c***********************************************************************
c***********************************************************************
cmat  tcoef = 0.0
cmat  tcont = 0.0
cmat  timag = 0.0
c
c===  calculate absorbing boundary
c
      do ix = -nxl, nxr
         abndry(ix) = 1.0
      enddo
c
      do ix = 1, nabs
         ab = base ** (nabs-ix+1)
         abndry(-nxl-1+ix) = ab
         abndry( nxr+1-ix) = ab
      enddo
c
c===  clear the extensions on pinc & pref
c
      do ix = nxr+1, nxr+nap-1
         pref( ix) = ( 0.0, 0.0 )
         pinc( ix) = ( 0.0, 0.0 )
      enddo
      do ix =  nxl+1, nxl+nap-1
         pref(-ix) = ( 0.0, 0.0 )
         pinc(-ix) = ( 0.0, 0.0 )
      enddo
c
c===  clear image
c
      do iz = 0, nz
         do ix = -nxc, nxc
            image(ix,iz) = 0.0
         enddo
      enddo
c
c===  initialize window function in case turned off
c
      iap_stride_p = 0
      rap_stride = 1.0
c
      do iap = 0, nap - 1
         wind(iap) = 1.0
      enddo
c
c***********************************************************************
c
c===  begin loop over omega
      do iw = 1, nw
c
c        write( LUP , * ) 'iw = ', iw
         w = omega(iw)
c
c======  determine stride, maximum aperture to use
c        strfac = vmax * 2*pi / dx  === lambda_max = vmax / f ========
c
c======  loop until more than xlam wavelengths are covered in region
c        NOTE: if: (xlam = 0) or (nap_use = nap),
c            then: (iap_stride = 1) and (nap_top = nap_use)
c            but ( nap_top=nap ) only if nap_use = nap
c
         do ix = 1, ( nap-1 ) / ( nap_use-1 )
            iap_stride = ix
            nap_top = iap_stride * ( nap_use-1 ) + 1
            if( nap_top .ge. ( xlam * strfac / w ) ) go to 100
         enddo
100      continue
c
c======  make sure nap_top is not bigger than nap ( - remainder nap_use )
c        and nap_top is not less than nap_use

         nap_top = min0( nap_top , nap - mod(nap-1, nap_use-1) )
         nap_top = max0( nap_top, nap_use )
         rap_stride   = float(iap_stride)
c
         if( iw .eq. 1 )write(LUP,*)' iw, f, iap_stride, nap_top, pts: '
c
         if( iap_stride .ne. iap_stride_p ) then
c
            write( LUP, '(i5,f12.5,3i8)' ) iw, w/6.2832, iap_stride
     &                 , nap_top, (nap_top-1)/iap_stride + 1
c
c=========  construct windowing function correction,
c           needed because table may be constructed
c           with an nap that is bigger than nap_top
c           (Note nap_top decreases as iaw goes up)
c           absorb rap_stride only for convenience
c
            if( nap_top .ne. nap ) then
               do iap = 0, nap_top-1, iap_stride
                   wind(iap) = ( cos(  pi*iap/nap_top ) + 1.0 )
     &                       / ( cos(  pi*iap/nap     ) + 1.0 )
     &                       * rap_stride
               enddo
            endif
c
         endif
c
         iap_stride_p = iap_stride
c
c======  clear incident and reflected fields
c
         do ix = -nxl, nxr
            pinc(ix) = ( 0.0, 0.0 )
            pref(ix) = ( 0.0, 0.0 )
         enddo
c
c======  begin loop over z, starting at jzmin, the highest point
c        (lowest depth)  technically should be jzmin+1
c
         do iz = jzmin, nz
c
c=========  calculate coefficient matrix
c
cmat        t0 = second()
c
c=========  begin loop over x
            do ix = -nxl, nxr
c
c============  new velocity for this x ?
c
c============  first: check above
c
               if( iz .eq. jzmin ) then
                  newvel = .true.
               else
                  newvel = vshot(ix,iz) .ne. vshot(ix,iz-1)
               endif
c
c============  second: check to the left
c
               if( newvel .and. ix .gt. 1 ) then
                  if( vshot(ix,iz) .eq. vshot(ix-1,iz) ) then
                     newvel = .false.
                     do iap = 0, nap_top-1, iap_stride
                        coeffs(ix,iap) = coeffs(ix-1,iap)
                     enddo
                  endif
               endif
c
c============  find index and constants for coefficient interpolation
c
               if( newvel ) then
c
                  awov  = w / vshot(ix,iz)
                  ia    = int( ( awov - awov0 ) / dawov ) + 1
                  beta  = ( awov - ( awov0 + (ia-1) * dawov ) ) / dawov
                  alpha = 1.0 - beta
c
c===============  begin loop over apertures
                  do iap = 0, nap_top-1, iap_stride
                     coeffs(ix,iap) = alpha * ctab(iap,ia)
     &                               + beta * ctab(iap,ia+1)
c
c==================  apply windowing (note rap_stride in wind)
c
                     coeffs(ix,iap) = coeffs(ix,iap) * wind(iap)
                  enddo
               endif
            enddo
c=========  end loop over x
c
cmat        t1 = second()
c
c***********************************************************************
c=========  initialize pinc and pref if still above the lowest station
c           (note pinc=pinc(iz-1) here, to be continued to pinc(iz))
c           ( "   pref=pref(iz-1) "   "    "    "    "   " pref(iz))
c***********************************************************************
c
            if ( (iz-1) .le. jzmax ) then
c
c============ source at shot station at center?
c
               if( (iz-1) .eq. jz(0) ) then
                  pinc(0) = ( 1.0 , 0.0)
               endif
c
c============  begin receiver at this depth? (note pref(0) = psi(0,iw) = 0 )
c
               do ix = -nxl, nxr
                  if( (iz-1) .eq. jz(ix) ) then
                    pref(ix) = psi(ix,iw)
                  endif
               enddo
c
            endif
c***********************************************************************
c
c=========  downward continue:
c                do aperture 0 and then loop over remaining apertures;
c                apply absorbing boundry;
c                then store back into the wave fields
c
c=========  incident field:
c                no need to continue until depth greater than source
c
            if( iz .ge. jz(0) ) then
c
               do ix = -nxl, nxr
                  wrkinc(ix) = pinc(ix) * conjg( coeffs(ix,0) )
               enddo
c
               do iap = iap_stride, nap_top-1, iap_stride
                  do ix = -nxl, nxr
                     wrkinc(ix) = wrkinc(ix) + conjg( coeffs(ix,iap) ) *
     &                      ( pinc(ix-iap) + pinc(ix+iap) )
                  enddo
               enddo
c
               do ix = -nxl, nxr
                  pinc(ix) = wrkinc(ix) * abndry(ix)
               enddo
c
            endif
c
c=========  reflected field:    check for continuation if
c                               above lowest station
c
            if ( iz .le. jzmax ) then
c
c============  receiver higher than current depth?
c
               do ix = -nxl, nxr
                  if( (iz-1) .ge. jz(ix) ) then
                      wrkref(ix) = pref(ix) * coeffs(ix,0)
                  endif
               enddo
c
               do iap = iap_stride, nap_top-1, iap_stride
                  do ix = -nxl, nxr
                  if( (iz-1) .ge. jz(ix) ) then
                        wrkref(ix) = wrkref(ix) + coeffs(ix,iap) *
     &                         ( pref(ix-iap) + pref(ix+iap) )
                     endif
                  enddo
               enddo
c
               do ix = -nxl, nxr
                  if( jz(ix) .le. iz ) then
                      pref(ix) = wrkref(ix) * abndry(ix)
                  endif
               enddo
c
            else
c
c============  always continue if below lowest station
c
               do ix = -nxl, nxr
                   wrkref(ix) = pref(ix) * coeffs(ix,0)
               enddo
c
               do iap = iap_stride, nap_top-1, iap_stride
                  do ix = -nxl, nxr
                     wrkref(ix) = wrkref(ix) + coeffs(ix,iap) *
     &                         ( pref(ix-iap) + pref(ix+iap) )
                  enddo
               enddo
c
               do ix = -nxl, nxr
                   pref(ix) = wrkref(ix) * abndry(ix)
               enddo
c
            endif
c
cmat        t2 = second()
c
c
c=========  compute image at z = iz*dz
c
            do ix = -nxl, nxr
               pref_r = real ( pref(ix) )
               pref_i = aimag( pref(ix) )
               pinc_r = real ( pinc(ix) )
               pinc_i = aimag( pinc(ix) )
               image(ix,iz) = image(ix,iz) +
     &                        ( pref_r * pinc_r + pref_i * pinc_i ) /
     &                        ( pinc_r * pinc_r + pinc_i * pinc_i + eps)
            enddo
c
cmat        t3 = second()
cmat        tcoef = tcoef + ( t1 - t0 )
cmat        tcont = tcont + ( t2 - t1 )
cmat        timag = timag + ( t3 - t2 )
c
         enddo
c======  end loop over z
c
      enddo
c===  end loop over omega
c***********************************************************************
cmat  write( LUP , '('' tcoef = '', f8.3, ''    tcont = '', f8.3
cmat &               , ''    timag = '', f8.3,''   total = '',f8.3)' )
cmat &                tcoef, tcont, timag, tcoef+tcont+timag
c***********************************************************************
      return
      end
c***********************************************************************
