C***********************************************************************
C                                                                      *
c                 copyright 2001, Amoco Production Company             *
c                             All Rights Reserved                      *
c                     an affiliate of BP America Inc.                  *
C***********************************************************************
C***********************************************************************
C                                                                      *
C  ROUTINE:       RKRAYN                                               *
C  ROUTINE TYPE:  SUBROUTINE                                           *
C  PURPOSE:                                                            *
C       Performs the time domain loop of the Runge-Kutta raytracer.    *
C       Calculates a Runge-Kutta update on the incident angle, THETA,  *
C       and updates the wavefront position, (XRAY,ZRAY), with an       *
C       Euler recursion.                                               *
C  Authors: Dan Whitmore and Ron Coleman                               *
C                                                                      *
C  CALLING FORMAT:                                                     *
C       CALL  RKRAYN (NRMAX, NZMAX, NRAY, NTSKIP, JT,                  *
C      &              DXOVER, DZOVER, VDSX, VDSZ, VDT, CSTAB,          *
C      &              THETA, WCOS, WSIN, WORK, XRAY, ZRAY,NXMAX )      *
C                                                                      *
C  PARAMETERS:                                                         *
C       NRMAX   Integer input scalar                                   *
C               Maximum number of rays and leading dimension of        *
C               matrices XRAY and ZRAY.                                *
C                                                                      *
C       NZMAX   Integer input scalar                                   *
C               Maximum number of Z's and leading dimension of         *
C               matrices VDSX, VDSZ, and VDT.                          *
C                                                                      *
C       NRAY    Integer input scalar                                   *
C               Number of rays                                         *
C                                                                      *
C       NTSKIP  Integer input scalar                                   *
C               Increment of time index (column index of XRAY and ZRAY)*
C                                                                      *
C       JT      Integer input scalar                                   *
C               Time index (column index of XRAY and ZRAY).            *
C                                                                      *
C       DXOVER  Real input scalar                                      *
C               Reciprocal of delta x.                                 *
C                                                                      *
C       DZOVER  Real input scalar                                      *
C               Reciprocal of delta z.                                 *
C                                                                      *
C       VDSX    Real input matrix of dimension NZMAX by NXMAX          *
C               VDSX = v * v * partial(d(1/v)/dx) * dt,                *
C               where v = interval velocity model                      *
C                                                                      *
C       VDSZ    Real input matrix of dimension NZMAX by NXMAX          *
C               VDSZ = v * v * partial(d(1/v)/dz) * dt,                *
C               where v = interval velocity model                      *
C                                                                      *
C       VDT     Real input matrix of dimension NZMAX by NXMAX          *
C               VDT = v * dt, where v = interval velocity model        *
C                                                                      *
C       CSTAB   Real input vector of implied length                    *
C               Packed cos/sin table as build by subroutine GCFTLP.    *
C                                                                      *
C       THETA   Real input/output vector of length NRAY                *
C               Incident angle on input.  Updated incident angle on    *
C               output.                                                *
C                                                                      *
C       WCOS    Real input/output vector of length NRAY                *
C               WCOS = cos( THETA ) on both input and output.          *
C                                                                      *
C       WSIN    Real input/output vector of length NRAY                *
C               WSIN = sin( THETA ) on both input and output.          *
C                                                                      *
C       WORK    Real output vector of length NRAY                      *
C               Gathered elements of VDT.                              *
C                                                                      *
C       XRAY    Real input/output matrix of dimension NRMAX by NTMAX   *
C               Wavefront x position at each dt.  Column JT is updated *
C               using column JT-NTSKIP.                                *
C                                                                      *
C       ZRAY    Real input/output matrix of dimension NRMAX by NTMAX   *
C               Wavefront z position at each dt.  Column JT is updated *
C               using column JT-NTSKIP.                                *
C                                                                      *
C       NXMAX   Integer input scalar                                   *
C               Maximum number of X's and second dimension of          *
C               matrices VDSX, VDSZ, and VDT.                          *
C                                                                      *
      subroutine  rkrayn (nrmax, nzmax, nray, ntskip, jt,
     &                   dxover, dzover, vdsx, vdsz, vdt, cstab,
     &                   theta, wcos, wsin, work, xray, zray, nxmax)
c
      integer nrmax, nzmax, nray, ntskip, jt
      real    dxover, dzover, vdsx(*), vdsz(*), vdt(*),
     &        xray(*), zray(*), theta(*),
     &        wcos(*), wsin(*), work(*)
#ifdef CRAY
      integer mask
      real    cstab(*), tabi
c
      parameter (mask = x'FFFFFFFF00000000')
#else
      real*8  cstab(*), tabi
      real    tabi2(2)
      equivalence (tabi, tabi2)
#endif
c
c-----------------------------------------------------------------------
c
      scale  = cstab(2)
      offset = cstab(3)
c
      j1 = (jt - 1 ) * nrmax
      j2 = j1 - ntskip * nrmax
      do 200 j = 1, nray
         j1 = j1 + 1
         j2 = j2 + 1
c
c        find temp node indices
c
         jx1 = ifix( xray(j2)*dxover + 0.5 )
         jz1 = ifix( zray(j2)*dzover + 1.5 )
         if(jz1.ge.nzmax-1) jz1 = nzmax-1
         if(jx1.ge.nxmax-1) jx1 = nxmax-1
         if(jz1.lt.1) jz1 = 1
         if(jx1.lt.1) jx1 = 1
         jv1 = jx1 * nzmax + jz1
c
c        compute thetmp
c
         thetmp  = theta(j) + wcos(j)*vdsx(jv1) - wsin(j)*vdsz(jv1)
cdan     itab    = ifix( scale * thetmp + offset )
cdan     tabi    = cstab(itab)
cdan     to restore old code uncomment 2 lines above and 
cdan     and delete the two lines below 
#ifdef CRAY
         itab    = ifix( scale * thetmp + offset )
         tabi    = cstab(itab)
         tcos    = and( tabi, mask )
         tsin    = shiftl( tabi, 32 )
#else
cdan     tcos    = tabi2(1)
cdan     tsin    = tabi2(2)
cdan     to restore old code uncomment 2 lines above and 
cdan     and delete the two lines below 
         tcos    = cos(thetmp)
         tsin    = sin(thetmp)
#endif
c
c        find temp node indices
c
         jx2 = ifix( (xray(j2) + tsin*vdt(jv1)) * dxover + 0.5 )
         jz2 = ifix( (zray(j2) + tcos*vdt(jv1)) * dzover + 1.5 )
         if(jz2.ge.nzmax-1) jz2 = nzmax-1
         if(jx2.ge.nxmax-1) jx2 = nxmax-1
         if(jz2.lt.1) jz2 = 1
         if(jx2.lt.1) jx2 = 1
         jv2 = jx2 * nzmax + jz2
c
c        compute Runge-Kutta update of theta
c

         theta(j) = 0.5 * ( theta(j) + thetmp
     &                        + tcos*vdsx(jv2) - tsin*vdsz(jv2) )
cdan     itab     = ifix( scale * theta(j) + offset )
cdan     tabi     = cstab(itab)
cdan     to restore old code uncomment 2 lines above and 
cdan     and delete the two lines below 
#ifdef CRAY
         itab     = ifix( scale * theta(j) + offset )
         tabi     = cstab(itab)
         wcos(j)  = and( tabi, mask )
         wsin(j)  = shiftl( tabi, 32 )
#else
cdan     wcos(j)  = tabi2(1)
cdan     wsin(j)  = tabi2(2)
cdan     to restore old code uncomment 2 lines above and 
cdan     and delete the two lines below 
         wcos(j)  = cos(theta(j))
         wsin(j)  = sin(theta(j))
#endif
c
c        update wavefront with an Euler recursion
c
         xray(j1) = xray(j2) + wsin(j)*.5*(vdt(jv1)+vdt(jv2))
         zray(j1) = zray(j2) + wcos(j)*.5*(vdt(jv1)+vdt(jv2))
  200 continue
c
      return
      end
