C***********************************************************************
C                 copyright 2001, Amoco Production Company             *
C                             All Rights Reserved                      *
C                     an affiliate of BP America Inc.                  *
C***********************************************************************
c-----------------------------------------------------------------------
c ***** GRADFILT *****
c Program Changes:
c - original written: September 18. 1995
c - re-written for to appease the USP gang: December 3, 1995
c - modified to output filter mask January 4, 1996
c - modified to make mask an option February 23, 1996
c - GRADFILT adapted from PSIFILT3  January 17 , 1997 to
c   filter velocity models base on gradients

c Program Description:
c GRADFILT -- velocity gradient filtering of velocity fields
c Read a velocity dataset and filter velocities so that 1) no
c sample to sample velocity variations exceed the gradient range
c specified on the command line. The main program options are
c the min/max allowable gradients, and the option to output
c a filter mask, rather than the filtered velocity data.

c-----------------------------------------------------------------------
c get machine dependent parameters
 

#include <f77/iounit.h>
#include <f77/lhdrsz.h>
#include <f77/sisdef.h>
#include <save_defs.h> 

c-----------------------------------------------------------------------
c dimension standard USP variables 

      integer     itr (SZLNHD)
     
      integer     nsamp, nsi, ntrc, nrec, iform
      integer     luin , luout, lbytes, nbytes, lbyout, obytes
      integer     rs, re, ist, iend, ns, ne, argis

      character   ntap*255, otap*255, name*7

      logical     verbos, mask

c Program Specific _ dynamic memory variables

      integer TraceSize, HeaderSize
      real    Trace(2), Trace_Mask(2)
      integer Headers(2), errcd1, errcd2, errcd3
c     integer memadr_Space,memadr_Trace,memadr_Headers
      pointer (memadr_Space, Trace_Mask)
      pointer (memadr_Trace, Trace)
      pointer (memadr_Headers, Headers)

c Program Specific _ static memory variables

      integer     ifmt_tmnem1,l_tmnem1,ln_tmnem1, hval1
      integer     ifmt_tmnem2,l_tmnem2,ln_tmnem2, hval2

      character   tmnem1*6, tmnem2*6
      integer dz1000, units
      real    lth, hth, lmaxg, lming, gmaxg, gming, dz

c-----------------------------------------------------------------------
c Initialize variables

      data abort/1/
      data dz/0/
      data name/"GRADFILT"/
      data tmnem1/"LinInd"/
      data tmnem2/"DphInd"/

c-----------------------------------------------------------------------
c give command line help if requested

      if ( argis ( '-?' ) .gt. 0 .or. 
     1     argis ( '-h' ) .gt. 0 .or. 
     2     argis ( '-help' ) .gt. 0 ) then
         call help()
         stop
      endif

c open printout file

#include <f77/open.h>

c get command line input parameters

      call cmdln (ntap, otap, rs, re, ns, ne, ist, iend, lming, 
     1 lmaxg, gming, gmaxg, lth, hth, dz, name, mask, verbos)

c open input and output files

      call getln (luin , ntap,'r', 0)
      call getln (luout, otap,'w', 1)

c  read input line header and save certain parameters

      call rtape (luin,itr,lbytes)

      if (lbytes.eq.0) then
         write(LER,*)name,': no line header on input dataset ',ntap
         write(LER,*)'FATAL'
         stop
      endif

      call saver (itr, 'NumSmp', nsamp, LINHED)
      call saver (itr, 'SmpInt', nsi  , LINHED)
      call saver (itr, 'NumTrc', ntrc , LINHED)
      call saver (itr, 'NumRec', nrec , LINHED)
      call saver (itr, 'Format', iform, LINHED)
      call saver (itr, 'TmSlIn', dz1000, LINHED)
      call saver (itr, 'UnitFl', units, LINHED)

       call savelu(tmnem1,ifmt_tmnem1,l_tmnem1,
     :           ln_tmnem1,TRACEHEADER)
       call savelu(tmnem2,ifmt_tmnem2,l_tmnem2,
     :           ln_tmnem2,TRACEHEADER)

c historical line header and print to printout file 

      call hlhprt (itr, lbytes, name, 7, LERR)

c-----------------------------------------------------------------------
c check user supplied boundary conditions and set defaults

c for records ...
      if (rs .eq. 0) rs = 1
      if (re .eq. 0 .or. re .gt. nrec) re = nrec

c for traces ...
      if (ns .eq. 0) ns = 1
      if (ne .eq. 0 .or. ne .gt. ntrc) ne = ntrc

c for depth (samples) ...
      if (dz .eq. -1.0) dz = dz1000 / 1000
      if (dz .eq. 0) then
         write(LER,*)' DZ is zero, need to use -dz on command line '
         write(LER,*)' FATAL'
         stop
      endif
      ist = 1 + (nint ( float(ist) / dz ))
      iend = 1 + (nint ( float(iend) / dz ))
      if (ist .eq. 0) ist = 1
      if (iend .le. ist) iend = nsamp
      if (iend .gt. nsamp) iend = nsamp

      print*, " gming=",gming," gmaxg=",gmaxg," lming=",lming,
     1 " lmaxg=",lmaxg," lth=",lth," hth=",hth," dz=",dz,
     2 " rs=",rs," re=",re

c modify line header to reflect actual record configuration output

      call savew (itr, 'NumRec', nrec, LINHED)
      call savew (itr, 'NumTrc', ntrc  , LINHED)
      call savew (itr, 'NumSmp', nsamp  , LINHED)

c number output bytes

      obytes = SZTRHD + SZSMPD * nsamp 

c save out hlh and line header

      call savhlh  (itr, lbytes, lbyout)
      call wrtape (luout, itr, lbyout)

c verbose output of all pertinent information before processing begins

      call verbal (ntap, otap, nrec, ntrc, nsamp,
     1      ist, iend, rs, re, ns, ne, mask, dz, 
     2      gmaxg, gming, lmaxg, lming, hth, lth)

c-----------------------------------------------------------------------
c dynamic record memory allocation:

      TraceSize = nsamp
      HeaderSize = ITRWRD
 
      call galloc (memadr_Trace, TraceSize * SZSMPD, errcd1, abort)
      call galloc (memadr_Space, TraceSize * SZSMPD, errcd2, abort)
      call galloc (memadr_Headers, HeaderSize * SZSMPD, errcd3, abort)
   
      if ( errcd1 .ne. 0 .or.
     1     errcd2 .ne. 0 .or.
     2     errcd3 .ne. 0 )then
         write(LERR,*)' '
         write(LERR,*)'Unable to allocate workspace:'
         write(LERR,*) 2*TraceSize+HeaderSize* SZSMPD, '  bytes'
         write(LERR,*)' '
         write(LER,*)' '
         write(LER,*)'Unable to allocate workspace:'
         write(LER,*) 2*TraceSize+HeaderSize* SZSMPD, '  bytes'
         write(LER,*)' '
         go to 999
      else
         write(LERR,*)' '
         write(LERR,*)'Allocating workspace:'
         write(LERR,*) 2*TraceSize+HeaderSize* SZSMPD, '  bytes'
         write(LERR,*)' '
      endif
 
c initialize record memory
 
      call vclr ( Trace, 1, TraceSize )
      call vclr ( Headers, 1, HeaderSize )
 
c-----------------------------------------------------------------------
c-----------------------------------------------------------------------
c BEGIN PROCESSING 

c JJ loop on records

      DO JJ = 1, nrec 

c KK loop on traces

         DO KK = 1, ntrc

c read input trace header and trace data:

            nbytes = 0
            call rtape (luin, itr, nbytes)
            if(nbytes .eq. 0) then
               write(LERR,*)'Premature EOF on input at:'
               write(LERR,*)'  trace= ',KK
               go to 999
            endif

            call saver2(itr,ifmt_tmnem1,l_tmnem1,
     :                 ln_tmnem1,hval1,TRACEHEADER)
            call saver2(itr,ifmt_tmnem2,l_tmnem2,
     :                 ln_tmnem2,hval2,TRACEHEADER)
c      print*, tmnem1,"=",hval1," ",tmnem2,"=",hval2

c write out unprocessed starting traces

         if (JJ .lt. rs .or. JJ .gt. re 
     1 .and. KK .lt. ns .or. KK .gt. ne ) then

          call wrtape (luout, itr, obytes)
         else

c load trace header and trace data into arrays

         call vmov ( itr, 1, Headers, 1, ITRWRD )
         call vmov ( itr(ITHWP1), 1, Trace, 1, nsamp)

c now that the trace is in memory, process it:

         call velfilt (Trace, Headers, Trace_Mask,
     1        nsamp, ist, iend, dz, gmaxg, gming, 
     2        lmaxg, lming, lth, hth, mask)

c write out filtered velocity trace or mask:    
 
         if (mask) then
            call vmov (Trace_Mask,1,itr(ITHWP1),1,nsamp)
         else 
            call vmov (Trace,1,itr(ITHWP1),1,nsamp)
         endif

         call vmov ( Headers, 1, itr(1), 1, ITRWRD )
         call wrtape (luout, itr, obytes)

        endif

       ENDDO

      ENDDO

c close data files 

      call lbclos (luin)
      call lbclos (luout)
      write(LERR,*)name,': Normal Termination'
      write(LER,*)name,': Normal Termination'
      stop

 999  continue

      call lbclos (luin)
      call lbclos (luout)
      write(LERR,*)name,': Abnormal Termination'
      write(LER,*)name,': Abnormal Termination'
      stop
      end

c -----------------  Subroutine -----------------------

      subroutine help()

c provide terse online help [detailed help goes in man page]

#include <f77/iounit.h>

c...5....0....5....0....5....0....5....0....5....0....5....0....5....0..
      write(LER,*)'                                                    '
      write(LER,*)' GRADFILT applies a gradient filter to a velocity   '
      write(LER,*)' dataset                                            '
      write(LER,*)'                                                    '
      write(LER,*)' The filter rule is:                                ' 
      write(LER,*)' Velocity variation from sample to sample must fall '
      write(LER,*)' within a user specified gradient range. Filtering  '
      write(LER,*)' is not performed if the velocity is within the     '
      write(LER,*)' specified range.                                   '
      write(LER,*)' Globally, the same filter test is performed, but   '
      write(LER,*)' overall gradient from top (or mudline) to current  '
      write(LER,*)' depth is checked to see if it falls within the     '
      write(LER,*)' specified range.                                   '
      write(LER,*)' The global filter testing is referenced to the     '
      write(LER,*)' depth where velocity first exceeds the low velocity'
      write(LER,*)' threshold. Currently, if the velocity exceeds the  '
      write(LER,*)' high threshold, nothing special happens.           '
      write(LER,*)' You can write either the filtered velocity data, or'
      write(LER,*)' a filter mask (with the -mask option). The mask is '
      write(LER,*)' simply a dataset with a sample value :             '
      write(LER,*)'     0 if no filtering was performed                '
      write(LER,*)'     1 if local filtering was performed             '
      write(LER,*)'     2 if global filtering was performed            '
      write(LER,*)'     3 if local and global filtering were performed '
      write(LER,*)'                                                    '
      write(LER,*)' -N[]   -- input data set                    (stdin)'
      write(LER,*)' -O[]   -- output data set                  (stdout)'
      write(LER,*)' -rs[]  -- start record number                   (1)'
      write(LER,*)' -re[]  -- end record number           (last record)'
      write(LER,*)' -ns[]  -- start trace number                    (1)'
      write(LER,*)' -ne[]  -- end trace number             (last trace)'
      write(LER,*)' -s[]   -- processing start depth                (1)'
      write(LER,*)' -e[]   -- processing end depth        (last sample)'
      write(LER,*)'  *** all data is written out ***                   '
      write(LER,*)' -gmaxg[]-- maximum global gradient            (100)'
      write(LER,*)' -gming[]-- minimum global gradient              (0)'
      write(LER,*)' -lmaxg[]-- maximum local gradient             (100)'
      write(LER,*)' -lming[]-- minimum local gradient            (-100)'
      write(LER,*)' -lth[] -- low velocity threshold                (0)'
      write(LER,*)' -hth[] -- high velocity threshold           (99999)'
      write(LER,*)' -dz[]  -- depth sample interval       (Dz1000/1000)'
      write(LER,*)' -mask  -- output filter mask, not data             '
      write(LER,*)' -V     -- verbos printout                          '
      write(LER,*)'                                                    '
      write(LER,*)' Usage:                                             '
      write(LER,*)' gradfilt -N[] -O[] -rs[] -re[] -ns[] -ne[]         '
      write(LER,*)'   -s[] -e[] -gming[] -gmaxg[] -lming[] -lmaxg[]    '
      write(LER,*)'   -lth[] -hth[] -dz[] -mask -V                     '
      write(LER,*)'                                                    '
      write(LER,*)' ==================================================='
      write(LER,*)'                                                    '
      
      return
      end

c ------------------------  subroutine cmdln  --------------------------
c pick up command line arguments 

      subroutine cmdln (ntap, otap, rs, re, ns, ne, ist, iend, lming, 
     1 lmaxg, gming, gmaxg, lth, hth, dz, name, mask, verbos)

#include <f77/iounit.h>

      integer    rs, re, ist, iend, ns, ne, argis
      real       lmaxg, lming, gmaxg, gming, lth, hth, dz
      character  ntap*(*), otap*(*), name*(*)
      logical    verbos, mask

           call argr4 ( '-dz', dz, -1.0, -1.0 )
           call argi4 ( '-e', iend, 0, 0 )
           mask = (argis('-mask') .gt. 0 )
           call argr4 ( '-gmaxg', gmaxg, 100.0, 100.0 )
           call argr4 ( '-gming', gming, 0.0, 0.0 )
           call argr4 ( '-lmaxg', lmaxg, 100.0, 100.0 )
           call argr4 ( '-lming', lming, -100.0, -100.0 )
           call argr4 ( '-lth', lth, 0.0, 0.0 )
           call argr4 ( '-hth', hth, 99999.0, 99999.0 )
           call argi4 ( '-ne', ne, 0, 0 )
           call argi4 ( '-ns', ns, 1, 1 )
           call argstr ( '-N', ntap, ' ', ' ' ) 
           call argstr ( '-O', otap, ' ', ' ' ) 
           call argi4 ( '-re', re, 0, 0 )
           call argi4 ( '-rs', rs, 1, 1 )
           call argi4 ( '-s', ist, 1, 1 )
           verbos = (argis('-V') .gt. 0 )

c check for extraneous arguments and abort if found to
c catch all manner of user typo's

      call xtrarg ( name, ler, .FALSE., .FALSE. )
      call xtrarg ( name, lerr, .FALSE., .TRUE. )

      return
      end

c ------------------------  subroutine verbal --------------------------
c verbal printout of pertinent program particulars

      subroutine verbal (ntap, otap, nrec, ntrc, nsamp,
     1      ist, iend, rs, re, ns, ne, mask, dz, 
     2      gmaxg, gming, lmaxg, lming, hth, lth)

#include <f77/iounit.h>

      integer    nsamp, ntrc, nrec
      integer    rs, re, ns, ne, ist, iend
      real       dz, lmaxg, lming, gming, gmaxg, lth, hth
      character  ntap*(*), otap*(*)
      logical    mask

      write(LERR,*) ' '
      write(LERR,*) ' Input Line Header Parameters: '
      write(LERR,*) ' '
      write(LERR,*) ' input data set name   =  ', ntap
      write(LERR,*) ' samples per trace     =  ', nsamp
      write(LERR,*) ' traces per record     =  ', ntrc
      write(LERR,*) ' number of records     =  ', nrec
      write(LERR,*) ' sample interval (ft)  =  ', dz
      write(LERR,*) ' '
      write(LERR,*) ' Command Line Parameters: '
      write(LERR,*) ' '
      write(LERR,*) ' output data set name    =  ', otap
      write(LERR,*) ' processing start record =  ', rs
      write(LERR,*) ' processing end record   =  ', re
      write(LERR,*) ' processing start trace  =  ', ns
      write(LERR,*) ' processing end trace    =  ', ne
      write(LERR,*) ' processing sample start = ', ist
      write(LERR,*) ' processing sample end   = ', iend
      write(LERR,*) ' maximum global gradient = ', gmaxg
      write(LERR,*) ' minimum global gradient = ', gming
      write(LERR,*) ' maximum local gradient  = ', lmaxg
      write(LERR,*) ' minimum local gradient  = ', lming
      write(LERR,*) ' low velocity threshold  = ', lth
      write(LERR,*) ' high velocity threshold = ', hth
      if ( mask )  write(LERR,*) ' filter mask output requested'
      write(LERR,*) ' '
      write(LERR,*) '========================================== '
      write(LERR,*) ' '

      return
      end


c-----------------------------------------------------------------------
c -----------------  Filter Subroutine -----------------------
      subroutine  velfilt (Trace, Headers, Trace_Mask,
     1        nsamp, ist, iend, dz, gmaxg, gming, 
     2        lmaxg, lming, lth, hth, mask)

#include <save_defs.h>
#include <f77/lhdrsz.h>

c variables passed from calling routine

      integer nsamp, ist, iend
      integer Headers(ITRWRD)
      real Trace(nsamp), Trace_Mask(nsamp)
      real dz, gmaxg, gming, lmaxg, lming, hth, lth
      logical mask

c local variables

      real vel1, vel2, minvel, maxvel
      real ggrad, gvel1, depth, gdepth1, gdepth2
      logical gflag,lflag

c initialize variables
      
      gflag = .false.
      lflag = .false.

      do i = 1, nsamp
         Trace_Mask(i) = 0.0
      enddo

c process data

      hdr_index = 0
      DO i = ist, iend-1

c look at two adjacent samples

         vel1 = Trace(i)
         vel2 = Trace(i+1)
         depth = (i-1) *dz

c if vel2 exceeds the max velocity threshold, set to hth ...

         if (vel2 .gt. hth) then
            vel2 = hth
            trace(i+1)=vel2
         endif

c test the second velocity if it exceeds lth,
c and record the depth/velocity at which vel2 first exceeds lth

         if (.not. gflag .and. vel2 .gt. lth) then
            gflag = .true.
            gdepth1 = depth + dz
            gvel1 = vel1
         endif
            
c once the low vel threshold is exceeded, start the filter tests

         if (gflag .and. lflag) then

c first test local gradient and filter if necessary

           minvel = vel1 + lming*dz
           maxvel = vel1 + lmaxg*dz

           if (vel2 .lt. minvel) then
              vel2 = minvel
              Trace(i+1) = vel2
              Trace_Mask(i+1) = 1.0
           endif

           if (vel2 .gt. maxvel) then
              vel2 = maxvel
              Trace(i+1) = vel2
              Trace_Mask(i+1) = 1.0
           endif

         endif

c now test global gradient

         if (gflag) then

           gdepth2 = i*dz
           minvel = gvel1 + ((gdepth2 - gdepth1) * gming)
           maxvel = gvel1 + ((gdepth2 - gdepth1) * gmaxg)
           ggrad = (vel2 - gvel1)/(gdepth2 - gdepth1) 

           if (ggrad .lt. gming) then
              Trace(i+1) = minvel
              Trace_Mask(i+1) = Trace_Mask(i+1) + 2
           endif

           if (ggrad .gt. gmaxg) then
              Trace(i+1) = maxvel
              Trace_Mask(i+1) = Trace_Mask(i+1) + 2
           endif

         endif

         lflag = .true.

      ENDDO

      return
      end
