C***********************************************************************
C                 copyright 2001, Amoco Production Company             *
C                             All Rights Reserved                      *
C                     an affiliate of BP America Inc.                  *
C***********************************************************************
C***********************************************************************
c Read a density field and compute the overburden
c      gradient (obg) at each depth point by summing densities.
c Output an overburden gradient field
c ppdenobg.F
c
c Mary Ann Thornton           v: 1.0         September 29, 1994
c
c************************************************************************
#include <save_defs.h>
#include <f77/iounit.h>
#include <f77/lhdrsz.h>
#include <f77/sisdef.h>
c-----
      parameter   (convm = 3.280833, convg = .433, ppgal = 0.052)
      integer     lhed (SZLNHD)
      integer * 2 itr  (SZLNHD)
      real        head (SZLNHD)
      real        obg  (SZLNHD), den(SZLNHD)

      integer     nsamp, nsi, ntrc, nrec, iform, obytes
      integer     luin, luout, lbytes, nbytes, lbyout
      integer     irs,ire,ns,ne

      character   ntap  * 100, otap * 100, name*8, version*4
      logical     verbos, hlp, query
      integer     argis
      equivalence (itr( 1), lhed (1), head(1))
      data lbytes/ 0 /, nbytes/ 0 /, name/'PPDENOBG'/, version/' 1.0'/ 
 
c-----
c     read program parameters from command line card image file
      query = ( argis ( '-?' ) .gt. 0 )
      hlp = ( argis ( '-h' ) .gt. 0 )
      if ( query .or. hlp )then
            call help()
            stop
      endif
c-----
c     open printout
#include <f77/mbsopen.h>
c-----
c     read command line arguments
      call gcmdln(ntap,otap,ns,ne,irs,ire,
     &            verbos)
c-----
c     open input and output files
      call getln(luin , ntap ,'r', 0)
      call getln(luout, otap ,'w', 1)
c-----
c     read line header from density file
      call rtape(luin, itr, lbytes)
      if(lbytes .eq. 0) then
         write(LOT,*)'ppdenobg: no line header read from unit ',luin
         write(LOT,*)' ( density field )'
         write(LOT,*)'JOB TERMINATED ABNORMALLY'
         stop
      endif
c     save values from density line header
      call saver(itr, 'NumSmp', nsamp, LINHED)
      call saver(itr, 'SmpInt', nsi  , LINHED)
      call saver(itr, 'NumTrc', ntrc , LINHED)
      call saver(itr, 'NumRec', nrec , LINHED)
      call saver(itr, 'Format', iform, LINHED)
      call saver(itr, 'Dx1000', idx,   LINHED)
      call saver(itr, 'Dz1000', idz,   LINHED)
      dx = idx/1000. 
      dz = idz/1000. 
      if(dz.le.0.0)then
         write(LOT,*)' Dz1000 in the line header cannot be zero'
         write(LOT,*)' for the pore pressure prediction programs'
         write(LOT,*)'JOB TERMINATED ABNORMALLY'
         stop
      endif
c------
c     print line header into printer listing
      call hlhprt(itr, lbytes, name, 8, LERR)
c-----
c     check validity of these arguments
      call cmdchk(ns,ne,irs,ire,ntrc,nrec)
c-----
c     modify line header to reflect actual number of traces output
      nrecc = ire - irs + 1
      call savew(itr, 'NumRec', nrecc, LINHED)
      jtr   = ne - ns + 1
      call savew(itr, 'NumTrc', jtr  , LINHED)
c----------------------
c     determine obytes as the number of bytes in output trace
      obytes = SZTRHD + nsamp * SZSMPD
c----------------------
c     save command line arguments in the historical part of line header
      call savhlh(itr,lbytes,lbyout)
c----------------------
c     write the output line header
      call wrtape(luout, itr, lbyout )
 
c-----
      if( verbos ) then
         call verbal(nsamp, nsi, ntrc, nrec, iform,
     &               ntap,otap,ns,ne,irs,ire,
     &               verbos)
      endif

c-----
c--------------------------------------------------
c--------------------------------------------------
c-----
c     BEGIN PROCESSING
c-----
c--------------------------------------------------
c--------------------------------------------------

c     convm=conversion factor for metric to english
c     convg=conversion factor for converting to grams/cubic_centimeter

c-----skip unwanted records
      call recskp(1,irs-1,luin,ntrc,itr)

c-----RECORD LOOP
      do 1000 jj = irs, ire
         call trcskp(jj,1,ns-1,luin ,ntrc,itr)
c--------TRACE LOOP
         do 1003  nn = ns, ne
c           read the traces into itr then move into den
            nbytes = 0
            call rtape( luin, itr, nbytes)
            if(nbytes .eq. 0) then
               write(LERR,*)'End of file on density input:',luin
               write(LERR,*)'  rec= ',jj,'  trace= ',nn
               go to 999
            endif
            call vmov(lhed(ITHWP1),1,den,1,nsamp)
c-----------SAMPLE LOOPS
c----       compute overburden by summing the densities
            obg(1) = den(1) * convg
            temp = den(1)
            do 1001 kk = 2,nsamp           
               temp = temp + den(kk)
               obg(kk) = (temp*convg) / float(kk)
 1001       continue
c----       
            call vmov (obg, 1, lhed(ITHWP1), 1, nsamp)
            call wrtape (luout, itr, obytes)
 1003    continue
c-----
c        skip to end of record
         call trcskp(jj,ne+1,ntrc,luin,ntrc,itr)
 1000 continue

c--------------------------------------------------
c--------------------------------------------------
c-----
c     END PROCESSING
c-----
c--------------------------------------------------
c--------------------------------------------------
 
  999 continue
 
c-----
c     close data files
      call lbclos(luin)
      call lbclos(luout)
c-----
      write(LERR,*)'end of ppdenobg, processed ',nrec,' record(s)',
     &               ' with ',ntrc, ' traces'
      end
 
C***********************************************************************
C***********************************************************************
C***********************************************************************
C***********************************************************************
      subroutine help
#include <f77/iounit.h>
         write(LER,*)
     :'***************************************************************'
        write(LER,*) ' '
        write(LER,*)'ppdenobg reads a density field and computes'
        write(LER,*)'the overburden gradient for each depth sample.'
#ifdef CRAYSYSTEM
        write(LER,*)'See manual pages by typing:   tman ppdenobg '
#else
        write(LER,*)'See manual pages by typing:   man ppdenobg ',
     &              'or:   mman ppdenobg'
#endif
        write(LER,*)'See pattern file by typing:   catpat ppdenobg'
        write(LER,*)'Build an executable script by typing: ',
     &              '     catpat ppdenobg > ppdenobg.job'
        write(LER,*)'Execute ppdenobg by typing ppdenobg and the ',
     &              'program parameters.'
        write(LER,*)'Note that each parameter is proceeded by -a ',
     &              'where "a" is a character(s) corresponding ',
     &              'to some parameter.'
        write(LER,*)'Users enter the following parameters, or use ',
     &              'the default values.'
        write(LER,*)' '
        write(LER,*)'-DEN[ntap]: input density field'    
        write(LER,*)'-OBG[otap]: output pressure field'  
        write(LER,*)'-ns[ns]   : start trace number       (default=1)'    
        write(LER,*)'-ne[ne]   : end trace number       (default=all)'
        write(LER,*)'-rs[irs]  : start record number      (default=1)'
        write(LER,*)'-re[ire]  : end record number      (default=all)'
        write(LER,*)'-V        : for verbose printout'
        write(LER,*)' '
        write(LER,*)'usage:  '
        write(LER,*)'ppdenobg -DEN[ntap] -OBG[otap] '
        write(LER,*)'        -ns[ns] -ne[ne] -rs[irs] -re[ire]'
        write(LER,*)'        [-V] '
        write(LER,*) ' '
        write(LER,*)
     :'***************************************************************'
      return
      end
 
C***********************************************************************
C***********************************************************************
C***********************************************************************
C***********************************************************************
      subroutine gcmdln(ntap,otap,ns,ne,irs,ire,
     &                    verbos)
c-----
c     get command arguments
c
c     ntap  - C*100    input file name velocity
c     otap  - C*100    output file name
c     ns    - int      starting trace index
c     ne    - int      ending trace index
c     irs   - int      starting record index
c     ire   - int      ending record index
c     verbos  L        verbose output or not
c-----
#include <f77/iounit.h>
      character   ntap*(*), otap*(*)
      integer     ns, ne, irs, ire
      logical     verbos
      integer     argis
 
c-------
c     last 2 arguments are values used when:
c     (1) if ONLY the key is present (no value attached to it)
c     (2) if NO key & no value are present
c-------
            call argstr('-DEN',  ntap,  ' ', ' ')
            call argstr('-OBG',  otap,  ' ', ' ')
            call argi4 ('-ns', ns ,   1  ,  1)
            call argi4 ('-ne', ne ,   0  ,  0)
            call argi4 ('-rs', irs ,  1  ,  1)
            call argi4 ('-re', ire ,  0  ,  0)
            verbos =   (argis('-V') .gt. 0)
 
c-------
      return
      end
 
C***********************************************************************
C***********************************************************************
C***********************************************************************
C***********************************************************************
      subroutine verbal(nsamp, nsi, ntrc, nrec, iform,
     &               ntap,otap,ns,ne,irs,ire,
     &               verbos)
c-----
c     verbose output of processing parameters
c
c     nsamp - I*4     number of samples in trace
c     nsi   - I*4     sample interval in ms
c     ntrc  - I*4     traces per record
c     nrec  - I*4     number of records per line
c     iform - I*4     format of data
c     ntap  - C*100   input file name velocity
c     otap  - C*100   output file name
c     ns    - Int     starting record
c     ne    - Int     ending record
c     irs   - Int     starting trace
c     ire   - Int     ending trace
c-----
#include <f77/iounit.h>
 
      integer     nsamp, nsi, ntrc, nrec,ns,ne,irs,ire
      character   ntap*100, otap*100
 
      write(LERR,*)' '
      write(LERR,*)' line header values after default check '
      write(LERR,*)' # of samples/trace          =  ', nsamp
      write(LERR,*)' sample interval             =  ', nsi
      write(LERR,*)' traces per record           =  ', ntrc
      write(LERR,*)' records per line            =  ', nrec
      write(LERR,*)' format of data              =  ', iform
      write(LERR,*)' input data set name density =  ', ntap
      write(LERR,*)' output data set name        =  ', otap
      write(LERR,*)' starting record             =  ', irs
      write(LERR,*)' ending record               =  ', ire
      write(LERR,*)' starting trace              =  ', ns
      write(LERR,*)' ending trace                =  ', ne
      write(LERR,*)' '
 
      return
      end
 
