C***********************************************************************
C                 copyright 2001, Amoco Production Company             *
C                             All Rights Reserved                      *
C                     an affiliate of BP America Inc.                  *
C***********************************************************************
C***********************************************************************
c Read a velocity field, read v1,z1 and v2,z2 points for creating a 
c      velocity trendline and output the velocity trendline as a USP
c      formatted dataset the same size as the velocity dataset.
c Output a velocity trendline in log space
c pptrn.F
c
c Mary Ann Thornton           v: 1.0         September 29, 1994
c      Initial release
c Mary Ann Thornton           v: 1.1           October 03, 1994
c      Modifications:
c      1. Allow up to 12 (6 pairs) v,z points for creating the trendline
c      2. Add a command line argument for water velocity
c      3. If the trendline velocity is less than the water velocity given,
c         set it equal to the water velocity.
c************************************************************************
#include <save_defs.h>
#include <f77/iounit.h>
#include <f77/lhdrsz.h>
#include <f77/sisdef.h>
c-----
      parameter   (convm = 3.280833, convg = .433, ppgal = 0.052)
      integer     lhed (SZLNHD), ipoints
      integer * 2 itr  (SZLNHD)
      real        head (SZLNHD)
      real        vn   (SZLNHD), v0 (SZLNHD)
      real        vectr(24)
      real        v1 ,z1 ,v2 ,z2 
      real        v3 ,z3 ,v4 ,z4 
      real        v5 ,z5 ,v6 ,z6 
      real        v7 ,z7 ,v8 ,z8 
      real        v9 ,z9 ,v10,z10 
      real        v11,z11,v12,z12 

      equivalence (v1 ,vectr(1 )),(z1 ,vectr(2 ))
      equivalence (v2 ,vectr(3 )),(z2 ,vectr(4 ))
      equivalence (v3 ,vectr(5 )),(z3 ,vectr(6 ))
      equivalence (v4 ,vectr(7 )),(z4 ,vectr(8 ))
      equivalence (v5 ,vectr(9 )),(z5 ,vectr(10))
      equivalence (v6 ,vectr(11)),(z6 ,vectr(12))
      equivalence (v7 ,vectr(13)),(z7 ,vectr(14))
      equivalence (v8 ,vectr(15)),(z8 ,vectr(16))
      equivalence (v9 ,vectr(17)),(z9 ,vectr(18))
      equivalence (v10,vectr(19)),(z10,vectr(20))
      equivalence (v11,vectr(21)),(z11,vectr(22))
      equivalence (v12,vectr(23)),(z12,vectr(24))

      integer     nsamp, nsi, ntrc, nrec, iform, obytes
      integer     luin, luout, lbytes, nbytes, lbyout

      character   ntap  * 100, otap * 100, name*5, version*4
      logical     verbos, hlp, query
      integer     argis
      equivalence (itr( 1), lhed (1), head(1))
      data lbytes/ 0 /, nbytes/ 0 /, name/'PPTRN'/, version/' 1.0'/ 
 
c-----
c     read program parameters from command line card image file
      query = ( argis ( '-?' ) .gt. 0 )
      hlp = ( argis ( '-h' ) .gt. 0 )
      if ( query .or. hlp )then
            call help()
            stop
      endif
c-----
c     open printout
#include <f77/mbsopen.h>
c-----
c     read command line arguments
      call gcmdln(ntap,otap,v1 ,z1 ,v2 ,z2,
     &                      v3 ,z3 ,v4 ,z4,
     &                      v5 ,z5 ,v6 ,z6,
     &                      v7 ,z7 ,v8 ,z8,
     &                      v9 ,z9 ,v10,z10,
     &                      v11,z11,v12,z12,
     &                      ipoints, watvel, verbos)
c-----
c     open input and output files
      call getln(luin , ntap ,'r', 0)
      call getln(luout, otap ,'w', 1)
c-----
      lbytes = 0
      call rtape(luin, itr, lbytes)
      if(lbytes .eq. 0) then
         write(LOT,*)'pptrn: no line header read from unit ',luin
         write(LOT,*)' ( velocity field )'
         write(LOT,*)'JOB TERMINATED ABNORMALLY'
         stop
      endif
c------
c     save values from velocity line header
      call saver(itr, 'NumSmp', nsamp, LINHED)
      call saver(itr, 'SmpInt', nsi  , LINHED)
      call saver(itr, 'NumTrc', ntrc , LINHED)
      call saver(itr, 'NumRec', nrec , LINHED)
      call saver(itr, 'Format', iform, LINHED)
      call saver(itr, 'Dx1000', idx,   LINHED)
      call saver(itr, 'Dz1000', idz,   LINHED)
      dx = idx/1000.
      dz = idz/1000.
      if(dz.le.0.0)then
         write(LOT,*)' Dz1000 in the line header cannot be zero.'
         write(LOT,*)'JOB TERMINATED ABNORMALLY'
         stop
      endif
c------
c     print line header into printer listing
      call hlhprt(itr, lbytes, name, 5, LERR)
c-----
c     modify line header to reflect actual number of traces output
c----------------------
c     determine obytes as the number of bytes in output trace
      obytes = SZTRHD + nsamp * SZSMPD
c----------------------
c     save command line arguments in the historical part of line header
      call savhlh(itr,lbytes,lbyout)
c----------------------
c     write the output line header
      call wrtape(luout, itr, lbyout )
 
c-----
      if( verbos ) then
         call verbal(nsamp, nsi, ntrc, nrec, iform,
     &               ntap,otap,
     &               v1 ,z1 ,v2 ,z2,
     &               v3 ,z3 ,v4 ,z4,
     &               v5 ,z5 ,v6 ,z6,
     &               v7 ,z7 ,v8 ,z8,
     &               v9 ,z9 ,v10,z10,
     &               v11,z11,v12,z12, watvel, ipoints, verbos)

      endif

c-----
c--------------------------------------------------
c--------------------------------------------------
c-----
c     BEGIN PROCESSING
c-----
c--------------------------------------------------
c--------------------------------------------------

c-----CREATE the velocity trendline
c     create the velocity_trend line in log space using the pairs of 
c     velocity,depth points from the command line
      ii = 1
      z = 0.0
      do 500 i = 1,ipoints-1
         call vtrend(vectr(ii),vectr(ii+1), vectr(ii+2), vectr(ii+3),
     &               z, dz, vn, i, ipoints, nsamp)
         ii = ii + 2
  500 continue

c-----RECORD LOOP
      do 1000 jj = 1,nrec
c--------TRACE LOOP
         do 1003  nn = 1,ntrc
            nbytes = 0
c           read a velocity trace into itr to get the trace header
c           then move the above calculated trendline into the data
c           portion of the trace and write the trace out so the 
c           velocity trendline dataset will be the same size as the velocity
c           dataset for use in subsequent pore pressure programs
            call rtape(luin, itr, nbytes)
            if(nbytes .eq. 0) then
               write(LERR,*)'End of file on velocity input:',luin
               write(LERR,*)'  rec= ',jj,'  trace= ',nn
               go to 999
            endif
            call vmov(lhed(ITHWP1),1,v0,1,nsamp)
            do 1002 kk=1,nsamp
               if(vn(kk).lt.watvel .or. v0(kk).lt.watvel)
     &            vn(kk) = watvel
 1002       continue
            call vmov (vn, 1, lhed(ITHWP1), 1, nsamp)
            call wrtape (luout, itr, obytes)
 1003    continue
c-----
 1000 continue

c--------------------------------------------------
c--------------------------------------------------
c-----
c     END PROCESSING
c-----
c--------------------------------------------------
c--------------------------------------------------
 
  999 continue
 
c-----
c     close data files
      call lbclos(luin)
      call lbclos(luout)
c-----
      write(LERR,*)'end of pptrn, processed ',nrec,' record(s)',
     &               ' with ',ntrc, ' traces'
      end
 
C***********************************************************************
C***********************************************************************
C***********************************************************************
C***********************************************************************
      subroutine help
#include <f77/iounit.h>
         write(LER,*)
     :'***************************************************************'
        write(LER,*) ' '
        write(LER,*)'pptrn reads a velocity field, up to 12 vel/dep '
        write(LER,*)'points for a velocity trendline and outputs a '
        write(LER,*)'velocity trendline in log space.'
#ifdef CRAYSYSTEM
        write(LER,*)'See manual pages by typing:   tman pptrn '
#else
        write(LER,*)'See manual pages by typing:   man pptrn ',
     &              'or:   mman pptrn'
#endif
        write(LER,*)'See pattern file by typing:   catpat pptrn'
        write(LER,*)'Build an executable script by typing: ',
     &              '     catpat pptrn > pptrn.job'
        write(LER,*)'Execute pptrn by typing pptrn and the ',
     &              'program parameters.'
        write(LER,*)'Note that each parameter is proceeded by -a ',
     &              'where "a" is a character(s) corresponding ',
     &              'to some parameter.'
        write(LER,*)'Users enter the following parameters, or use ',
     &              'the default values.'
        write(LER,*)' '
        write(LER,*)'-VEL[ntap]: input velocity field'    
        write(LER,*)'-VTR[otap]: output velocity trendline field'  
        write(LER,*)'-wv[zl]   : water velocity '
        write(LER,*)'-va[va]   : va/za and vb/zb are the '
        write(LER,*)'-za[za]   :    velocity/depth points used'
        write(LER,*)'-vb[vb]   :    to compute the velocity'
        write(LER,*)'-zb[zb]   :    trendline          (No default)'
        write(LER,*)' '
        write(LER,*)'-vc[vc]   : ' 
        write(LER,*)'-zc[zc]   : '
        write(LER,*)'-vd[vd]   : Up to 6 pairs of velocity/depth'
        write(LER,*)'-zd[vd]   :    points (12 points) may be entered.'
        write(LER,*)' '
        write(LER,*)'-ve[ve]   : '
        write(LER,*)'-ze[ze]   : '
        write(LER,*)'-vf[vf]   : ' 
        write(LER,*)'-zf[zf]   : Only two velocity/depth pairs are'
        write(LER,*)' '
        write(LER,*)'-vg[vg]   :    required.'
        write(LER,*)'-zg[zg]   : '
        write(LER,*)'-vh[vh]   : '
        write(LER,*)'-zh[zh]   : '
        write(LER,*)' '
        write(LER,*)'-vi[vi]   : '
        write(LER,*)'-zi[zi]   : '
        write(LER,*)'-vj[vj]   : '
        write(LER,*)'-zj[zj]   : '
        write(LER,*)' '
        write(LER,*)'-vk[vk]   : '
        write(LER,*)'-zk[zk]   : '
        write(LER,*)'-vl[vl]   : '
        write(LER,*)'-zl[zl]   : '
        write(LER,*)'-V        : for verbose printout'
        write(LER,*)' '
        write(LER,*)'usage:  '
        write(LER,*)'pptrn -VEL[ntap] -VTR[otap] -wv[watvel]'
        write(LER,*)'        -va[va] -za[za] -vb[vb] -zb[zb]'
        write(LER,*)'        -vc[vc] -zc[zc] -vd[vd] -zd[zd]'
        write(LER,*)'        -ve[ve] -ze[ze] -vf[vf] -zf[zf]'
        write(LER,*)'        -vg[vg] -zg[zg] -vh[vh] -zh[zh]'
        write(LER,*)'        -vi[vi] -zi[zi] -vj[vj] -zj[zj]'
        write(LER,*)'        -vk[vk] -zk[zk] -vl[vl] -zl[zl]'
        write(LER,*)'        [-V] '
        write(LER,*) ' '
        write(LER,*)
     :'***************************************************************'
      return
      end
 
C***********************************************************************
C***********************************************************************
C***********************************************************************
C***********************************************************************
      subroutine gcmdln(ntap,otap,v1,z1,v2 ,z2 ,v3 ,z3 ,v4 ,z4,
     &                            v5,z5,v6 ,z6 ,v7 ,z7 ,v8 ,z8,
     &                            v9,z9,v10,z10,v11,z11,v12,z12,
     &                  ipoints, watvel, verbos)
c-----
c     get command arguments
c
c     ntap  - C*100    input file name velocity
c     otap  - C*100    output file name
c     v1,z1 - real     1st point for computing vel. trendline
c     v2,z2 - real     2nd point for computing vel. trendline
c     verbos  L        verbose output or not
c-----
#include <f77/iounit.h>
      character ntap*(*), otap*(*)
      logical   verbos
      integer   argis, ipoints
      real      v1,z1,v2,z2,v3,z3,v4,z4,v5,z5,v6,z6,v7,z7,v8,z8,v9,z9
      real      v10,z10,v11,z11,v12,z12
      real      watvel
 
c-------
c     last 2 arguments are values used when:
c     (1) if ONLY the key is present (no value attached to it)
c     (2) if NO key & no value are present
c-------
            call argstr('-VEL',  ntap,  ' ', ' ')
            call argstr('-VTR',  otap,  ' ', ' ')
            call argr4 ('-va', v1 ,0.0  ,0.0)
            call argr4 ('-za', z1 ,0.0  ,0.0)
            call argr4 ('-vb', v2 ,0.0  ,0.0)
            call argr4 ('-zb', z2 ,0.0  ,0.0)
            call argr4 ('-vc', v3 ,0.0  ,0.0)
            call argr4 ('-zc', z3 ,0.0  ,0.0)
            call argr4 ('-vd', v4 ,0.0  ,0.0)
            call argr4 ('-zd', z4 ,0.0  ,0.0)
            call argr4 ('-ve', v5 ,0.0  ,0.0)
            call argr4 ('-ze', z5 ,0.0  ,0.0)
            call argr4 ('-vf', v6 ,0.0  ,0.0)
            call argr4 ('-zf', z6 ,0.0  ,0.0)
            call argr4 ('-vg', v7 ,0.0  ,0.0)
            call argr4 ('-zg', z7 ,0.0  ,0.0)
            call argr4 ('-vh', v8 ,0.0  ,0.0)
            call argr4 ('-zh', z8 ,0.0  ,0.0)
            call argr4 ('-vi', v9 ,0.0  ,0.0)
            call argr4 ('-zi', z9 ,0.0  ,0.0)
            call argr4 ('-vj', v10,0.0  ,0.0)
            call argr4 ('-zj', z10,0.0  ,0.0)
            call argr4 ('-vk', v11,0.0  ,0.0)
            call argr4 ('-zk', z11,0.0  ,0.0)
            call argr4 ('-vl', v12,0.0  ,0.0)
            call argr4 ('-zl', z12,0.0  ,0.0)
            call argr4 ('-wv', watvel,1480., 1480.)
            verbos =   (argis('-V') .gt. 0)
c-----
c     Make sure 4 valid points have been received
      if(v1.eq.0.0 .or. v2.eq.0.0 .or. z2.eq.0.0)then
        write(LOT,*)' You must enter va,za and vb,zb for building'
        write(LOT,*)' the velocity trendline; za and zb cannot both'
        write(LOT,*)' be equal to zero.'
        write(LOT,*)' JOB TERMINATED ABNORMALLY'
        stop
      endif
      ipoints = 2
c-----  
c     Count the points of points received
      if(v3 .gt.0.0 .and. v4 .gt.0.0)ipoints = 4
      if(v5 .gt.0.0 .and. v6 .gt.0.0)ipoints = 6
      if(v7 .gt.0.0 .and. v8 .gt.0.0)ipoints = 8
      if(v9 .gt.0.0 .and. v10.gt.0.0)ipoints = 10
      if(v11.gt.0.0 .and. v12.gt.0.0)ipoints = 12
c-----  
      return
      end
 
C***********************************************************************
C***********************************************************************
C***********************************************************************
C***********************************************************************
      subroutine verbal(nsamp, nsi, ntrc, nrec, iform,
     &               ntap,otap,
     &               v1,z1,v2,z2,v3,z3,v4,z4,v5,z5,v6,z6,
     &               v7,z7,v8,z8,v9,z9,v10,z10,v11,z11,v12,z12,
     &               watvel, ipoints, verbos)
c-----
c     verbose output of processing parameters
c
c     nsamp - I*4     number of samples in trace
c     nsi   - I*4     sample interval in ms
c     ntrc  - I*4     traces per record
c     nrec  - I*4     number of records per line
c     iform - I*4     format of data
c     ntap  - C*100   input file name velocity
c     otap  - C*100   output file name
c     watvel- real    water velocity  
c     v1    - real    1st point for making trendline
c     z1    - real    1st depth 
c     v2    - real    2nd point for making trendline
c     z2    - real    2nd depth
c     v3-v12- real    up to 6 pairs of vel/dep points
c     z3-z12- real
c-----
#include <f77/iounit.h>
 
      integer     nsamp, nsi, ntrc, nrec
      real        v1,z1,v2,z2,v3,z3,v4,z4,v5,z5,v6,z6,v7,z7,v8,z8,v9,z9
      real        v10,z10,v11,z11,v12,z12

      character   ntap*100, otap*100
 
      write(LERR,*)' '
      write(LERR,*)' line header values after default check '
      write(LERR,*)' # of samples/trace          =', nsamp
      write(LERR,*)' sample interval             =', nsi
      write(LERR,*)' traces per record           =', ntrc
      write(LERR,*)' records per line            =', nrec
      write(LERR,*)' format of data              =', iform
      write(LERR,*)' input data set name velocity=', ntap
      write(LERR,*)' output data set name        =', otap
      write(LERR,*)' water velocity              =', watvel
      write(LERR,*)' velocity/depth points       =', ipoints
      write(LERR,*)' Velocity trendline points   =', v1,z1,' ',v2,z2
      if(v3.ne.0.0 .and. v4.ne.0.0)then
      write(LERR,*)'                              ', v3,z3,' ',v4,z4
      endif
      if(v5.ne.0.0 .and. v6.ne.0.0)then
      write(LERR,*)'                              ', v5,z5,' ',v6,z6
      endif
      if(v7.ne.0.0 .and. v8.ne.0.0)then
      write(LERR,*)'                              ', v7,z7,' ',v8,z8
      endif
      if(v9.ne.0.0 .and. v10.ne.0.0)then
      write(LERR,*)'                              ', v9,z9,' ',v10,z10
      endif
      if(v11.ne.0.0 .and. v12.ne.0.0)then
      write(LERR,*)'                              ', v11,z11,' ',v12,z12
      endif
 
      return
      end
 
C***********************************************************************
C***********************************************************************
C***********************************************************************
C***********************************************************************
      subroutine vtrend(v1, z1, v2, z2, z, dz, vn, i, ipoints, nsamp)
#include <f77/iounit.h>
      integer nstrt
      real    z, dz, v1, z1, v2, z2, vn(*)
      real    a1, a2, temp

      nstrt = 0
      if(i.eq.1)then
         nstrt = nstrt + 1
      else
         nstrt = z1 / dz + 1
      endif

      nend = z2 / dz
      if(i.eq.ipoints-1)then
         nend = nsamp
      endif

      a1 = alog(v1)
      a2 = alog(v2)
      do 900 kk = nstrt,nend
         temp = ((z-z1)/(z2-z1)) * (a2 -a1) + a1
         vn(kk) = exp(temp)
         z = z + dz
  900 continue

cmat      write(LERR,*)'vtrend: v1,       z1,        v2,        z2,
cmat     &       nstrt,    nend'
cmat      write(LERR,*) v1, z1, v2, z2, nstrt, nend
cmat      write(LERR,100) (kk,vn(kk),kk=nstrt,nend)
cmat  100 format(i10, f12.5)

      return
      end
