C***********************************************************************
C                 copyright 2001, Amoco Production Company             *
C                             All Rights Reserved                      *
C                     an affiliate of BP America Inc.                  *
C***********************************************************************
C***********************************************************************
c Read a velocity field and output a density field using Gardner's equation
c velocities will be converted to feet/sec if they are in metric
c densities will be output in grams/cubic_centimeter
c watvel must be given in same units as the velocity field
c presgraph2den.F
c
c Mary Ann Thornton                 V: 1.0            September 27, 1994
c Mary Ann Thornton                 V: 1.1               August 15, 1995
c    Make a tolerance factor (1 meter/second) for the check on water
c    velocity.
c    Add 2 new parameters: salt velocity and salt density
c    If both are entered, check for salt velocity (within 1 meter/second)
c    and set the density to user selected salt density at that point.
c**********************************************************************c
#include <save_defs.h>
#include <f77/iounit.h>
#include <f77/lhdrsz.h>
#include <f77/sisdef.h>
c-----
c-----conv is conversion factor for meters to feet conversion
c-----ppgal is conversion factor for pounds/gallon from psi/ft
      real        conv, ppgal, watmin, watmax
      parameter   (conv = 3.280833, ppgal = 0.052)
      parameter   (watmin = 1000., watmax = 6000.)
      integer     lhed (SZLNHD)
      integer * 2 itr  (SZLNHD)
      real        head (SZLNHD)
      real        vel  (SZLNHD), den(SZLNHD)
      real        watvel, watden, scale, expo, saltvel, saltden
      real        phi0, obc, rhom, rhof
      real        k3,k,pe,phi,rho,accum

      integer     nsamp, nsi, ntrc, nrec, iform, obytes
      integer     luin , luout, lbytes, nbytes, lbyout
      integer     irs,ire,ns,ne

      character   ntap * 100, otap * 100, name*7, version*4
      logical     verbos, hlp, query, met, eng
      integer     argis 
      equivalence (itr( 1), lhed (1), head(1))
      data lbytes/ 0 /, nbytes/ 0 /, name/'VEL2DEN'/, version/' 1.1'/ 
 
      met = .false.
      eng = .false.
c-----
c     read program parameters from command line card image file
      query = ( argis ( '-?' ) .gt. 0 )
      hlp = ( argis ( '-h' ) .gt. 0 )
      if ( query .or. hlp )then
            call help()
            stop
      endif
c-----
c     open printout
#include <f77/mbsopen.h>
c-----
c     read command line arguments
      call gcmdln(ntap,otap,ns,ne,irs,ire,met,
     &            eng,watvel,watden,scale,expo,saltvel,saltden,
     &            phi0, obc, rhom, rhof, carbvelmin, verbos)

c-----
c     open input and output files
      call getln(luin , ntap,'r', 0)
      call getln(luout, otap,'w', 1)
c-----
c     read line header
      call rtape  ( luin, itr, lbytes)
      if(lbytes .eq. 0) then
         write(LOT,*)'VEL2DEN: no line header read from unit ',luin
         write(LOT,*)'JOB TERMINATED ABNORMALLY'
         stop
      endif
c------
c     save values from line header
      call saver(itr, 'NumSmp', nsamp, LINHED)
      call saver(itr, 'SmpInt', nsi  , LINHED)
      call saver(itr, 'NumTrc', ntrc , LINHED)
      call saver(itr, 'NumRec', nrec , LINHED)
      call saver(itr, 'Format', iform, LINHED)
      call saver(itr, 'Dx1000', idx,   LINHED)
      call saver(itr, 'Dz1000', idz,   LINHED)
      dx = idx/1000.
      dz = idz/1000.

      if(dz.le.0.0)then
        write(LOT,*)' Dz1000 in the line header cannot be zero'
        write(LOT,*)' when used for the pore pressure prediction'
        write(LOT,*)' programs.'
        write(LOT,*)' JOB TERMINATED ABNORMALLY'
      endif
c------
c     print line header into printer listing
      call hlhprt(itr, lbytes, name, 7, LERR)
c-----
c     check validity of these arguments
      call cmdchk(ns,ne,irs,ire,ntrc,nrec)
c-----
c     modify line header to reflect actual number of traces output
      nrecc = ire - irs + 1
      call savew(itr, 'NumRec', nrecc, LINHED)
      jtr   = ne - ns + 1
      call savew(itr, 'NumTrc', jtr  , LINHED)
c----------------------
c     determine obytes as the number of bytes in output trace
      obytes = SZTRHD + nsamp * SZSMPD
c----------------------
c     save command line arguments in the historical part of line header
      call savhlh(itr,lbytes,lbyout)
c----------------------
c     write the output line header
      call wrtape(luout, itr, lbyout )
 
c-----
      if( verbos ) then
         call verbal(nsamp, nsi, ntrc, nrec, iform,ntap,otap,
     &                  ns,ne,irs,ire,met,eng,watvel,watden,
     &                  scale,expo,
     &                  saltvel,saltden,phi0,obc,rhom,rhof,
     &                  carbvelmin,verbos)
      endif

c-----
c--------------------------------------------------
c--------------------------------------------------
c-----
c     BEGIN PROCESSING
c-----
c--------------------------------------------------
c--------------------------------------------------

c-----met=metric
c-----conv=conversion factor 
      if(met)then
         convert = conv
      else
         convert = 1.0
      endif
      carbvelmin    = carbvelmin*convert
      watvel = watvel*convert
      saltvel= saltvel*convert
      sv1    = saltvel-convert
      sv2    = saltvel+convert

c-----do additional parameter checks
      if(watvel*convert .gt. watmax)then
         write(LOT,*)'Warning: water velocity greater than ', watmax
      endif
      if(watvel*convert .lt. watmin)then
         write(LOT,*)'Warning: water velocity less than ', watmin
      endif
c-----skip unwanted records
      call recskp(1,irs-1,luin,ntrc,itr)

c-----RECORD LOOP
      do 1000 jj = irs, ire
         call trcskp(jj,1,ns-1,luin,ntrc,itr)
c--------TRACE LOOP
         do 1003  nn = ns, ne
c           read the traces into itr then move
            nbytes = 0
            call rtape( luin, itr, nbytes)
            if(nbytes .eq. 0) then
               write(LERR,*)'End of file on input:',luin
               write(LERR,*)'  rec= ',jj,'  trace= ',nn
               go to 999
            endif
            call vmov(lhed(ITHWP1),1,vel,1,nsamp)
c-----------SAMPLE LOOP
c----       if velocity is in metric units, convert to feet
c----       compute density for ea. sample using Gardner's equation


c----       Search down trace for first non-water velocity and branch
c----       out.  Stuff in water density as we go.
            watdep=0.
            kks=1
            do kk = 1, nsamp
               if(vel(kk).gt. (watvel+0.1) )goto 1050
               watdep=(kk-1)*dz
               kks = kk
               den(kk)=watden
            enddo
 1050       continue


c----       initialize integration of densities.
            accum=watdep*watden


c----       calculate densities via gardner and presgraph and use most dense
c----       unless salt is detected.
            do 1001 kk = kks,nsamp

               vel(kk) = convert*vel(kk)

               rho_gard = scale*vel(kk)**expo

               depth = (kk-1) * dz * convert
               k3 = .038 * (depth - watdep/4)**.33
               k = ( 1. + k3 + k3 ) / 3. 
               if (accum .eq. 0.) then
                  pe = rhof * .433 * depth
               else
                  if (depth .eq. 0.) then
                     pe = 0.
                  else
                     pe = (accum / depth - rhof) *.433 * depth
                  endif
               endif
               phi = phi0 * exp (-pe/obc)
               rho = rhof * phi + rhom * (1. - phi)

c---           here is where we take the most dense
c---           if we are in a carbonate
               if(vel(kk) .ge. carbvelmin) then
                  if(rho_gard .gt. rho) then
                     rho = rho_gard
                  endif
               endif

c---           here is where we take the salt density
               if(vel(kk).eq.saltvel) rho = saltden

               den(kk) = rho

               accum = accum + rho * dz

 1001       continue
c----       
            call vmov (den, 1, lhed(ITHWP1), 1, nsamp)
            call wrtape (luout, itr, obytes)
 1003    continue
c-----
c        skip to end of record
         call trcskp(jj,ne+1,ntrc,luin,ntrc,itr)
 1000 continue

c--------------------------------------------------
c--------------------------------------------------
c-----
c     END PROCESSING
c-----
c--------------------------------------------------
c--------------------------------------------------
 
  999 continue
 
c-----
c     close data files
      call lbclos(luin)
      call lbclos(luout)
c-----
      write(LERR,*)'end of presgraph2den, processed ',nrec,' record(s)',
     &               ' with ',ntrc, ' traces'
      end
 
C***********************************************************************
C***********************************************************************
C***********************************************************************
C***********************************************************************
      subroutine help
#include <f77/iounit.h>
         write(LER,*)
     :'***************************************************************'
        write(LER,*) ' '
        write(LER,*)'presgraph2den reads a velocity field and outputs ' 
        write(LER,*)'a density field using Gardner equation.'
#ifdef CRAYSYSTEM
        write(LER,*)'See manual pages by typing:   tman presgraph2den '
#else
        write(LER,*)'See manual pages by typing:   man presgraph2den ',
     &              'or:   mman presgraph2den'
#endif
        write(LER,*)'See pattern file by typing:   catpat presgraph2den'
        write(LER,*)'Build an executable script by typing: ',
     &              '     catpat presgraph2den > presgraph2den.job'
        write(LER,*)'Execute presgraph2den by typing presgraph2den ',
     &              'and the program parameters.'
        write(LER,*)'Note that each parameter is proceeded by -a ',
     &              'where "a" is a character(s) corresponding ',
     &              'to some parameter.'
        write(LER,*)'Users enter the following parameters, or use ',
     &              'the default values.'
        write(LER,*)' '
        write(LER,*)'-VEL[ntap]  : input velocity field'    
        write(LER,*)'-DEN[otap]  : output density field'  
        write(LER,*)'-ns[ns]     : start trace number       (default=1)'    
        write(LER,*)'-ne[ne]     : end trace number       (default=all)'
        write(LER,*)'-rs[irs]    : start record number      (default=1)'
        write(LER,*)'-re[ire]    : end record number      (default=all)'
        write(LER,*)'-wv[watvel] : velocity of water     (default=5000)'
        write(LER,*)'-wd[watden] : density of water     (default=1.026)'
        write(LER,*)'-a [scale ] : Gardner scale constant(default= .23)'
        write(LER,*)'-b [expo  ] : Gardner exponent      (default= .25)'
        write(LER,*)'-sv[saltvel]: velocity of salt    (default: 14850)'
        write(LER,*)'-sd[saltden]: density of salt      (default: 2.16)'
        write(LER,*)'-phi0[phi0] : seafloor porosity     (default: .41)'
        write(LER,*)'-obc[obc]   : overburden gradient  (default: 5200)'
        write(LER,*)'-rhom[rhom] : matrix density       (default (2.65)'
        write(LER,*)'-rhof[rhof] : fluid density        (default (1.02)'
        write(LER,*)'-carb[carbvelmin] : minimum velocity '
        write(LER,*)' to allow Gardner switch to occur (default (7000.)'
        write(LER,*)'-met        : velocity is in metric units'
        write(LER,*)'-eng        : velocity is in english units'
        write(LER,*)'-V  include on command line for verbose printout'
        write(LER,*)' '
        write(LER,*)'usage:  '
        write(LER,*)'presgraph2den -VEL[ntap] -DEN[otap] '
        write(LER,*)'        -ns[ns] -ne[ne] '
        write(LER,*)'        -rs[irs] -re[ire] -a[scale] -b[expo]'
        write(LER,*)'        -sv[saltvel -sd[saltden]'
        write(LER,*)'        -phi0[phi0] -obc[obc]'
        write(LER,*)'        -rhom[rhom] -rhof[rhof]'
        write(LER,*)'        -wv[watvel] -wd[watden] [-met -eng] [-V] '
        write(LER,*) ' '
        write(LER,*)
     :'***************************************************************'
      return
      end
 
C***********************************************************************
C***********************************************************************
C***********************************************************************
C***********************************************************************
      subroutine gcmdln(ntap,otap,ns,ne,irs,ire,met,eng,
     &                  watvel,watden,scale,expo,saltvel,saltden,
     &                  phi0,obc,rhom,rhof,carbvelmin,verbos)
c-----
c     get command arguments
c
c     ntap  - C*100    input file name
c     otap  - C*100    output file name
c     ns    - int      starting trace index
c     ne    - int      ending trace index
c     irs   - int      starting record index
c     ire   - int      ending record index
c     scale - real     Gardner equation scaling constant
c     expo  - real     Gardner equation exponent
c     watvel- real     velocity of water  
c     watden- real     density of sea-water (depth variable)
c     met   - L        flag indicating metric units
c     eng   - L        flag indicating english units
c     saltvel real     velocity of salt
c     saltden real     density of salt
c     phi0    real     porosity (fraction) at the sea bottom (or surface)
c     obc     real     overburden compaction (psi)
c     rhom    real     density of matrix
c     rhof    real     density of formation water (g/cc)
c     verbos  L        verbose output or not
c-----
#include <f77/iounit.h>
      character   ntap*(*), otap*(*)
      integer     ns, ne, irs, ire
      logical     verbos, met,eng
      integer     argis
      real        watvel,watden,scale,expo,saltvel,saltden
      real        phi0,obc,rhom,rhof
 
c-------
c     last 2 arguments are values used when:
c     (1) if ONLY the key is present (no value attached to it)
c     (2) if NO key & no value are present
c-------
            call argstr('-VEL',  ntap, ' ', ' ')
            call argstr('-DEN',  otap, ' ', ' ')
            call argi4 ('-ns' , ns ,   1  ,  1)
            call argi4 ('-ne' , ne ,   0  ,  0)
            call argi4 ('-rs' , irs ,  1  ,  1)
            call argi4 ('-re' , ire ,  0  ,  0)
            call argr4 ('-wv' , watvel ,5000.,5000.)
            call argr4 ('-wd' , watden ,1.026,1.026)
            call argr4 ('-a'  , scale  ,.23  ,.23  )
            call argr4 ('-b'  , expo   ,.25  ,.25  )
            call argr4 ('-sv' , saltvel ,14850.,14850.)
            call argr4 ('-sd' , saltden ,2.16,2.16)
            call argr4 ('-phi0' , phi0 ,.41,.41)
            call argr4 ('-obc' , obc , 5200.,5200.)
            call argr4 ('-rhom' , rhom ,2.65,2.65)
            call argr4 ('-rhof' , rhof ,1.02,1.02)
            call argr4 ('-carb' , carbvelmin ,7000,7000.)
            verbos =   (argis('-V') .gt. 0)
            met    =   (argis('-met') .gt. 0)
            eng    =   (argis('-eng') .gt. 0)
           

            if(watvel.lt.4500.)then
              if(eng)then
                 write(LOT,*) 'Warning: Units are english but '
                 write(LOT,*)           water velocity is less than 4500'
              endif
            endif

            if(.not.met .and. .not.eng)then
              if(watvel .lt. 2000.) then
                 write(LOT,*) 'Warning: Units not specified but '
                 write(LOT,*) '         water velocity less than 2000 '
                 write(LOT,*) '         therefore metric assumed'
                 met=.TRUE.
              else
                 write(LOT,*) 'Warning: Units not specified but '
                 write(LOT,*) '         water velocity less than 2000 '
                 write(LOT,*) '         therefore metric assumed'
                 eng=.TRUE.
              endif
            endif

c-------
      return
      end
 
C***********************************************************************
C***********************************************************************
C***********************************************************************
C***********************************************************************
      subroutine verbal(nsamp, nsi, ntrc, nrec, iform,ntap,otap,
     &                  ns,ne,irs,ire,met,eng,watvel,watden,
     &                  scale,expo,
     &                  saltvel,saltden,phi0,obc,rhom,rhof,
     &                  carbvelmin,verbos)
c-----
c     verbose output of processing parameters
c
c     nsamp - I*4     number of samples in trace
c     nsi   - I*4     sample interval in ms
c     ntrc  - I*4     traces per record
c     nrec  - I*4     number of records per line
c     iform - I*4     format of data
c     ntap  - C*100   input file name
c     otap  - C*100   output file name
c     ns    - Int     starting record
c     ne    - Int     ending record
c     irs   - Int     starting trace
c     ire   - Int     ending trace
c     scale - real    Gardner's scaling constant
c     expo  - real    Gardner's exponenet
c     watvel- real    velocity of water  
c     saltvel real    velocity of salt  
c     saltden real    density of salt  
c     phi0    real    porosity (fraction) at the sea bottom (or surface)
c     obc     real    overburden compaction (psi)
c     rhom    real    density of matrix
c     rhof    real    density of formation water (g/cc)
c     met   - L       flag indicating metric units
c     eng   - L       flag indicating english units
c-----
#include <f77/iounit.h>
 
      integer     nsamp, nsi, ntrc, nrec,ns,ne,irs,ire
      logical     eng,met
      character   ntap*100, otap*100
      real        watvel,scale,expo,saltvel,saltden
 
      write(LERR,*)' '
      write(LERR,*)' line header values after default check '
      write(LERR,*)' # of samples/trace          =  ', nsamp
      write(LERR,*)' sample interval             =  ', nsi
      write(LERR,*)' traces per record           =  ', ntrc
      write(LERR,*)' records per line            =  ', nrec
      write(LERR,*)' format of data              =  ', iform
      write(LERR,*)' input data set name         =  ', ntap
      write(LERR,*)' output data set name        =  ', otap
      write(LERR,*)' starting record             =  ', irs
      write(LERR,*)' ending record               =  ', ire
      write(LERR,*)' starting trace              =  ', ns
      write(LERR,*)' ending trace                =  ', ne
      write(LERR,*)' water velocity              =  ', watvel
      write(LERR,*)' sea water density           =  ', watden
      write(LERR,*)' salt velocity               =  ', saltvel
      write(LERR,*)' salt density                =  ', saltden
      write(LERR,*)' Gardner scaling constant    =  ', scale
      write(LERR,*)' Gardner exponent            =  ', expo
      write(LERR,*)' Porosity at sea floor       =  ', phi0
      write(LERR,*)' Matrix density              =  ', rhom
      write(LERR,*)' Fluid density               =  ', rhof
      write(LERR,*)' Carbonate velocity          =  ', carbvelmin

      if(met)then
         write(LERR,*)' Units are metric '
      endif
      if(eng)then
        write(LERR,*)' Units are English '
      endif

      write(LERR,*)' '
 
      return
      end
 
