C***********************************************************************
C                 copyright 2001, Amoco Production Company             *
C                             All Rights Reserved                      *
C                     an affiliate of BP America Inc.                  *
C***********************************************************************
c-----------------------------------------------------------------------
c ***** PSIFILT *****
c Program Changes:
c - original written: September 18. 1995
c - re-written for to appease the USP gang: December 3, 1995
c - modified to output filter mask January 4, 1996
c - modified to make mask an option February 23, 1996

c Program Description:
c PSIFILT -- hydrostatic filtering (ala Thomsen) of pressure fields
c Read a pressure dataset and filter pressures so that 1) no
c pressure regressions exist, and 2) pressure increases at the
c minimum gradient supplied on the command line.
c-----------------------------------------------------------------------
c-----------------------------------------------------------------------

c get machine dependent parameters
 

#include <f77/iounit.h>
#include <f77/lhdrsz.h>
#include <f77/sisdef.h>
#include <save_defs.h> 

c-----------------------------------------------------------------------
c-----------------------------------------------------------------------

c dimension standard USP variables 

      integer     itr (SZLNHD)
     
      integer     nsamp, nsi, ntrc, ntrco, nrec, nreco, iform
      integer     luin , luout, lbytes, nbytes, lbyout, obytes
      integer     ist, iend, ns, ne, argis

      character   ntap*255, otap*255, name*7

      logical     verbos, met, mask

c Program Specific _ dynamic memory variables

c for records:
      integer RecordSize, HeaderSize, errcd2, errcd3, errcd4
      integer Headers
      real    Record, Record_Mask
      pointer (memadr_Record, Record(2))
      pointer (memadr_Space, Record_Mask(2))
      pointer (memadr_Headers, Headers(2))

c Program Specific _ static memory variables

      integer dz1000, units
      real    convm, nhg, dz

c-----------------------------------------------------------------------
c-----------------------------------------------------------------------

c Initialize variables

      data abort/1/
      data dz/0/
      data convm/3.280833/
      data name/"PSIFILT"/

c give command line help if requested

      if ( argis ( '-?' ) .gt. 0 .or. 
     :     argis ( '-h' ) .gt. 0 .or. 
     :     argis ( '-help' ) .gt. 0 ) then
         call help()
         stop
      endif

c open printout file

#include <f77/open.h>

c get command line input parameters

      call cmdln (ntap, otap, ns, ne, ist, iend, 
     :     nhg, met, dz, name, mask, verbos)

c open input and output files

      call getln (luin , ntap,'r', 0)
      call getln (luout, otap,'w', 1)

c  read input line header and save certain parameters

      call rtape (luin,itr,lbytes)

      if (lbytes.eq.0) then
         write(LER,*)name,': no line header on input dataset ',ntap
         write(LER,*)'FATAL'
         stop
      endif

      call saver (itr, 'NumSmp', nsamp, LINHED)
      call saver (itr, 'SmpInt', nsi  , LINHED)
      call saver (itr, 'NumTrc', ntrc , LINHED)
      call saver (itr, 'NumRec', nrec , LINHED)
      call saver (itr, 'Format', iform, LINHED)
      call saver (itr, 'TmSlIn', dz1000, LINHED)
      call saver (itr, 'UnitFl', units, LINHED)

      if (nrec .gt. 1) then
         write(LER,*)name,': More than 1 record in ',ntap
         write(LER,*)'Swapping record and trace numbers...'
         ntrc=nrec
         nrec=1
      endif

c historical line header and print to printout file 

      call hlhprt (itr, lbytes, name, 7, LERR)

c check user supplied boundary conditions and set defaults

      if (ns .eq. 0) ns = 1
      if (ne .eq. 0 .or. ne .gt. ntrc) ne = ntrc

      if (dz .eq. -1.0) dz = dz1000 / 1000
      ist = nint ( float(ist) / dz )
      iend = nint ( float(iend) / dz )
      if (ist .eq. 0) ist = 1
      if (iend .eq. 0 .or. iend .gt. nsamp) iend = nsamp

      nreco = 1
      ntrco = ne - ns + 1

      if (met) dz = dz * convm

      write(LER,*)name,': Command Line Parameters... '
      write(LER,*)' ntap = ',ntap 
      write(LER,*)' otap = ',otap 
      write(LER,*)' ns = ',ns 
      write(LER,*)' ne = ',ne 
      write(LER,*)' ist = ',ist 
      write(LER,*)' iend = ',iend 
      write(LER,*)' nhg = ',nhg 
      write(LER,*)' dz = ',dz 
      if (met) write(LER,*)' metric depth units (m) '
      if (mask) write(LER,*)' filter mask requested '

c modify line header to reflect actual record configuration output
c NOTE: in this case the sample limits ist and iend are used to 
c       limit processing only.   All data samples are actually passed.

      call savew (itr, 'NumRec', nreco, LINHED)
      call savew (itr, 'NumTrc', ntrco  , LINHED)

c number output bytes

      obytes = SZTRHD + SZSMPD * nsamp 

c save out hlh and line header

      call savhlh  (itr, lbytes, lbyout)
      call wrtape (luout, itr, lbyout)

c verbose output of all pertinent information before processing begins

      call verbal (ntap, otap, nsamp, nsi, ntrc, nrec, iform, ist, 
     :     iend, ns, ne, verbos, met, mask, dz, nhg)

c-----------------------------------------------------------------------
c-----------------------------------------------------------------------
c dynamic record memory allocation:

      RecordSize = ntrco * nsamp
      HeaderSize = ntrco * ITRWRD
 
      call galloc (memadr_Record, RecordSize * SZSMPD, errcd2, abort)
      call galloc (memadr_Space, RecordSize * SZSMPD, errcd3, abort)
      call galloc (memadr_Headers, HeaderSize * SZSMPD, errcd4, abort)
   
      if ( errcd2 .ne. 0 .or.
     :     errcd3 .ne. 0 .or.
     :     errcd4 .ne. 0 )then
         write(LERR,*)' '
         write(LERR,*)'Unable to allocate workspace:'
         write(LERR,*) 2*RecordSize+HeaderSize* SZSMPD, '  bytes'
         write(LERR,*)' '
         write(LER,*)' '
         write(LER,*)'Unable to allocate workspace:'
         write(LER,*) 2*RecordSize+HeaderSize* SZSMPD, '  bytes'
         write(LER,*)' '
         go to 999
      else
         write(LERR,*)' '
         write(LERR,*)'Allocating workspace:'
         write(LERR,*) 2*RecordSize+HeaderSize* SZSMPD, '  bytes'
         write(LERR,*)' '
      endif
 
c initialize record memory
 
      call vclr ( Record, 1, RecordSize )
      call vclr ( Headers, 1, HeaderSize )
 
c-----------------------------------------------------------------------
c-----------------------------------------------------------------------
c BEGIN PROCESSING 
c
c load record to memory

      tr_index = 1 - nsamp
      hdr_index = 1 - ITRWRD
      call trcskp ( JJ, 1, ns-1, luin, ntrc, itr )
      write(LER,*)'Loading traces into memory...'

      DO KK = 1, ntrco

         nbytes = 0
         call rtape (luin, itr, nbytes)
         if(nbytes .eq. 0) then
            write(LERR,*)'Premature EOF on input at:'
            write(LERR,*)'  trace= ',KK
            go to 999
         endif

c set array load points for this trace

         tr_index = tr_index + nsamp
         hdr_index = hdr_index + ITRWRD

c load trace header and trace data into arrays

         call vmov ( itr(ITHWP1), 1, Record(tr_index), 1, nsamp )
         call vmov ( itr, 1, Headers(hdr_index), 1, ITRWRD )
 
      ENDDO
      write(LER,*)'Loaded ',KK,' traces into memory'

      call trcskp ( JJ, ne+1, ntrc, luin, ntrc, itr )

c now that the record is in memory, process it:

      write(LER,*)'Filtering ',ntrco,' traces ...'

      call hsfilt (Record, Headers, Record_Mask,
     :        nsamp, ntrco, ist, iend, dz, nhg)

      write(LER,*)'Done filtering.'

c reset array load points for writing trace data
 
      tr_index = 1 - nsamp
      hdr_index = 1 - ITRWRD

c write output data
 
      DO KK = 1, ntrco
 
         tr_index = tr_index + nsamp
         hdr_index = hdr_index + ITRWRD
 
         if (mask) then
            call vmov (Record_Mask(tr_index),1,itr(ITHWP1),1,nsamp)
         else 
            call vmov (Record(tr_index),1,itr(ITHWP1),1,nsamp)
         endif

         call vmov ( Headers(hdr_index), 1, itr(1), 1, ITRWRD )
         call wrtape (luout, itr, obytes)
 
      ENDDO

c close data files 

      call lbclos (luin)
      call lbclos (luout)
      write(LERR,*)name,': Normal Termination'
      write(LER,*)name,': Normal Termination'
      stop

 999  continue

      call lbclos (luin)
      call lbclos (luout)
      write(LERR,*)name,': Normal Termination'
      write(LER,*)name,': Normal Termination'
      stop
      end

c -----------------  Subroutine -----------------------

      subroutine help()

c provide terse online help [detailed help goes in man page]

#include <f77/iounit.h>

c...5....0....5....0....5....0....5....0....5....0....5....0....5....0..
      write(LER,*)'                                                    '
      write(LER,*)' ================================================== '
      write(LER,*)'                                                    '
      write(LER,*)' Command Line Arguments for program: psifilt        '
      write(LER,*)'                                                    '
      write(LER,*)' PSIFILT applies a hydrostatic filter to a pressure '
      write(LER,*)' dataset (in PSI units NOT gradient, PPG units)     '
      write(LER,*)'                                                    '
      write(LER,*)' The filter rule is: pressure must increase         '
      write(LER,*)' with depth  at a rate faster than the fluid        '
      write(LER,*)' gradient specified in the command line.            '
      write(LER,*)'                                                    '
      write(LER,*)' You can write either the filtered pressure data, or'
      write(LER,*)' a filter mask (with the -mask option). The mask is '
      write(LER,*)' simply a dataset with a sample value of 1 at depths'
      write(LER,*)' where the pressure data would have been filtered,  '
      write(LER,*)' and a sample value of 0 where no filtering was     '
      write(LER,*)' required.                                          '
      write(LER,*)'                                                    '
      write(LER,*)' NOTE: psifilt will process datasets as if they were'
      write(LER,*)' a single n-trace record; if the data is not in this'
      write(LER,*)' form going in, it will be coming out!              '
      write(LER,*)'                                                    '
      write(LER,*)' Parameter      Description                 Default '
      write(LER,*)'                                                    '
      write(LER,*)' -N[]   -- input data set                    (stdin)'
      write(LER,*)' -O[]   -- output data set                  (stdout)'
      write(LER,*)' -s[]   -- process start time (ms)               (1)'
      write(LER,*)' -e[]   -- process end time (ms)       (last sample)'
      write(LER,*)' -ns[]  -- start trace number                    (1)'
      write(LER,*)' -ne[]  -- end trace number             (last trace)'
      write(LER,*)' -nhg[] -- hydrostatic gradient              (0.465)'
      write(LER,*)' -dz[]  -- depth sample interval       (TmSlIn/1000)'
      write(LER,*)' -met   -- depth units in meters              (feet)'
      write(LER,*)' -dz    -- depth sample interval       (TmSlIn/1000)'
      write(LER,*)' -mask  -- output filter mask, not data       (feet)'
      write(LER,*)' -V     -- verbos printout                          '
      write(LER,*)'                                                    '
      write(LER,*)' Usage:                                             '
      write(LER,*)'    psifilt -N[] -O[] -nhg[] -dz[] -s[] -e[] -ns[]  '
      write(LER,*)'            -ne[] -met -mask -V                     '
      write(LER,*)'                                                    '
      write(LER,*)' ==================================================='
      write(LER,*)'                                                    '
      
      return
      end

c -----------------  Subroutine -----------------------

c pick up command line arguments 

      subroutine cmdln ( ntap, otap, ns, ne, ist, iend, 
     :     nhg, met, dz, name, mask, verbos )

#include <f77/iounit.h>

      integer    ist, iend, ns, ne, argis

      real       nhg, dz

      character  ntap*(*), otap*(*), name*(*)

      logical    verbos, mask, met

           call argr4 ( '-dz', dz, -1.0, -1.0)
           call argi4 ( '-e', iend, 0, 0 )
           mask = (argis('-mask') .gt. 0)
           met = (argis('-met') .gt. 0)
           call argr4 ( '-nhg', nhg, 0.465, 0.465 )
           call argi4 ( '-ne', ne, 0, 0 )
           call argi4 ( '-ns', ns, 0, 0 )
           call argstr ( '-N', ntap, ' ', ' ' ) 
           call argstr ( '-O', otap, ' ', ' ' ) 
           call argi4 ( '-s', ist, 1, 1 )
           verbos = (argis('-V') .gt. 0)

c check for extraneous arguments and abort if found to
c catch all manner of user typo's

      call xtrarg ( name, ler, .FALSE., .FALSE. )
      call xtrarg ( name, lerr, .FALSE., .TRUE. )

      return
      end

c -----------------  Subroutine -----------------------

c verbal printout of pertinent program particulars

      subroutine verbal ( ntap, otap, nsamp, nsi, ntrc, nrec, iform, 
     :     ist, iend, ns, ne, verbos, met, mask, dz, nhg)

#include <f77/iounit.h>

      integer    nsamp, ntrc, iform, ist, iend, ns, ne, nsi
      real       dz, nhg
      character  ntap*(*), otap*(*)
      logical    met, mask, verbos

      write(LERR,*)' '
      write(LERR,*)' Input Line Header Parameters'
      write(LERR,*)' '
      write(LERR,*) ' input data set name   =  ', ntap
      write(LERR,*) ' samples per trace     =  ', nsamp
      write(LERR,*) ' traces per record     =  ', ntrc
      write(LERR,*) ' number of records     =  ', nrec
      write(LERR,*) ' data format           =  ', iform
      write(LERR,*) ' sample interval (ft)  =  ', dz
      write(LERR,*)' '
      write(LERR,*)' '
      write(LERR,*)' Command Line Parameters '
      write(LERR,*)' '
      write(LERR,*) ' output data set name  =  ', otap
      write(LERR,*) ' start trace           =  ', ns
      write(LERR,*) ' end trace             =  ', ne
      write(LERR,*) ' processing sample start = ', ist
      write(LERR,*) ' processing sample end   = ', iend
      write(LERR,*) ' normal pressure gradient = ', nhg
      if ( mask )  write(LERR,*) ' filter mask output requested'
      if ( met )  write(LERR,*) ' Input depth units are metric (m) '
      if ( verbos )  write(LERR,*) ' verbose printout requested'
      write(LERR,*)' '
      write(LERR,*)'========================================== '
      write(LERR,*)' '

      return
      end


c-----------------------------------------------------------------------
c-----------------------------------------------------------------------
c -----------------  Filter Subroutine -----------------------
      subroutine hsfilt (Record, Headers, Record_Mask,
     :        nsamp, ntrco, ist, iend, dz, nhg)

#include <save_defs.h>
#include <f77/lhdrsz.h>

c variables passed from calling routine

      integer nsamp, ntrco, ist, iend
      integer Headers(ITRWRD*ntrco)
      real Record(nsamp, ntrco), Record_Mask(nsamp,ntrco)
      real dz, nhg

c local variables

      real psi1, psi2, minpsi2, mindpsi

c initialize variables

      mindpsi = nhg*dz

      do i = 1, nsamp
         do j = 1, ntrco
            Record_Mask(i,j) = 0.0
         enddo
      enddo

c process data

      hdr_index = 0
      DO j = 1, ntrco
         DO i = ist, iend-1
            psi1 = Record(i,j)
            psi2 = Record(i+1,j)
            minpsi2 = psi1 + mindpsi
            if (psi2 .lt. minpsi2) then
               Record(i+1,j) = minpsi2
               Record_Mask(i,j) = 1.0
               psi2 = minpsi2
            endif
         ENDDO
      ENDDO
      return
      end
