c***********************************************************************
c                                                                      *
c                 copyright 2001, Amoco Production Company             *
c                             All Rights Reserved                      *
c                     an affiliate of BP America Inc.                  *
c***********************************************************************
c NAME: GTGRID   GENERATE GRIDS IN TIME                REV 1.0  AUG 95 *
c***********************************************************************
c
c  PURPOSE:
c       GTGRID generates a piecewise regular grid and a simple grid
c       in time, based on analagous grids in depth.
c
c  LANGUAGE:
c       FORTRAN 77
c
c  HISTORY:
c       ORIGINAL                AUG 95          J. Cooperstein, CETech
c
c  CALLING FORMAT:
c
c       call gtgrid( nzt, nz, nzseg, izsnz, zsdz, zsslor, zssloi
c     &              , zmax, dtout, zsdt, t1grid, t2grid, ierr )
c
c  PARAMETERS:
c
c       nzt     integer input scalar
c               expected length(-1) of the output irregular grid t1grid
c               (same as that of the irregular z-grid)
c
c       nz      integer input/output scalar
c               length of output regular grid t2grid
c               if nz=0 on input it is set as tmax/dtout
c
c       nzseg   integer input scalar
c               Number of Z segments.
c
c       izsnz   integer input vector of length nzseg
c               izsnz(i) is the number of z steps in the i-th z segment
c
c       zsdz    real input vector of length nzseg
c               zsdz(i) is the delta z for the i-th z segment.
c
c       zsslor  real input vector of length nzseg
c               contains the reflected slowness for each segment, if
c               prestack. Otherwise, contains twice the slowness.
c
c       zssloi  real input vector of length nzseg
c               contains the incident slowness for each segment, if
c               prestack.  Otherwise, contains zeros.
c
c       zmax    real input scalar
c               largest z-value for the regular z-grid; a largest value
c               of t will be used for the regular t-grid which is correct
c               for this depth.  If zmax=0, tmax is taken to be the 
c               largest value of t for the irregular grid, t2grid
c
c       dtout   real input/output scalar
c               delta t for the regular output grid t2grid.
c               if dtout=0 on input it is set as tmax/nz
c
c       zsdt    real output vector of length nzseg
c               zsdt(i) is the delta t for the i-th z segment.
c
c       t1grid  real output vector of dimension nzt + 1
c               irregular grid constructed by mapping the irregular z grid
c               nzt = sum[ izsnz(i), i = 1, nzseg ]
c
c       t2grid  real output vector of dimension nz + 1
c               regular grid with same starting and ending point as t1grid
c               entries spaced by dtout
c
c  DESCRIPTION:
c       GTGRID calculates vectors of values in time analagous to those
c       used for depth.  The grid t1grid is a piece wise regularly
c       spaced grid.  The grid t2grid covers the same range in time with
c       equal increments.
c
c  SUBPROGRAMS CALLED:
c       GGRID (libmbs)
c       VRAMP (MathAdvantage)
c
c  ERROR CONDITIONS:
c       If the length of the generated output grid t1grid is not 
c       consistent with that expected, ierr = 1
c
c*********************************************************************
c
      subroutine gtgrid( nzt, nz, nzseg, izsnz, zsdz, zsslor, zssloi
     &              , zmax, dtout, zsdt, t1grid, t2grid, ierr )
c
      implicit none
c
c input parameters:
c
      integer  nzt, nz, nzseg, izsnz(nzseg)
      real  zsdz(nzseg), zsslor(nzseg), zssloi(nzseg), zmax
c
c output parameters:
c
      real    t1grid(*), t2grid(*), zsdt(nzseg), dtout, ierr
c
c local parameters:
c
      integer izseg, ncheck
      real dttop, dztop, ttop, ztop, tmax
c
c**********************************************************************
      ierr      = 0
      ncheck    = 0
c
      do izseg = 1, nzseg
          zsdt(izseg) = zsdz(izseg) * ( zsslor(izseg) + zssloi(izseg) )
          ncheck = ncheck + izsnz(izseg)
      enddo
c
      call ggrid( nzseg, izsnz, zsdt, t1grid(2) )
c
      if( zmax .gt. 0.0 ) then
         ttop   = 0.0
         ztop   = 0.0
         do izseg = 1, nzseg
            dztop = izsnz(izseg) * zsdz(izseg)
            dttop = izsnz(izseg) * zsdt(izseg)
            if( (ztop+dztop) .le. zmax ) then
               tmax = ttop + dttop * ( zmax - ztop ) / dztop
            endif
            ztop = ztop + dztop
            ttop = ttop + dttop
         enddo
      else
         tmax = t1grid(nzt+1)
      endif
c
      if( nz .eq. 0 ) then
         if( dtout .eq. 0 ) then
            ierr = 2
            return
         endif
         nz = int( tmax / dtout )
      else
         if( nz. eq. 0 ) then
            ierr = 3
            return
         endif
         dtout = tmax / float( nz )
      endif
c
      t2grid(1) = 0.0
      call vramp( 0.0, dtout, t2grid, 1, nz + 1 )
c
      if( ncheck .ne. nzt ) ierr = 1
c
      return
      end
c**********************************************************************
