C***********************************************************************
C                 copyright 2001, Amoco Production Company             *
C                             All Rights Reserved                      *
C                     an affiliate of BP America Inc.                  *
C***********************************************************************
C***********************************************************************
C NAME: PWMVZN    PLANE WAVE MIGRATION - V OF Z        REV 9.9  NOV 95 *
C***********************************************************************
C
C  PURPOSE:
C       PWMVZN performs a V of Z plane wave migration.
C
C  USAGE:
C       pwmvzn [-Nintape] [-Oouttape] [-Ccard] -VC[vcards] [-Sstretch] \
C              [-TIME] [-V] [-h]
C
C       -Nintape
C               Specifies 'intape' as the input SIS data set.  If
C               omitted, standard input is used.
C
C       -Oouttape
C               Specifies 'outtape' as the output SIS data set.  If
C               omitted, standard output is used.
C
C       -Ccard
C               Specifies 'card' as the input card file.  If omitted,
C               pwmvzn.crd is used.
C
C       -VCvcards
C               Specifies 'vcards' as the input velocity file.
C
C       -Sstretch
C               Specifies 'stretch' as a multiplier for the depths and
C               velocities.
C
C       -TIME
C               Causes the program to generate an output grid based on
C               time instead of depth, with a spacing such that dt is
C               given by DTMS*.001, If DTMS is not given in the card
C               file, it is taken as the sampling time interval in the
C               input file line header.
C
C       -V
C               Causes the program to generate verbose print output.
C
C       -h
C               Causes the program to write help information to the
C               standard output and then halts.
C
C  FILES:
C       intape
C               Input standard SIS data set.
C
C       outtape
C               Output standard SIS data set.
C
C       card
C               Input card file.
C
C       vcards
C               Input velocity cards.
C
C       pfile
C               Output print file.  This file contains informational
C               output and error messages.  The print file name is
C               PW.xxxxx.yyyyy, where xxxxx and yyyyy are 5 digit
C               numbers that are generated from the process ID.
C
C  EXAMPLE:
C       The following command line will execute pwmvzn with an input SIS
C       data set named pwmvzn.inp, an output SIS data set named
C       pwmvzn.out, an input card file named pwmvzn.crd, a velocity card
C       file called velcards and with the verbose output option set:
C
C    pwmvzn -Npwmvzn.inp -Opwmvzn.out -Cpwmvzn.crd -VCvelcards -S1.2 -V
C
C       The following is an example of a card file:
C
C          1111111111222222222233333333334444444444555555555566666666667
C 1234567890123456789012345678901234567890123456789012345678901234567890
C
C TIME       ENDTIMEMS PADTIMEMS BEGTIMEMS      DTMS
C TIME            2000       200       100
C FREQUENCY  MINFREQHZ  F2FREQHZ  F3FREQHZ MAXFREQHZ
C FREQUENCY         20        25        55        60
C DEPTH      MAX DEPTH   DELTA Z
C DEPTH          10000        50
C WIDTH         DELTAX   BEG REC   END REC   REC INC  BEGTRACE  ENDTRACE
C WIDTH             80
C MODEL                REF VELOC BEAMWIDTH PADTRACE
C MODEL                     8000        10
C   # ANGLES     START INCREMENT
C          3        -8         8
C
C       The following is an example of a velocity card file:
C
C          1111111111222222222233333333334444444444555555555566666666667
C 1234567890123456789012345678901234567890123456789012345678901234567890
C
C NOVEL     # VELOCITY
C NOVEL              4
C VELOCITIES REFLECTED  INCIDENT END DEPTH
C          1      6000      6000      2000
C          2      5000      5100      4000
C          3      7000      7000      5000
C          4      8000      8000      8000
C   OR:
C MODEL              4
C VELOCITIES REFLECTED  INCIDENT END DEPTH
C          1      6000      6000      2000
C          2      5000      5100      4000
C          3      7000      7000      5000
C          4      8000      8000      8000
C
C  ERROR CONDITIONS:
C       If an error is detected, a error message containing a error code
C       is written and the program aborts with a stop code of 1.
C
C       The error code is a four digit number of the form xyzz, where
C       x indicates in which PWMVZN subroutine the error was detected, y
C       indicates which (if any) library subroutine reported the error,
C       and zz indicates the nature of the error.  The following are the
C       error codes:
C
C       (1) First digit (x) - PWMVZN subroutine where error was
C           detected:
C               1 - PWMVZN main program
C               2 - PWINIT
C               3 - PWPARM
C               4 - PWRCRD
C               5 - PWMIGR
C
C       (2) Second digit (y) - library subroutine which reported the
C           error:
C               0 - None, error is reported by PWMVZN routine.
C               1 - OPENPR
C               2 - VZRDVM
C               3 - GCFTLP
C               4 - RDANGL
C               5 - RDREC
C               6 - VZDCIP
C               7 - WRREC
C
C       (3) Last two digits (zz) - error.  If y is greater than zero,
C           then zz contains the absolute value of the error code
C           returned by the routine identified by y.  Otherwise, zz has
C           the following meanings:
C               01 - velocity card file was not specified
C               02 - error when attempting to read line header
C               03 - dimension exceeds array size
C               04 - unexpected end-of-file
C               05 - invalid parameter value
C
C  HISTORY:
C       DEC 87          REL 1.0         R.D. Coleman, QTC
C
C       JAN 88          REL 1.1         M.A. Thornton
C               Added GAMOCO and COF TAPEIO
C
C       FEB 88          REL 1.2         M.A. Thornton
C               If INC.VEL = 0., then INC.VEL = REF.VEL
C
C       APR 88          REL 1.3         M.A. Thornton
C               (1) Changed RFFTM to RFFTMR.  Ron Coleman changed the
C               names because MATHADV came out with a different RFFTM
C               (it has a different call list).  RFFTMR is in DANLIB.
C               (2) Added call to HLHPRT.
C
C       MAY 88          REL 2.0         R.D. Coleman, QTC
C               Replaced FFT calls in PWREAD with SCILIB looping mixed
C               radix FFT_S (RFFTMLT and CFFTMLT).  Replaced call to
C               NRFFTM with call to NRFFT5.  Increased MAXNX to 1027
C               to increase the size of depth for FFT work space and
C               decreased the MAXNW tO 512 (=MAXNT/2).
C
C       JUN 88          REL 3.0         R.D. Coleman, QTC
C               Replaced XFSMP with CCUINT (CAL-coded cubic interpo-
C               lation routine) for performing the resampling.
C               CCUINT is in DANLIB.
C
C       JUL 88          REL 4.0         R.D. Coleman, QTC
C               Replaced XVZPKL with XVZPK2 and replaced IPWR2 with
C               NRFFT5.  XVZPK2 and NRFFT5 are in DANLIB.  This change
C               allows NK to have values other than a power of two.
C               Specifically, NK will be set to the smallest even
C               number greater than or equal to NX that is of the form
C               2**I * 3**J * 5**K.  (NX is the number of traces.)
C
C       AUG 88          REL 5.0         R.D. Coleman, QTC
C               Major restructuring and clean-up of the code.
C               (1) Eliminated one COMMON block.
C               (2) Replaced COMMON_s with INCLUDE statements.
C               (3) Replaced subroutines PWCRD1, PWCRD2, PWCRD3, PWCRD4,
C               PWCRD5, PWCRD6, and PWCRD7 with PWRCRD.  PWRCRD calls
C               the general input routines RDCARD, RDVMOD, and RDANGL.
C               These routines are in DANLIB.
C               (4) Restructured error messages.
C               (5) Doubled the maximum number of samples per trace.
C
C       SEP 88          REL 5.1         T.G. Mattson, QTC
C               Added the capability to pad the number of traces.
C
C       APR 89          REL 5.2         M.A. THORNTON
C               DOUBLE NO. TRACES, INCREASE NO. OUTPUT DEPTH SAMPLES
C
C       AUG 89          REL 5.3         M.A. THORNTON
C               Added stretch parameter to adjust velocities and depths
C
C       SEP 89          REL 5.4         M.A. THORNTON
C               REWROTE PWRCRD TO DO SIMPLE READING OF CARDS
C               CHANGED THE NAME TO PWMVZNSS, AND EVENTUALLY ADDED THE
C               SORT AND SUM ROUTINES TO THIS PROGRAM
C               PROGRAM NO LONGER CALLS RDCARD
C
C       JAN 90          REL 6.0         M.A. THORNTON
C               Changed pwmvzss to read velocity cards from separate
C               file program will be called pwmvzn, pwmvzss may or
C               may not be completed at another time.
C
C       FEB 90          REL 6.1         M.A. THORNTON
C               Removed # velocities from the model card and introduced
C               a new card (NOVEL) to contain the number of velocities.
C               Added .01 to the nz computation in pwrcrd
C
C       MAY 90          REL 6.2         M.A. THORNTON
C               Corrected internal name of the crd file to pwmvzn.crd
C
C       MAY 90          REL 7.0         R.D. Coleman, QTC
C               ADDED SUBROUTINE PWITAB AND CHANGED CALL TO XVZPK2 TO
C               CALL TO XVZPK3.  PHASE SHIFT VECTORS ARE NOW CALCULATED
C               BY TABLE LOOKUP.
C
C       NOV 90          REL 7.1         R.D. Coleman, QTC
C               (1) Did general clean up and minor restructuring.
C               (2) Changed name of common file to pwmvzn.h
C               (3) Fixed PWITAB so that the complex function table
C                   requires less space
C               (4) Split the functionality of RDVMOD between two new
C                   routines - PWRDVM and PWVMOD.
C               (5) Calculate the two way travel time (TT2WAY) and use
C                   it as the default ENDTIME.
C               (6) Stop migrating whenever an angle becomes critical.
C
C       NOV 90          REL 8.0         R.D. Coleman, QTC
C               Wave field and image data is now maintained in natural
C               order as opposed to FFT order.  Subroutine XVZPK3
C               replaced by XVZPK4.
C
C       JAN 91          REL 8.1         R.D. Coleman, QTC
C               Corrected K limits calculation.
C
C       MAR 91          REL 8.2         M.A. Thornton
C               Use saver and savew for lineheader words manipulation
C
C	MAR 91		REL 8.3		J.C. Childress
C		Changed the calling sequence of 4 subroutines to allow
C	        for Dynamic memory allocation with HPALLOC.  The four
C		subroutines that changed are:  PWDCIM, PWREAD, PWRSMP
C		and PWWRIT.  The main program, PWMVZN, computes the
C		required amount of memory for 3 arrays:  DATA, RIMAGE
C		and DEPTH.  The length of each array is stored in
C		LDATA, LRIMAGE and LDEPTH respectively.  The three
C		length parameters replace the array names in the named
C		common block "PWCOMM".
C
C       JUL 91          REL 8.4         M.A. Thornton
C		Corrected code to not crash when first depth is a zero
C		This change is in PWRDVM subroutine                    
C
C       SEP 91          REL 8.5         M.A. Thornton
C		Moved code to sun for maintenance/distribution there  
C               Removed the "docpack" stuff from the beginning - added
C               version and ppname and lprt to the routine pwinit.F so
C               mbs_d_stamp would stamp the date in printout during
C               execution.
C
C       NOV 91          REL 9.0         R.D. Coleman, CETech
C               Restructured code - replaced much of the code with
C               libmbs calls and removed the COMMON block.
C
C       FEB 92          REL 9.1         R.D. Coleman, CETech
C               Made portable version, uses standard .h files, and
C               added DEBUG.
C
C       JUL 92          REL 9.2         R.D. Coleman, CETech
C               Fixed the length calculation for WRK2 and changed the
C               call to OPENPR to use the full program name.
C
C       OCT 92          REL 9.3         R.D. Coleman, CETech
C               Added output for Z = 0.0.
C
C       FEB 93          REL 9.4         R.D. Coleman, CETech
C		Corrected error in calculation of LWRK2 for SUN version.
C		Also, forced depths at segment boundaries to the next
C		higher integer multiple of the output delta Z.
C
C       MAY 93          REL 9.5         M.A. Thornton          
C		Recompiled to take advantage of new gtfltr.f library routine
C		Added include file for the HP logical unit
C		Changed line header size to be SZLNHD     
C       JUNE 93         REL 9.6         M.A. Thornton          
C		Recompiled to take advantage of new vzrdvm2.f library routine
C               which will accept the NOVEL template and NOVEL card or just
C               the MODEL card
C       OCTOBER 93      REL 9.7         M.A. Thornton          
C		Recompiled to add auditing
C       AUGUST  95      REL 9.8         J. Cooperstein, CETech
C               Added -TIME switch to permit resampling by time instead of
C               depth.  Calls new library subroutine GTGRID.  Modified
C               main program and CMINIT.
C       NOV  95         REL 10.2         Jerry Cooperstein, CETech
C               Modified the -TIME switch, so that a fixed
C               delta-t = DTMS*001 is used. If DTMS is not given in the card
C               file, it is taken as the sampling time interval in the
C               input file line header.
C               GTGRID modified so that the maximum time corresponds
C               properly to the maximum depth, using two-way travel time.
C       NOV 95          REL 10.2         Mary Ann Thornton               
C               Added Dz1000 to the output line header along
C               with Jerry's changes listed above.        
C
C-----------------------------------------------------------------------
C
C  DATA DICTIONARY
C       The table below shows usage of the major variables.  Symbolic
C       constants, such as LUPRT, are documented in the file pwmvzn.h.
C
C                    USAGE BY ROUTINE
C                    M  I  P  R  D  M
C                    A  N  A  C  U  I
C                    I  I  R  R  M  G
C  TYPE    VARIABLE  N  T  M  D  P  R  DESCRIPTION
C
C  REAL    AK    ()  D     O  O  I  I   Wave number vector
C  REAL    BWIDTH    D     O  O  I  I   Beam width
C  REAL    CXFTAB()  D     O        I   Complex function table
C  LOGICAL DEBUG     D  O               Debug flag
C  REAL    DK        D     O  O  I  I   Delta K
C  REAL    DZ        D     O  O  I      Delta Z
C  REAL    FILTR ()  D     O  O  I      Freguency filter
C  REAL    FMAX            D  O  I      Maximum frequency (Hz)
C  INTEGER HEADER()  D                  Trace header array
C  INTEGER IDT             D  O  I      Sampling interval used
C  INTEGER IDT0            D  I  I      Sampling interval from line header
C  INTEGER IERR      D  O  O  O     O   Error flag
C  INTEGER INCREC    D     O     I      Record increment
C  COMPLEX IMAGE ()  D+             O*  Complex image (in WRK1 in main)
C  INTEGER IREC1     D     O  O  I      First input record to keep
C  INTEGER IREC2     D     O  O  I      Last  input record to keep
C  INTEGER ISMP1     D     O  O  I      First sample to keep
C  INTEGER ISMP2     D     O  O  I      Last  sample to keep
C  INTEGER ITRC1     D     O  O  I      First trace to keep
C  INTEGER ITRC2     D     O  O  I      Last  trace to keep
C  INTEGER IW1       D     O  O  I      First frequency to keep
C  INTEGER IZSNZ ()  D     O     I  I   Number of Z_s per Z segment
C  INTEGER JREC      D                  Index of  input record
C  INTEGER KREC      D                  Index of output record
C  INTEGER LUINP     D  O  I            LUN of  input SIS dataset
C  INTEGER LUOUT     D  O  I            LUN of output SIS dataset
C  INTEGER MK        D              I   Leading dimension of IMAGE
C  INTEGER MT        D              I   Leading dimension of input matrix
C  INTEGER NK        D     O  O  I  I   Number of K_s (wavenumbers)
C  REAL    NREC            D  I  I      Number of  input records
C  REAL    NREC2           D  O  I      Number of output records
C  INTEGER NSMP      D     O  I  I      Number of samples per trace
C  INTEGER NT        D     O  O  I      Number times after padding
C  INTEGER NTOFF     D     O  O  I      Padding at start of trace
C  INTEGER NTRC      D     O  I  I      Number of traces per record
C  INTEGER NW        D     O  O  I  I   Number of frequencies kept
C  INTEGER NX        D     O  O  I      Number of X_s
C  INTEGER NZ        D     O  O  I      Number of output Z_s
C  INTEGER NZSEG     D     O     I  I   Number of Z segments
C  INTEGER NZT       D     O     I  I   Number of Z_s for migration grid
C  REAL    OMEGA ()  D     O  O  I  I   Angular frequency vector
C  COMPLEX PSI   ()  D+             I*  Wave field (in WRK2 in main)
C  REAL    STRCH     D  O  I  I  I      Velocity stretch factor
C  REAL    THETA ()  D     O  O  I  I   Angle vector (scalar in PWMIGR)
C  REAL    TT2WAY          D  I  I      Minimum two way travel time
C  REAL    VELREF    D     O  O  I  I   Reference velocity
C  LOGICAL VERBOS    D  O  I  I         Verbose flag
C  REAL    WORK  ()  D+             W*  Work vector (in WRK0 in main)
C  REAL    WRK0  ()  D              W*  Medium work space
C  REAL    WRK1  ()  D              O*  Large work space 1
C  REAL    WRK2  ()  D              I*  Large work space 2
C  REAL    Z1GRID()  D     O            Z grid for migration
C  REAL    Z2GRID()  D     O            Z grid for output
C  REAL    ZSDZ  ()  D     O     I  I   Delta Z per Z segment
C  REAL    ZSSLOI()  D     O     I  I   Incident  slowness per Z segment
C  REAL    ZSSLOR()  D     O     I  I   Reflected slowness per Z segment
C
C  KEY:
C  D = DECLARED; I = INPUT; O = OUTPUT; W = WORKSPACE
C
C  * Indicates that the CALL and SUBROUTINE use different names
C  + Indicates that the space is declared under a different name
C
C-----------------------------------------------------------------------
C
      PROGRAM PWMVZN
C
#include <localsys.h>
#include <f77/lhdrsz.h>
#include <f77/sisdef.h>
C
#include <pwmvzn.h>
C
      INTEGER IERR, INCREC, IREC1, IREC2, JREC, KREC, ISMP1, ISMP2,
     &        ITRC1, ITRC2, IW1, LHDR, LUINP, LUOUT, LWRK0, LWRK1,
     &        LWRK2, MK, MT, NBYTES, NEXTRC, NK, NSMP, NT, NTOFF, NTRC,
     &        NW, NX, NZ, NZSEG, NZT, IZSNZ(MAXSEG), JERR
      LOGICAL VERBOS, DEBUG, TIMELOG
      REAL    DK, DZ, BWIDTH, STRCH, VELREF, AK(MAXNK), FILTR(MAXNW),
     &        OMEGA(MAXNW), THETA(MAXREC), Z1GRID(MAXNZT),
     &        Z2GRID(MAXNZ), ZSDZ(MAXSEG), ZSSLOI(MAXSEG),
     &        ZSSLOR(MAXSEG),
     &        ZSDT(MAXSEG), T1GRID(MAXNZT), T2GRID(MAXNZT)
C
C===  SYSTEM DEPENDENT DECLARATIONS
C
#ifdef CRAY
      REAL    CXFTAB(NTAB+5)
#else
      REAL*8  CXFTAB(NTAB+5)
#endif
C
C===  DECLARATIONS FOR DYNAMICALLY ALLOCATED ARRAYS
C
      INTEGER HEADER(1)
      REAL    WRK0(1), WRK1(1), WRK2(1)
      POINTER (PWRK0, WRK0), (PWRK1, WRK1), (PWRK2, WRK2),
     &        (PHDR, HEADER)
C
      DATA LUINP, LUOUT / -1, -1 /

C
C-----------------------------------------------------------------------
C
C===  INITIALIZE, OPEN FILES, ETC.
C
      CALL PWINIT( VERBOS, DEBUG, TIMELOG, STRCH, LUINP, LUOUT, IERR )
      IF( IERR .NE. 0 ) GO TO 800
      if( DEBUG ) print *, 'finished PWINIT'
C
C===  READ LINE HEADER & JOB PARAMETERS, WRITE OUTPUT LINE HEADER
C
      CALL PWPARM( LUINP, LUOUT, VERBOS, STRCH, IREC1, IREC2, INCREC,
     &             NTRC, ITRC1, ITRC2, NSMP, ISMP1, ISMP2,
     &             NT, NTOFF, NW, IW1, NX, NK, NZ, NZT, NZSEG,
     &             DK, DZ, BWIDTH, VELREF, THETA, OMEGA, FILTR, AK,
     &             IZSNZ, ZSDZ, ZSSLOR, ZSSLOI, Z1GRID, Z2GRID,
     &             CXFTAB, TIMELOG, T1GRID, T2GRID, ZSDT,
     &             IERR )
      IF( IERR .NE. 0 ) GO TO 800
      if( DEBUG ) print *, 'finished PWPARM'
C
C===  Compute the memory requirements and allocate memory with HPALLOC.
C
      MT = NT + 3
      MK = NK + 1
C
      LHDR  = NX * ITRWRD
      LWRK0 = 7 * MAX0( NK, NZ, NZT+1 )
      LWRK1 = MAX0( MT*NK, 2*MK*(NZT+1) )
#ifdef CRAY
      LWRK2 = MAX0( 2*NT*NX+2*NT, 4*NW*NK+2*NK, 4*NK*(NZT+1)+2*NK, 
     &              NX*NZ )
#else
      LWRK2 = MAX0( 9*NT/2+18*NX+41, 6*NK+18*NW+41, 6*NK+18*(NZT+1)+41,
     &              NX*NZ, 2*NW*NK )
#endif
C
      CALL GALLOC( PHDR , LHDR *ISZBYT, JERR, 'ABORT' )
      CALL GALLOC( PWRK0, LWRK0*ISZBYT, JERR, 'ABORT' )
      CALL GALLOC( PWRK1, LWRK1*ISZBYT, JERR, 'ABORT' )
      CALL GALLOC( PWRK2, LWRK2*ISZBYT, JERR, 'ABORT' )
      if( DEBUG ) print *, 'finished GALLOC'
      if( DEBUG ) print *, 'ISZBYT, LWRK0, LWRK1, LWRK2, LHDR = ', 
     &          iszbyt, lwrk0, lwrk1, lwrk2, lhdr
C
C===  LOOP OVER ANGLE (RECORD)
C
      NBYTES = NSMP * SZSAMP + SZDTHD
      NEXTRC = IREC1
      KREC   = 0
      DO 120 JREC = 1, IREC2
         if( DEBUG ) print *, 'rec = ', jrec
C
         IF (JREC .EQ. NEXTRC) THEN
            IF (VERBOS) WRITE (LUPRT, *) ' PROCESSING RECORD ', JREC
            NEXTRC = NEXTRC + INCREC
            KREC   = KREC  + 1
C
C======     READ RECORD DATA
C======     NOTE: WRK1 WILL NOW CONTAIN THE INPUT DATA IN (T,X)
C
            CALL RDREC( LUINP, LUPRT, JREC, NTRC, ITRC1, ITRC2,
     &                  NSMP, ISMP1, ISMP2, NTOFF, NT, 1, NX, MT,
     &                  WRK0, HEADER, WRK1, IERR )
            if( DEBUG ) print *, 'finished RDREC'
            IF( IERR .NE. 0 ) THEN
               IERR = 1500 + IABS( IERR )
               GO TO 800
            ENDIF
C
C======     FORWARD 2D FFT
C======     NOTE: WRK2 WILL NOW CONTAIN THE FILTER INPUT DATA IN (K,W)
C
            CALL FFT2DF( MT, NT, NX, NK, NW, IW1, FILTR, WRK1, WRK2 )
            if( DEBUG ) print *, 'finished FFT2DF'
C
C======     PERFORM MIGRATION
C======     NOTE: WRK1 WILL NOW CONTAIN THE COMPLEX IMAGE IN (K,Z1)
C======           WHERE Z1 IS Z WITH THE INTERNAL IRREGULAR GRID
C
            CALL PWMIGR( MK, NK, NW, NZSEG, NZT, DK, THETA(JREC),
     &                   BWIDTH, VELREF, OMEGA, AK, IZSNZ, ZSDZ, ZSSLOR,
     &                   ZSSLOI, CXFTAB, WRK0, WRK2, WRK1, IERR )
            if( DEBUG ) print *, 'finished PWMIGR'
            IF( IERR .NE. 0 ) GO TO 800
C
C======     INVERSE FFT
C======     NOTE: WRK1 WILL NOW CONTAIN THE REAL IMAGE IN (X,Z1)
C
            CALL FFTIXR( MK, NK, NX, NZT+1, WRK2, WRK1 )
            if( DEBUG ) print *, 'finished FFTIXR'
C
C======     RESAMPLE
C======     NOTE: WRK2 WILL NOW CONTAIN THE REAL IMAGE IN (Z,X), or
C              (T,X) IF TIMELOG = .TRUE.
C
            IF( TIMELOG ) THEN
               CALL RESAMP( NX, 1, NZ, NZT+1, NX, NZ, 1, T1GRID, T2GRID,
     &                      WRK0, WRK1, WRK2 )
            ELSE
               CALL RESAMP( NX, 1, NZ, NZT+1, NX, NZ, 1, Z1GRID, Z2GRID,
     &                      WRK0, WRK1, WRK2 )
            ENDIF
C
            if( DEBUG ) print *, 'finished RESAMP'
C
C======     WRITE RECORD
C
            CALL WRREC( LUOUT, LUPRT, KREC, NZ, 1, NX, NZ,
     &                  WRK0, HEADER, WRK2, IERR )
            if( DEBUG ) print *, 'finished WRREC'
            IF( IERR .NE. 0 ) THEN
               IERR = 1700 + IABS( IERR )
               GO TO 800
            ENDIF
C
         ELSE
C
C======     SKIP RECORD
C
            IF (VERBOS) WRITE (LUPRT, *) ' SKIPPING RECORD ', JREC
            CALL SKIPT (LUINP, NTRC, NBYTES)
C
         ENDIF
C
  120 CONTINUE
C
C===  CLOSE FILES, CLEAN-UP, & EXIT
C
      IERR = 0
C
  800 CONTINUE
      CALL MEXIT( IERR, LUPRT, LUINP, LUOUT )
C
      END
C*********************************************************************
C NAME: PWINIT                                                       *
C*********************************************************************
C
C  PURPOSE:
C      PWINIT INITIALIZES THE PROGRAM INCLUDING OPENING ALL REQUIRED
C      FILES.
C
C-----------------------------------------------------------------------
C
      SUBROUTINE PWINIT( VERBOS, DEBUG, TIMELOG, STRCH, LUINP, LUOUT
     &                 , IERR )
C
#include <pwmvzn.h>
#include <f77/hp.h>
C
C  SUBROUTINE ARGUMENTS
C
      LOGICAL VERBOS, DEBUG, TIMELOG
      REAL    STRCH
      INTEGER LUINP, LUOUT, IERR
C
C  LOCAL VARIABLES
C
      CHARACTER*128 NTAPE, OTAPE, CARDS, VCARDS
      INTEGER       ARGIS, ICOPEN, LUTRM, NLINES, II
C
C-----------------------------------------------------------------------
C
  901 FORMAT (/' ', 'PWMVZN COMMAND LINE ARGUMENTS',
     &        /' ',
     &        /' ', 'INPUTS:',
     &        /' ', '  -N[ntape]   - INPUT  DATASET NAME',
     &        /' ', '  -O[otape]   - OUTPUT DATASET NAME',
     &        /' ', '  -C[cards]   - CARD FILE NAME',
     &        /' ', '  -VC[cards]  - VELOCITY CARD FILE NAME',
     &        /' ', '  -S[stretch] - STRETCH DEPTHS, VELOCITIES',
     &        /' ', '  -TIME       - OUTPUT TIME, NOT DEPTH',
     &        /' ', '  -V          - VERBOSE PRINTOUT',
     &        /' ',
     &        /' ', 'USAGE:',
     &        /' ', '  pmmvzn -N[] -O[] -C[] -VC[] -S[] -TIME'
     &                        ' -V'/)
  902 FORMAT (/' ', ' INPUT DATASET  = ' / A128,
     &        /' ', ' OUTPUT DATASET = ' / A128)
  903 FORMAT (/' ', '***** ERROR: MISSING VELOCITY CARD FILE NAME ****')
C
C-----------------------------------------------------------------------
C
C  IF HELP, WRITE INSTRUCTIONS TO SCREEN
C
CMAT  LUTRM = 0
      LUTRM = LER
      IF (ARGIS( '-h' ) .GT. 0) THEN
         WRITE (LUTRM, 901)
         IERR = -1
         RETURN
      ENDIF
C
C  CREATE UNIQUE PRINT FILE NAME FROM THE PROCESS ID & OPEN PRINT FILE
C
      CALL OPENPR (LUSUR, LUPRT, PPNAME, IERR)
      IF (IERR .NE. 0) THEN
         IERR = 2100 + IABS( IERR )
         RETURN
      ENDIF
C
#include <mbsdate.h>
C
      NLINES = 1
      CALL GAMOCO (TITLE, NLINES, LUPRT)
C
      DEBUG  = ARGIS( '-D' ) .GT. 0
      VERBOS = ARGIS( '-V' ) .GT. 0 .OR. DEBUG
C
      TIMELOG = ARGIS( '-TIME' ) .GT. 0
      IF (VERBOS) WRITE (LUPRT, *) ' ENTER SUBROUTINE PWINIT'
C
C  GET REMAINING COMMAND LINE ARGUMENTS
C
      CALL ARGSTR ('-N', NTAPE , ' ', ' ')
      CALL ARGSTR ('-O', OTAPE , ' ', ' ')
      CALL ARGSTR ('-C', CARDS , ' ', ' ')
      CALL ARGSTR ('-VC',VCARDS, ' ', ' ')
      CALL ARGR4  ('-S', STRCH, 1.0, 1.0)
C
C  SET LUTRM
C
      IF (NTAPE .EQ. ' ') LUTRM = 2
C
C  DEFAULT FOR PIPES
C
      LUINP = 0
      LUOUT = 1
C
C  OPEN I/O FILES
C
      IF (NTAPE .NE. ' ') CALL LBOPEN (LUINP, NTAPE, 'r')
      IF (OTAPE .NE. ' ') CALL LBOPEN (LUOUT, OTAPE, 'w')
C
      IF (CARDS .NE. ' ') THEN
         OPEN (UNIT=LUCRD, FILE=CARDS, STATUS='OLD')
      ELSE
         II = ICOPEN ('-pwmvzn.crd', LUCRD)
      ENDIF
C
C  OPEN VELOCITY CARD FILE
C
      IF (VCARDS .NE. ' ') THEN
         OPEN (UNIT=LUVEL, FILE=VCARDS, STATUS='OLD')
      ELSE
         WRITE(LUPRT, 903)
         IERR = 2001
         RETURN
      ENDIF
C
      WRITE (LUPRT, 902) NTAPE, OTAPE
C
      IERR = 0
      RETURN
      END
C***********************************************************************
C NAME: PWPARM                                                         *
C***********************************************************************
C
C  PURPOSE:
C      PWPARM READS THE INPUT LINE HEADER, JOB PARAMETERS, AND WRITES
C      THE OUTPUT LINE HEADER.  IN ADDITION, PWPARM CALCULATES SOME JOB
C      PARAMETERS.
C
C-----------------------------------------------------------------------
C
      SUBROUTINE PWPARM( LUINP, LUOUT, VERBOS, STRCH, IREC1, IREC2,
     &                   INCREC, NTRC, ITRC1, ITRC2, NSMP, ISMP1, ISMP2,
     &                   NT, NTOFF, NW, IW1, NX, NK, NZ, NZT, NZSEG,
     &                   DK, DZ, BWIDTH, VELREF, THETA, OMEGA, FILTR,
     &                   AK, IZSNZ, ZSDZ, ZSSLOR, ZSSLOI,
     &                   Z1GRID, Z2GRID, CXFTAB, TIMELOG,
     &                   T1GRID, T2GRID, ZSDT, IERR )
C
#include <f77/lhdrsz.h>
#include <pwmvzn.h>
C
C  SUBROUTINE ARGUMENTS
C
      LOGICAL VERBOS, TIMELOG
      INTEGER LUINP, LUOUT, IREC1, IREC2, INCREC, NTRC, ITRC1, ITRC2,
     &        NSMP, ISMP1, ISMP2, NT, NTOFF, NW, IW1, NX, NK,
     &        NZ, NZT, NZSEG, IZSNZ(*), IERR
      REAL    STRCH, DK, DZ, BWIDTH, VELREF, THETA(*),
     &        OMEGA(*), FILTR(*), AK(*), ZSDZ(*), ZSSLOR(*), ZSSLOI(*),
     &        Z1GRID(*), Z2GRID(*),
     &        CXFTAB(*), T1GRID(*), T2GRID(*), ZSDT(*)
C
C  LOCAL VARIABLES
C
      INTEGER     IDT, IDT0, JUNK, LINHED, NBYTES, NREC, NREC2,
     &            IHEAD(SZLNHD), lbyout
      REAL        FMAX, TT2WAY, VREFL(MAXSEG), VINCI(MAXSEG),
     &            ZSTEP(MAXSEG), DTOUT
C
C-----------------------------------------------------------------------
C
  901 FORMAT (/' ', 'INPUT LINE HEADER PARAMETERS:')
  902 FORMAT (/' ', 'OUTPUT LINE HEADER PARAMETERS:')
  903 FORMAT (/' ', '   NUMBER OF SEISMIC RECORDS IN THIS JOB =', I5,
     2        /' ', '   NUMBER OF TRACES PER SEISMIC RECORD   =', I5,
     3        /' ', '   NUMBER OF DATA SAMPLES PER TRACE      =', I5,
     4        /' ', '   SAMPLE INTERVAL (MILLISECONDS)        =', I5)
  991 FORMAT (/' ', '***** ERROR - LINE HEADER READ ERROR *****')
  992 FORMAT (/' ', '***** ERROR - DIMENSIONS EXCEED ARRAY SIZES *****')
  993 FORMAT (/' ', '***** ERROR - FATAL ERROR REPORTED BY GCFTLP ****')
C
C-----------------------------------------------------------------------
C
      IF (VERBOS) WRITE (LUPRT, *) ' ENTER SUBROUTINE PWPARM'
C
C=======================================================================
C                       PROCESS LINE HEADER
C=======================================================================
C
      NBYTES = 0
      CALL RTAPE (LUINP, IHEAD, NBYTES)
      IF (NBYTES .EQ. 0) THEN
         WRITE (LUPRT, 991)
         IERR = 3002
         RETURN
      ENDIF
C
C  GET PARAMETERS FROM LINE HEADER
C
      LINHED = 0
      CALL SAVER(IHEAD, 'NumTrc', NTRC, LINHED)
      CALL SAVER(IHEAD, 'NumRec', NREC, LINHED)
      CALL SAVER(IHEAD, 'SmpInt', IDT0, LINHED)
      CALL SAVER(IHEAD, 'NumSmp', NSMP, LINHED)
C%%%
      IF (VERBOS) THEN
         WRITE (LUPRT, 901)
         WRITE (LUPRT, 903) NREC, NTRC, NSMP, IDT0
      ENDIF
C%%%
      IF (NREC .GT. MAXREC .OR. NSMP .GT. MAXSMP) THEN
         WRITE (LUPRT, *) ' NREC, MAXREC, NSMP, MAXSMP = ',
     &                      NREC, MAXREC, NSMP, MAXSMP
         WRITE (LUPRT, 992)
         IERR = 3003
         RETURN
      ENDIF
C
C=======================================================================
C                 READ PARAMETERS FROM CARD FILE
C=======================================================================
C
      CALL GETDZ( LUCRD, DZ, IERR )
      IF( IERR .NE. 0 ) THEN
         IERR = 3100 + IABS( IERR )
         RETURN
      ENDIF
C
      CALL VZRDVM2( PRESTACK, LUVEL, LUPRT, MAXSEG, STRCH, DZ, TT2WAY,
     &              NZSEG, VREFL, VINCI, ZSTEP, IERR )
      IF( IERR .NE. 0 ) THEN
         IERR = 3200 + IABS( IERR )
         RETURN
      ENDIF
C
      CALL PWRCRD( VERBOS, STRCH, TT2WAY, NREC, NTRC, NSMP, IDT0,
     &             IDT, NREC2, IREC1, IREC2, INCREC, ITRC1, ITRC2,
     &             ISMP1, ISMP2, NT, NTOFF, NW, IW1, NX, NK, NZ,
     &             DK, DZ, FMAX, BWIDTH, VELREF,
     &             THETA, OMEGA, FILTR, AK, IERR )
      IF( IERR .NE. 0 ) RETURN
C
      CALL VZVMOD( PRESTACK, FMAX, NZSEG, VREFL, VINCI, ZSTEP,
     &             NZT, IZSNZ, ZSDZ, ZSSLOR, ZSSLOI )
      IF( NZT+1 .GT. MAXNZT ) THEN
         WRITE (LUPRT, 992)
         IERR = 3003
         RETURN
      ENDIF
c
c===  SET UP TIME GRIDS
c
      DTOUT = 0.0
      IF( TIMELOG ) THEN
         DTOUT = 0.001 *  FLOAT( IDT )
         if( DTOUT .GT. 0.0 ) NZ = 0
         CALL GTGRID( NZT, NZ, NZSEG, IZSNZ, ZSDZ, ZSSLOR, ZSSLOI,
     &              Z2GRID(NZ+1), DTOUT, ZSDT, T1GRID, T2GRID, IERR )
         IF( VERBOS )
     &   WRITE( LUPRT,* )' OUTPUT = TIME, DT = ', DTOUT, ' NZ = ', NZ
     &                  , ' TMAX = ', DTOUT * NZ
         IF( IERR .NE. 0 ) THEN
            WRITE( LUPRT, * )' ERROR IN GTGRID: INCOMPATIBLE DIMENSIONS'
            RETURN
         ENDIF
      ENDIF
C
C=======================================================================
C                  UPDATE AND OUTPUT LINE HEADER
C=======================================================================
C
      CALL HLHPRT (IHEAD, NBYTES, PPNAME, LEN( PPNAME ), LUPRT)
C
      CALL SAVEW(IHEAD, 'NumTrc', NX   , LINHED)
      CALL SAVEW(IHEAD, 'NumRec', NREC2, LINHED)
      CALL SAVEW(IHEAD, 'SmpInt', IDT  , LINHED)
      CALL SAVEW(IHEAD, 'NumSmp', NZ   , LINHED)
      mdz = dz*1000.
      call savew(ihead, 'Dz1000', mdz, LINHED)
C
C     ADD THE COMMAND LINE ARGUMENTS TO THE LINE HEADER
C
      CALL SAVHLH (IHEAD, NBYTES, LBYOUT)
      CALL WRTAPE (LUOUT, IHEAD, LBYOUT)
C%%%
      IF (VERBOS) THEN
         WRITE (LUPRT, 902)
         WRITE (LUPRT, 903) NREC2, NX, NZ, IDT
      ENDIF
C%%%
C
C=======================================================================
C                  COMPUTE Z GRIDS AND PHASE SHIFT TABLE
C=======================================================================
C
      Z1GRID(1) = 0.0
      CALL GGRID( NZSEG, IZSNZ, ZSDZ, Z1GRID(2) )
      CALL VRAMP( 0.0, DZ, Z2GRID, 1, NZ+1 )
C
      CALL GCFTLP (JUNK, JUNK, 0, NTAB, TABMIN, TABMAX, ITABFG,
     &             CXFTAB, IERR)
C
      IF (IERR .NE. 0) THEN
         WRITE (LUPRT, 993)
         IERR = 3300 + IABS( IERR )
         RETURN
      ENDIF
C
C=======================================================================
C                  IF VERBOS, DUMP PARAMETERS TO PRINT FILE
C=======================================================================
C
      IF (VERBOS) THEN
         CALL PWDUMP( IDT0, IDT, NREC, NREC2, IREC1, IREC2, INCREC,
     &                 NTRC, ITRC1, ITRC2, NSMP, ISMP1, ISMP2,
     &                 NT, NTOFF, NW, IW1, NX, NK, NZ, NZT, NZSEG,
     &                 DK, DZ, FMAX, BWIDTH, VELREF, STRCH, TT2WAY,
     &                 THETA, OMEGA, FILTR, AK,
     &                 IZSNZ, ZSDZ, ZSSLOR, ZSSLOI )
      ENDIF
C%%%
      IERR = 0
      RETURN
      END
C***********************************************************************
C NAME: PWRCRD                                                         *
C***********************************************************************
C
C  PURPOSE:
C      PWRCRD READS AND PROCESSES THE CONTENTS OF THE CARD FILE.
C
C-----------------------------------------------------------------------
C
      SUBROUTINE PWRCRD( VERBOS, STRCH, TT2WAY, NREC, NTRC, NSMP, IDT0,
     &                   IDT, NREC2, IREC1, IREC2, INCREC, ITRC1, ITRC2,
     &                   ISMP1, ISMP2, NT, NTOFF, NW, IW1, NX, NK, NZ,
     &                   DK, DZ, FMAX, BWIDTH, VELREF,
     &                   THETA, OMEGA, FILTR, AK, IERR )
C
#include <pwmvzn.h>
C
C  SUBROUTINE ARGUMENTS
C
      LOGICAL VERBOS
      INTEGER NREC, NTRC, NSMP, IDT0, IDT, NREC2, IREC1, IREC2, INCREC,
     &        ITRC1, ITRC2, ISMP1, ISMP2, NT, NTOFF, NW, IW1, NX, NK,
     &        NZ, IERR
      REAL    STRCH, TT2WAY, DK, DZ, FMAX, BWIDTH, VELREF, THETA(*),
     &        OMEGA(*), FILTR(*), AK(*)
C
C  LOCAL VARIABLES
C
      INTEGER I, IROUND, ITBEG, ITEND, ITPAD, J, K, KBIAS, KREC, MT,
     &        NRFFT5, NTPAD, NXPAD
      REAL    F1, F2, F3, F4, ZMAX, DT, DX
      CHARACTER*80 CARD
C
      IROUND(I,J) = J * IFIX( FLOAT( I ) / FLOAT( J ) + 0.5 )
C
C-----------------------------------------------------------------------
C
  900 FORMAT (A80)
  901 FORMAT (10X, 4I10) 
  902 FORMAT (10X, 4F10.0) 
  903 FORMAT (10X, 2F10.0) 
  904 FORMAT (10X,  F10.0, 5I10  ) 
  905 FORMAT (10X,   10X , 2F10.0, I10) 
  911 FORMAT (/' ', 'TIME       ENDTIMEMS PADTIMEMS BEGTIMEMS',
     &              '      DTMS')
  912 FORMAT (/' ', 'FREQUENCY  MINFREQHZ  F2FREQHZ  F3FREQHZ',
     &              ' MAXFREQHZ')
  913 FORMAT (/' ', 'DEPTH      MAX DEPTH   DELTA Z')
  914 FORMAT (/' ', 'WIDTH         DELTAX   BEG REC   END REC',
     &              '   REC INC  BEGTRACE  ENDTRACE')
  915 FORMAT (/' ', 'MODEL                REF VELOC BEAMWIDTH',
     &              '  PADTRACE')
  921 FORMAT ( ' ', 'TIME      ', 4I10  )
  922 FORMAT ( ' ', 'FREQUENCY ', 4F10.1)
  923 FORMAT ( ' ', 'DEPTH     ', 2F10.1)
  924 FORMAT ( ' ', 'WIDTH     ',  F10.1, 5I10  )
  925 FORMAT ( ' ', 'MODEL     ',   10X , 2F10.1, I10)
  991 FORMAT (/' ', '***** ERROR - UNEXPECTED END OF CARD FILE *****')
  992 FORMAT (/' ', '***** ERROR - INVALID PARAMETER VALUE *****')
  993 FORMAT (/' ', '***** ERROR - DIMENSIONS EXCEED ARRAY SIZES *****')
C
C-----------------------------------------------------------------------
C
      IF (VERBOS) WRITE (LUPRT, *) ' ENTER SUBROUTINE PWRCRD'
C
C=======================================================================
C                 READ SCALAR PARAMETERS FROM CARD FILE
C=======================================================================
C
C  READ TIME CARD
C
   10 CONTINUE
         READ (LUCRD, 900, ERR=810) CARD
         IF (CARD(1:4) .NE. 'TIME') GO TO 10
C
      READ (LUCRD, 901) ITEND, ITPAD, ITBEG, IDT
C
C  READ FREQUENCY CARD
C
   20 CONTINUE
         READ (LUCRD, 900, ERR=810) CARD
         IF (CARD(1:9) .NE. 'FREQUENCY') GO TO 20
C
      READ (LUCRD, 902) F1, F2, F3, F4
C
C  READ DEPTH CARD
C
   30 CONTINUE
         READ (LUCRD, 900, ERR=810) CARD
         IF (CARD(1:5) .NE. 'DEPTH') GO TO 30
C
      READ (LUCRD, 903) ZMAX, DZ
C
C  READ WIDTH CARD
C
   40 CONTINUE
         READ (LUCRD, 900, ERR=810) CARD
         IF (CARD(1:5) .NE. 'WIDTH') GO TO 40
C
      READ (LUCRD, 904) DX, IREC1, IREC2, INCREC, ITRC1, ITRC2
C
C  READ MODEL CARD
C
   50 CONTINUE
         READ (LUCRD, 900, ERR=810) CARD
         IF (CARD(1:5) .NE. 'MODEL') GO TO 50
C
      READ (LUCRD, 905) VELREF, BWIDTH, NXPAD
C
C=======================================================================
C                        PROCESS TIME PARAMETERS
C=======================================================================
C
      IF (IDT   .LE. 0) IDT   = IDT0
      IF (ITEND .LE. 0) ITEND = 1000.0 * TT2WAY + 0.5
      IF (ITPAD .LE. 0) ITPAD = 0
C
      ITBEG = IROUND( ITBEG, IDT )
      ITEND = IROUND( ITEND, IDT )
      ITPAD = IROUND( ITPAD, IDT )
C
C     CALCULATE:
C        ISMP1 = INDEX OF FIRST TIME SAMPLE TO USE
C        ISMP2 = INDEX OF LAST  TIME SAMPLE TO USE
C        NTOFF = NUMBER OF ZERO SAMPLES BEFORE FIRST SAMPLE
C        NTPAD = NUMBER OF ZERO SAMPLES AFTER LAST SAMPLE (PAD)
C
C        NOTE: FIRST SAMPLE IN TRACE CORRESPONDS TO TIME = 0
C
      ISMP2 = ITEND / IDT + 1
      NTPAD = ITPAD / IDT
      IF (ITBEG .GE. 0) THEN
         NTOFF = 0
         ISMP1 = ITBEG / IDT + 1
      ELSE
         NTOFF = - ITBEG / IDT
         ISMP1 = 1
      ENDIF
C
C     IF ISMP2 > NUMBER OF SAMPLES, ADJUST ISMP2 AND NTPAD
C
      IF (ISMP2 .GT. NSMP) THEN
         NTPAD = NTPAD + ISMP2 - NSMP
         ISMP2 = NSMP
      ENDIF
C
C     CALCULATE NT = NUMBER OF TIME SAMPLES FOR FFT, PAD AS NEEDED
C
      MT    = ISMP2 - ISMP1 + 1 + NTOFF + NTPAD
      NT    = NRFFT5( MT )
      DT    = 0.001 * FLOAT( IDT )
      NTPAD = NTPAD + NT - MT
C
C     RECALCULATE ITPAD AND ITEND
C
      ITPAD = IDT * NTPAD
      ITEND = IDT * (ISMP2 - 1)
C
      WRITE (LUPRT, 911)
      WRITE (LUPRT, 921) ITEND, ITPAD, ITBEG, IDT
C
      IF (ITEND .LE. ITBEG) GO TO 820
      IF (NT    .GT. MAXNT) GO TO 830
C
C=======================================================================
C                     PROCESS FREQUENCY PARAMETERS
C=======================================================================
C
      IF (F2 .LE. 0) F2 = F1
      IF (F3 .LE. 0) F3 = F4
      WRITE (LUPRT, 912)
      WRITE (LUPRT, 922) F1, F2, F3, F4
C
      CALL GTFLTR( F1, F2, F3, F4, NT, DT, MAXNW,
     &             NW, IW1, OMEGA, FILTR, IERR )
      IF (IERR .EQ. 3) GO TO 830
      IF (IERR .NE. 0) GO TO 820
C
      FMAX = OMEGA(NW) / (2.0 * PI)
C
C=======================================================================
C                       PROCESS DEPTH PARAMETERS
C=======================================================================
C
      IF (STRCH .NE. 0.0) THEN
         ZMAX = ZMAX * STRCH
         DZ   = DZ   * STRCH
      ENDIF
C
      WRITE (LUPRT, 913)
      WRITE (LUPRT, 923) ZMAX, DZ
C
      IF (ZMAX .LE. DZ .OR. DZ .LE. 0.0) GO TO 820
C
      NZ = IFIX( ZMAX / DZ + 1.01)
C
      IF (NZ .GT. MAXNZ) GO TO 830
C
C=======================================================================
C                       PROCESS WIDTH PARAMETERS
C=======================================================================
C
      IF (IREC1  .LE. 0) IREC1  = 1
      IF (IREC2  .LE. 0 .OR. IREC2 .GT. NREC) IREC2  = NREC
      IF (INCREC .LE. 0) INCREC = 1
      IF (ITRC1  .LE. 0) ITRC1  = 1
      IF (ITRC2  .LE. 0) ITRC2  = NTRC
C
      NREC2 = (IREC2 - IREC1) / INCREC + 1
      IREC2 = IREC1 + (NREC2 - 1) * INCREC
C
      WRITE (LUPRT, 914)
      WRITE (LUPRT, 924) DX, IREC1, IREC2, INCREC, ITRC1, ITRC2
C
      IF (IREC1 .GT. IREC2 .OR. ITRC1 .GT. ITRC2) GO TO 820
C
C     CALCULATE REMAINING WIDTH PARAMETERS
C     NOTE: NK IS FORCED TO BE EVEN BY USING NRFFT5 INSTEAD OF NCFFT5
C     BECAUSE THERE MAY STILL BE CODE LEFT FROM THE RADIX 2 FFT DAYS
C     THAT EXPECTS NK TO BE EVEN.
C
      NX    = ITRC2 - ITRC1 + 1
      NK    = NRFFT5( NX + NXPAD)
      DK    = 2.0 * PI / (NK * DX)
      NXPAD = NK - NX
C
      IF (NX .GT. MAXNX .OR. NK .GT. MAXNK) GO TO 830
C
      KBIAS = (NK + 1) / 2
      DO 120 I = 1, NK
	 K = I - KBIAS
         AK(I) = K * DK
  120 CONTINUE
C
C=======================================================================
C                       PROCESS MODEL PARAMETERS
C=======================================================================
C
      WRITE (LUPRT, 915)
      WRITE (LUPRT, 925) VELREF, BWIDTH, NXPAD
C
      IF (VELREF .LE. 0.0 .OR. BWIDTH .LE. 0.0) GO TO 820
C
C=======================================================================
C                              READ ANGLES
C=======================================================================
C
      READ (LUCRD, 900) CARD
      CALL RDANGL (LUCRD, LUPRT, LUPRT, NREC, KREC, THETA, IERR)
      IF (IERR .NE. 0) GO TO 840
C
      IERR = 0
      RETURN
C
C=======================================================================
C                              ERROR EXITS
C=======================================================================
C
  810 CONTINUE
      WRITE (LUPRT, 991)
      IERR = 4004
      RETURN
C
  820 CONTINUE
      WRITE (LUPRT, 992)
      IERR = 4005
      RETURN
C
  830 CONTINUE
      WRITE (LUPRT, 993)
      IERR = 4003
      RETURN
C
  840 CONTINUE
      IERR = 4400 + IABS( IERR )
      RETURN
C
      END
C***********************************************************************
C NAME: PWDUMP                                                         *
C***********************************************************************
C
C  PURPOSE:
C      PWDUMP WRITES SELECTED JOB PARAMETERS TO LUPRT.
C
C-----------------------------------------------------------------------
C
      SUBROUTINE PWDUMP( IDT0, IDT, NREC, NREC2, IREC1, IREC2, INCREC,
     &                   NTRC, ITRC1, ITRC2, NSMP, ISMP1, ISMP2,
     &                   NT, NTOFF, NW, IW1, NX, NK, NZ, NZT, NZSEG,
     &                   DK, DZ, FMAX, BWIDTH, VELREF, STRCH, TT2WAY,
     &                   THETA, OMEGA, FILTR, AK,
     &                   IZSNZ, ZSDZ, ZSSLOR, ZSSLOI )
C
#include <pwmvzn.h>
C
C  SUBROUTINE ARGUMENTS
C
      INTEGER IDT0, IDT, NREC, NREC2, IREC1, IREC2, INCREC,
     &        NTRC, ITRC1, ITRC2, NSMP, ISMP1, ISMP2, NT, NTOFF,
     &        NW, IW1, NX, NK, NZ, NZT, NZSEG, IZSNZ(*)
      REAL    DK, DZ, FMAX, BWIDTH, VELREF, STRCH, TT2WAY, THETA(*),
     &        OMEGA(*), FILTR(*), AK(*), ZSDZ(*), ZSSLOR(*), ZSSLOI(*)
C
C  LOCAL VARIABLES
C
      INTEGER I
      REAL    VELI, VELR
C
C-----------------------------------------------------------------------
C
  900 FORMAT (/' ', 'JOB PARAMETERS:')
  901 FORMAT (/' ', 'NUMBER OF INPUT RECORDS     =', I5,
     2        /' ', 'NUMBER OF TRACES PER RECORD =', I5,
     3        /' ', 'NUMBER OF SAMPLES PER TRACE =', I5,
     4        /' ', 'FIRST RECORD                =', I5,
     5        /' ', 'LAST  RECORD                =', I5,
     6        /' ', 'RECORD INCREMENT            =', I5,
     7        /' ', 'NUMBER OF OUTPUT RECORDS    =', I5,
     8        /' ', 'START TRACE IN EACH RECORD  =', I5,
     9        /' ', 'END   TRACE IN EACH RECORD  =', I5,
     &        /' ', 'INPUT  SAMPLE RATE (MS)     =', I5,
     1        /' ', 'OUTPUT SAMPLE RATE (MS)     =', I5)
  902 FORMAT (/' ', 'NUMBER OF T-S               =', I5,
     2        /' ', 'NUMBER OF W-S               =', I5,
     3        /' ', 'NUMBER OF X-S               =', I5,
     4        /' ', 'NUMBER OF K-S, DELTA K      =', I5, E14.5,
     5        /' ', 'NUMBER OF Z-S, DELTA Z      =', I5, E14.5)
  903 FORMAT (/' ', 'INDEX OF FIRST SAMPLE       =', I5,
     2        /' ', 'INDEX OF LAST  SAMPLE       =', I5,
     3        /' ', 'LEADING  ZERO PAD           =', I5)
  904 FORMAT (/' ', 'INDEX OF FIRST OMEGA        =', I5)
  905 FORMAT (/' ', 'MAXIMUM FREQUENCY           =', F9.3,
     2        /' ', 'REFERENCE VELOCITY          =', F9.3,
     3        /' ', 'BEAMWIDTH                   =', F9.3,
     4        /' ', 'VELOCITY STRETCH FACTOR     =', F9.3,
     5        /' ', 'TWO WAY TRAVEL TIME         =', F9.3)
  906 FORMAT (/' ', 'NUMBER OF Z SEGMENTS        =', I5,
     2        /' ', 'TOTAL NUMBER OF Z STEPS     =', I5)
  907 FORMAT (/' ', '    I   NZ        DZ   VEL REF   VEL INC'/)
  908 FORMAT ( ' ', 2I5, 3F10.3)
  909 FORMAT (/' ', '    I     OMEGA    FILTER'/)
  910 FORMAT ( ' ', I5, 2F10.3)
  911 FORMAT (/' ', '    I     THETA'/)
  912 FORMAT ( ' ', I5, F10.3)
C
C-----------------------------------------------------------------------
C
      WRITE (LUPRT, 900)
      WRITE (LUPRT, 901) NREC, NTRC, NSMP, IREC1, IREC2, INCREC, NREC2,
     &                   ITRC1, ITRC2, IDT0, IDT
      WRITE (LUPRT, 902) NT, NW, NX, NK, DK, NZ, DZ
      WRITE (LUPRT, 903) ISMP1, ISMP2, NTOFF
      WRITE (LUPRT, 904) IW1
      WRITE (LUPRT, 905) FMAX, VELREF, BWIDTH, STRCH, TT2WAY
      WRITE (LUPRT, 906) NZSEG, NZT
C
      WRITE (LUPRT, 907)
      DO 210 I = 1, NZSEG
         VELR = 1.0 / ZSSLOR(I)
         VELI = 1.0 / ZSSLOI(I)
         WRITE (LUPRT, 908) I, IZSNZ(I), ZSDZ(I), VELR, VELI
  210 CONTINUE
C
      WRITE (LUPRT, 909)
      DO 220 I = 1, NW
         WRITE (LUPRT, 910) I, OMEGA(I), FILTR(I)
  220 CONTINUE
C
      WRITE (LUPRT, 911)
      DO 230 I = 1, NREC
         WRITE (LUPRT, 912) I, THETA(I)
  230 CONTINUE
C
      RETURN
      END
C*********************************************************************
C NAME: PWMIGR                                                       *
C*********************************************************************
C
C  PURPOSE:
C      PWMIGR DOWNWARD CONTINUES THE WAVE FIELD AND IMAGES.
C
C-----------------------------------------------------------------------
C
      SUBROUTINE PWMIGR( MK, NK, NW, NZSEG, NZT, DK, THETA, BWIDTH,
     &                   VELREF, OMEGA, AK, IZSNZ, ZSDZ, ZSSLOR, ZSSLOI,
     &                   CXFTAB, WORK, PSI, IMAGE, IERR )
C
#include <pwmvzn.h>
C
C  SUBROUTINE ARGUMENTS
C
      INTEGER MK, NK, NW, NZSEG, NZT, IZSNZ(NZSEG), IERR
      REAL    DK, THETA, BWIDTH, VELREF, OMEGA(NW), AK(NK), ZSDZ(NZSEG),
     &        ZSSLOR(NZSEG), ZSSLOI(NZSEG), CXFTAB(*), WORK(*),
     &        IMAGE(*)
      COMPLEX PSI(NK,NW)
C
C  LOCAL VARIABLES
C
      INTEGER IZSEG, JW, MZSEG, KLIM(2)
      REAL    A, A1, A2, BW, P, P1, P2, PABS
C
C-----------------------------------------------------------------------
C
  901 FORMAT (/' ', '**** ERROR: FATAL ERROR REPORTED BY VZDCIP0 ****')
C
C-----------------------------------------------------------------------
C
      CALL VCLR( IMAGE, 1, 2*MK*(NZT+1) )
C
      BW = ABS( BWIDTH ) + 10.0
C
      A  =  THETA       * DEG2RAD
      A1 = (THETA - BW) * DEG2RAD
      A2 = (THETA + BW) * DEG2RAD
C
      IF (A1 .LT. -HALFPI) A1 = -HALFPI
      IF (A2 .LT. -HALFPI) A2 = -HALFPI
      IF (A1 .GT.  HALFPI) A1 =  HALFPI
      IF (A2 .GT.  HALFPI) A2 =  HALFPI
C
      P  = SIN( A  ) / VELREF
      P1 = SIN( A1 ) / VELREF
      P2 = SIN( A2 ) / VELREF
C
C  LIMIT DEPTH OF MIGRATION IF ANGLE BECOMES CRITICAL
C
      PABS = ABS( P )
      DO 110 IZSEG = 1, NZSEG
         IF (PABS .GT. ZSSLOI(IZSEG)) THEN
            MZSEG = IZSEG - 1
            IF (MZSEG .GT. 0) GO TO 200
	    GO TO 800
         ENDIF
  110 CONTINUE
      MZSEG = NZSEG
C
C  LOOP OVER FREQUENCY
C
  200 CONTINUE
      DO 210 JW = 1, NW
         KLIM(1) = NINT( 2.0 * P1 * OMEGA(JW) / DK )
         KLIM(2) = NINT( 2.0 * P2 * OMEGA(JW) / DK )
         CALL VZDCIP0( MK, NK, 1, MZSEG, NZT, OMEGA(JW), AK, P, KLIM,
     &                 IZSNZ, ZSDZ, ZSSLOR, ZSSLOI, CXFTAB, WORK,
     &                 PSI(1,JW), IMAGE, IERR )
C
         IF (IERR .NE. 0) THEN
            WRITE (LUPRT, 901)
            IERR = 5600 + IABS( IERR )
            RETURN
         ENDIF
  210 CONTINUE
C
C  EXIT SUBROUTINE
C
  800 CONTINUE
      IERR = 0
      RETURN
      END
