C***********************************************************************
C                 copyright 2001, Amoco Production Company             *
C                             All Rights Reserved                      *
C                     an affiliate of BP America Inc.                  *
C***********************************************************************
C  ROUTINE:       VTTD                                                 *
C  ROUTINE TYPE:  MAIN                                                 *
C  PURPOSE:  CONVERT A TIME SECTION TO A DEPTH SECTION                 *
C                            -OR-                                      *
C          - CONVERT A DEPTH SECTION TO A TIME SECTION                 *
C  AUTHOR:   Mary Ann Thornton                  ORIGIN DATE: 90/06/11  *
C  ROUTINES CALLED:                                                    *
C      CMDLIN          - GET COMMAND LINE ARGUMENTS                    *
C      LBOPEN          - OPEN TAPE                                     *
C      OPENPR          - OPEN PRINT FILE                               *
C      GAMOCO          - PRINT TORCH AND OVAL                          *
C      ICOPEN  INTEGER - OPEN CARD FILE                                *
C      RTAPE           - READ TAPE                                     *
C      MOVE            - MOVE CHARACTER STRING                         *
C      XMATRX          - BUILD VELOCITY MATRIX                         *
C      HLHPRT          - PRINT & UPDATE HISTORICAL LINEHEADER          *
C      WRTAPE          - WRITE TAPE                                    *
C      CCUINT          - CUBIC INTERPOLATOR                            *
C      LBCLOS          - CLOSE TAPE                                    *
C  FILES:                                                              *
C      LCRD   ( INPUT  SEQUENTIAL ) - INPUT CARD FILE                  *
C      LERR   ( OUTPUT SEQUENTIAL ) - PRINT FILE                       *
C      LLIST  ( OUTPUT SEQUENTIAL ) - PARENT PID LIST                  *
C      LMXC   ( INPUT  SEQUENTIAL ) - MODEL FILE                       *
C  STOP CODES:                                                         *
C      200      ( 1) - PRINT FILE ERRORS, and OUT OF BOUNDS ARRAYS     *
C      50       ( 5) - CARD FILE ERRORS                                *
C      75       ( 5) - TAPEIO ERRORS                                   *
C      100      ( 3) - OTHER ERRORS                                    *
C      500      ( 1) - ERROR IN VELOCITY MATRIX                        *
C      =BLANK=  ( 5) - NO ERRORS                                       *
C       +------------------------------------------------------+       *
C       |             OTHER DOCUMENTATION DETAILS              |       *
C       +------------------------------------------------------+       *
C  ERROR HANDLING:  ???                                                *
C  GENERAL DESCRIPTION:  Original program read mxc file and built a    *
C               Matrix, this version replaced mxc with velocity tape   *
C  REVISED BY:  MARY ANN THORNTON             REVISION DATE: 89/02/20  *
C            -  ADD MAXIMUM TIME AND DEPTH PARAMETERS AND MAKE KSAMP   *
C            -  CORRESPOND                                             *
C  REVISED BY:  MARY ANN THORNTON             REVISION DATE: 89/02/28  *
C            -  MAKE THE INPUT & OUTPUT TABLES THE SAME (LONGEST)      *
C  REVISED BY:  MARY ANN THORNTON             REVISION DATE: 89/03/03  *
C            -  CORRECT INPUT & OUTPUT TABLES ALL WRONG BEFORE         *
C  REVISED BY:  MARY ANN THORNTON             REVISION DATE: 89/03/23  *
C            -  CHECK FOR INVALID DTMS VALUE IN HEADER WORD 13         *
C  REVISED BY:  MARY ANN THORNTON             REVISION DATE: 89/03/23  *
C            -  Moved code to Sun for Maintenance from the Sun         *
C            -  Corrected error found in reading vel.tape lineheader   *
C            -  Changed CALL MOVES to CALL VCLR AND CALL VMOV          *
C  REVISED BY:  CURTIS KRUSE                  REVISION DATE: 91/??/??  *
C               Added 2 checks to make sure not getting samples below  *
C               end of tape                                            *
C  REVISED BY:  Mary Ann Thornton             REVISION DATE: 91/??/??  *
C               Added 2 checks from Curtis's change above to the code  *
C  REVISED BY:  Mary Ann Thornton  V3.0       REVISION DATE: 91/??/??  *
C               removed DX parameter (not necessary) replaced the cubic*
C               interpolator with the linear interpolator.             *
C  REVISED BY:  Mary Ann Thornton  V3.1       REVISION DATE: 92/03/26  *
C               Call openpr with full program name for OS 6.1          *
C  REVISED BY:  Mary Ann Thornton  V3.2       REVISION DATE: 92/05/12  *
C               Rename topen to cmdlin for new sun compiler            *
C  REVISED BY:  Gary Murphy        V3.3       REVISION DATE: 92/06/12  *
C               Added verbos print statements for negative velocities  *
C  REVISED BY:  Mary Ann Thornton  V3.4       REVISION DATE: 92/07/30  *
C               Allowed the program to run when the velocity tape does
C               not match the seismic data precisely. The last velocity
C               trace or the last velocity sample will be used when the
C               velocity tape is not as big in size as the seismic data
C               Corrected error in trace numbers when doing the depth 
C               to time conversion.  Cleaned up the code and tried to
C               make the error messages more meaningful to a user.
C  REVISED BY:  Mary Ann Thornton  V3.5       REVISION DATE: 92/08/25  
C               Added statements to rewind the velocity tape in between
C               input records, so vttd would work on multi-record input
C  REVISED BY:  Mary Ann Thornton  V3.6       REVISION DATE: 92/11/02  
C               Copied vttd from ~mbs to ~usp so vttdusp could be
C               eliminated - recompiled to see if everything works as
C               a usp group member - code will be in mbs and usp for now
C  REVISED BY:  Mary Ann Thornton  V3.7       REVISION DATE: 92/11/09  
C               Added a change to the way the table is calculated for
C               the time to depth conversion to make it more accurate
C               The values will be the average slowness over the 
C               boundaries at the base of the layers.
C               Also increased the maximum sample limit and resampled
C               data on input to 1ms data when doing time to depth
C               conversion. 
C  REVISED BY:  Mary Ann Thornton  V3.8       REVISION DATE: 93/02/25  
C               Remove the call to lbopen after the call to rwd.  The
C               tape does not need to be reopened.  Each time lbopen was
C               called a new lu number was assigned, and after 100
C               calls, the program bombs with too many files open.
C  REVISED BY:  Dan Whitmore       V3.9       REVISION DATE: 93/03/12 
C               Added option to allow input velocity tape to be in time.
C  REVISED BY:  Gary Murphy        V4.0       REVISION DATE: 93/03/12
c               added 3d option to allow multirecord input without
c               rewinding the velocity tape.  added lbclos call after
c               lbopen in velocity tape handling.
c               added velocity tape input option (vt).
C  REVISED BY:  Mary Ann Thornton  V4.1       REVISION DATE: 93/03/23  
C               removed a call to vmov and replaced with a do loop
C               for moving data in at the end of traces when it is a	
C               velocity tape. Also increased line header size.
C  REVISED BY:  Mary Ann Thornton  V4.2       REVISION DATE: 93/04/02  
C               Changed comments that get printed to describe how to use
C               the program and added check for dt on command line.
C               dt must be entered on command line.
C  REVISED BY:  Mary Ann Thornton  V4.3       REVISION DATE: 93/05/24  
C               Corrected an error in 3d option.  the 'endif' was in the
C               wrong place when checking to see if running in 3d mode
C               When in 3d mode, a call to rtape at the end of each record    
C               loop read a velocity trace off the next record, when this
C               call to rtape was really made for reading the line 
C               header after rewinding the velocity tape before
C               going to the next data record.
C  REVISED BY:  Mary Ann Thornton  V4.4       REVISION DATE: 93/06/10  
C               increased the mxsam to 12000 to allow more space for the
C               work arrays when requesting a large amount of time on
C               output, and changed the error message to be more 
C               informative to a user.
C  REVISED BY:  Gary Murphy        V4.5       REVISION DATE: 93/07/13  
C               Put in crude fix to make sure output velocity tape     
C               never contains zeros.                                  
C  REVISED BY:  Mary Ann Thornton  V4.6       REVISION DATE: 93/07/21  
C               Remove fix of version 4.5.                             
C               Change code when doing depth to time conversion on a
C               seismic dataset to extend the trace with zeroes rather
C               than extend the trace with the last data sample BEFORE
C               the conversion is done.
C               Also, added many more comments to try to make code more
C               understandable, and added checks where there was a 
C               chance for the arrays to go out of bounds.
C               Changed cmdlin to make the comments better, which
C               requires a change be made to vttd.pat to match comments
C               Changed calculation of nzvt:(added +1.0 to calculation)
C  REVISED BY:  Mary Ann Thornton  V4.7       REVISION DATE: 93/08/26  
C               Changed to issue a warning message when the velocity   
C               dataset is shorter or longer than the dataset to be    
C               converted. Also made a change to allow 1 trace records 
C               to be handled correctly whether or not the 3d option is
C               selected.                                              
C  REVISED BY:  Mary Ann Thornton  V4.8       REVISION DATE: 93/10/25  
C               Changed to include the command line arguments in L.header
C  REVISED BY:  Mary Ann Thornton  V4.9       REVISION DATE: 94/01/04  
C               Added dx1000 to the output line header and dz1000 when
C               output is in depth.                                    
C               A new command line argument was added for dx.  If it is
C               missing, pgm will look for dx1000 in input velocity field,
C               If that is missing, too, pgm will look for dx1000 in the
C               input dataset to be converted.  If that is blank, then
C               proceed with nothing put in dx1000 location.  Also if 3d
C               option is requested, 'ILClIn' was placed in line header.
C               The format statements were revised for easier and more
C               informative reading. The dx and dz values among the inputs
C               and the output will be compared, and the user notified
C               of mismatches, but program will continue execution.
C  REVISED BY:  Mary Ann Thornton  V5.0       REVISION DATE: 94/03/22  
C               Switch the calls ccuint and fliint for seismic and velocity
C               input.
C  REVISED BY:  Mary Ann Thornton  V5.1       REVISION DATE: 94/03/24  
C               Remove the limit on traces in; there is no need for it.    
C  REVISED BY:  Mary Ann Thornton  V5.2       REVISION DATE: 94/11/17  
C               Add a -lhdt parameter to force the program to use the  
C               sample rate from the input dataset for -dt
C  REVISED BY:  Paul Garrosino     V5.3       REVISION DATE: 96/03/05 
C               Allow velocity tape to be a pipe (IKP ability)
C               He established a symbolic link vttdusp -> vttd
C  REVISED BY:  Mary Ann Thornton  V5.4       REVISION DATE: 96/04/22  
C               If 3d, and velocity field has more traces than seismic,
C               read past the last traces in the velocity field record
C               before beginning the next record.
C  REVISED BY:  Gary Murphy        V5.5       REVISION DATE: 97/01/13 
C               Bumped limit for number of samples.
C  REVISED BY:  Jerry Ehlers       V6.0       REVISION DATE: 99/07/28
C               Use sinc interpolation & total rewrite
C  REVISED BY:  Jerry Ehlers       V6.1       REVISION DATE: 00/02/16
C               Add -4d and -qc options
C  REVISED BY:  Jerry Ehlers       V6.1       REVISION DATE: 00/05/19
C               Make running sums double precision
C***********************************************************************
#include <save_defs.h>
#include <f77/iounit.h>
#include <f77/lhdrsz.h>
#include <f77/sisdef.h>

      
      parameter (mxsam=32000,lhead=szlnhd)
      parameter (lcrd=25,llist=27,lmxc=28)
      parameter (mxxxx=mxsam+itrwrd,nmax=8)
      parameter (init=1)
c     
#include <f77/pid.h>

      real*8    sum
      dimension ihead(lhead),ivhd(lhead)
      integer*2 itr(szlnhd)
      dimension w(mxsam,-nmax:nmax),iz(mxsam),zz(4*mxsam),tmp(mxsam)
      dimension tablz(mxsam),tabl1(mxsam),tabl2(mxsam)
      dimension rxx(mxxxx),data(mxsam)
      dimension vel(mxxxx),veld(mxsam)
c     
      integer*2 irx(lntrhd),ivx(lntrhd)
      integer pipe
c     
      logical verbos,threed,fourd,vtin,lhdt
      character*1 parr(66)
      character*2 pname
      character*4 version
      character*4 ppname
      character*128 ntap,otap,nvel
c     
      equivalence (ppname,pname)
      equivalence (ihead(1),itr(1))
      equivalence (rxx(1),irx(1)),(rxx(ithwp1),data(1))
      equivalence (vel(1),ivx(1)),(vel(ithwp1),veld(1))
      
      
      data version/' 6.1'/
      data ppname/'VTTD'/
      data parr/
     1     ' ',' ',' ',' ',' ',' ',' ',' ',' ',' ',
     2     'V','E','L','O','C','I','T','Y',' ','T',
     3     'A','P','E',' ','T','I','M','E','/','D',
     4     'E','P','T','H',' ','C','O','N','V','E',
     5     'R','S','I','O','N',' ',' ',' ',' ',' ',
     6     ' ',' ',' ',' ',' ',' ',' ',' ',' ',' ',
     7     ' ',' ',' ',' ',' ',' '/
      data pipe/3/

c     
      icode= 0
      lpprt= 0
      ierr = 0
      call cmdlin(ntap,otap,nvel,iflag,dx,dz,dtms,zmax,tmax,itype,
     *     ipipi,ipipo,LER,threed,fourd,vtin,lhdt,nsinc,nqc,verbos)

cccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc
c     open printfile
cccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc
      call openpr(llist,LERR,ppname,jerr)
      if (jerr.ne.0) then
         write(LER,*) 'unable to open print files.'
         stop 200
      endif
c#include <f77/open.h>
      nlin=1
      call gamoco(parr,nlin,LERR)
cccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc

      if (ipipi.eq.0) then
         call lbopen(lu1,ntap,'r')
      else
         lu1=0
      endif

      if (ipipo.eq.0) then
         call lbopen(lu2,otap,'w')
      else
         lu2=1 
      endif
c     
c     open velocity tape
c
      if (nvel.ne.' ') then
         call lbopen(lu3,nvel,'r')
      elseif (threed.or.fourd) then
         call sisfdfit (lu3,pipe)
      else
         write(LERR,*)' '
         write(LERR,*)' VTTD: You cannot pipe the velocity data'
         write(LERR,*)'       into this routine unless using the'
         write(LERR,*)'       -3d or -4d option.  No backing up '
         write(LERR,*)'       on pipes allowed.'
         write(LERR,*)' FATAL'
         write(LER,*)' '
         write(LER,*)' VTTD: You cannot pipe the velocity data'
         write(LER,*)'       into this routine unless using the'
         write(LER,*)'       -3d or -4d option.  No backing up '
         write(LER,*)'       on pipes allowed.'
         write(LER,*)' FATAL'
         stop
      endif

      write(LERR,8001) ntap,otap,nvel
 8001 format(' INPUT DATASET = ',/,A128,/,' OUTPUT DATASET = '/,A128/,
     *     ' VELOCITY DATASET ='/,A128)
cccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc
c     check for command line arguments
ccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc
      if ((iflag.eq.0).and.(dz.eq.0.0)
     *     .and.(zmax.le.0.0).and.(tmax.le.0.0)) then
         write(LERR,*)'  all parameters have been defaulted'
         write(LERR,*)'  JOB TERMINATED...................'
         stop 50
      elseif (dz.le.0.0) then
         write(LERR,*)' dz must be > 0.0 - JOB TERMINATED '
         stop 100
      endif

      if (nsinc.gt.nmax) then
         write(LERR,*)' nsn must be <= 8 - JOB TERMINATED '
         stop 110
      endif

      if (fourd) then
         if (threed) then
            write(LERR,*)
     *        ' Cannot have both -3d and -4d options - JOB TERMINATED '
            stop 110
         endif
         write(LER,*) "running in 4d mode"
         write (LERR,*) "running in 4d mode"
      elseif (threed) then
         write(LER,*) "running in 3d mode"
         write (LERR,*) "running in 3d mode"
      endif

      iflag=max(1,min(2,iflag))
      itype=max(1,min(2,itype))
ccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc
c     read the lineheader of the tape to be converted
ccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc
      jeof = 0
      call rtape(lu1,ihead,jeof)
      if (jeof.eq.0) goto 1000
      call saver(ihead,'NumSmp',KSAMP,LINHED)
      call saver(ihead,'NumTrc',ltr,LINHED)
      call saver(ihead,'SmpInt',ISI,LINHED)
      call saver(ihead,'NumRec',NREC,LINHED)
      call saver(ihead,'Format',IFORM,LINHED)
      call saver(ihead,'Dx1000',idx,LINHED)
      call saver(ihead,'Dz1000',idz,LINHED)
c     dx1 = trace spacing of input tape if available
c     dz1 = depth spacing of input tape if available
      dx1 = idx/1000.0
      dz1 = idz/1000.0
      len=4
      call hlhprt(ihead,jeof,ppname,len,LERR)
      if (ksamp.gt.mxsam) then
         write(LERR,*)' ERROR*****max # samples allowed is ',mxsam
         stop 100
      endif
ccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc
c     read the velocity tape lineheader
ccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc
      jeof2 = 0
      call rtape(lu3,ivhd,jeof2)
      if (jeof2.eq.0) goto 1020
      call saver(ivhd,'NumSmp',KSAMPV,LINHED)
      call saver(ivhd,'NumTrc',ltrV,LINHED)
      call saver(ivhd,'SmpInt',ISIV,LINHED)
      call saver(ivhd,'NumRec',NRECV,LINHED)
      call saver(ivhd,'Format',IFORMV,LINHED)
      call saver(ivhd,'Dx1000',idx,LINHED)
      call saver(ivhd,'Dz1000',idz,LINHED)
      if (threed.and.nrecv.le.1) then
         write(LERR,*)'******NOTICE******'
         write(LERR,*)' You have asked for the -3D option, but have',
     *                ' only one record on your velocity field'
         write(LER,*)'******NOTICE******'
         write(LER,*)' You have asked for the -3D option, but have',
     *           ' only one record on your velocity field'
      elseif (fourd.and.nrecv*ltrv.ne.1.and.nrecv*ltrv.ne.nrec) then
         write(LERR,*)'******NOTICE******'
         write(LERR,*)' You have asked for the -4D option, but have'
         write(LERR,*)' more than 1 velocity trace but not equal to'
         write(LERR,*)' nrec of seismic data'
         write(LER,*)'******NOTICE******'
         write(LER,*)' You have asked for the -4D option, but have'
         write(LER,*)' more than 1 velocity trace but not equal to'
         write(LER,*)' nrec of seismic data'
      endif
c     dx2 = trace spacing of the velocity field to use if available
c     dz2 = depth spacing of the velocity field to use if available
      dx2 = idx/1000.0
      dz2 = idz/1000.0
ccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc
c     check the parameters
c     if 1-trace records are coming in, let ntr=nrec and output only 1
c     record with ntr traces
ccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc
c     dataset to convert:
      ntr =ltr
      mrec=nrec
      if (ltr.eq.1.and..not.fourd) then
         ntr =nrec
         mrec=1
         call savew(ihead,'NumTrc',ntr,linhed)
         call savew(ihead,'NumRec',mrec,linhed)
      endif
c     check dx1000 and dz1000 and put in line header (and ILClIn if 3d)
      if (dx1.ne.dx2) then
         write(LERR,*)'trace spacing on the input tapes do not match'
         write(LER,*) 'trace spacing on the input tapes do not match'
      endif
      if (dx.le.0.0) dx = dx2
      if (dx.le.0.0) dx = dx1
      if (dx.gt.0.0) then
         idx = dx * 1000.0
         call savew(ihead,'Dx1000',idx,linhed)
         if (threed) then
            call savew(ihead,'ILClIn',dx,linhed)
         endif
         if (dx.ne.dx1) then
            write(LERR,*)'******NOTICE******'
            write(LERR,*)'The trace spacing of the output does not',
     *           ' match the trace spacing of the input tape.'
            write(LER,*)'******NOTICE******'
            write(LER,*)'The trace spacing of the output does not',
     *           ' match the trace spacing of the input dataset.'
         endif
         if (dx.ne.dx2) then
            write(LERR,*)'******NOTICE******'
            write(LERR,*)'The trace spacing of the output does not',
     *           ' match the trace spacing of the velocity field.'
            write(LER,*)'******NOTICE******'
            write(LER,*)'The trace spacing of the output does not',
     *           ' match the trace spacing of the velocity field.'
         endif
      endif

      if (iflag.eq.1) then
         idz = dz * 1000.0
         call savew(ihead,'Dz1000',idz,LINHED)  
      endif
C
C     dataset containing velocity field:
C
      if (ltrv.gt.1) then
         ntrv=ltrv
      else
         ntrv=max(1,nrecv)
      endif

      if (fourd) then
         ntrv=1
         if (ltrv*nrecv.eq.1.and.nrec.gt.1) then
            write(LER ,*)' ******WARNING******'
            write(LER ,*)' The dataset containing the velocity field'
            write(LER ,*)' has only one which will be used for the'
            write(LER ,*)' entire conversion.'
            write(LERR,*)' ******WARNING******'
            write(LERR,*)' The dataset containing the velocity field'
            write(LERR,*)' has only one which will be used for the'
            write(LERR,*)' entire conversion.'
         endif
      elseif (ntrv.lt.ntr.and.ntrv.eq.1) then
         write(LER ,*)' ******WARNING******'
         write(LER ,*)' The dataset containing the velocity field'
         write(LER ,*)' has only one which will be used for the'
         write(LER ,*)' entire conversion.'
         write(LERR,*)' ******WARNING******'
         write(LERR,*)' The dataset containing the velocity field'
         write(LERR,*)' has only one which will be used for the'
         write(LERR,*)' entire conversion.'
      elseif (ntrv.lt.ntr) then
         write(LER ,*)' ******WARNING******'
         write(LER ,*)' The dataset containing the velocity field'
         write(LER ,*)' contains less traces than the dataset to be'
         write(LER ,*)' converted.  The last trace will be used '
         write(LER ,*)' multiple times.  For more accurate results, use' 
         write(LER ,*)' vtrsiz to resize the velocity field dataset.'
         write(LERR,*)' ******WARNING******'
         write(LERR,*)' The dataset containing the velocity field'
         write(LERR,*)' contains less traces than the dataset to be'
         write(LERR,*)' converted.  The last trace will be used '
         write(LERR,*)' multiple times.  For more accurate results, use'
         write(LERR,*)' vtrsiz to resize the velocity field dataset.'
      elseif (ntrv.gt.ntr) then
         write(LER ,*)' ******NOTICE******'
         write(LER ,*)' The dataset containing the velocity field '
         write(LER ,*)' contains more traces than the dataset to be'
         write(LER ,*)' converted.  Only the first ',ltr,' traces ' 
         write(LER ,*)' will be used.' 
         write(LERR,*)' ******NOTICE******'
         write(LERR,*)' The dataset containing the velocity field '
         write(LERR,*)' contains more traces than the dataset to be'
         write(LERR,*)' converted.  Only the first ',ltr,' traces ' 
         write(LERR,*)' will be used.' 
      endif
      if (ksampv.gt.mxsam) then
         write(LERR,*)' ERROR*****max # samples allowed is ',mxsam
         stop 100
      endif
      
ccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc
c     
c     time to depth conversion  (iflag=1)
c     
ccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc
      if (iflag.eq.1) then
         if (zmax.le.0.0) then
            write(LERR,*)' MAXIMUM DEPTH MUST BE ENTERED'
            stop 100
         endif
         if (dtms.le.0.0) then
            if (lhdt) then
               call saver(ihead,'SmpInt',idtms,linhed)
               if (idtms.le.0) then
                  write(LERR,*)' Delta-t is less than or equal to 0'
                  write(LERR,*)' Delta-t must be entered'
                  stop 100
               else
                  dtms = idtms
               endif
            endif
         endif
         idtms=dtms
         if (dtms.gt.32.0) dtms=dtms/1000.0
         dt=dtms/1000.0
         dtin = dt
         if (idtms.le.0) then
            write(LERR,*)' INVALID DTMS PARAMETER '
            stop 100
         endif
         call savew(ihead,'SmpInt',idtms,linhed)
c
c        if user asks to use more time in than we have, we will use
c        only as much as we have.
c
         if (tmax.le.0.0) tmax=dtms*ksamp
         tm = dtms*ksamp
         tmax=min(tm,tmax)
         itmax=tmax
         if (idtms.gt.32) then
            ksin=tmax/dtms
         else
            ksin=itmax/idtms
         endif
         nzout=zmax/dz+1.0
         nout =nzout
         call savew(ihead,'NumSmp',nzout,linhed)
         if (verbos) then
            write(LERR,*) ' tm,dtms,ksamp=',tm,dtms,ksamp
            write(LERR,*) ' itmax,tmax,idtms,ksin=', itmax,tmax,
     *           idtms,ksin
            write(LERR,*) ' dt= ',dt
         endif
         if (vtin) then
            write(LERR,8011) idtms,dx1,itmax,ksin,dx,dz,zmax,nzout
         else
            write(LERR,8111) idtms,dx1,itmax,ksin,dx,dz,zmax,nzout
         endif
         if (itype.ge.2) then
            write(LERR,8121) dx2,nrecv,ltrv,ksampv
            if (isiv.ne.idtms) then
               write(LERR,*)'WARNING--the delta-t of the velocity ',
     *              'dataset (',isiv,') does not match that of the ',
     *              'seismic dataset (',idtms,')'
            endif
         else
            write(LERR,8131) dx2,dz2,nrecv,ltrv,ksampv
            if (dz.ne.dz2) then
               write(LERR,*)'******NOTICE******'
               write(LERR,*)'The depth spacing of the output does',
     *              ' not match that of the velocity field.'
               write(LER,*)'******NOTICE******'
               write(LER,*)'The depth spacing of the output does',
     *              ' not match that of the velocity field.'
            endif
         endif  
 8011    format(
     *     ' THE INPUT DATASET CONTAINS VELOCITY DATA TO BE CONVERTED',
     *     ' FROM TIME TO DEPTH'/,
     *     '   Input spacing in milliseconds          ', 10X,'=', I15 ,/
     *     '   Input trace spacing in feet/meters     ', 10X,'=',F15.5,/
     *     '   Maximum time of input to use           ', 10X,'=', I15 ,/
     *     '   Number of samples on input to use      ', 10X,'=', I15 ,/
     *     ' THE OUTPUT DATASET CONTAINS VELOCITY DATA IN DEPTH',/
     *     '     Output trace spacing in feet/meters  ', 10x,'=',F15.5,/
     *     '     Output depth spacing in feet/meters  ', 10x,'=',F15.5,/
     *     '     Maximum depth on output              ', 10x,'=',F15.5,/
     *     '     Number of samples on output          ', 10x,'=', I15)
 8111    format(
     *     ' THE INPUT DATASET CONTAINS SEISMIC DATA TO BE CONVERTED',
     *     ' FROM TIME TO DEPTH',/,
     *     '   Input spacing in milliseconds          ', 10X,'=', I15 ,/
     *     '   Input trace spacing in feet/meters     ', 10X,'=',F15.5,/
     *     '   Maximum time of input to use           ', 10X,'=', I15 ,/
     *     '   Number of samples on input to use      ', 10X,'=', I15 ,/
     *     ' THE OUTPUT DATASET CONTAINS SEISMIC DATA IN DEPTH',/
     *     '     Output trace spacing in feet/meters  ', 10x,'=',F15.5,/
     *     '     Output depth spacing in feet/meters  ', 10x,'=',F15.5,/
     *     '     Maximum depth on output              ', 10x,'=',F15.5,/
     *     '     Number of samples on output          ', 10x,'=', I15)
 8121    format(
     *     ' THE INPUT VELOCITY FIELD IS IN TIME',/
     *     '       Trace spacing in feet/meters       ', 10x,'=',F15.5,/
     *     '       Number of records                  ', 10x,'=',I15,/  
     *     '       Number of traces                   ', 10x,'=',I15,/  
     *     '       Number of samples                  ', 10x,'=',I15)  
 8131    format(
     *     ' THE INPUT VELOCITY FIELD IS IN DEPTH',/
     *     '       Trace spacing in feet/meters       ', 10x,'=',F15.5,/
     *     '       Depth spacing in feet/meters       ', 10x,'=',F15.5,/
     *     '       Number of records                  ', 10x,'=',I15,/  
     *     '       Number of traces                   ', 10x,'=',I15,/  
     *     '       Number of samples                  ', 10x,'=',I15)  
c
ccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc
c     
c     depth to time conversion  (iflag=2)
c     
ccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc
c
      else
         if (tmax.le.0.0) then
            write(LERR,*)' MAXIMUM TIME MUST BE ENTERED'
            stop 100
         endif
         itmax=tmax
         if (dz.le.0.0) then
            write(LERR,*)' DEPTH SPACING MUST BE ENTERED'
            stop 100
         endif
         if (dtms.le.0.0) then
            if (lhdt) then
               call saver(ihead,'SmpInt',idtms,linhed)
               if (idtms.le.0) then
                  write(LERR,*)' Delta-t is less than or equal to 0'
                  write(LERR,*)' Delta-t must be entered'
                  stop 100
               else
                  dtms = idtms
               endif
            endif
         endif
         dt=dtms/1000.0
         dtin = dt
         idtms = dtms
         if (idtms.le.0) then
            write(LERR,*)' INVALID DTMS PARAMETER '
            stop 100
         endif
         call savew(ihead,'SmpInt',idtms,linhed)
         if (zmax.le.0.0) then
            zmax=dz*(ksamp-1)
         endif
         ksin = 1.0+zmax/dz
c
c        if user asks to use more samples in than we have, 
c        limit to the number of samples we have.
c
         if (ksin.gt.ksamp) ksin=ksamp
         if (idtms.gt.32) then
            ntout = tmax/dtms+1
         else
            ntout = itmax/idtms+1
         endif
         nout = ntout
         call savew(ihead,'NumSmp',ntout,linhed)
         if (verbos) write(LERR,*)' DT',dt
c
c        check for dz mismatches between velocity(in depth) & input
c
         if (itype.le.1) then
            if (dz1.ne.dz2) then
               write(LERR,*)'******NOTICE******'
               write(LERR,*)'The depth spacing of the input does',
     *              ' not match that of the velocity field.'
               write(LER,*)'******NOTICE******'
               write(LER,*)'The depth spacing of the input does',
     *              ' not match that of the velocity field.'
            endif
         endif
c        check for dz mismatches between inputs and output
         if (dz.ne.dz1.and.dz.ne.dz2) then
            write(LERR,*)'******NOTICE******'
            write(LERR,*)'The depth spacing of the output does',
     *           ' not match that of the input.'
            write(LER,*)'******NOTICE******'
            write(LER,*)'The depth spacing of the output does',
     *           ' not match that of the input.'
         endif
         if (vtin) then
            write(LERR,8012) dz,dx1,zmax,ksin,dx,idtms,itmax,ntout
         else
            write(LERR,8112) dz,dx1,zmax,ksin,dx,idtms,itmax,ntout
         endif
         write(LERR,8036) mrec,ntr
         if (itype.ge.2) then
            write(LERR,8121) dx2,nrecv,ltrv,ksampv
         else
            write(LERR,8131) dx2,dz2,nrecv,ltrv,ksampv
         endif 
 8012    format(
     *     ' THE INPUT DATASET CONTAINS VELOCITY DATA TO BE CONVERTED',
     *     ' FROM DEPTH TO TIME',/
     *     '   Input depth spacing in feet/meters     ', 10X,'=',F15.5,/
     *     '   Input trace spacing in feet/meters     ', 10X,'=',F15.5,/
     *     '   Maximum depth of input to use          ', 10X,'=',F15.5,/
     *     '   Number of samples on input to use      ', 10X,'=',I15  ,/
     *     ' THE OUTPUT DATASET CONTAINS VELOCITY DATA IN TIME',/
     *     '     Output trace spacing in feet/meters  ', 10x,'=',F15.5,/
     *     '     Output time spacing in milliseconds  ', 10x,'=',I15  ,/
     *     '     Maximum time on output               ', 10x,'=',I15  ,/
     *     '     Number of samples on output          ', 10x,'=',I15)
 8112    format(
     *     ' THE INPUT DATASET CONTAINS SEISMIC DATA TO BE CONVERTED',
     *     ' FROM DEPTH TO TIME',/
     *     '   Input depth spacing in feet/meters     ', 10x,'=',F15.5,/
     *     '   Input trace spacing in feet/meters     ', 10X,'=',F15.5,/
     *     '   Maximum depth of input to use          ', 10x,'=',F15.5,/
     *     '   Number of samples on input to use      ', 10x,'=',I15  ,/
     *     ' THE OUTPUT DATASET CONTAINS SEISMIC DATA IN TIME',/
     *     '     Output trace spacing in feet/meters  ', 10x,'=',F15.5,/
     *     '     Output time spacing in milliseconds  ', 10x,'=',I15  ,/
     *     '     Maximum time on output               ', 10x,'=',I15  ,/
     *     '     Number of samples on output          ', 10x,'=',I15)
 8036    format(
     *     '     Number of records on output          ', 10x,'=', I15,/,
     *     '     Number of traces per record on output', 10x,'=', I15)
      endif
c
c     put command line into the historical part of line header
c
      call savhlh(itr,jeof,lbyout)
      call wrtape(lu2,ihead,lbyout)
c      
ccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc
c
c     start processing dataset                                       
c      
c     build input and output time tables for resampling input data
c      
ccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc
c      
c     time to depth conversion (iflag=1)
c     depth to time conversion (iflag=2)
c     determine amount of oversampling desired
c     build tables for interpolation
c      
      if (iflag.eq.1) then
         dold=dt
         if (vtin) then
            newksin=ksin
         else
            dt=min(0.001,dt/2.0)
            newksin=tmax/(1000.0*dt)
         endif
         dnew=dt
         do k=1,mxsam
            tabl2(k) = (k-1)*dz
            tablz(k) = (k-1)*dz
         enddo
      else
         dold=dz
         if (vtin) then
            newksin=ksin
         else
            dz=dz/2.0
            newksin=1.0+zmax/dz
         endif
         dnew=dz
         do k=1,mxsam
            tabl2(k) = (k-1)*dt
            tablz(k) = (k-1)*dz
         enddo
      endif
c     
c     are the arrays long enough to handle this?  (mxsam)
c     
      if (newksin.gt.mxsam) then
         write(LERR,*) 'Check Point 1: newksin',newksin,' >',mxsam
         if (iflag.eq.1) then
            write(LERR,*) '   tmax,dt,dz=',tmax,dt,dz
         else
            write(LERR,*) '   zmax,dt,dz=',zmax,dt,dz
         endif
         goto 2000
      endif
c.......................................................................
c     loop over input records
c.......................................................................
      ncount=0
      do mr=1,mrec
         if (verbos) write(LERR,*) '  PROCESSING RECORD ',mr
c......................................................................
c        loop over input traces
c.......................................................................
         do l=1,ntr
c......................................................................
c           read next input trace.
c.......................................................................
            jeof=0
            call rtape(lu1,rxx,jeof)
            if (jeof.eq.0) goto 1500
            call saver(irx,'DphInd',idi,TRCHED)
            call saver(irx,'LinInd',ili,TRCHED)
c.......................................................................
c           read next velocity trace
c
c           if 4d: turn off reading anymore if only 1 trace (ntrv=0)
c.......................................................................
            if (l.le.ntrv.and.mr.le.nrecv) then
               if (fourd.and.ltr*nrecv.eq.1) ntrv=0
               jeofv = 0
               call rtape(lu3,vel,jeofv)
               if (jeofv.eq.0) goto 1520
               call saver(ivx,'DphInd',idiv,TRCHED)
               call saver(ivx,'LinInd',iliv,TRCHED)
c     
               if (itype.ge.2) then
c     
c                 convert input velocity from time to depth
c     
                  sum=0.0
                  do k=1,ksampv
                     tabl1(k)=sum
                     sum=sum+veld(k)*dtin/2.0
                  enddo 
                  nzvt = tabl1(ksampv)/dz + 1.0
                  if (nzvt.gt.mxsam) then
                     write(LERR,*) 'Check Point 2: nzvt',nzvt,' >',mxsam
                     write(LERR,*) '   dtin,dz=',dtin,dz
                     goto 2000
                  endif
c     
                  call fliint(tabl1,veld,ksampv,tablz,tmp,nzvt,
     *                 iz,zz,init)
                  call vmov(tmp,1,veld,1,nzvt)
c     
c                 extended the velocity
c     
                  do k=nzvt+1,mxsam
                     veld(k) = veld(nzvt)
                  enddo
               else
c     
c                 interpolate velocity to oversample delta, 
c                 extended last sample
c     
                  if (dz2.ne.dz) then
                     do k=1,ksampv
                        tabl1(k)=(k-1)*dz2
                     enddo
                     nzvt = tabl1(ksampv)/dz + 1.0
                     if (nzvt.gt.mxsam) then
                        write(LERR,*) 'Check Point 3: nzvt',nzvt,' >',
     *                       mxsam
                        write(LERR,*) '   dz,dz2=',dz,dz2
                        goto 2000
                     endif
c     
                     call fliint(tabl1,veld,ksampv,tablz,tmp,nzvt,
     *                    iz,zz,init)
                     call vmov(tmp,1,veld,1,nzvt)
                  else
                     nzvt=ksampv
                  endif
                  do k=nzvt+1,mxsam
                     veld(k) = veld(nzvt)
                  enddo
               endif
c     
c              build input table (tabl1) for converting input
c              by calculating dz's in terms of dt's
c              or by calculating dt's in terms of dz's
c     
               k=1
               sum=0.0
               tabl1(k)=sum
               sold=1.0/veld(1)
               if (iflag.eq.1) then
                  do while(tabl1(k).lt.zmax.and.k.lt.mxsam)
                     k=k+1
                     res=(tabl1(k-1)+0.5*dt/sold)/dz+1.0
                     ind=res
                     res=res-ind
                     if (ind.ge.mxsam) then
                        write(LERR,*) 'Check Point 4: ind',ind,' >',
     *                       mxsam
                        write(LERR,*) '   dt,dz=',dt,dz
                        goto 2000
                     endif
c     
c                    check divisor for zero before doing the divide
c     
                     if (veld(ind).le.0.0.or.veld(ind+1).le.0.0) then
                        write(LER,*) 'velocity < zero on trace ',l,
     *                       ' sample',ind,' record ',mr
                        write(LER,*)'Job Terminated'
                        write(LERR,*) 'velocity < zero on trace ',l,
     *                       ' sample',ind,' record ',mr
                        write(LERR,*) 'Job terminated'
                        icode = 200
                        goto 5000
                     endif
                     snew=(1-res)/veld(ind)+res/veld(ind+1)
                     sum=sum+0.5*dt*(1.0/sold+1.0/snew)/2.0
                     sold=snew
                     tabl1(k)=sum
                  enddo
                  len=k
                  ltabl2=1.0+tabl1(len)/dz
               else
                  do while(tabl1(k).lt.tmax/1000.0.and.k.lt.mxsam)
                     k=k+1
c     
c                    check divisor for 0 value before dividing
c     
                     if (veld(k).le.0.0) then
                        write(LER,*)'velocity < zero on trace',l,
     *                       ' sample ',k,' record ',mr
                        write(LER,*)'job terminated'
                        write(LERR,*) 'velocity < zero on trace ',l,
     *                       ' sample ',k,' record ',mr
                        write(LERR,*) 'job terminated'
                        icode = 200
                        goto 5000
                     endif
                     snew=1.0/veld(k)
                     sum=sum+2.0*dz*(sold+snew)/2.0
                     sold=snew
                     tabl1(k)=sum
                  enddo
                  len = k
                  ltabl2 = 1.0+tabl1(len)/dt
               endif

               if (ltabl2.gt.mxsam) then
                  write(LERR,*) 'Check Point 5: ltabl2',ltabl2,' >',
     *                 mxsam
                  write(LERR,*) '   dt,dz=',dt,dz
                  goto 2000
               endif
            endif
c.......................................................................
c  
c           Process Data
c
c           1) oversample input trace with sinc function if seismic
c           2) extend trace to "len" samples
c           3) resample with liner for non-seismic & cubic for seismic
c     
c           each output trace is different in length depending upon 
c           the velocities at that x location.
c.......................................................................
            if (nqc.gt.0) then
               if (mod(ncount,nqc).eq.0) then
                  write(LERR,8200) l,ili,idi,iliv,idiv
 8200             format(' QC: trc=',I5,'   Seismic:',2I8,
     *                '   Velocity', 2I8)
               endif
               ncount=ncount+1
            endif
c
            if (vtin) then
               call vmov(data,1,tmp,1,ksin)
               do i=ksin+1,len
                  tmp(i) = tmp(ksin)
               enddo
               call fliint(tabl1,tmp,len,tabl2,data,ltabl2,iz,zz,init)
            else
               call oversamp(data,ksin,dold,tmp,newksin,dnew,nsinc,
     *              iz,zz,w,.false.)
               do i=newksin+1,len
                  tmp(i) = 0.0
               enddo
               call ccuint(tabl1,tmp,len,tabl2,data,ltabl2,iz,zz,init)
            endif
c.......................................................................
c   
c           Write Output
c
c           If trace is shorter than nzout, extend it to nzout.  
c           If converted trace is seismic data, zero extend it
c           else extend the last velocity value
c.......................................................................
            if (ltabl2.lt.nout) then
               lword = nout-ltabl2
               if (vtin) then
                  do kk = 1,lword
                     data(ltabl2+kk) = data(ltabl2)
                  enddo
               else
                  call vclr(data(ltabl2+1),1,lword)
               endif
            endif
            irx(106) = mr
            irx(107) = l
            jbytes = nout*szsmpd + sztrhd
            call wrtape(lu2,rxx,jbytes)
            if (jbytes.eq.0) goto 1600
         enddo
c.......................................................................
c  
c        Reposition Velocity iIput
c
c     
c        if 2d: rewind velocity input if more than one seismic record
c        if 3d: position velocity to beginning of next record
c        if 4d: simply continue on
c.......................................................................
         if (threed) then
            do mmm=ntr+1,ntrv
               jeofv= 0
               call rtape(lu3,vel,jeofv)
               if (jeofv.eq.0) goto 200
            enddo
 200        continue
         elseif (.not.fourd) then
            call rwd(lu3)
            jeof2 = 0
            call rtape(lu3,ivhd,jeof2)
            if (jeof2.eq.0) goto 1020
         endif
      enddo
c.......................................................................
c     finished
c.......................................................................
      write(LERR,8451) mrec
      write(LER,8451) mrec
 8451 format(i5,' records processed -- job complete')
      icode = 0
      goto 5000
cccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc
c     error handling                               
c     line header errors                             
cccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc
 1000 write(LERR,1010)
 1010 format(2x,'ERROR reading line header from input data')
      call lbclos(lu1)
      call lbclos(lu2)
      stop 75
c
 1020 write(LERR,1030) 
 1030 format(2x,'ERROR reading line header from velocity field')
      call lbclos(lu1)
      call lbclos(lu2)
      call lbclos(lu3)
      stop 75
cccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc
c     tapeio errors
cccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc
 1500 write(LERR,1510)mr,l
 1510 format(' ERROR reading record',i5,' trace',i5,' input data')
      icode = 75
      goto 5000
c
 1520 write(LERR,1530)mr,l
 1530 format(' ERROR reading record',i5,' trace',i5,' velocity field')
      icode = 75
      goto 5000
c
 1600 write(LERR,1610)mr,l
 1610 format(' ERROR writing output record',i5,' trace',i5)
      icode = 75
      goto 5000
cccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc
c     data longer than array limits of program
cccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc
 2000 write(LER,*)' Internal arrays are not large enough to accomodate'
      write(LER,*)' the amount of time you have asked for on output.'
      write(LER,*)' You may ask for a shorter amount of time [-t] '
      write(LER,*)' and try again.'
      write(LER,*)' *** Job terminated. ***'
      write(LERR,*)' Internal arrays are not large enough to accomodate'
      write(LERR,*)' the amount of time you have asked for on output'
      write(LERR,*)' You may ask for a shorter amount of time [-t] '
      write(LERR,*)' and try again.'
      write(LERR,*)' *** Job terminated. ***'
      icode = 200
cccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc
c     end of job                            
cccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc
 5000 call lbclos(lu1)
      call lbclos(lu2)
      call lbclos(lu3)
      if (icode.eq.200) stop 200
      if (icode.eq.75)  stop 75
      end
cccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc
c
c     subroutine cmdlin                        
c      
cccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc
      subroutine cmdlin(ntap,otap,nvel,iflag,dx,dz,dtms,zmax,tmax,itype,
     *     ipipi,ipipo,LER,threed,fourd,vtin,lhdt,nsinc,nqc,verbos)
      integer argis,nqc
      logical help,verbos,threed,fourd,vtin,lhdt
      character*128 ntap,otap,nvel
c     set defaults to no pipes
      ipipi=0
      ipipo=0
      verbos=.false.
      threed=.false.
      vtin  =.false.
      help  = (argis('-h').gt.0).or.(argis( '-?').gt.0)
      if (help) then
         write(LER,*)'      COMMAND LINE ARGUMENTS:'
         write(LER,*)' TIME TO DEPTH CONVERSION OF SEISMIC DATA'
         write(LER,*)' OR VELOCITY DATA USING A VELOCITY FIELD'
         write(LER,*)' '
         write(LER,*)' INPUT '
         write(LER,*)'-N[]   ..INPUT DATASET NAME'
         write(LER,*)'-O[]   ..OUTPUT DATASET NAME'
         write(LER,*)'-VT[]  ..VELOCITY FIELD NAME'
         write(LER,*)'-type[]..VELOCITY FIELD TYPE FLAG'
         write(LER,*)'         (=1: v(z,x), = 2: v(t,x))'
         write(LER,*)'-flag[]..CONVERSION TYPE FLAG       '
         write(LER,*)'         (=1: Time to Depth =2: Depth to Time)'
         write(LER,*)'-vt    ..FLAG INDICATES THE DATA TO BE'
         write(LER,*)'         CONVERTED IS VELOCITY DATA NOT SEISMIC'
         write(LER,*)'          This flag must be present when velocity'
         write(LER,*)'          data is being converted rather than'
         write(LER,*)'          seismic data'
         write(LER,*)'-3d    ..FLAG INDICATES THE DATA IS 3d'
         write(LER,*)'          This flag must be present when a 3d'
         write(LER,*)'          dataset is being converted, and its'
         write(LER,*)'          corresponding velocity dataset is'
         write(LER,*)'          also 3d (multi-record)'
         write(LER,*)'-4d    ..FLAG INDICATES THE DATA IS 4d'
         write(LER,*)'          This flag must be present when a 4d'
         write(LER,*)'          dataset of gathers is being converted,'
         write(LER,*)'          and its corresponding velocity dataset'
         write(LER,*)'          is either a single Vz trace or a Vxyz '
         write(LER,*)'          with ntrc*nrec = nrec of the gathers'
         write(LER,*)'-lhdt  ..If this flag is present on the command'
         write(LER,*)'          line, the program is forced to use'
         write(LER,*)'          the sample rate from the input dataset'
         write(LER,*)'          line header.  However,caution should'
         write(LER,*)'          be used because this value is not'
         write(LER,*)'          always present in the line header'
         write(LER,*)'-dt[]  ..SAMPLE RATE(ms) OF THE TIME DATASETS'
         write(LER,*)'          The coordinates of the datasets '
         write(LER,*)'              should match.'
         write(LER,*)'          NO DEFAULT.'
         write(LER,*)'-t[]   ..MAXIMUM TIME OF TIME DATASETS'
         write(LER,*)'          Depth to Time Conversion : NO DEFAULT'
         write(LER,*)'          Time to Depth Conversion default :'
         write(LER,*)'             Maximum time may be calculated using'
         write(LER,*)'             the -dt from the command line'
         write(LER,*)'             and the number of samples from'
         write(LER,*)'             the input line header'
         write(LER,*)'-dx[]  ..DELTA X (Trc spacing) OF OUTPUT DATASET'
         write(LER,*)'          This value*1000 will be placed'
         write(LER,*)'          in the output line header. DEFAULT is'
         write(LER,*)'          the Dx1000 value from input velocity'
         write(LER,*)'          field line header or from the input'
         write(LER,*)'          dataset to be converted.'
         write(LER,*)'-dz[]  ..DELTA Z OF THE DEPTH DATASETS'
         write(LER,*)'           The coordinates of the datasets '
         write(LER,*)'               should match.'
         write(LER,*)'           NO DEFAULT'
         write(LER,*)'-z[]   ..MAXIMUM DEPTH OF DEPTH DATASETS'
         write(LER,*)'           Time to Depth Conversion : NO DEFAULT'
         write(LER,*)'           Depth to Time Conversion default :'
         write(LER,*)'             Maximum depth may be calculated'
         write(LER,*)'             using the -dz parameter from the'
         write(LER,*)'             command line and the number of'
         write(LER,*)'             samples from the input line header'
         write(LER,*)'-nsn[] ..SIZE OF SINC FUNCTION'
         write(LER,*)'           Seismic data oversampling sinc '
         write(LER,*)'           function length (-nsn:nsn) : default 8'
         write(LER,*)'-qc[]  ..FREQUENCY OF QC DUMPS'
         write(LER,*)'           LIs and DIs will be dumped for the '
         write(LER,*)'           Seismic and velocity trace every '
         write(LER,*)'           nqcth seismic trace (default 1000 '
         write(LER,*)'           if only -qc specified) '
         write(LER,*)'-V    .. VERBOSE PRINTOUT'
         write(LER,*)'USAGE:'
         write(LER,*)'vttd -N[] -O[] -VT[] -[vt] -[3d] -[4d] -type[] '
         write(LER,*)'    -flag[]  -dt[] -t[] -dz[] -z[] -nsn[] -v '
         stop
      endif
      call argstr('-N',NTAP,' ',' ')
      call argstr('-O',OTAP,' ',' ')
      call argstr('-VT',NVEL,' ',' ')
      call argI4 ('-flag',iflag,0,0)
      call argI4 ('-type',itype,0,0)
      call argI4 ('-nsn',nsinc,8,8)
      call argI4 ('-qc',nqc,1000,0)
      call argR4 ('-dx',dx,0.0,0.0)
      call argR4 ('-dz',dz,0.0,0.0)
      call argR4 ('-dt',dtms,0.0,0.0)
      call argR4 ('-z',zmax,0.0,0.0)
      call argR4 ('-t',tmax,0.0,0.0)
      verbos = (argis('-V').gt.0)
      vtin   = (argis('-vt').gt.0)
      threed = (argis('-3d').gt.0)
      fourd  = (argis('-4d').gt.0)
      lhdt   = (argis('-lhdt').gt.0)
c     make the ntap a pipe
      if (ntap.eq.' ') ipipi=1
c     make the otap a pipe
      if (otap.eq.' ') ipipo=1

      return
      end
