C***********************************************************************
C                 copyright 2001, Amoco Production Company             *
C                             All Rights Reserved                      *
C                     an affiliate of BP America Inc.                  *
C***********************************************************************
c*********************************************************************
c
c
c  regsys - corrects multicomponent seismic aquired with various 
c           irregular orientations so that all vectors refer to the
c           same (regularized) coordinate system.  regsys assumes that
c           all orientation information is already encoded correctly in
c           usp header words SrComp, SrXAzm, SrYRot, SrZRot, and 
c                            RcComp, RcXAzm, RcYRot, RcZRot
c  	    If this polarization information is not already in the usp
c           headers, Use usp_sps, sps_aps, and aps_usp to get it in.
c           Use usp_aqp to generate a plotfile for visual checking.
c
c*********************************************************************

#define MAXCOMP	9
#define MAXFILE	3

#define MAXTRLEN	3000

#include <save_defs.h>
#include <f77/iounit.h>
#include <f77/lhdrsz.h>
#include <f77/sisdef.h>

#ifdef SUNSYSTEM
      implicit   none
#endif

      character name*100
      character getname * 100

      integer jerr
      integer argis, nblen, usprtrace, uspwtrace, uspclose
*      integer uspseek

      integer ii
c
c    Unfortunately, Fortran has no graceful mechanism for
c    allocating a two-dimensional array where we don't know
c    at compile time the dimension of the fast (first) argument.
c
c    One approach is to simply hard-wire a maximum trace length.
c    (That's what we do here.)
c
c    There are two ways to allocate memory dynamically:
c    1) Allocate the memory in a top-level routine that does little
c       more than allocate the memory and then pass the arrays down
c       to a subroutine that contains the main body of the user code;
c    2) Use C-style #defines to wrap the 2D array into a 1D array;
c       the 1D array can be allocated in the usual way.
c       For example:
c       #define DATA(A,B)	data((A)+(B)*nsamp)
c       If you do this beware the 72-character line-length limit!
c

      real mctrace(MAXTRLEN, MAXCOMP)

      integer maxcomp, maxfile

      character mode*100
      character clparam*100
      character wclparam*100
      integer dptw
      integer ncomp
      integer ncompmax
      integer complist(2, MAXCOMP)
      integer allcomps
      integer ikpoffs
      integer ikpsock
      integer seekalot
      integer seekstat
      integer incon, ierror

      integer numsmp, mcnumtrc, mcnumrec, mcns, mcrs, pt
      logical verbos
      real dt
      integer lheader(SZLNHD)
      integer lhedlength, trlength, trdataoff
      character infile*100, oufile*100

      integer cinp, dmctrsp, outcon

*************************************************************
*     regsys declarations

      integer ifmt_NumCmp, l_NumCmp, ln_NumCmp, numcmp
      integer ifmt_MCTrSp, l_MCTrSp, ln_MCTrSp, mctrsp
      integer ifmt_MCLE(16), l_MCLE(16), ln_MCLE(16), mcle(16)

      integer ifmt_SrComp, l_SrComp, ln_SrComp
      integer ifmt_RcComp, l_RcComp, ln_RcComp
      integer ifmt_SrXAzm, l_SrXAzm, ln_SrXAzm, srxazm
      integer ifmt_RcXAzm, l_RcXAzm, ln_RcXAzm, rcxazm

*      integer ifmt_SrYRot, l_SrYRot, ln_SrYRot
*      integer ifmt_SrZRot, l_SrZRot, ln_SrZRot
*      integer ifmt_SrcLoc, l_SrcLoc, ln_SrcLoc, srcloc
*      integer ifmt_LinInd, l_LinInd, ln_LinInd, linind
*      integer ifmt_SrPtXC, l_SrPtXC, ln_SrPtXC, srptxc
*      integer ifmt_SrPtYC, l_SrPtYC, ln_SrPtYC, srptyc
*      integer ifmt_RcPtXC, l_RcPtXC, ln_RcPtXC, rcptxc
*      integer ifmt_RcPtYC, l_RcPtYC, ln_RcPtYC, rcptyc
*      integer ifmt_RcYRot, l_RcYRot, ln_RcYRot
*      integer ifmt_RcZRot, l_RcZRot, ln_RcZRot
*      integer ifmt_SoPtNm, l_SoPtNm, ln_SoPtNm, soptnm
*      integer ifmt_RecInd, l_RecInd, ln_RecInd

      character*6    mcmnem
      integer        srcomp, srcmp, kc
      integer        rccomp, rccmp, jc
      integer        isamp
      integer        mctnsr(4,4), icmp, jcmp, mcofst 
      real           srazm(4), rcazm(4), azdif, theta
      real           sinth, costh, azmth0, result, r(2,2)
      real           pi, deg_rad

      data mctnsr, pi       /
     &     16*0  , 3.1415927/
      deg_rad = pi/180.
c*********************************************************************
c
c    Get this program's name
c
      name = getname()
c*********************************************************************
c
c    Self-doc
c
c*********************************************************************
 
      if ( argis ( '-?' ) .gt. 0 .or. argis ( '-h' ) .gt. 0
     1          .or. argis ( '-help' ) .gt. 0 ) then
 
          write(LER,*)
     1'***************************************************************'
          write(LER,*) ' '
          write(LER,*) name(1:nblen(name)),
     1    ' regularizes the coordinate system of multicomponent',
     2    ' seismic data, using orientation information ASSUMED to be',
     3    ' already in the usp headers.',' ',
     4    ' Usage:',
     5    ' regsys -Nseismic_infile -Oseismic_outfile -Aazimuth_of_+x',
     6    '        (azimuth default = 90.0 = East)  -V  -h'
           stop
      endif
c
c    Open up the log file
c
#include <f77/open.h>

*     get commandline
      call cmdln (azmth0, verbos)

 100  if (azmth0 .lt. 0.0) then
         azmth0 = azmth0 + 360.
         go to 100
      endif
 200  if (azmth0 .gt. 360.0) then
         azmth0 = azmth0 - 360.
         go to 200
      endif

      write(LER ,105) azmth0, azmth0 + 90.
      write(LERR,105) azmth0, azmth0 + 90.
 105  format(' Seismic data will be transformed to a REGular SYStem'/
     1       ' of orthogonal right-handed coordinates, with:'/
     1       ' +x at azimuth ', f5.1/
     2       ' +y at azimuth ', f5.1/
     3       ' +z DOWN')
c
c    Reserve space for up to maxfile uspio input and output files
c    (in sum), each with up to maxcomp components. (Files you
c    open and close directly yourself are your own business and
c    don't count here.)
c
      maxfile = 2
      maxcomp = 4

      call uspioinit(name,maxfile,maxcomp)

c******************************************************************
c******************************************************************
c******************************************************************
c
c uspinput: find all the necessary inputs, verify they exist,
c process their headers, and figure out how we're going to read
c what we need from them.
c
c******************************************************************
c
c INPUT arguments to uspinput:
c
c mode (read / write mode):
c 'r'	open an already existing file for reading only
c 'rw'  open an already existing file for reading and writing
c 'ra'  open an already existing file for reading, writing, and
c       appending. Output will start at the end of the file
c       (although nothing will prevent you seeking back to the
c       beginning and overwriting the original data if you wish).
c
c  Pass through will be disabled if the mode is 'rw' or 'ra'.
c
      mode = 'r'
c
c clparam ("Input filename commandline parameter suffix"): this string
c is appended to '-N'; the composite string forms the mnemonic which
c identifies the input filename on the command line. If clparam is
c whitespace (as in this example), then uspsinput will look for '-N'
c (as is usual practice for scalar USP programs that only need a single
c input). If instead clparam were 'XY', then uspsinput would look for
c '-NXY' on the command line. The filename found will be returned in
c the variable "infile" (discussed with the other arguments returned
c from uspinput below).
c
c If an input file name is specified, then uspinput will attempt
c to open that file for input. If the file is not there, and the
c input is not being opened for scalar (non-MC) access, then uspinput
c will look for multiple subscripted files according to Leon's MC
c filename convention. (Refer to the man page for rottnsr.)
c
c If no input file name is specified on the command line, the place
c uspinput looks for input depends on whether the program is running
c under IKP or not. If not in IKP, the only possible fallback is to
c standard input. If in IKP, the fallback is to the "IKP single input"
c socket number specified by ikpsock (which is quite likely 0 meaning
c standard input, but doesn't have to be). If that is not connected
c or grounded, and the input is not scalar, then the final fallback
c is to one socket per component, with the socket number for each
c component being given by the formula
c (10 * Source Comp No + Receiver Comp No + ikpoffs).
c
c If you specify the same "clparam" for different inputs, uspinput
c will check the command line for repeat occurrences of the same
c argument. For example, if "clparam" is ' ' and the command
c line shows "-N file1 -N file2", then the first time uspinput is
c called "file1" will be found, and the second time "file2".
c
c If instead the command line showed "-N11 file1 -N12 file2", the first
c call to uspinput with clparam = '11' would find file1. A second
c call with the same clparam would attempt to open standard input.
c The first call to uspinput with clparam = '12' would find file2.
c A second call with clparam = '12' would again attempt to open
c standard input, which would be an error if it had already been
c opened for another connection previously.
c
      clparam = ' '
c
c wclparam ("Windowing commandline parameter suffix"): This string is
c appended to '-ns', '-ne', '-rs', '-re', '-pt', and '-wi' and the
c composite strings form the mnemonics which determine how to "window"
c the data. If wclparam is entirely whitespace you'll get the usual USP
c defaults.
c
c '-ns', '-ne' are the MC equivalents of the standard USP start trace,
c end trace in a gather parameters; '-rs', '-re' are the MC equivalents
c of the standard USP start gather, end gather in a line parameters. The
c dimensions are in terms of the MC dataset, and the "traces" being 
c counted are multicomponent traces.
c
c For example, if MCTrSp is 2 and there are three components, then the
c dataset would be organized:
c
c trace 1, component 1, gather 1
c trace 2, component 1, gather 1
c trace 3, component 1, gather 1
c ...
c trace N, component 1, gather 1
c trace 1, component 2, gather 1
c trace 2, component 2, gather 1
c trace 3, component 2, gather 1
c ...
c trace N, component 2, gather 1
c trace 1, component 3, gather 1
c ...
c trace N, component 3, gather 1
c trace 1, component 1, gather 2
c ...
c
c For this dataset the number of gathers as written in the USP line
c header would be 3 times the true MC number of gathers (so that programs
c that know nothing about MC datasets could still read this data). If you
c specified '-ns 2', then ALL of gather 1 as shown above would be skipped,
c 3N scalar traces, not just the N scalar traces in component 1 gather 1.
c
c Note your program would see the above dataset organized as:
c
c MC trace 1, gather 1
c MC trace 2, gather 1
c MC trace 3, gather 1
c ...
c MC trace N, gather 1
c MC trace 1, gather 2
c ...
c
c '-pt', if present, specifies that traces outside the window should
c be passed through; '-wi', if present, specifies that traces outside
c the window should be discarded. (The input and output files appear
c to be the same size to the user program in either case. In the
c first case the actual output file has the same dimensions as the
c actual input file, whereas in the second case the output file has the
c dimensions of the portion of the input visible to the user program.)
c
c If you specify wclparam for multiple input files, uspinput will only
c look for the associated windowing parameters on the command line ONCE.
c After that it will re-use whatever values it found before. (The idea
c is that if you specify one '-rs', say, you'll want it to apply to
c ALL the associated '-N' input files, no matter how many there happen
c to be.)
c
      wclparam = ' '
c
c dptw ("default pass through or window"): In the absence of anything
c on the command line, 0 means don't pass through unused data, 1 means
c do pass it through unchanged. '-wi(wclparam suffix)' on the command
c line overrides, and means to truncate anyway. '-pt(wclparam suffix)'
c overrides, and means to pass through anyway. If dptw is -1, that
c means don't even LOOK on the command line for any windowing
c parameters; the user program wants to handle ALL such
c windowing / pass-through etc details for this file for itself.
c
      dptw = 0
c
c ncomp is the number of components wanted. This means the number
c of source/receiver combinations, ie for split-shear data
c (11,12,21,22), ncomp would equal 4.
c
c On return, ncomp is set to the number of components found. (This only
c happens if allcomps is not 0; see the allcomps entry below.) For
c scalar input, ncomp should be 1 (and see "complist" below).
c
c If "allcomps" is not 0, then ncomps can be 0. That means no
c components are being _specifically_ requested; up to ncompmax
c components that can be "found" will be returned, and ncomp updated
c appropriately. (Note ncomp will never be _smaller_ on return than
c it was on entry. If components specifically asked for were not found
c it's an error!)
c
      ncomp = 0
c
c    ncompmax is the maximum number of components that can be returned.
c
      ncompmax = MAXCOMP
c
c complist is an array of dimension (2,ncompmax) specifying for each
c component:
c row 1:  Source comp #
c row 2:  Rec comp #
c For SCALAR input, set the source component number of the single
c component to -1; the IKP socket number to use for this scalar
c input inside IKP will be in "ikpsock" (see below).
c
c    (11,12,21,22):
c
c     complist(1,1) = 1
c     complist(2,1) = 1

c     complist(1,2) = 1
c     complist(2,2) = 2

c     complist(1,3) = 2
c     complist(2,3) = 1

c     complist(1,4) = 2
c     complist(2,4) = 2
c
c allcomps is either 0 or 1. 0 means "only tell me about the
c components I asked for". 1 means "tell me about all the components
c you could find". If allcomps is 1, then ncomp will be set equal to
c the number of components found, and complist will be updated to
c include the additional components that were found beyond the ones
c asked for. (The ncomp ones asked for will always be in the first
c ncomp slots, in the exact order and position they occur in complist.)
c
c    In this example we'll accept any other components that happen to
c    be there, up to a total of MAXCOMP.
c
      allcomps = 1
c
c ikpoffs ("IKP socket offset") is a number to add to the standard
c socket numbers (10 * Source Comp No + Receiver Comp No + ikpoffs)
c for IKP purposes for this file. Usually ikpoffs will just be 0;
c the purpose of this parameter is to allow multiple multicomponent
c file input from inside IKP.
c
      ikpoffs = 0
c
c ikpsock is the ikp socket number to look for a single input on.
c If this is not grounded, then all the input will be looked for
c here instead of in multiple sockets. Usually ikpsock will be 0,
c meaning standard input. (Note even if it's not used in the program,
c XIKP still expects stdin, socket 0, to be connected to something
c in the IKP network, even if only /dev/null!)
c
      ikpsock = 0
c
c seekalot guides uspinput in deciding how to buffer the data.
c
c A value of 0 means "this program will mostly plod through the data
c reading in order (alternated perhaps with passes through the data
c writing in order)". If you're going to mostly plod with only the
c occasional long seek, that would also count as 0. (Basically, if
c you would have left the sisio buffers on yourself, then seekalot
c should be 0.)
c
c uspseek is smart enough not to flush sisio's buffers unnecessarily,
c so don't worry about short forward seeks on reading: if that's
c all you're going to do, then seekalot can also be 0, because
c usprtrace will avoid the buffer flush by reading the intervening
c MC traces and discarding them for you. (Repeated short forward seeks
c on writing, however, count as bona-fide disruptive seeks. In that
c case seekalot should probably be 1.) If you would prefer to have
c the large buffers OFF so the intervening MC traces don't have to be
c read at all, then DON'T make seekalot 0!
c
c A value of 1 means "this program will jump around within a gather
c a lot, but not often between different gathers". If the program
c is going to furiously alternate reading and writing nearby MC traces
c (for example, read a MC trace, write it back out, read the next MC trace,
c write it back out, etc) that also counts as 1. (This mode will
c turn off the sisio large buffers unless a MC gather will be contained
c within the span of uspio's in-core transpose buffer. In that case
c it leaves them on.)
c
c If the program will jump at random all over the data, even between
c different MC gathers and lines, then seekalot should be 2. This will
c always turn off sisio large buffers.
c
      seekalot = 0

      call uspinput(mode,
     1      clparam, wclparam, dptw,
     1      ncomp, maxcomp, complist, allcomps,
     1      ikpoffs, ikpsock,
     1      seekalot, seekstat,
     1      numsmp, mcnumtrc, mcnumrec, dt, mcns, mcrs, pt,
     1      lheader, lhedlength,
     1      trlength, trdataoff, infile,
     1      incon, ierror)

c*******************************************************************
c
c OUTPUT arguments from uspinput:
c
c
c seekstat can be 0, 1, or 2.
c
c 0 means "feel free to seek all you want".
c
c 1 means "you are reading or writing to a pipe or socket, so
c seeking backwards is not allowed". (It's OK to seek forwards.
c If you're reading it will read the intervening MC traces and
c discard them; if you're writing it will write null MC traces for
c you.)
c
c 2 means "pass-through is enabled so no seeking is allowed at all".
c (If pass through is enabled, just calling uspseek will generate
c an error.)
c
c
c numsmp, mcnumtrc, mcnumrec, dt specify the dataset dimensions:
c  numsmp = number of samples in a single component of a trace (NumSmp),
c  dt = FLOATING POINT sampling rate (SmpInt)
c  mcnumtrc = number of MC traces in a gather (NumTrc)
c  mcnumrec = number of gathers in a line  (NumRec)
c
c
c mcns, mcrs tell the program the MC trace number and gather number,
c respectively, of the first MC trace the user program sees. (These are
c not from any header; they are simply counted from the beginning of
c the file starting with 1, 2, 3, etc.) The user program may need
c this information to modify values from the line header specifying
c values like initial trace offsets, etc; they may no longer be
c correct after windowing.
c
c
c pt tells the user program whether the output will be windowed
c (0) or passed through (1). The user program may need to know
c this for modifying values in an outgoing line header. (The standard
c USP dimensional line-header keywords "NumTrc" and "NumRec" will
c automatically be set appropriately by "uspoutput"; the user program
c can't do that for itself because the correct values to use will
c depend on how the MC dimension is scrambled into the other
c dimensions, as specified by the line header keyword "MCTrSp". That
c information may not be available to the user program.)
c
c 
c lheader is the line header going with the first component
c (if the different components come from multiple files)
c or the line header of the input dataset
c (if the different components come from one MC file).
c
c
c lhedlength is the length in bytes of the line header.
c
c
c trlength is the length in full words (floats) of a single component of
c an MC trace, including the trace header. You'll need to allocate
c trlength * SZSMPD * the number of components (returned ncomp)
c bytes to store a single MC trace.
c
c
c trdataoff is the offset in full words (floats) from the start of
c each single component of the trace to the start of the corresponding
c component's seismic data.
c
c
c infile is the "root input file name", that is, the filename that
c followed the appropriate '-N' construction on the command line.
c You may find this useful if you want to construct an appropriate
c filename for auxilliary output. Note if there was no '-N' field
c on the command line uspinput will be forced to invent a name.
c If the input is from standard input, the name will be 'stdin'.
c If from a single IKP socket it will be of the form 'IKP.sN',
c where N is the socket number. If MC input through one or more
c IKP sockets, the name will be of the form 'IKP.iN', where N is
c the socket offset.
c
c
c incon is the "connection number", the number you'll have to use
c to refer to this connection later.
c
c
c ierror will be incremented once for each error that was found. (For
c example, it will be incremented once for each component that isn't
c found.) Unless ierror is zero this input cannot be used, and
c attempting to do so will result in an error.
c
c
c******************************************************************
c******************************************************************
c******************************************************************
*     header stuff here

      call savelu('MCTrSp',ifmt_MCTrSp,l_MCTrSp,ln_MCTrSp,LINEHEADER)
      call savelu('NumCmp',ifmt_NumCmp,l_NumCmp,ln_NumCmp,LINEHEADER)
      call saver2(lheader,ifmt_NumCmp,l_NumCmp, ln_NumCmp,
     1            numcmp , LINEHEADER)

*      write(LER,*) 'NumCmp', numcmp

      call saver2(lheader,ifmt_MCTrSp,l_MCTrSp, ln_MCTrSp,
     1            mctrsp , LINEHEADER)
*      write(LER,*) 'MCTrSp', mctrsp 
      do icmp = 1, numcmp
         mcmnem(1:5) = 'MCLE0'
         if (icmp .le. 9) then
            write(mcmnem(6:6),'(i1)') icmp
         else
            write(mcmnem(5:6),'(i2)') icmp
         endif
         call savelu(mcmnem,ifmt_MCLE(icmp),l_MCLE(icmp),
     &               ln_MCLE(icmp),LINEHEADER)
         call saver2(lheader, ifmt_MCLE(icmp), l_MCLE(icmp),
     &               ln_MCLE(icmp), mcle(icmp), LINEHEADER)
      enddo

*      write(LER,*) 'MCList:', (mcle(icmp), icmp=1,numcmp)

*     decode MCLEnn 
      do icmp = 1, numcmp
         rccomp = mod(mcle(icmp),10)
         srcomp = (mcle(icmp)-rccomp)/10
         mctnsr(srcomp, rccomp) = icmp
      enddo

      write(LERR,1005) ((mctnsr(srcomp,rccomp), rccomp=1,4),srcomp=1,4)
 1005 format(' components (src,rcvr) listed as:'/(4i5))

*     check for the presence of BOTH horizontal receiver components
 1100     do 1109 srcomp = 1, 3
                if(mctnsr(srcomp,1) .eq. 0 .and.
     1             mctnsr(srcomp,2) .eq. 0      ) go to 1109
                if (mctnsr(srcomp,1) .eq. 0) then
                   write(LER ,1105) mctnsr(srcomp,2)
                   write(LERR,1105) mctnsr(srcomp,2)
 1105                 format(' BOTH horizontal receiver components must'
     1                       ' be present to regularize the coordinate'
     1                       ' system. I see only :',i3)
                elseif (mctnsr(srcomp,2) .eq. 0) then
                   write(LER ,1105) mctnsr(srcomp,1)
                   write(LERR,1105) mctnsr(srcomp,1)
                endif
 1109     enddo

c      write(LER,*) ncomp
c      write(LER,1125) (complist(1,ii), complist(2,ii), ii=1,numcmp)
c 1125 format('complist:', (3x,2i1))

c******************************************************************
c****************************************************<**************
c******************************************************************
c
c uspoutput: Create an output MC file from scratch and write out
c            the line header.
c
c
c******************************************************************
c
c INPUT arguments for uspoutput:
c
c mode (read / write mode):
c 'w'	create a new file for writing only
c 'rw'  create a new file for reading and writing
c 'ra'  create a new file for reading and writing and appending,
c       meaning that you don't have to specify the final file
c       size up front. After you're done the line header will
c       automatically be updated to reflect the final size.
c       You DO have to write out an integral number of gathers.
c
      mode = 'w'
c
c clparam ("Output filename commandline parameter suffix"): this string
c is appended to '-O'; the composite string forms the mnemonic which
c identifies the output filename on the command line. If clparam is
c whitespace (as in this example), then uspsinput will look for '-O'
c (as is usual practice for scalar USP programs that only need a single
c input). If instead clparam were 'YZ', then uspsinput would look for
c '-OYZ' on the command line. The output filename found on the command
c line will be returned in the variable "oufile"; see discussion below.
c
c If no output file name is specified on the command line, the place
c uspoutput next looks depends on whether the program is running
c under IKP or not. If not in IKP, the only possible fallback is to
c standard output. If in IKP, the fallback is to the "IKP single output"
c socket number specified by the argument "ikpsock" (this is quite
c likely 1, meaning standard output, but doesn't have to be).
c
c If you specify the same "clparam" for different outputs, uspoutput
c will check the command line for repeated occurrences of the same
c argument. (See the discussion of "clparam" for uspinput.)
c
c If an appropriate '-O' is found on the command line, the name(s) of
c the actual file(s) opened depend on the transpose mode (mctrsp).
c (Refer to the discussion under wclparam below.)
c
c If the "multi-component trace spacing" mctrsp is set to 0 (possibly
c by the user specifying "-mc 0" on the command line) then the file
c name from '-O' is used as a root name and a .IJ ending is attached,
c where I is the source component number (1 to 3) and J is the receiver
c component number (1 to 3).
c
c If mctrsp is 0 but there is no '-O' filename template, then the output
c will be into a single MC file (with single-component gathers) as if
c mctrsp had been set to 2. (Ditto if the program is running under IKP
c but the single output is available.)
c
      clparam = ' '
c
c wclparam ("Windowing commandline parameter suffix"): This string is
c appended to '-mc' and the composite string forms the mnemonic which
c determines how to transpose the data. A single integer should follow,
c corresponding to the line-header keyword MCTrSp (MultiComponent
c Trace Spacing). If no "-mc" is specified, you'll get the default:
c whatever is set by the program in dmctrsp below, or if that is
c negative the same as the corresponding input (if one exists), or
c if there isn't one then a single-component-gather file (mctrsp=2).
c 
      wclparam = ' '
c
c ncomp is the number of components wanted.
c
c    Already set by uspinput
c
c complist is an array of dimension (2,ncomp) specifying for each
c component:
c row 1:  Source comp #
c row 2:  Rec comp #
c For SCALAR output, set the source component number of the single
c component to -1; the IKP socket number to use for this scalar
c output inside IKP will be in "ikpsock" (see below).
c
c    Already set by uspinput
c
c cinp gives the "corresponding input". Whether or not to do
c pass through, etc, depends on the setting for the corresponding
c input.
c
c If there is a corresponding input the parameters that define the
c data sample rate and dimensions, udt, umcnumtrc, and umcnumrec,
c will be read from there instead of from the uspoutput command line.
c These will be RETURNED as well, so don't pass zeroes in
c those slots!) The trace-data length unumsmp, the total trace length
c trlength, the number of components ncomp, and the component list
c complist will all be checked to make sure they match. If they don't
c you'll get an error.
c
c If there is no corresponding input set cinp to 0.
c
      cinp = incon
c
c dmctrsp gives the default MCTrSp to use.
c Choices are:
c -1: Use the standard default, ie:
c     if there was "corresponding input", use that MCTrSp, or
c     if not, dmctrsp = 2.
c 0: Use Leon-style multi-file MC output. (See the rottnsr man page.)
c 1: Use "component fast" single-file output. (That is, consecutive scalar
c    traces cycle through the components in the order given in complist.)
c 2: Use "single-component gathers" single-file output. (That is,
c    consecutive scalar gathers [aka "records"] cycle through the components
c    in the order given in complist.)
c 3: Use "single-component lines" single-file output. (That is, an entire
c    line's worth of component 1, followed by an entire line's worth of
c    component 2, etc, as if several separate scalar files had been
c    concatenated together, minus the intervening line headers.)
c
c In any case, whatever is specified on the command line overrides.
c (Unless in IKP, in which case the configuration of output pipes
c can override; if the single-output pipe is open, then you'll get
c single-file output instead of multiple-file output no matter what.)
c
      dmctrsp = -1
c
c ikpoffs ("IKP socket offset") is a number to add to the standard
c socket numbers (10 * Source Comp No + Receiver Comp No + 5 + ikpoffs)
c for IKP purposes for this file. Usually ikpoffs will be 0;
c the purpose of this parameter is to allow multiple multicomponent
c file output from inside IKP.
c
      ikpoffs = 0
c
c ikpsock is the ikp socket number to look for a single output on.
c If this is not grounded, then all the output will be looked for
c here instead of in multiple sockets. Usually ikpsock will be 1,
c meaning standard output. (Note even if it's not used in the program,
c XIKP still expects stdout, socket 1, to be connected to something
c in the IKP network, even if only /dev/null!)
c
      ikpsock = 1
c
c seekalot guides uspoutput in deciding how to buffer the data.
c A value of 0 means "this program will mostly plod through the data
c writing in order (alternated perhaps with passes through the data
c reading in order)". A value of 1 means "this program will jump around
c within a gather a lot, but not often between different gathers". If
c the program is going to furiously alternate reading and writing
c nearby MC traces (for example, read a MC trace, write it back out, read
c the next MC trace, write it back out, etc) that also counts as 1.
c If the program will jump at random all over the data, even between
c different MC gathers and lines, then seekalot should be 2.
c
      seekalot = 0
c
c numsmp, mcnumtrc, mcnumrec, dt specify the dataset dimensions:
c  numsmp = number of samples in a scalar trace (NumSmp),
c  dt = FLOATING-POINT sampling rate (SmpInt)
c  mcnumtrc = number of MC traces in a gather (NumTrc)
c  mcnumrec = number of gathers in a line  (NumRec)
c
c If there is a "corresponding input", dt, mcnumtrc and mcnumrec will be
c SET FOR YOU. Any previous value they may have had will be
c ignored.
c
c If you're appending, there's no reason not to set mcnumrec to zero.
c
c    numsmp was already set by uspinput;
c    since we have a corresponding input, we do not need
c    to set dt, mcnumtrc, and mcnumrec: they'll be returned as output
c    arguments instead.
c
c lheader is the line header. You can either provide a header yourself
c or ask that uspoutput create one for you. In either case, the lheader
c you provide should be allocated big enough to contain a USP line
c header.
c
c    lheader was already returned from uspinput
c
c lhedlength is the length in bytes of the line header. If 0, then
c you are asking for uspoutput to create a header for you from
c scratch.
c
c    lhedlength was already returned from uspinput
c
c trlength is the length in full words (floats) of a single scalar trace,
c including the trace header. You'll need to allocate
c trlength * SZSMPD * ncomp (as returned)
c bytes to store a single MC trace. Note you need to SET trlength.
c It is not returned. Until variable length headers are introduced,
c just use trlength = ITRWRD + unumsmp.
c
c If there is a "corresponding input", trlength must match the trace
c length for the corresponding input.
c
c    trlength was already set by uspinput
c
      call uspoutput (mode, clparam, wclparam,
     1   ncomp, complist, cinp, dmctrsp,
     1   ikpoffs, ikpsock, seekalot, seekstat,
     1   numsmp, mcnumtrc, mcnumrec, dt,
     1   lheader, lhedlength, trlength, oufile,
     1   outcon, ierror)

c*******************************************************************
c
c OUTPUT arguments from uspoutput:
c
c seekstat is the same as for uspinput.
c
c If there is a "corresponding input", dt, mcnumtrc and mcnumrec will be
c SET to the values they have in the corresponding input. Don't pass
c numeric values to uspoutput for these variables lest they be
c unexpectedly redefined!
c
c
c oufile is the "root output file name", which you may find useful
c if you want to construct a filename for auxilliary output. Note
c if there was no '-O' field on the command line uspoutput will be
c forced to invent a name. If the output is to standard output, the
c name will be 'stdout'. If to a single IKP socket it will be of
c the form 'IKP.sN', where N is the socket number. If MC output
c through one or more IKP sockets, the name will be of the form
c 'IKP.oN', where N is the socket offset.
c
c
c outcon is the "connection number", the number you'll have to use
c to refer to this connection later.
c
c
c ierror will be incremented once for each error that was found.
c Unless ierror is zero this output cannot be used, and
c attempting to do so will result in an error.
c
c
c******************************************************************
c******************************************************************
c******************************************************************
*  get trace header offsets
*     args:       mnemonic, type,      offset,   count,    buffer
      call savelu('SrComp',ifmt_SrComp,l_SrComp,ln_SrComp,TRACEHEADER)
      call savelu('RcComp',ifmt_RcComp,l_RcComp,ln_RcComp,TRACEHEADER)
      call savelu('SrXAzm',ifmt_SrXAzm,l_SrXAzm,ln_SrXAzm,TRACEHEADER)
      call savelu('RcXAzm',ifmt_RcXAzm,l_RcXAzm,ln_RcXAzm,TRACEHEADER)

*      call savelu('SrPtXC',ifmt_SrPtXC,l_SrPtXC,ln_SrPtXC,TRACEHEADER)
*      call savelu('SrPtYC',ifmt_SrPtYC,l_SrPtYC,ln_SrPtYC,TRACEHEADER)
*      call savelu('RcPtXC',ifmt_RcPtXC,l_RcPtXC,ln_RcPtXC,TRACEHEADER)
*      call savelu('RcPtYC',ifmt_RcPtYC,l_RcPtYC,ln_RcPtYC,TRACEHEADER)

*      call savelu('SrYRot',ifmt_SrYRot,l_SrYRot,ln_SrYRot,TRACEHEADER)
*      call savelu('SrZRot',ifmt_SrZRot,l_SrZRot,ln_SrZRot,TRACEHEADER)
*      call savelu('RcYRot',ifmt_RcYRot,l_RcYRot,ln_RcYRot,TRACEHEADER)
*      call savelu('RcZRot',ifmt_RcZRot,l_RcZRot,ln_RcZRot,TRACEHEADER)

*      call savelu('SoPtNm',ifmt_SoPtNm,l_SoPtNm,ln_SoPtNm,TRACEHEADER)
*      call savelu('RecInd',ifmt_RecInd,l_RecInd,ln_RecInd,TRACEHEADER)
*      call savelu('SrcLoc',ifmt_SrcLoc,l_SrcLoc,ln_SrcLoc,TRACEHEADER)
*      call savelu('LinInd',ifmt_LinInd,l_LinInd,ln_LinInd,TRACEHEADER)

      write (LER, *) 'Input: ', infile(1:nblen(infile))
      write (LER, *) 'Output: ', oufile(1:nblen(oufile))

c**********************************************************************
c
c    Now read a MC trace, write a MC trace, repeat.
c
c    Use "usprtrace" to read, "uspwtrace" to write.
c
c**********************************************************************
 
c
c    Arguments to usprtrace and uspwtrace:
c
c
c**********************************************************************
c
c    The first argument is the integer connection number that you
c    got back from either uspinput or uspoutput.
c
c    The second argument is the multicomponent trace array for it
c    to read from or write to.
c
c    The third argument is the DIMENSIONED length of the MC trace.
c    usprtrace and uspwtrace need to know this so they can find
c    the ncomp components in the multi-component trace.
c
c    The return value is:
c
c    0: everything OK
c   -1: something very bad
c    1: you just went off the end of the data
c

      do 2099 ii = 1, mcnumtrc * mcnumrec
*      do 2099 ii = 1, 5

          write (LERR, *) 'Reading MC trace ', ii
          if (usprtrace(incon, mctrace, MAXTRLEN) .ne. 0) then
              write (LERR, *) 'Trouble reading MC trace ', ii
          endif

*         do for each source component
 2190     do 2199 srcomp = 1, 3
 2200        do 2209 rccomp = 1, 2
                icmp = mctnsr(srcomp,rccomp)
                if (icmp .eq. 0) go to 2199

                mcofst = MAXTRLEN * (icmp-1) * 2
                call saver2(mctrace,ifmt_SrXAzm,mcofst + l_SrXAzm, 
     1                      ln_SrXAzm, srxazm , TRACEHEADER)
                call saver2(mctrace,ifmt_SrComp,mcofst + l_SrComp, 
     1                      ln_SrComp, srcmp , TRACEHEADER)
                if (srcmp .ne. srcomp) then
                   write(LERR,2205) ii
 2205              format(' Inconsistency in source indexing, MC trace'
     1                    ,i5)
                   stop
                endif


                call saver2(mctrace,ifmt_RcXAzm,mcofst + l_RcXAzm, 
     1                      ln_RcXAzm, rcxazm , TRACEHEADER)
                call saver2(mctrace,ifmt_RcComp,mcofst + l_RcComp, 
     1                      ln_RcComp, rccmp , TRACEHEADER)
                if (rccmp .ne. rccomp) then
                   write(LERR,2207) ii
 2207              format(' Inconsistency in reciever indexg, MC trace'
     1                    ,i5)
                   stop
                endif

*                write(LERR,*) srcomp, rccomp, srxazm, rcxazm 

                call getazm(srcomp, srxazm, srazm(srcomp))
                call getazm(rccomp, rcxazm, rcazm(rccomp))

*                write(LERR,*) srcomp, rccomp, srxazm, rcxazm, 
*     1                        srazm(srcomp), rcazm(rccomp)

 2209        enddo
*             write(LERR,*) ' MC trace', ii,': got rcvr comps.'

*            check that these receiver components are nearly orthogonal
             azdif = abs(rcazm(2)-rcazm(1))
             if (abs(azdif-90.) .ge. 5.0) then
*               there may be a problem of non-orthogonal comps.
*               check to see that we are not wrapped around 360:
                if (rcazm(1) .lt. rcazm(2)) then
                   azdif = abs(rcazm(1)+180.-rcazm(2))
                else
                   azdif = abs(rcazm(1)-rcazm(2)-180.)
                endif
*                now check again:
                if (abs(azdif-90.) .ge. 5.0) then
                   write(LER ,2305) ii
                   write(LERR,2305) ii
 2305              format(' Horizontal components on MC trace', i5,
     1                    ' are non-orthogonal!'/
     1                    ' I QUIT!!')
                   stop
                endif
             endif
*            write(LERR,*) ' MC trace', ii,': rcvr comps closely orthog'

*            check that these receiver components refer to a 
*            right-handed system, with z down.
             azdif = rcazm(2) - rcazm(1)
             if(azdif .lt. 85.0 .and.
     1          azdif .gt. 95.0      ) go to 2400
             if(azdif .lt. -265.0 .and.
     1          azdif .gt. -275.0    ) go to 2400
*            this was a left-handed coordinate system; invert:
             do isamp = 1, numsmp
                mctrace(trdataoff + isamp,2) = 
     1                                  - mctrace(trdataoff + isamp,2)
             enddo
             rcazm(2) = rcazm(2) + 180.
             if (rcazm(2) .gt. 360.) rcazm(2) = rcazm(2) - 360.
             write(LERR,2315) ii
 2315        format(' MC trace',i3,': had a left-handed (z down)',
     1              ' coordinate system; component 2 was inverted.')

*            check that rcvr coordinate system is exactly orthogonal
*            (src coords ASSUMED exactly orthog below!)
 2400        azdif = rcazm(2) - rcazm(1)
             if(azdif. lt. 0.0) azdif = azdif + 360.0
             theta = azdif - 90.
             if (abs(theta) .gt. 1.0) then
*               orthogonalize, fixing direction of component 1
*               since azdif is with +/- 5 degs of 90, theta is small
*               deformation matrix is:|  1    sinth |
*                                     | costh   0   |, where...
                sinth = sin(theta*deg_rad)
                costh = cos(theta*deg_rad)
                do isamp = trdataoff+1, trdataoff+numsmp
                   mctrace(isamp, 1) = mctrace(isamp, 1) + 
     2                                 sinth*mctrace(isamp, 2)
                   mctrace(isamp, 2) = costh*mctrace(isamp, 2)
                enddo
*           write(LERR,*) ' MC trace', ii,': rcvrs now exactly orthog.' 
             endif
             rcxazm = (rcazm(1)*10)
*             write(LERR,*) srcomp, rccomp, srxazm, rcxazm

*            check that rcvr coords are aligned wit>h src coords.
             theta = (rcxazm - srxazm)/10.
             if (abs(theta) .gt. 1.0) then
*                then rotate rcvrs into coincidence w srcs.
*                rotation matrix is:|  costh    sinth |
*                                   | -sinth    costh |, where...
                theta = theta*deg_rad
                sinth = sin(theta)
                costh = cos(theta)
                do isamp = trdataoff+1, trdataoff+numsmp
                   mctrace(isamp, 1) = costh*mctrace(isamp, 1) + 
     1                                 sinth*mctrace(isamp, 2)
                   mctrace(isamp, 2) = sinth*mctrace(isamp, 1) - 
     1                                 costh*mctrace(isamp, 2)
                enddo

*               re-write 1-header
                icmp = mctnsr(srcomp,1)
                mcofst = MAXTRLEN * (icmp-1) * 2
                call savew2(mctrace,ifmt_SrXAzm,mcofst + l_SrXAzm, 
     1                      ln_SrXAzm, srxazm , TRACEHEADER)
                call savew2(mctrace,ifmt_RcXAzm,mcofst + l_RcXAzm, 
     1                      ln_RcXAzm, srxazm , TRACEHEADER)
             endif

*            re-write 2-header
             icmp = mctnsr(srcomp,1)
             mcofst = MAXTRLEN * (icmp-1) * 2
             call savew2(mctrace,ifmt_SrXAzm,mcofst + l_SrXAzm, 
     1                   ln_SrXAzm, srxazm , TRACEHEADER)
             call savew2(mctrace,ifmt_RcXAzm,mcofst + l_RcXAzm, 
     1                   ln_RcXAzm, srxazm , TRACEHEADER)

*             write(LERR,*) srcomp, 1, srxazm, srxazm
*             write(LERR,*) srcomp, 2, srxazm, srxazm
 2199     enddo
*         end of source-component loop

*         at this point, all source- and receiver- comps are referred 
*         to the same right-handed (z down) orthogonal coord system.
*         check if this src-rcvr coord sys is properly aligned:

          theta = (srxazm/10. - azmth0)
          if (abs(theta) .gt. 1.0) then
             theta = theta*deg_rad
             sinth = sin(theta)
             costh = cos(theta)
*            rotation matrix is: r = |  costh    sinth |
*                                    | -sinth    costh |
             r( 1, 1 ) = costh
             r( 1, 2 ) = sinth
             r( 2, 1 ) = -sinth 
             r( 2, 2 ) = costh  
*            do for each sample
 2240        do 2249 isamp = trdataoff+1, trdataoff+numsmp

*****          do for each output component             
 2250          do 2259 srcomp = 1, 3
 2260             do 2269 rccomp = 1, 2
                     jcmp = mctnsr(srcomp,rccomp)
                     if (jcmp .eq. 0) go to 2259

*****                tensor rotation:
                     result = 0.0
 2270                do 2279 jc = 1, 2
 2280                   do 2289 kc = 1, 2
                           icmp = mctnsr(jc, kc)
                           if (icmp .eq. 0) go to 2259
                           result = result +
     &                     mctrace(isamp,icmp)*r(jc,srcomp)*r(kc,rccomp)
 2289                   enddo
 2279                enddo
                     mctrace(isamp, jcmp) = result

 2269             enddo
 2259          enddo

 2249       continue
*           end of sample loop
*           rotation complete

*           write trace header
            srxazm = azmth0*10.
 2310       do icmp = 1, ncomp
               mcofst = MAXTRLEN * (icmp-1) * 2
               call savew2(mctrace,ifmt_SrXAzm,mcofst + l_SrXAzm, 
     1                     ln_SrXAzm, srxazm , TRACEHEADER)
               call savew2(mctrace,ifmt_RcXAzm,mcofst + l_RcXAzm, 
     1                     ln_RcXAzm, srxazm , TRACEHEADER)
            enddo
*           at this point, all source- and receiver- comps are referred 
*           to the same right-handed (z down) orthogonal coord system,
*           properly aligned as specified by -A

         endif
*        
         write (LERR, *)
         write (LERR, *) 'Writing MC trace ', ii
         if (uspwtrace(outcon, mctrace, MAXTRLEN) .ne. 0) then
            write (LERR, *) 'Trouble writing MC trace ', ii
         endif

 2099 enddo
*     end of MC trace loop

c
c    usprtrace and uspwtrace will print an error message if you
c    try to read or write outside the legal bounds of the data, with
c    one exception: if you increment ONCE off the end of the data
c    (while either reading or writing), they will give a nonzero
c    error code but will NOT print any message nor abort the program.
c
c    Thus if you don't want to keep track of how many MC traces there
c    are, you can just "keep going until you get a nonzero return
c    code", and that will work. (Keep in mind that if you allow
c    appending, then you can write anywhere! If it's off the current
c    end of the data, you'll just make the data that much bigger.
c    If you try to read beyond what's there, though, that will
c    always be an error.)
c
 
********************************************************************* 

c
c    Finally, make SURE to call uspclose to close your connections
c    before exiting. Some of your data may end up missing if you forget
c    to do this.
c

      if (uspclose(incon)) then
          write (LERR, *) 'Trouble closing input'
      endif

      if (uspclose(outcon)) then
          write (LERR, *) 'Trouble closing output'
      endif

      stop      
      end
*********************************************************************
*********************************************************************
*********************************************************************
********************************************************************
      subroutine getazm (jcomp, ixazm, azm)
*     rationalize azimuth

      azm = ixazm/10.0
      if (jcomp .eq. 2) then
*     Note: what is stored is the azimuth of the x-axis of the
*           implied right-handed coordinate system (z down),
*           corresponding to this y-axis.
         azm = azm + 90.
      endif

 100  if (azm .lt. 0.0) then
         azm = azm + 360.
         go to 100
      endif
 
 200  if (azm .gt. 360.0) then
         azm = azm - 360.
         go to 200
      endif

      azm10 = azm*10.0 +.5
      ixazm = azm10
      if (jcomp .eq. 2) then
         ixazm = ixazm - 900
      endif

      return
      end
