C***********************************************************************
C                 copyright 2001, Amoco Production Company             *
C                             All Rights Reserved                      *
C                     an affiliate of BP America Inc.                  *
C***********************************************************************
**********************************************************************c
c
c rottnsr reads seismic trace data from a 2s x 2r set of input files,
c     rotates these about the orthogonal axis, and writes the results 
c     in a 2x2 set of output files.
c
c**********************************************************************c
c
c     SZSMPM	- maximum number of trace samples
c     SZLNHD	- line file size = (SZTRHD + SZSAMP*MAXSMP)/2
c     SZTRHD	- size of trace header in pipes in bytes
c     SZSMPD   - size of sample in pipe (in bytes)
c        LER  is the current window.
c        LOT  is used for error reporting; also comes to the window.
c        LERR is a standard output file; will appear in directory as
c             ROTT.pid1.
c
c     declare variables
c
#include <save_defs.h>
#include <f77/iounit.h>
#include <f77/lhdrsz.h>
#include <f77/sisdef.h>
 
c declare standard usp variables

      integer * 2  ibufij( SZLNHD )                           
      integer      lhedij( SZLNHD )
      integer      nsamp, nsi, ntrc, nrec, iform
      integer      linij, loutij, lbytes, nbytes, lbyout
      integer      luin( 3, 3 ), luout( 3, 3 )
      integer      irs, ire, ns, ne
      integer      argis, pipei(4), pipeo(4)
             
      real         headij( SZLNHD )
      real         trcij ( SZLNHD )

      character    chaxis*1, chdgrs*3, chext*3, name*7
      character*255 casert, ifile( 3, 3 ), ofile( 3, 3 )
      character    otap*255

      logical      verbos, query, IKP

c declare program specific variables

      integer  ncmp(2), idgrs, icomp, recnum, trcnum, leotap
      integer  luphi, jc, jcomp, JJ, KK, length, NumEntries

      real     trc( SZLNHD , 3, 3 ), r( 3, 3 ), trcout( SZLNHD , 3, 3 )
      real     pi, degrad, theta, costh, sinth, degree 
      real     Record(SZLNHD), Angle(SZLNHD)

      character phitap*255, iwd*6

c set up usefull equivalences

      equivalence ( ibufij( 1 ),  lhedij( 1 ), headij( 1 ) )

c initialize variables

      data lbytes / 0 /
      data nbytes / 0 /
      data name/'ROTTNSR'/
      data pi/3.14159/
      data pipei/0,3,5,7/
      data pipeo/1,4,6,8/

      pi = 4. * atan(1.0)
      degrad = pi/180.

c issue command line help if requested 

      query = ( argis( '-?' ) .gt. 0 .or. argis( '-h' ) .gt. 0 )
      if ( query ) then
         call help()
         stop
      endif

c open printout files

#include <f77/open.h>
 
c read command line arguments

      call cmdln ( casert, otap, phitap, ns, ne, irs, ire, verbos,
     :     iaxis, idgrs, iwd, IKP )


c open angle file if attached

      if ( phitap .ne. ' ' ) then
         call alloclun(luphi)
         length = lenth(phitap)
         open ( luphi, file=phitap(1:length), status='old', err=990)
         call ReadAngles(luphi, Record, Angle, NumEntries)
      endif

c restrict angle

      IF ( phitap .ne. ' ' ) then
         do i = 1, NumEntries
 40         if (Angle(i) .gt. 90. ) then
               Angle(i) = Angle(i) - 180.
               if ( verbos ) write (LER ,45) Record(i)
               write (LERR,45) Record(i)
 45            format(' Rotation angle at record ',f7.2,' adjusted
     : to lie between +- 90 degrees')
               go to 40
            elseif (Angle(i) .lt. -90 ) then
               Angle(i) = Angle(i) + 180.
               if ( verbos ) write (LER ,45)
               write (LERR,45)
               go to 40
            endif
         enddo
      ELSE
         do while ( iabs(idgrs) .gt. 90 )
            if (idgrs .gt. 90 ) then
               idgrs = idgrs - 180
               if ( verbos ) 
     :              write (LER ,*)' Rotation angle adjusted to lie betwe
     :en +- 90 degrees'
               write (LERR,*)' Rotation angle adjusted to lie between +-
     : 90 degrees'
            elseif (idgrs .lt. -90 ) then
               idgrs = idgrs + 180
               if ( verbos ) 
     :              write (LER ,*)' Rotation angle adjusted to lie betwe
     :en +- 90 degrees'
               write (LERR,*)' Rotation angle adjusted to lie between +-
     : 90 degrees'
            endif
         enddo
      ENDIF
      
      if ( phitap .eq. ' ' ) then
         if ( verbos ) write(LER, 60)  casert, idgrs, iaxis
         write(LERR,60)  casert, idgrs, iaxis
 60      format(/' Case-root:'/a80/
     :        ' (Right-handed) rotation angle:',i4,' degrees',
     :        ' about positive axis', i2)
      endif

c get components for this case

      call strend(casert,lstr)
      write(chdgrs, '(i3)') idgrs
      if ( chdgrs(2:2) .eq. '-' ) then
         chdgrs(1:1) = '-'
         chdgrs(2:2) = '0'
      endif
      if ( chdgrs(1:1) .eq. ' ' ) chdgrs(1:1) = '+'
      if ( chdgrs(2:2) .eq. ' ' ) chdgrs(2:2) = '0'

c select axes

      DO jc=1,2
         if (iaxis .eq. 1) then
            if (jc .eq. 1) then
               jcomp = 2
            else
               jcomp = 3
            endif
         elseif (iaxis .eq. 2) then
            if (jc .eq. 1) then
               jcomp = 1
            else
               jcomp = 3
            endif
         elseif (iaxis .eq. 3) then
            if (jc .eq. 1) then
               jcomp = 1
            else
               jcomp = 2
            endif
         endif
         ncmp(jc) = jcomp
      ENDDO

c do for each component

      lc = 0

      DO ic = 1, 2

         icomp = ncmp(ic)  
 
         DO jc = 1, 2

            jcomp = ncmp(jc)
         
c get logical unit number for input 

            if (IKP) then
               lc = lc + 1
               call sisfdfit (linij , pipei(lc))
               konin = lc*2 -1
               if (konin .eq. 1) konin = 0
               ij = 10*icomp + jcomp
               if (verbos) write(LER ,121) konin, ij
               write(LERR,121) konin, ij
 121           format (/' Input stream',i3,' assumed to contain',
     &              ' component',i5)
            else

c construct filename.extension

               ext = .1*icomp +.01*jcomp
               write(chext,'(f3.2)') ext
               ifile( icomp, jcomp )=casert
               ifile( icomp, jcomp )(lstr+1:lstr+3) = chext
               if (verbos) write(LER, 122) ifile(icomp, jcomp)
               write(LERR,122) ifile(icomp, jcomp)
 122           format (/' Input file required:'/a80)
               call getln(linij,  ifile( icomp, jcomp ), 'r', 0)
            endif

            luin( icomp, jcomp ) = linij           
         
c read line header of input, check, and output 

            call rtape (linij, ibufij, lbytes )            
            if (lbytes .eq. 0) then
               write(LOT,*)'ROTTNSR: no header read from unit ',linij
               write(LOT,*)'FATAL'
               stop
            endif

c save certain lineheader parameters

            call saver(ibufij, 'NumSmp', nsamp, LINHED)
            call saver(ibufij, 'SmpInt', nsi  , LINHED)
            call saver(ibufij, 'NumTrc', ntrc , LINHED)
            call saver(ibufij, 'NumRec', nrec , LINHED)
            call saver(ibufij, 'Format', iform, LINHED)

            call savelu('TrcNum',ifmt,l_TrcNum,length,TRACEHEADER)
            call savelu(iwd,ifmt,l_RecNum,length,TRACEHEADER)
            call savelu('SrcLoc',ifmt,l_SrcLoc,length,TRACEHEADER)
            call savelu('RecInd',ifmt,l_RecInd,length,TRACEHEADER)
            call savelu('DphInd',ifmt,l_DphInd,length,TRACEHEADER)
            call savelu('DstSgn',ifmt,l_DstSgn,length,TRACEHEADER)
            call savelu('StaCor',ifmt,l_StaCor,length,TRACEHEADER)
            call savelu('WDepDP',ifmt,l_WDepDP,length,TRACEHEADER)  
            call savelu('ShtDep',ifmt,l_ShtDep,length,TRACEHEADER)

c ensure that command line values are compatible with data set

            call cmdchk ( ns, ne, irs, ire, ntrc, nrec )

c modify line header to reflect actual number of traces output

            nrecc=ire - irs + 1
            ntrcc=ne-ns+1
            call savew(ibufij, 'NumRec', nrecc, LINHED)
            call savew(ibufij, 'NumTrc', ntrcc, LINHED)

c write progname (from command line) to historical line hdr

            call savhlh( ibufij, lbytes, lbyout)

c skip unwanted records

            call recskp(1, irs-1, luin(icomp,jcomp), ntrc, ibufij )


c get logical unit number for output file

            if (IKP) then
               call sisfdfit (loutij, pipeo(jc))
            else

               write(chaxis,'(i1)') iaxis

c construct filename.extension

               if ( otap .eq. ' ' ) then
                  ofile( icomp, jcomp ) = casert
                  if ( phitap .eq. ' ' ) then
                     ofile( icomp, jcomp )(lstr+1:lstr+2) = '.R'
                     ofile( icomp, jcomp )(lstr+3:lstr+3) = chaxis
                     ofile( icomp, jcomp )(lstr+4:lstr+6) = chdgrs
                     ofile( icomp, jcomp )(lstr+7:lstr+9) = chext
                  else
                     ofile( icomp, jcomp )(lstr+1:lstr+2) = '.r'
                     ofile( icomp, jcomp )(lstr+3:lstr+3) = chaxis
                     ofile( icomp, jcomp )(lstr+4:lstr+8) = '_var_'
                     ofile( icomp, jcomp )(lstr+9:lstr+11) = chext
                  endif
               else
                  leotap = lenth(otap)
                  ofile( icomp, jcomp ) = otap(1:leotap)
                  if ( phitap .eq. ' ' ) then
                     ofile( icomp, jcomp )(leotap+1:leotap+2) = '.R'
                     ofile( icomp, jcomp )(leotap+3:leotap+3) = chaxis
                     ofile( icomp, jcomp )(leotap+4:leotap+6) = chdgrs
                     ofile( icomp, jcomp )(leotap+7:leotap+9) = chext
                  else
                     ofile( icomp, jcomp )(leotap+1:leotap+2) = '.r'
                     ofile( icomp, jcomp )(leotap+3:leotap+3) = chaxis
                     ofile( icomp, jcomp )(leotap+4:leotap+8) = '_var_'
                     ofile( icomp, jcomp )(leotap+9:leotap+11) = chext
                  endif
               endif

               call getln(loutij, ofile( icomp, jcomp ), 'w', 1)
               if ( verbos ) write(LER, 123) ofile(icomp, jcomp), loutij
               write(LERR,123) ofile(icomp, jcomp), loutij
 123           format (/' Output file produced:',/a80/
     :                  ' Logical unit:',i5/)
            endif
            luout(icomp, jcomp ) = loutij

c write line header

            call wrtape( loutij, ibufij, lbyout )
            if ( verbos ) then

c verbose output of all pertinent information before processing begins

               call verbal(nsamp, nsi, ntrcc, nrecc, iform,
     :              ifile( icomp, jcomp ), ofile( icomp, jcomp ),
     :              icomp, iaxis, idgrs, iwd, phitap, IKP)
            endif                
         ENDDO
      ENDDO

c BEGIN PROCESSING

c ASSUME that: irs, ire, nsamp, ntrcc, lbytes, nbytes,
c              iform, ns
c are the SAME for each component!

      if ( phitap .eq. ' ' ) then
         phi = float(idgrs) * degrad
         cosphi = cos(phi)
         sinphi = sin(phi)
 
         if ( iaxis .eq. 1) then
            r( 2, 2 ) = cosphi
            r( 2, 3 ) = sinphi
            r( 3, 2 ) = -sinphi
            r( 3, 3 ) = cosphi  
         elseif ( iaxis .eq. 2) then
            r( 1, 1 ) = cosphi
            r( 1, 3 ) = -sinphi
            r( 3, 1 ) = sinphi
            r( 3, 3 ) = cosphi
         elseif ( iaxis .eq. 3) then
            r( 1, 1 ) = cosphi
            r( 1, 2 ) = sinphi
            r( 2, 1 ) = -sinphi 
            r( 2, 2 ) = cosphi  
         endif
      endif

      if ( verbos ) write(LER, 135)  irs, ire, ns, ne
      write(LERR,135)  irs, ire, ns, ne
 135  format(/    ' Start record:',i5/
     :     ' End record:  ',i5/
     :     ' Start trace: ',i5/
     :     ' End Trace:   ',i5)

      DO JJ = irs, ire

c skip to start trace for all components
         
         do ic = 1, 2
            icomp = ncmp(ic)
            do jc = 1, 2
               jcomp = ncmp(jc)
               call trcskp ( JJ, 1, ns-1, luin(icomp,jcomp), ntrc, 
     :              ibufij )
            enddo
         enddo

         DO KK = ns, ne

            do ic = 1, 2
               icomp = ncmp(ic)
               do jc = 1, 2
                  jcomp = ncmp(jc) 
  
c read trace info (trace header + trace values in trcij)

                  nbytes = 0
                  call rtape( luin(icomp,jcomp), ibufij, nbytes)
                  call vmov ( lhedij(ITHWP1), 1, trcij, 1, nsamp)
                  if(nbytes .eq. 0) then
                     write(LERR,*)'End of file on input:'
                     write(LERR,*)'  rec= ',JJ,'  trace= ',KK
                     go to 9000
                  elseif (KK .eq. ns .and. verbos ) then
                     write(LERR,*)'  rec= ',JJ,'  trace= ',KK
                  endif

                  recnum = ibufij(l_RecNum)
                  trcnum = ibufij(l_TrcNum)

c save trace values as tensor
               
                  do n = 1, nsamp
                     trc( n, icomp, jcomp ) = trcij( n )
                  enddo

               enddo
            enddo

c retrieve angle for this record and form rotation matrix if 
c angle file attached otherwise use constant angle and matrix
c from above

            if ( phitap .ne. ' ' ) then
               call GetAngle ( recnum, Record, Angle, NumEntries, 
     :              degree )
               write(LERR,*) ' Record = ',recnum,' rotation angle = '
     :              ,degree, ' degrees'
               theta = degree * degrad
               costh = cos(theta)
               sinth = sin(theta)
               if ( iaxis .eq. 1) then
                  r( 2, 2 ) = costh
                  r( 2, 3 ) = sinth
                  r( 3, 2 ) = -sinth
                  r( 3, 3 ) = costh  
               elseif ( iaxis .eq. 2) then
                  r( 1, 1 ) = costh
                  r( 1, 3 ) = -sinth
                  r( 3, 1 ) = sinth
                  r( 3, 3 ) = costh
               elseif ( iaxis .eq. 3) then
                  r( 1, 1 ) = costh
                  r( 1, 2 ) = sinth
                  r( 2, 1 ) = -sinth
                  r( 2, 2 ) = costh
               endif
            endif
      
c rotate traces from all components "like a tensor" thru 
c same angle at all samples

            do n = 1, nsamp
               do ic = 1, 2
                  icomp = ncmp(ic)
                  do jc = 1, 2
                     jcomp = ncmp(jc)
                     result = 0.0
                     do kc = 1, 2
                        kcomp = ncmp(kc)
                        do mc = 1, 2
                           mcomp = ncmp(mc)
                           result = result + 
     :                  r(icomp,kcomp)*trc(n,kcomp,mcomp)*r(jcomp,mcomp)
                        enddo
                     enddo
                     trcout( n, icomp, jcomp ) = result
                  enddo
               enddo
            enddo

c write output data for each component

            do ic = 1, 2
               icomp = ncmp(ic)
               do jc = 1, 2
                  jcomp = ncmp(jc)
                  do n = 1, nsamp
                     trcij( n ) = trcout( n, icomp, jcomp )
                  enddo

c assign header values

                  ibufij(l_WDepDP) = 10*icomp + jcomp

                  if ( phitap .ne. ' ' ) then
                     ibufij(l_ShtDep) = nint(degree) 
                  else
                     ibufij(l_ShtDep) = idgrs
                  endif

                  call vmov   (trcij, 1, lhedij(ITHWP1), 1, nsamp)
                  call wrtape (luout(icomp,jcomp), ibufij, nbytes)
               enddo
            enddo

            if(verbos) write(LERR,*)'ri ',recnum,' trace ',trcnum
         ENDDO

c skip to end of record on all input components

         do ic = 1, 2
            icomp = ncmp(ic)
            do jc = 1, 2
               jcomp = ncmp(jc)
               call trcskp ( JJ, ne+1, ntrc, luin(icomp,jcomp), ntrc, 
     :              ibufij )
            enddo
         enddo

      ENDDO
 
c close data files

9000  continue

      do ic = 1, 2
         icomp = ncmp(ic)
         do jc = 1, 2
            jcomp = ncmp(jc)
            call lbclos ( luin( icomp, jcomp ) )
            call lbclos (luout( icomp, jcomp ) )
         enddo
      enddo

      if ( phitap .ne. ' ' ) close(luphi)

      write( LERR, * ) ' '
      write( LERR, * ) ' end of rottnsr, processed ',nrec,' record(s)'
      write( LERR, * ) ' with ',ntrc , ' traces.'
      write( LERR, * ) ' '
      write( LER, * ) ' '
      write( LER, * ) ' rottnsr: Normal Termination'
      write( LER, * ) ' '
      stop


 990  continue
      write(LERR,*)' '
      write(LERR,*)' ROTTNSR: Error opening Angle file'
      write(LERR,*)'          Check existance/permissions and '
      write(LERR,*)'          rerun.'
      write(LERR,*)' '
      write(LER,*)' '
      write(LER,*)' ROTTNSR: Error opening Angle file'
      write(LER,*)'          Check existance/permissions and '
      write(LER,*)'          rerun.'
      write(LER,*)' '
      stop

      end
