C***********************************************************************
C                 copyright 2001, Amoco Production Company             *
C                             All Rights Reserved                      *
C                     an affiliate of BP America Inc.                  *
C***********************************************************************
**********************************************************************c
c
c rotvctr reads seismic trace data from a 1s x 2r set of input files,
c     rotates these about the orthogonal axis, and writes the results 
c     in a 1x2 set of output files.
c
c**********************************************************************c
c
c     SZSMPM	- maximum number of trace samples
c     SZLNHD	- line file size = (SZTRHD + SZSAMP*MAXSMP)/2
c     SZTRHD	- size of trace header in pipes in bytes
c     SZSMPD   - size of sample in pipe (in bytes)
c        LER  is the current window.
c        LOT  is used for error reporting; also comes to the window.
c        LERR is a standard output file; will appear in directory as
c             ROTV.pid1.
c
c     declare variables
c
#include <save_defs.h>
#include <f77/iounit.h>
#include <f77/lhdrsz.h>
#include <f77/sisdef.h>

c declare standard usp variables
 
      integer * 2  ibufij( SZLNHD )                           
      integer  lhedij( SZLNHD )
      integer  nsamp, nsi, ntrc, nrec, iform
      integer  linij, loutij, lbytes, nbytes, lbyout
      integer  irs, ire, ns, ne
      integer  luin( 3, 3 ), luout( 3, 3 )
      integer  argis, pipei(2), pipeo(2)

      real     headij( SZLNHD )
      real     trcij ( SZLNHD )

      character    otap*255

      logical      verbos, query, IKP

c declare program specific variables

      integer  jcmp(2), iaxis, idgrs, isrc, recnum, trcnum , leotap
      integer  luphi, jc, jcomp, JJ, KK, length, NumEntries

      real     trc( SZLNHD , 3, 3 )
      real     r( 3, 3 ), trcout( SZLNHD , 3, 3 )
      real     pi, degrad, theta, costh, sinth, degree 
      real     Record(SZLNHD), Angle(SZLNHD)

      character     chaxis*1, chdgrs*3, chext*3, name*7, iwd*6
      character*255 casert, phitap, ifile( 3, 3 ), ofile( 3, 3 )
 
      equivalence ( ibufij( 1 ),  lhedij( 1 ), headij( 1 ) )

c initialize variables

      data lbytes / 0 /
      data nbytes / 0 /
      data name/'ROTVCTR'/
      data pipei/0,3/
      data pipeo/1,4/

      pi = 4. * atan(1.0)
      degrad = pi / 180.

c issue command line help if requested 

      query = ( argis( '-?' ) .gt. 0 .or. argis( '-h' ) .gt. 0 )
      if ( query ) then
         call help()
         stop
      endif
 
c open printout files

#include <f77/open.h>
 
c read command line arguments

      call cmdln ( casert, otap, phitap, ns, ne, irs, ire, verbos,
     :     isrc, iaxis, idgrs, iwd, IKP)

c open angle file it attached

      if ( phitap .ne. ' ' ) then
         call alloclun(luphi)
         length = lenth(phitap)
         open ( luphi, file=phitap(1:length), status='old', err=990)
         call ReadAngles(luphi, Record, Angle, NumEntries)
      endif

c restrict angle

      IF ( phitap .ne. ' ' ) then
         do i = 1, NumEntries
 40         if (Angle(i) .gt. 90. ) then
               Angle(i) = Angle(i) - 180.
               if ( verbos ) write (LER ,45) Record(i)
               write (LERR,45) Record(i)
 45            format(' Rotation angle at record ',f7.2,' adjusted
     : to lie between +- 90 degrees')
               go to 40
            elseif (Angle(i) .lt. -90 ) then
               Angle(i) = Angle(i) + 180.
               if ( verbos ) write (LER ,45)
               write (LERR,45)
               go to 40
            endif
         enddo
      ELSE
 50      if (idgrs .gt. 90 ) then
            idgrs = idgrs - 180
            if ( verbos ) write (LER ,55)
            write (LERR,55)
 55         format(' Rotation angle adjusted to lie between +- 90 degree
     :s')
            go to 50
         elseif (idgrs .lt. -90 ) then
            idgrs = idgrs + 180
            if ( verbos ) write (LER ,55)
            write (LERR,55)
            go to 50
         endif
      ENDIF
      
      if ( phitap .eq. ' ' ) then
         if ( verbos ) write(LER, 60)  casert, isrc, idgrs, iaxis
         write(LERR,60)  casert, isrc, idgrs, iaxis
 60      format(/' Case-root:'/a80/
     :        ' Source-component:',i2/
     :        ' (Right-handed) rotation angle:',i4,' degrees',
     :        ' about positive axis', i2)
      endif

c get components for this case

      call strend(casert,lstr)
      write(chdgrs, '(i3)') idgrs
      if ( chdgrs(2:2) .eq. '-' ) then
         chdgrs(1:1) = '-'
         chdgrs(2:2) = '0'
      endif
      if ( chdgrs(1:1) .eq. ' ' ) chdgrs(1:1) = '+'
      if ( chdgrs(2:2) .eq. ' ' ) chdgrs(2:2) = '0'

c do for each component

      DO jc = 1, 2

c select axes

         if (iaxis .eq. 1) then
            if (jc .eq. 1) then
               jcomp = 2
            else
               jcomp = 3
            endif
         elseif (iaxis .eq. 2) then
            if (jc .eq. 1) then
               jcomp = 1
            else
               jcomp = 3
            endif
         elseif (iaxis .eq. 3) then
            if (jc .eq. 1) then
               jcomp = 1
            else
               jcomp = 2
            endif
         endif
         jcmp(jc) = jcomp
         
c get logical unit number for input 

         if (IKP) then
            call sisfdfit (linij , pipei(jc))
         else

c construct filename.extension

            ext = .1*isrc +.01*jcomp
            write(chext,'(f3.2)') ext
            write(chaxis,'(i1)') iaxis
            ifile( isrc, jcomp )=casert
            ifile( isrc, jcomp )(lstr+1:lstr+3) = chext
            if ( verbos ) write(LER, 121) ifile(isrc, jcomp)
            write(LERR,121) ifile(isrc, jcomp)
 121        format (/' Input file required:'/a80)
            call getln(linij,  ifile( isrc, jcomp ), 'r', 0)
         endif
         
         luin( isrc, jcomp ) = linij
         
c read line header of input, check, and output 

         call rtape (linij, ibufij, lbytes )            
         if (lbytes .eq. 0) then
            write(LOT,*)'ROTVCTR: no header read from unit ',linij
            write(LOT,*)'FATAL'
            stop
         endif
         
c save certain lineheader parameters

         call saver(ibufij, 'NumSmp', nsamp, LINHED)
         call saver(ibufij, 'SmpInt', nsi  , LINHED)
         call saver(ibufij, 'NumTrc', ntrc , LINHED)
         call saver(ibufij, 'NumRec', nrec , LINHED)
         call saver(ibufij, 'Format', iform, LINHED)
         
         call savelu('TrcNum',ifmt,l_TrcNum,length,TRACEHEADER)
         call savelu(iwd,ifmt,l_RecNum,length,TRACEHEADER)
         call savelu('SrcLoc',ifmt,l_SrcLoc,length,TRACEHEADER)
         call savelu('RecInd',ifmt,l_RecInd,length,TRACEHEADER)
         call savelu('DphInd',ifmt,l_DphInd,length,TRACEHEADER)
         call savelu('DstSgn',ifmt,l_DstSgn,length,TRACEHEADER)
         call savelu('StaCor',ifmt,l_StaCor,length,TRACEHEADER)
         call savelu('WDepDP',ifmt,l_WDepDP,length,TRACEHEADER)  
         call savelu('ShtDep',ifmt,l_ShtDep,length,TRACEHEADER)

c ensure that command line values are compatible with data set

         call cmdchk ( ns, ne, irs, ire, ntrc, nrec )

c modify line header to reflect actual number of traces output

         nrecc=ire - irs+1
         ntrcc=ne-ns+1
         call savew(ibufij, 'NumRec', nrecc, LINHED)
         call savew(ibufij, 'NumTrc', ntrcc, LINHED)

c write progname (from command line) to historical line hdr

         call savhlh( ibufij, lbytes, lbyout)

c skip unwanted records

         call recskp(1, irs-1, luin(isrc,jcomp), ntrc, ibufij )

c get logical unit number for output file

         if (IKP) then
            call sisfdfit (loutij, pipeo(jc))
         else 

c construct filename.extension

            if ( otap .eq. ' ' ) then
               ofile( isrc, jcomp ) = casert
               if ( phitap .eq. ' ' ) then
                  ofile( isrc, jcomp )(lstr+1:lstr+2) = '.r'
                  ofile( isrc, jcomp )(lstr+3:lstr+3) = chaxis
                  ofile( isrc, jcomp )(lstr+4:lstr+6) = chdgrs
                  ofile( isrc, jcomp )(lstr+7:lstr+9) = chext
               else
                  ofile( isrc, jcomp )(lstr+1:lstr+2) = '.r'
                  ofile( isrc, jcomp )(lstr+3:lstr+3) = chaxis
                  ofile( isrc, jcomp )(lstr+4:lstr+8) = '_var_'
                  ofile( isrc, jcomp )(lstr+9:lstr+11) = chext
               endif
            else
               leotap = lenth(otap)
               ofile( isrc, jcomp ) = otap(1:leotap)
               if ( phitap .eq. ' ' ) then
                  ofile( isrc, jcomp )(leotap+1:leotap+2) = '.r'
                  ofile( isrc, jcomp )(leotap+3:leotap+3) = chaxis
                  ofile( isrc, jcomp )(leotap+4:leotap+6) = chdgrs
                  ofile( isrc, jcomp )(leotap+7:leotap+9) = chext
               else
                  ofile( isrc, jcomp )(leotap+1:leotap+2) = '.r'
                  ofile( isrc, jcomp )(leotap+3:leotap+3) = chaxis
                  ofile( isrc, jcomp )(leotap+4:leotap+8) = '_var_'
                  ofile( isrc, jcomp )(leotap+9:leotap+11) = chext
               endif
            endif
            
            call getln(loutij, ofile( isrc, jcomp ), 'w', 1)
            if (verbos) write(LER, 123) ofile(isrc, jcomp), loutij
            write(LERR,123) ofile(isrc, jcomp), loutij
 123        format (/' Output file produced:',/a80/
     :           ' Logical unit:',i5/)
         endif

         luout(isrc, jcomp ) = loutij

c write line header

         call wrtape( loutij, ibufij, lbyout )

c verbose output of all pertinent information before processing begins

         call verbal(nsamp, nsi, ntrcc, nrecc, iform,
     :        ifile( isrc, jcomp ), ofile( isrc, jcomp ),
     :        isrc, iaxis, idgrs, iwd, phitap, IKP )
         
      ENDDO

c BEGIN PROCESSING

c ASSUME that: irs, ire, nsamp, ntrcc, lbytes, nbytes,
c              iform, ns
c are the SAME for each component!

      if ( phitap .eq. ' ' ) then
         theta = float(idgrs) * degrad
         costh = cos(theta)
         sinth = sin(theta)
 
         if ( iaxis .eq. 1) then
            r( 2, 2 ) = costh
            r( 2, 3 ) = sinth
            r( 3, 2 ) = -sinth
            r( 3, 3 ) = costh  
         elseif ( iaxis .eq. 2) then
            r( 1, 1 ) = costh
            r( 1, 3 ) = -sinth
            r( 3, 1 ) = sinth
            r( 3, 3 ) = costh
         elseif ( iaxis .eq. 3) then
            r( 1, 1 ) = costh
            r( 1, 2 ) = sinth
            r( 2, 1 ) = -sinth
            r( 2, 2 ) = costh
         endif
      endif
      
      if ( verbos ) write(LER, 135)  irs, ire, ns, ne
    
      write(LERR,135)  irs, ire, ns, ne
 135  format(/    ' Start record:',i5/
     :     ' End record:  ',i5/
     :     ' Start trace: ',i5/
     :     ' End Trace:   ',i5)

      DO JJ = irs, ire

c skip to start trace for all components
         
         do jc = 1, 2
            jcomp = jcmp(jc)
            call trcskp ( JJ, 1, ns-1, luin(isrc,jcomp), ntrc, 
     :           ibufij )
         enddo

         DO KK = ns, ne

c read trace from each component

            do jc = 1, 2

c read trace info (trace header + trace values in trcij)

               nbytes = 0
               jcomp = jcmp(jc)   
               call rtape( luin(isrc,jcomp), ibufij, nbytes)
               if(nbytes .eq. 0) then
                  write(LERR,*)'End of file on input:'
                  write(LERR,*)'  rec= ',JJ,'  trace= ',KK
                  go to 9000
               elseif (KK .eq. ns .and. verbos ) then
                  write(LERR,*)'  rec= ',JJ,'  trace= ',KK
               endif
               call vmov ( lhedij(ITHWP1), 1, trcij, 1, nsamp)
               
               recnum = ibufij(l_RecNum)
               trcnum = ibufij(l_TrcNum)
               
c save trace values as tensor
               
               do n = 1, nsamp
                  trc( n, isrc, jcomp ) = trcij( n )
               enddo
            enddo

c retrieve angle for this record and form rotation matrix if 
c angle file attached otherwise use constant angle and matrix
c from above

               if ( phitap .ne. ' ' ) then
                  call GetAngle ( recnum, Record, Angle, NumEntries, 
     :                 degree )
                  write(LERR,*) ' Record = ',recnum,' rotation angle = '
     :                 ,degree, ' degrees'
                  theta = degree * degrad
                  costh = cos(theta)
                  sinth = sin(theta)
                   if ( iaxis .eq. 1) then
                     r( 2, 2 ) = costh
                     r( 2, 3 ) = sinth
                     r( 3, 2 ) = -sinth
                     r( 3, 3 ) = costh  
                  elseif ( iaxis .eq. 2) then
                     r( 1, 1 ) = costh
                     r( 1, 3 ) = -sinth
                     r( 3, 1 ) = sinth
                     r( 3, 3 ) = costh
                  elseif ( iaxis .eq. 3) then
                     r( 1, 1 ) = costh
                     r( 1, 2 ) = sinth
                     r( 2, 1 ) = -sinth
                     r( 2, 2 ) = costh
                  endif
               endif
      
c rotate traces from all components "like a vector" thru 
c same angle at all samples

            do n = 1, nsamp
               do jc = 1, 2
                  jcomp = jcmp(jc)
                  result = 0.0
                  do kc = 1,2
                     kcomp = jcmp(kc)
                     result = result + r(jcomp,kcomp)*trc(n,isrc,kcomp)
                  enddo
                  trcout( n, isrc, jcomp ) = result
               enddo
            enddo

c write output data for each component

            do jc = 1, 2
               jcomp = jcmp(jc)
               do n = 1, nsamp
                  trcij( n ) = trcout( n, isrc, jcomp )
               enddo

c assign header values

               ibufij(l_WDepDP) = 10*isrc + jcomp

               if ( phitap .ne. ' ' ) then
                  ibufij(l_ShtDep) = nint(degree) 
               else
                  ibufij(l_ShtDep) = idgrs
               endif

               call vmov   (trcij, 1, lhedij(ITHWP1), 1, nsamp)
               call wrtape (luout(isrc,jcomp), ibufij, nbytes)
            enddo

            if(verbos) write(LERR,*)'ri ',recnum,' trace ',trcnum

         ENDDO

c skip to end of record on all input components

         do jc = 1,2
            jcomp = jcmp(jc)
            call trcskp ( JJ, ne+1, ntrc, luin(isrc,jcomp), ntrc, 
     :           ibufij )
         enddo

      ENDDO
 
c close data files

9000  continue

      do jc=1,2
         jcomp = jcmp(jc)
         call lbclos ( luin( isrc, jcomp ) )
         call lbclos (luout( isrc, jcomp ) )
      enddo

      if ( phitap .ne. ' ' ) close(luphi)

      write( LERR, * ) ' '
      write( LERR, * ) ' end of rotvctr, processed ',nrec,' record(s)'
      write( LERR, * ) ' with ',ntrc , ' traces.'
      write( LERR, * ) ' '
      write( LER, * ) ' '
      write( LER, * ) ' rotvctr: Normal Termination'
      write( LER, * ) ' '
      stop

 990  continue
      write(LERR,*)' '
      write(LERR,*)' ROTVCTR: Error opening Angle file'
      write(LERR,*)'          Check existance/permissions and '
      write(LERR,*)'          rerun.'
      write(LERR,*)' '
      write(LER,*)' '
      write(LER,*)' ROTVCTR: Error opening Angle file'
      write(LER,*)'          Check existance/permissions and '
      write(LER,*)'          rerun.'
      write(LER,*)' '
      stop

      end
