C***********************************************************************
C                 copyright 2001, Amoco Production Company             *
C                             All Rights Reserved                      *
C                     an affiliate of BP America Inc.                  *
C***********************************************************************
c**********************************************************************c
c
c rad2slnt reads seismic trace plane wave data from an input file,
c either interpolates either to a constant angle grid (or in reverse
c mode to constant ray parameter) and
c writes the results to an output file
c
c
c**********************************************************************c
c
c     declare variables
c
#include <save_defs.h>
#include <f77/iounit.h>
#include <f77/lhdrsz.h>
#include <f77/sisdef.h>
 
      integer     itr  ( SZLNHD )
      integer     itrv ( SZLNHD )
      integer     lhed ( SZLNHD )
      integer     lhedv( SZLNHD )
      real        head ( SZLNHD )
      real        headv( SZLNHD )
      integer     nsamp, nsi, ntrc, nrec, iform,obytes
      integer     luin , luout, lbytes, nbytes, lbyout
#include <f77/pid.h>
      integer     recnum, static
      real        tri ( SZLNHD ), weight ( SZLNHD )
      real        vel ( SZLNHD )
      real        p   ( SZLNHD )
      real        tabl1 (SZLNHD), tabl2(SZLNHD), zz(4*SZLNHD)
      integer     iz(SZLNHD), key(SZLNHD)
      integer     mute(SZLNHD)

      integer     IVREF,NP,IPSTRT,IPINC

c------
c  dynamic memory allocation for big arrays, eg whole records

      real        bigar1
      pointer     (wkadr1, bigar1(1))
      real        itrhdr
      pointer     (wkadri, itrhdr(1))
c------


      character     ntap * 256, otap * 256, name*8
      character     vtap * 256, tag * 1
	
      logical       verbos, query, heap1, rev, TV
      logical       split, neg, first
      integer       argis, pipe
 
c     equivalence ( itr(129), tri (1) )
      equivalence ( itr(  1), lhed(1), head(1) )
      equivalence ( itrv(  1), lhedv(1), headv(1) )

      data lbytes / 0 /, nbytes / 0 /, name/'RAD2SLNT'/
      data rev/.false./
      data pipe/3/
      data TV/.false./
      data split/.false./
      data neg/.false./
      data first/.true./

      deg = 180. / 3.14159265
      rad = 1. / deg
c-----
c     read program parameters from command line card image file
c-----
      query = ( argis ( '-?' ) .gt. 0 )
      if ( query )then
            call help()
            stop
      endif
 
c-----
c     open printout files
c-----
#include <f77/open.h>
 
      call gcmdln(ntap,otap,vrefi,angs,ange,dang,vtap,
     1            taper,xmax,dxrec,vmute,rev,verbos,TV)
c-----
c     get logical unit numbers for input and output
c-----
      call getln(luin , ntap,'r', 0)
      call getln(luout, otap,'w', 1)

      if (TV) then
         if (vtap .ne. ' ') then
             call getln(luvel, vtap, 'r',-1)
         else
             write(LERR,*)'rad2slnt assumed to be running inside IKP'
             call sisfdfit (luvel, pipe)
         endif
         if(luvel .lt. 0)   then
            write(LERR,*)
     1      'rad2slnt error: velocity file -v not accessible'
         endif
      endif

c-----
c     read line header of input
c     save certain parameters
c-----
      call rtape  ( luin, itr, lbytes)
      if(lbytes .eq. 0) then
         write(LER,*)'rad2slnt: no header read from unit ',luin
         write(LER,*)'FATAL'
         stop
      endif
      call saver(itr, 'NumSmp', nsamp, LINHED)
      call saver(itr, 'SmpInt', nsi  , LINHED)
      call saver(itr, 'NumTrc', ntrc , LINHED)
      call saver(itr, 'NumRec', nrec , LINHED)
      call saver(itr, 'Format', iform, LINHED)
      call saver(itr, 'UnitSc', unitsc, LINHED)
      if (unitsc .eq. 0.0) then
          write(LERR,*)'********************************************'
          write(LERR,*)'WARNING: sample unit scaler in LH = ',unitsc
          write(LERR,*)'         will set to .001 (millisec default)'
          write(LERR,*)'********************************************'
          unitsc = .001
          call savew(itr, 'UnitSc', unitsc, LINHED)
      endif

      do  kk = 1, ntrc
          key (kk) = kk
      enddo
c-----
c     read line header of velocity
c     save certain parameters
c-----
      IF (TV) THEN
         call rtape  ( luvel, itrv, lbytev)
         if(lbytev .eq. 0) then
            write(LER,*)'rad2slnt: no header read from vel unit ',luvel
            write(LER,*)'FATAL'
            stop
         endif
         call saver(itrv, 'NumSmp', nsampv, LINHED)
         call saver(itrv, 'SmpInt', nsiv  , LINHED)
         call saver(itrv, 'NumTrc', ntrcv , LINHED)
         call saver(itrv, 'NumRec', nrecv , LINHED)
         call saver(itrv, 'Format', iformv, LINHED)

         if (nsiv .ne. nsi) then
            write(LERR,*)'rad2slnt: FATAL ERROR'
            write(LERR,*)'Velocity sample interval not equal to input'
            write(LER ,*)'rad2slnt: FATAL ERROR'
            write(LER ,*)'Velocity sample interval not equal to input'
            stop
         endif
         if (ntrcv*nrecv .ne. nrec) then
            write(LERR,*)'WARNING from rad2slnt:'
            write(LERR,*)'Number velocity trace different from input'
            write(LERR,*)'number of recs. Will discard unused vel'
            write(LERR,*)'traces or repeat last trace'
         endif
         if (nsampv .ne. nsamp) then
            write(LERR,*)'WARNING from rad2slnt:'
            write(LERR,*)'Number velocity samples different from input'
            write(LERR,*)'input data. Will adjust length of vel traces'
            write(LER ,*)'WARNING from rad2slnt:'
            write(LER ,*)'Number velocity samples different from input'
            write(LER ,*)'input data. Will adjust length of vel traces'
         endif
      ENDIF

      ntrco = ntrc
      nrecc = nrec

c******
      IF (.not. TV) THEN

      if ( rev) then

         call saver (itr, 'CDPFld' , ntrco , LINHED)
         call saver (itr, 'WatVel' , ivref , LINHED)
         call saver (itr, 'MutVel' , ivmute, LINHED)
         call saver (itr, 'RATTrc' , iangs , LINHED)
         call saver (itr, 'RATFld' , idang , LINHED)
         call saver (itr, 'Dx1000' , idxrec, LINHED)

         if (vrefi .ne. 0.) ivref = vrefi
         vref  = ivref
         if (angs .eq. 0.) angs  = iangs
         if (dang .eq. 0.) dang = idang
         if (dang .eq. 0.) then
            write(LER,*)'rad2slnt error: reverse mode (slnt -> radonf)'
            write(LER,*)'Angle increment is zero!  Probably original'
            write(LER,*)'value was a fraction which got rounded to zero'
            write(LER,*)'when stuffed into line header.  Rerun rad2slnt'
            write(LER,*)'specifying -da[] on cmd line'
            stop
         endif
         if (ange .eq. 0.) then
             ange  = angs + (ntrco-1) * dang
         else
             dang = abs (ange - angs) / float(ntrco-1)
         endif
         ps    = sin (rad * angs) / vref
         pe    = sin (rad * ange) / vref
         pang  = (pe - ps) / float(ntrco - 1)
         vs    = 1. / abs(ps)
         ve    = 1. / abs(pe)
         ixmax = xmax
         itmax = 1000. * pe * xmax
         itmin = 1000. * ps * xmax

         write(LERR,*)' '
         write(LERR,*)'slnt input:'
         write(LERR,*)'Reference Velocity=  ',vref
         write(LERR,*)'Start angle       =  ',angs
         write(LERR,*)'Angle increment   =  ',dang
         write(LERR,*)'Number angles     =  ',ntrco
         write(LERR,*)'Mute velocity     =  ',ivmute
         write(LERR,*)' '
         write(LERR,*)'Max distance      =  ',xmax
         write(LERR,*)'ENd angle         =  ',ange
         write(LERR,*)'Start ray parameter=  ',ps
         write(LERR,*)'End ray parameter  =  ',pe
         write(LERR,*)'Min moveout time   =  ',itmin
         write(LERR,*)'Max moveout time   =  ',itmax
         write(LERR,*)' '

         call savew (itr, 'MxRSEL' , ixmax , LINHED)
         call savew (itr, 'MxTrSt' , itmax , LINHED)
         call savew (itr, 'MnTrSt' , itmin , LINHED)

c******
      else
c******
         call saver (itr, 'Crew04' , tag   , LINHED)
         if (tag .eq. 'F') then

            call saver (itr, 'WatVel' , ivref , LINHED)
            call saver (itr, 'MxTrSt' , itmax , LINHED)
            call saver (itr, 'MnTrSt' , itmin , LINHED)
            tp1  = float (itmin) / 10000000.
            tp2  = float (itmax) / 10000000.
            vref = float (ivref)
            vs = 1. / tp1
            ve = 1. / tp2

         else

            call saver (itr, 'MxTrOf' , ixmax , LINHED)
            call saver (itr, 'MxGrEl' , mmax  , LINHED)
            call saver (itr, 'MnGrEl' , mmin  , LINHED)

   
            xmax = ixmax
            tmax = float(mmax)
            tmin = float(mmin)
            tmxa = amax1 (abs(tmax),abs(tmin))
   
            if (tmin .ne. 0.) then
                vs   = 1000. * xmax / tmin
            else
                vs   = 10000000.
            endif
            if (tmax .ne. 0.) then
                ve   = 1000. * xmax / tmax
            else
                ve   = 10000000.
            endif
   
            if (tmxa .eq. 0.) then
               vref = 0.
            else
               vref = 1000. * xmax / tmxa
            endif

         endif
     
         ps   = 1. / vs
         pe   = 1. / ve
         pang = (pe - ps) / float(ntrc-1)

         write(LERR,*)' '
         write(LERR,*)'radonf input:'
         write(LERR,*)'Max distance      =  ',xmax
         write(LERR,*)'Max time (ms)     =  ',tmax
         write(LERR,*)'Min time (ms)     =  ',tmin
         write(LERR,*)'Reference velocity=  ',vref
         write(LERR,*)'Start ray param   =  ',ps
         write(LERR,*)'End ray param     =  ',pe
         write(LERR,*)'Ray param inc     =  ',pang
         write(LERR,*)' '

         if (vrefi .ne. 0.) vref = vrefi

         if (angs .eq. 0.) then
            angs = deg * asin ( ps * vref )
         endif
         if (ange .eq. 0.) then
            ange = deg * asin ( pe * vref )
         endif
         if (dang .eq. 0.) then
            dang   = (ange - angs) / float(ntrc-1)
            ntrco = ntrc
         else
            ntrco = abs(ange - angs) / dang + 1
         endif

         ivref  = vref
         ivmute = vmute
         iangs  = angs
         idang  = dang
         idxrec = dxrec
         IPSTRT = angs
         IPINC  = dang
         NP     = ntrc

         write(LERR,*)'slnt output:'
         write(LERR,*)'Reference velocity=  ',vref
         write(LERR,*)'Mute velocity     =  ',vmute
         write(LERR,*)'Start angle       =  ',angs
         write(LERR,*)'End   angle       =  ',ange
         write(LERR,*)'Number angles     =  ',ntrco
         write(LERR,*)'Angle increment   =  ',dang
         write(LERR,*)'Shot spacing      =  ',dxrec
         write(LERR,*)' '

         call savew (itr, 'CDPFld' , ntrco , LINHED)
         call savew (itr, 'WatVel' , ivref , LINHED)
         call savew (itr, 'MutVel' , ivmute, LINHED)
         call savew (itr, 'RATTrc' , iangs , LINHED)
         call savew (itr, 'RATFld' , idang , LINHED)
         call savew (itr, 'Dx1000' , idxrec, LINHED)

      endif
c******
      ENDIF

      call savew(itr, 'NumRec', nrecc , LINHED)


c     For trace header values we take mnemonics and build a
c     set of pointers to an I*2 array equivalenced to the
c     RTAPE  Integer array (headers + data)
c     TRACEHEADER is a value in the include file <sisdef.h> that
c     refers to the trace header
 
      call savelu('TrcNum',ifmt_TrcNum,l_TrcNum,ln_TrcNum,TRACEHEADER)
      call savelu('RecNum',ifmt_RecNum,l_RecNum,ln_RecNum,TRACEHEADER)
      call savelu('SrcLoc',ifmt_SrcLoc,l_SrcLoc,ln_SrcLoc,TRACEHEADER)
      call savelu('SrcPnt',ifmt_SrcPnt,l_SrcPnt,ln_SrcPnt,TRACEHEADER)
      call savelu('SoPtNm',ifmt_SoPtNm,l_SoPtNm,ln_SoPtNm,TRACEHEADER)
      call savelu('RecInd',ifmt_RecInd,l_RecInd,ln_RecInd,TRACEHEADER)
      call savelu('DphInd',ifmt_DphInd,l_DphInd,ln_DphInd,TRACEHEADER)
      call savelu('DstSgn',ifmt_DstSgn,l_DstSgn,ln_DstSgn,TRACEHEADER)
      call savelu('DstUsg',ifmt_DstUsg,l_DstUsg,ln_DstUsg,TRACEHEADER)
      call savelu('StaCor',ifmt_StaCor,l_StaCor,ln_StaCor,TRACEHEADER)
      call savelu('SrPtXC',ifmt_SrPtXC,l_SrPtXC,ln_SrPtXC,TRACEHEADER)
      call savelu('SrPtYC',ifmt_SrPtYC,l_SrPtYC,ln_SrPtYC,TRACEHEADER)
      call savelu('RcPtXC',ifmt_RcPtXC,l_RcPtXC,ln_RcPtXC,TRACEHEADER)
      call savelu('RcPtYC',ifmt_RcPtYC,l_RcPtYC,ln_RcPtYC,TRACEHEADER)
      call savelu('SrRcMX',ifmt_SrRcMX,l_SrRcMX,ln_SrRcMX,TRACEHEADER)
      call savelu('SrRcMY',ifmt_SrRcMY,l_SrRcMY,ln_SrRcMY,TRACEHEADER)
      call savelu('CDPBCX',ifmt_CDPBCX,l_CDPBCX,ln_CDPBCX,TRACEHEADER)
      call savelu('CDPBCY',ifmt_CDPBCY,l_CDPBCY,ln_CDPBCY,TRACEHEADER)
      call savelu('VPick2',ifmt_VPick2,l_VPick2,ln_VPick2,TRACEHEADER)


      call hlhprt (itr, lbytes, name, 8, LERR)

c-----
c     modify historical line header
c-----

      obytes = SZTRHD + nsamp * SZSMPD
      call savhlh(itr,lbytes,lbyout)
      call wrtape ( luout, itr, lbyout                 )

      ntrcm = max0 (ntrc, ntrco)

      do  j = 1, ntrco
          weight (j) = 1.0
      enddo

      IF (.not. TV) THEN

c-----
c     build interpolation tables
c-----
      if (rev) then
         taper = taper / 100.
         itaper = ntrco * taper
         left = ntrco - itaper
         do  j = left, ntrco
             scl = float(ntrco - j) / float(itaper)
             weight (j) = scl
         enddo
      endif

      ENDIF
c---------------------------------------------------
c  malloc only space we're going to use
      heap1 = .true.
 
c--------------------------
c  note: these don't
c  have to be the same size
 
      item1 = ntrcm  * nsamp * SZSMPD
      itemi = ntrcm  * ITRWRD * SZSMPD
 
c  note also SZSMPD is the native
c  size of a float or int in bytes
c--------------------------
 
c--------
c  galloc - general allocation (machine independent since it uses C
c  malloc internally
c  inputs to galloc are pointer, number of bytes to allocate
c  outputs are error codes:
c     errcod = 1  (allocation succeeded)
c     errcod = 0  (allocation failed)
c--------
 
      call galloc (wkadr1, item1, errcd1, abort1)
      if (errcd1 .ne. 0.) heap1 = .false.

      call galloc (wkadri, itemi, errcdi, aborti)
      if (errcdi .ne. 0.) heap1 = .false.
 
      if (.not. heap1) then
         write(LERR,*)' '
         write(LERR,*)'Unable to allocate workspace:'
         write(LERR,*) item1,'  bytes'
         write(LERR,*) itemi,'  bytes'
         write(LERR,*)' '
         go to 999
      else
         write(LERR,*)' '
         write(LERR,*)'Allocating workspace:'
         write(LERR,*) item1,'  bytes'
         write(LERR,*) itemi,'  bytes'
         write(LERR,*)' '
      endif

c-----
c     verbose output of all pertinent information before
c     processing begins
c-----
c     if( verbos ) then
            call verbal(nsamp, nsi, ntrc, nrec, iform,
     1                  ntap,otap)
c     end if

c-----
c     BEGIN PROCESSING
c     read trace, interpolate or decimate, write to output file
c-----
                     icinit = 1
c-----
c     process desired trace records
c-----
      DO   JJ = 1, nrec
 
            call vclr (bigar1, 1,  ntrcm*nsamp)

            if ( TV ) then
               nbytev = 0
               call rtape( luvel, itrv, nbytev)
               if(nbytev .ne. 0) then
                  call vmov (lhedv(ITHWP1), 1, vel, 1, nsampv)
                  if (nsampv .lt. nsamp) then
                     do  ii = nsampv+1, nsamp
                         vel (ii) = vel (nsampv)
                     enddo
                  endif
               endif
            endif

            do   kk = 1, ntrc

                  nbytes = 0
                  call rtape( luin, itr, nbytes)
                  if(nbytes .eq. 0) then
                     write(LERR,*)'End of file on input:'
                     write(LERR,*)'  rec= ',jj,'  trace= ',kk
                     go to 999
                  endif
                  call vmov (lhed(ITHWP1), 1, tri, 1, nsamp)

                  call saver2(lhed,ifmt_StaCor,l_StaCor, ln_StaCor,
     1                        static , TRACEHEADER)
                  call saver2(lhed,ifmt_DstUsg,l_DstUsg, ln_DstUsg,
     1                        idis   , TRACEHEADER)
                  p (kk) = float (idis) / 10000000.

                  if (rev) then
                     call saver2(lhed,ifmt_VPick2,l_VPick2, ln_VPick2,
     1                           mute(kk), TRACEHEADER)
                  else
                     mute (kk) = nsamp
                     do  ii = nsamp, 1, -1
                         if (tri(ii) .eq. 0.0) then
                            mute (kk) = ii
                         else
                            go to 27
                         endif
                     enddo
27                   continue
                  endif

                  IF(static .eq. 30000) call vclr (tri, 1, nsamp)

                  istrc = (kk-1) * nsamp
                  call vmov (tri, 1, bigar1(istrc+1), 1, nsamp)
                  ishdr = (kk-1) * ITRWRD
                  call vmov (lhed, 1, itrhdr(ishdr+1),1,ITRWRD)

            enddo

c-----------
c first sort p's into smallest (most neg) to largest (most pos)
c then build interpolation tables
c decide whether all positive p's, or all neg p's, or mix of both
c-----------
            IF (first) THEN

                call tables (p, tabl1, tabl2, tri, pmin, pamin,
     1                       pmax, pamax, split, neg, key, lapmin,
     2                       isym, ileft, irite, ntrc)

            ENDIF
c-----------
c do ray param to angle, or
c do angle to ray param
c interpolation

            call rectrp  (ntrc, ntrco, ntrcm, nsamp, bigar1, vel,
     1                    tabl1, tabl2, zz, iz, icinit, TV, rev, p,
     2                    iax, split, neg, pamax, key, first, lapmin)

            first = .false.

            IF (TV .and. JJ .eq. 1) THEN
               vref = 1. / pmax
            ENDIF
 
            do  kk = 1, ntrco

               istrc = (kk-1) * nsamp
               ishdr = (kk-1) * ITRWRD
               scl = weight(kk)
               call vsmul (bigar1(istrc+1), 1,scl,
     1                     tri, 1, nsamp)
               call vmov (itrhdr(ishdr+1),1,lhed, 1,ITRWRD)

               if (.not.rev) then
                  call savew2(lhed,ifmt_VPick2,l_VPick2, ln_VPick2,
     1                        mute(kk), TRACEHEADER)
               endif

               IF (TV) THEN

                  ip = tabl1(kk)
                  call savew2(lhed,ifmt_DstSgn,l_DstSgn, ln_DstSgn,
     1                        ip     , TRACEHEADER)

               ELSE

               if (rev) then
                  ip = tabl1(kk)
                  call savew2(lhed,ifmt_DstUsg,l_DstUsg, ln_DstUsg,
     1                        ip     , TRACEHEADER)
               else

                  ang = angs + (kk-1) * dang
                  pk   = sin (rad * ang) / vref
                  if (pk .ge. 0.) then
                     i125 = 1000. * dxrec * (jj - 1) * pk
                  else
                     i125 = 1000. * dxrec * (nrec - jj) * pk
                  endif
                  call savew2(lhed,ifmt_StaCor,l_StaCor, ln_StaCor,
     1                        i125   , TRACEHEADER)

               endif

               ENDIF

               if (rev) then
                  do  ii = nsamp, mute(kk), -1
                      tri(ii) = 0.0
                  enddo
               endif
               call vmov (tri, 1, lhed(ITHWP1), 1, nsamp)

               call wrtape (luout, itr, obytes)

            enddo
 
            if(verbos)write(LER,*)'rad2slnt: ri ',recnum,' processed'

      ENDDO
c-----
c     close data files
c-----
  999 continue

      call lbclos ( luin )
      call lbclos ( luout )

            write(LERR,*)'end of rad2slnt, processed',nrec,' record(s)',
     :               ' with ',ntrc, ' traces'
            write(LER ,*)'end of rad2slnt, processed',nrec,' record(s)',
     :               ' with ',ntrc, ' traces'
      end
 
C***********************************************************************
      subroutine help
#include <f77/iounit.h>
         write(LER,*)
     :'***************************************************************'
        write(LER,*) ' '
        write(LER,*)
     :'rad2slnt either interpolates or decimates seismic data'
        write(LER,*)
     :'see manual pages for details ( online by typing uman prg )'
        write(LER,*)' '
        write(LER,*)
     :'execute rad2slnt by typing rad2slnt and the program parameters'
        write(LER,*)
     :'note that each parameter is proceeded by -a where "a" is '
        write(LER,*)
     :'a character(s) corresponding to some parameter.'
        write(LER,*)
     :'users enter the following parameters, or use the default values'
        write(LER,*)' '
        write(LER,*)
     :' -N [ntap]    (no default)      : input data file name'
        write(LER,*)
     :' -O [otap]    (no default)      : output data file name'
        write(LER,*)
     :' -v [vtap]    (no velocity)     : optional velocity field'
        write(LER,*) ' '
        write(LER,*)
     :' -rv[vref] (def= line hdr) :  reference velocity (ft,m/s)'
        write(LER,*)
     :' -mv[vmute] (def= 0)       :  mute velocity (ft,m/s)'
        write(LER,*)
     :' -sa[angs] (def= input)    :  start angle (or ray param)'
        write(LER,*)
     :' -ea[ange] (def= input)    :  end angle (or ray param)'
        write(LER,*)
     :' -da[dang] (def= input)    :  angle (or ray param) increment'
        write(LER,*)
     :' -sp[dxrec] (def= none)    :  shot spacing'
        write(LER,*) ' '
        write(LER,*)
     :' -TV  intepolated angles are time varying'
        write(LER,*)
     :' -R  include on command line to go from angle to ray paramter'
        write(LER,*)
     :' (default is forward direction: ray parameter to angle'
        write(LER,*)
     :' -V  include on command line if verbose printout is desired'
        write(LER,*) ' '
        write(LER,*)
     :'usage:   rad2slnt -N[] -O[] -rv[] -mv[] -sa[] -ea[] -da[]'
        write(LER,*)
     :'                   [-TV -v[] -R -V]'
        write(LER,*) ' '
        write(LER,*)
     :'***************************************************************'
      return
      end
 
C***********************************************************************
      subroutine gcmdln(ntap,otap,vrefi,angs,ange,dang,vtap,
     1                  taper,xmax,dxrec,vmute,rev,verbos,TV)
c-----
c     get command arguments
c
c     ntap  - C*100     input file name
c     otap  - C*100     output file name
c     sii   - R*4       input sample interval override
c     sio   - R*4       output sample interval
c     verbos  L         verbose output or not
c-----
#include <f77/iounit.h>
      character   ntap*(*), otap*(*), vtap*(*)
      real        vrefi,angs,ange,dang,vmute,dxrec
      logical     verbos, rev, TV
      integer     argis
 
            call argstr( '-N', ntap, ' ', ' ' )
            call argstr( '-O', otap, ' ', ' ' )
            call argstr( '-v', vtap, ' ', ' ' )
            call argr4( '-rv', vrefi,  0., 0. )
            call argr4( '-mv', vmute , 0., 0. )
            call argr4( '-sa', angs,   0., 0. )
            call argr4( '-ea', ange,   0., 0. )
            call argr4( '-da', dang,   0., 0. )
            call argr4( '-sp', dxrec,  0., 0. )
            call argr4( '-tp', taper,  5., 5. )
            call argr4( '-xm', xmax ,  999999., 999999. )
            rev    = (argis('-R') .gt. 0)
            TV     = (argis('-TV') .gt. 0)

            if (TV .AND. (vtap(1:1) .eq. ' ') ) then
               write(LERR,*)'rad2slnt:  cmd line error'
               write(LERR,*)'For TV option must have velocity field'
               write(LERR,*)'(enter using -v[] )'
               write(LER ,*)'rad2slnt:  cmd line error'
               write(LER ,*)'For TV option must have velocity field'
               write(LER ,*)'(enter using -v[] )'
               stop
            endif
            IF (.not. TV) THEN
c           if (rev .and. xmax .eq. 999999.) then
c              write(LERR,*)'rad2slnt:  cmd line error'
c              write(LERR,*)'In reverse mode must enter max absolute'
c              write(LERR,*)'offset, xmax, using -xm[] cmd line entry'
c              write(LER ,*)'rad2slnt:  cmd line error'
c              write(LER ,*)'In reverse mode must enter max absolute'
c              write(LER ,*)'offset, xmax, using -xm[] cmd line entry'
c              stop
c           endif
            ENDIF
            
            verbos = (argis('-V') .gt. 0)

      return
      end
 
      subroutine verbal(nsamp, nsi, ntrc, nrec, iform,
     1                  ntap,otap)
c-----
c     verbose output of processing parameters
c
c     nsamp - I*4 number of samples in trace
c     sii   - R*4       input sample interval override
c     sio   - R*4  output sample interval
c     nsi   - I*4 sample interval in ms
c     ntrc  - I*4 traces per record
c     nrec  - I*4 number of records per line
c     iform - I*4 format of data
c     ntap  - C*120     input file name
c     otap  - C*120     output file name
c-----
#include <f77/iounit.h>
      integer     nsamp, nsi, ntrc, nrec
      real        sio, sii
      character   ntap*(*), otap*(*)
 
            write(LERR,*)' '
            write(LERR,*)' line header values after default check '
            write(LERR,*) ' # of samples/trace =  ', nsamp
            write(LERR,*) ' input sample interval    =  ', nsi
            write(LERR,*) ' traces per record  =  ', ntrc
            write(LERR,*) ' records per line   =  ', nrec
            write(LERR,*) ' format of data     =  ', iform
            write(LERR,*) ' input sample interval   = ',sii
            write(LERR,*) ' output sample interval   = ',sio
            write(LERR,*) ' input data set name =  ', ntap
            write(LERR,*) ' output data set name=  ', otap
            write(LERR,*)' '
            write(LERR,*)' '
 
      return
      end
 
