C***********************************************************************
C                 copyright 2001, Amoco Production Company             *
C                             All Rights Reserved                      *
C                     an affiliate of BP America Inc.                  *
C***********************************************************************
      subroutine addcon(n,m,a,iact,nact,z,u,relacc,indxbd,space)

c-----
c     This subroutine adds a constraint to the active set during
c     constrained optimisation of a function of several variables
c     subject to linear constraints on the values of the variables.
c
c     See, Powell, M.J.D.,1989, "Tolmin: A Fortran Package for Linearly
c     Constrained Optimization Calculations" University of Cambridge,
c     Dept. of Applied Mathematics and Theoretical Physics report NA2.
c
c---- Subroutine inputs
c
c     n = The number of variables
c     m = The number of linear constraints (excluding simple bounds)
c     a = The n * m array of coefficients of equality constraints
c         such that  transpose[a] * [x] = [b]
c     z = Working matrix such that z * z(transpose) = inverse of the
c         second derivatives of the active gradients.
c     u = The diagonal elements of the Goldfarb - Idnani upper triangular
c         factorization matrix.
c     relacc = The relative accuracy of the computation. It is determined
c              by the program to be close to the relative precision of
c              the computer arithmetic.
c     indxbd = The index of the constraint that determines the search
c              step size.
c
c---- Subroutine outputs
c
c     iact = The m + 2 * n vector of active constraints
c     nact = The number of active constraints
c     space = working space of (2 * n) variables
c
c---- Subroutines Required
c
c     ddot
c     dmmove
c
c-----

      integer n,m
      real*8 a(n * m),z(n * n),u(*),space(2 * n)
      real*8 relacc
      integer iact(*)
      integer nact,indxbd

      real*8 temp,wcos,wsin,wpiv,tempa,tempb,sum,sumabs
      integer np,icon,inewbd,i,j,jp,ipiv
      integer iztc,icgrad,ij,ijp,inp

c-----
c     Define entry to working space
c-----

      iztc = 0
      icgrad = n + 1

c-----
c     Initialise variable
c-----

      np = nact + 1
      icon = iact(indxbd)
      iact(indxbd) = iact(np)
      iact(np) = icon

c------
c     form ztc when the new constraint is a bound
c------

      if(icon.gt.m) then

        inewbd = icon - m

        if(inewbd.le.n) then

          temp = -1.0d0

          else

          inewbd = inewbd - n
          temp = 1.0d0

        endif

        ij = inewbd

        do j = 1,n

          space(iztc + j) = temp * z(ij)
          ij = ij + n

        enddo

c------
c     form ztc for an ordinary constraint
c------

      else

        ij = (icon - 1) * n + 1
        call dmmove(n,a(ij),space(icgrad))
        ij = 1

        do j = 1,n

          call ddot(n,z(ij),space(icgrad),space(iztc + j))
          ij = ij + n

        enddo

      endif

c-----
c     find any Givens rotations to apply to the last columns of z.
c-----

      j = n

   40 jp = j

      j = j - 1

      if(j.gt.nact) then

        if(space(iztc + jp).eq.0.0d0) goto 40

        if(dabs(space(iztc + jp)).le.relacc * dabs(space(iztc + j)))
     :then

          temp = dabs(space(iztc + j))

          else if(dabs(space(iztc + j)).le.
     :relacc * dabs(space(iztc + jp))) then

          temp = dabs(space(iztc + jp))

          else

          temp = dabs(space(iztc + jp)) * dsqrt(1.0d0 +
     :(space(iztc + j) / space(iztc + jp)) * * 2)

        endif

        wcos = space(iztc + j) / temp
        wsin = space(iztc + jp) / temp
        space(iztc + j) = temp

c-----
c     apply the rotation when the new constraint is a bound
c-----

        if(icon.gt.m) then

          ij = (j - 1) * n + 1
          ijp = (jp - 1) * n + 1

          do i = 1,n

            temp = wcos * z(ijp) - wsin * z(ij)
            z(ij) = wcos * z(ij) + wsin * z(ijp)
            z(ijp) = temp
            ij = ij + 1
            ijp = ijp + 1

          enddo

          ijp = (jp - 1) * n + inewbd
          z(ijp) = 0.0d0

c-----
c     else apply the rotation for an ordinary constraint
c-----

          else

          wpiv = 0.0d0
          ij = (j - 1) * n + 1
          ijp = (jp - 1) * n + 1

          do i = 1,n

            tempa = wcos * z(ijp)
            tempb = wsin * z(ij)
            temp = dabs(space(icgrad + i - 1)) *
     :                 (dabs(tempa) + dabs(tempb))

            if(temp.gt.wpiv) then

              wpiv = temp
              ipiv = i

            endif

            z(ij) = wcos * z(ij) + wsin * z(ijp)
            z(ijp) = tempa - tempb
            ij = ij + 1
            ijp = ijp + 1

          enddo

c-----
c     ensure orthogonality of z(.,jp) to cgrad
c-----

          ijp = (jp - 1) * n + 1
          call ddot(n,z(ijp),space(icgrad),sum)
          ijp = (jp - 1) * n + ipiv

          if(sum.ne.0.0d0) then

            z(ijp) = z(ijp) - sum / space(icgrad + ipiv - 1)

          endif

        endif

        goto 40

      endif

c-----
c     test for linear independence in the proposed new active set
c-----

      if(space(iztc + np).eq.0.0d0) return

      if(icon.le.m) then

        sum = 0.0d0
        sumabs = 0.0d0
        inp = (np - 1) * n + 1

        do i = 1,n

          temp = z(inp) * space(icgrad + i - 1)
          sum = sum + temp
          sumabs = sumabs + dabs(temp)
          inp = inp + 1

        enddo

        if(dabs(sum).le.relacc * sumabs) return

      endif

c-----
c     set the new diagonal element of u and return
c-----

      u(np) = 1.0d0 / space(iztc + np)
      nact = np
      return
      end
