C***********************************************************************
C                 copyright 2001, Amoco Production Company             *
C                             All Rights Reserved                      *
C                     an affiliate of BP America Inc.                  *
C***********************************************************************
      subroutine adjtol(n,m,a,b,xl,xu,x,iact,nact,xbig,relacc,tol,meql)

c-----
c     This subroutine adjusts the relative tolerance of constraint
c     resdiduals during minimisation of a differentiable function of
c     several variables, subject to linear constraints on the values of
c     the variables.
c
c     See, Powell, M.J.D.,1989, "Tolmin: A Fortran Package for Linearly
c     Constrained Optimization Calculations" University of Cambridge,
c     Dept. of Applied Mathematics and Theoretical Physics report NA2.
c
c---- Subroutine inputs
c
c     n = The number of variables
c     m = The number of linear constraints (excluding simple bounds)
c     a = The n * m array of coefficients of equality constraints
c         such that  transpose[a] * [x] = [b]
c     b = The length m vector of constraints.
c     xl = The vector of lower bounds on x
c     xu = The vector of upper bounds on x
c     x = The vector of variables to be minimised.
c     relacc = The relative accuracy of the computation. It is determined
c              by the program to be close to the relative precision of
c              the computer arithmetic.
c
c---- Subroutine outputs
c
c     iact = The m + 2 * n vector of active constraints
c     nact = The final number of active constraints
c     xbig = The array of largest absolute values of the variables.
c     tol = The relative tolerance of constraint residuals.  This is
c           reduced during the search until tol <= relacc
c     meql = The number of independent inequalities.
c
c-----

      integer n,m
      real*8 a(n * m),b(*),xl(*),xu(*),x(*),xbig(*)
      real*8 relacc,tol
      integer iact(*)
      integer nact,meql

      real*8 viol,res,resabs
      integer kl,k,j,jm,i,ij

c-----
c     Set viol to the greatest relative constraint residual of the first
c     nact constraints
c-----

      viol = 0.0d0

      if(nact.gt.meql) then

        kl = meql + 1

        do k = kl,nact

          j = iact(k)

          if(j.le.m) then

            res = b(j)
            resabs = dabs(b(j))
            ij = (j - 1) * n + 1

            do i = 1,n

              res = res - a(ij) * x(i)
              resabs = resabs + dabs(a(ij) * xbig(i))
              ij = ij + 1

            enddo

          else

            jm = j - m

            if(jm.le.n) then

              res = x(jm) - xl(jm)
              resabs = xbig(jm) + dabs(xl(jm))

            else

              jm = jm - n
              res = xu(jm) - x(jm)
              resabs = xbig(jm) + dabs(xu(jm))

            endif

          endif

          if(res.gt.0.0d0) viol = dmax1(viol,res / resabs)

        enddo

      endif

c-----
c     Adjust tol
c-----

      tol = 0.1d0 * dmin1(tol,viol)

      if(tol.le.relacc + relacc) then

        tol = relacc

        do i = 1,n

          xbig(i) = dabs(x(i))

        enddo

      endif

      return
      end
