C***********************************************************************
C                 copyright 2001, Amoco Production Company             *
C                             All Rights Reserved                      *
C                     an affiliate of BP America Inc.                  *
C***********************************************************************
      subroutine conres(n,m,a,b,xl,xu,x,iact,nact,g,z,u,xbig,bres,d,
     :ztg,relacc,tol,stepcb,sumres,meql,msat,mtot,indxbd,space)

c-----
c     This subroutine calculates the constraint residuals and sets the
c     search direction and its components during minimisation of a
c     differentiable function of several variables, subject to linear
c     constraints on the values of the variables.
c
c     See, Powell, M.J.D.,1989, "Tolmin: A Fortran Package for Linearly
c     Constrained Optimization Calculations" University of Cambridge,
c     Dept. of Applied Mathematics and Theoretical Physics report NA2.
c
c---- Subroutine inputs
c
c     n = The number of variables
c     m = The number of linear constraints (excluding simple bounds)
c     a = The n * m array of coefficients of equality constraints
c         such that  transpose[a] * [x] = [b]
c     b = The length m vector of constraints.
c     xl = The vector of lower bounds on x
c     xu = The vector of upper bounds on x
c     x = The vector of variables to be minimised.
c     relacc = The relative accuracy of the computation. It is determined
c              by the program to be close to the relative precision of
c              the computer arithmetic.
c
c---- Subroutine outputs
c
c     iact = The m + 2 * n vector of active constraints
c     nact = The final number of active constraints
c     g = The array of function gradients at X.
c     z = Working matrix such that z * z(transpose) = inverse of the
c         second derivatives of the active gradients.
c     u = The diagonal elements of the Goldfarb - Idnani upper triangular
c         factorization matrix.
c     xbig = The array of largest absolute values of the variables.
c     bres = The array oc constraint residuals.
c     d = The array of the components of the stepl length of the line
c         search in the directions with respect to X.
c     ztg = The vector obtained from z(transpose) * g
c     tol = The relative tolerance of constraint residuals.  This is
c           reduced during the search until tol <= relacc
c     stepcb = The step length of the line search
c     sumres = Sum of residuals at constraint violations.
c     meql = The number of independent inequalities.
c     msat = The number of satisfied constraints
c     mtot = The total number of constraints and bounds
c     indxbd = The index of the constraint that determines the search
c              step size.
c     space = working space of (5 * n) elements
c
c---- Subroutines Required
c
c     addcon
c     delcon
c     getd
c     newcon
c     sdegen
c     sdirn
c     stepbd
c     ddot
c     dzero
c     dmmove
c     dmxadd
c
c-----

      integer n,m
      real*8 a(n * m),b(*),xl(*),xu(*),x(*),g(*),z(n * n)
      real*8 u(*),xbig(*),bres(*),d(*),ztg(*),space(5 * n)
      real*8 relacc,tol,stepcb,sumres
      integer iact(*)
      integer nact,meql,msat,mtot,indxbd

      real*8 res,resabs,temp,sum,ddotg
      integer idiff,i,msatk,mdeg,kl,k,j,jm,ij

c-----
c     Define entry points into working space
c     space 1 to n used for gm in getd
c     space n + 1 to space 5 * n working space for use in getd
c-----

      idiff = mtot - msat

c-----
c     Calculate and partition the residuals of the inactive constraints,
c     and set the gradient vector when seeking feasibility.
c-----

      if(idiff.gt.0) then

        call dzero(n,g)
        sumres = 0.0d0

      endif

      msatk = msat
      mdeg = nact
      msat = nact
      kl = meql + 1

      do 50 k = kl,mtot

        j = iact(k)

c-----
c     Calculate the residual of the current constraint
c-----

        if(j.le.m) then

          res = b(j)
          resabs = dabs(b(j))
          ij = (j - 1) * n + 1

          do i = 1,n

            res = res - x(i) * a(ij)
            resabs = resabs + dabs(xbig(i) * a(ij))
            ij = ij + 1

          enddo

        else

          jm = j - m

          if(jm.le.n) then

            res = x(jm) - xl(jm)
            resabs = dabs(xbig(jm)) + dabs(xl(jm))

          else

            jm = jm - n
            res = xu(jm) - x(jm)
            resabs = dabs(xbig(jm)) + dabs(xu(jm))

          endif

        endif

        bres(j) = res

c-----
c     Set temp to the relative residual
c-----

        temp = 0.0d0
        if(resabs.ne.0.0d0) temp = res / resabs

        if(k.gt.msatk.and.temp.lt.0.0d0) then

          if(temp + relacc.ge.0.0d0) then

            if(j.le.m) then

              sum = dabs(b(j))
              ij = (j - 1) * n + 1

              do i = 1,n

                sum = sum + dabs(x(i) * a(ij))
                ij = ij + 1

              enddo

            else

              jm = j - m

              if(jm.le.n) then

                sum = dabs(x(jm)) + dabs(xl(jm))

              else

                sum = dabs(x(jm - n)) + dabs(xu(jm - n))

              endif

            endif

            if(dabs(res).le.sum * relacc) temp = 0.0d0

          endif

        endif

c-----
c     Place the residual in the appropriate position
c-----

        if(k.le.nact) goto 50

        if(k.le.msatk.or.temp.ge.0.0d0) then

          msat = msat + 1

          if(msat.lt.k) then

            iact(k) = iact(msat)

          endif

          if(temp.gt.tol) then

            iact(msat) = j

          else

            mdeg = mdeg + 1
            iact(msat) = iact(mdeg)
            iact(mdeg) = j

          endif

c------
c      Update the gradient and sumres if the constraint is violated when
c      seeking feasibility
c------

          else

          if(j.le.m) then

            ij = (j - 1) * n + 1
            call dmxadd(g,a(ij),g,n,1)

          else

            j = j - m

            if(j.le.n) then

              g(j) = g(j) - 1.0d0

            else

              g(j - n) = g(j - n) + 1.0d0

            endif

          endif

          sumres = sumres + dabs(res)

        endif

   50 continue

c-----
c     Seek the next search direction unless conres was called from getfes
c     and feasibility has been achieved.
c-----

      stepcb = 0.0d0
      if(idiff.gt.0.and.msat.eq.mtot) return
      call getd(n,m,a,iact,nact,g,z,u,d,ztg,relacc,ddotg,meql,mdeg,
     :space,space(n + 1))

c-----
c     Calculate the (bound on the) step - length due to the constraints
c-----

      if(ddotg.lt.0.0d0) then

        call stepbd(n,m,a,iact,bres,d,stepcb,ddotg,mdeg,msat,mtot,
     :indxbd)

      endif

      if(idiff.eq.0) sumres = ddotg
      return
      end
