C***********************************************************************
C                 copyright 2001, Amoco Production Company             *
C                             All Rights Reserved                      *
C                     an affiliate of BP America Inc.                  *
C***********************************************************************
      subroutine delcon(n,m,a,iact,nact,z,u,relacc,idrop)

c-----
c     This subroutine deletes a constraint from the active set during
c     minimisation of a differentiable function of several variables,
c     subject to linear constraints on the values of the variables.
c
c     See, Powell, M.J.D.,1989, "Tolmin: A Fortran Package for Linearly
c     Constrained Optimization Calculations" University of Cambridge,
c     Dept. of Applied Mathematics and Theoretical Physics report NA2.
c
c---- Subroutine inputs
c
c     n = The number of variables
c     m = The number of linear constraints (excluding simple bounds)
c     a = The n * m array of coefficients of equality constraints
c         such that  transpose[a] * [x] = [b]
c     relacc = The relative accuracy of the computation. It is determined
c              by the program to be close to the relative precision of
c              the computer arithmetic.
c     idrop = The index of the constraint to be dropped
c
c---- Subroutine outputs
c
c     iact = The m + 2 * n vector of active constraints
c     nact = The final number of active constraints
c     z = Working matrix such that z * z(transpose) = inverse of the
c         second derivatives of the active gradients.
c     u = The diagonal elements of the Goldfarb - Idnani upper triangular
c         factorization matrix.
c
c---- Subroutines required
c
c     ddot
c
c----

      integer n,m
      real*8 a(n * m),z(n * n),u(*)
      real*8 relacc
      integer iact(*)
      integer nact,idrop

      real*8 rjjp,ujp,temp,denom,wcos,wsin,wpiv,tempa,tempb,sum
      integer nm,isave,j,jp,i,ibd,icon,ipiv,ij,iicon,ibdj,ijp
      integer ibdjp,ipivjp

      nm = nact - 1
      if(idrop.eq.nact) goto 60
      isave = iact(idrop)

c-----
c     Cycle through the constraint exchanges that are needed
c-----

      do j = idrop,nm

        jp = j + 1
        icon = iact(jp)
        iact(j) = icon

c-----
c     Calculate the (j,jp) element of r
c-----

        if(icon.le.m) then

          ij = (j - 1) * n + 1
          iicon = (icon - 1) * n + 1
          call ddot(n,z(ij),a(iicon),rjjp)

          else

          ibd = icon - m

          if(ibd.le.n) then

            ibdj = (j - 1) * n + ibd
            rjjp = -z(ibdj)

            else

            ibd = ibd - n
            ibdj = (j - 1) * n + ibd
            rjjp = z(ibdj)

          endif

        endif

c-----
c     Calculate the parameters of the next rotation
c-----

        ujp = u(jp)
        temp = rjjp * ujp
        denom = dabs(temp)
        if(denom * relacc.lt.1.0d0) denom = dsqrt(1.0d0 + denom * denom)
        wcos = temp / denom
        wsin = 1.0d0 / denom

c-----
c     Rotate z when a bound constraint is promoted
c-----

        if(icon.gt.m) then

          ij = (j - 1) * n + 1
          ijp = (jp - 1) * n + 1

          do i = 1,n

            temp = wcos * z(ijp) - wsin * z(ij)
            z(ij) = wcos * z(ij) + wsin * z(ijp)
            z(ijp) = temp
            ij = ij + 1
            ijp = ijp + 1

          enddo

          ibdjp = (jp - 1) * n + ibd
          z(ibdjp) = 0.0d0

c-----
c     Rotate z when an ordinary constraint is promoted
c-----

        else

          wpiv = 0.0d0
          ij = (j - 1) * n + 1
          ijp = (jp - 1) * n + 1
          iicon = (icon - 1) * n + 1

          do i = 1,n

            tempa = wcos * z(ijp)
            tempb = wsin * z(ij)
            temp = dabs(a(iicon)) * (dabs(tempa) + dabs(tempb))

            if(temp.gt.wpiv) then

              wpiv = temp
              ipiv = i

            endif

            z(ij) = wcos * z(ij) + wsin * z(ijp)
            z(ijp) = tempa - tempb
            ij = ij + 1
            ijp = ijp + 1
            iicon = iicon + 1

          enddo

c-----
c     Ensure orthogonality to promote constraint
c-----

          ijp = (jp - 1) * n + 1
          iicon = (icon - 1) * n + 1
          call ddot(n,z(ijp),a(iicon),sum)
          ipivjp = (jp - 1) * n + ipiv
          iicon = (icon - 1) * n + ipiv

          if(sum.ne.0.0d0) then

            z(ipivjp) = z(ipivjp) - sum / a(iicon)

          endif

        endif

c-----
c     Set the new diagonal elements of u
c-----

        u(jp) = -denom * u(j)
        u(j) = ujp / denom

      enddo

      iact(nact) = isave
   60 nact = nm
      return
      end
