C***********************************************************************
C                 copyright 2001, Amoco Production Company             *
C                             All Rights Reserved                      *
C                     an affiliate of BP America Inc.                  *
C***********************************************************************
      subroutine dilalg(xin1,xin2,u,x,delta,ilim,itrial,ierr,
     :                  freq,filter) 
 
c*********************************************************************** 
c     This subroutine determines the roots of the function u(x) = 0 
c     using the Illinois Algorithm. 
c 
c     See Bjorck and Anderson, "Numerical Methods.", pages 230 and 503 
c 
c**** Subroutine inputs 
c 
c     xin1 = The first initial value of the independent variable, x. 
c     xin2 = The second initial value of the independent variable, x. 
c     u = The function to be optimised. This is an external function name 
c         in the main program. 
c     delta = The desired maximum tolerance in change of x between 
c             iterations 
c     ilim = The maximum number of trials allowed before termination. 
c 
c**** Subroutine outputs 
c 
c     x = The value of the independent variable at the root u(x) = 0 
c     itrial = The number of trials 
c     ierr = Completion code 
c            ierr = 0       Normal completion. u(x) is less than flim. 
c            ierr = 1       ugrd(x) = 0.0 
c            ierr = 2       Maximum number of trials has been exceeded 
c            ierr = 3       The initial values do not straddle the root. 
c*********************************************************************** 
c     This subroutine was written by M.D. Bush in Cairo, February 1990 
c     Updated for wedgest Sept. 2002
c*********************************************************************** 
 
      implicit none

      integer nfreq,iflag1
      real*8 summeas,summeasp,tmeas,ampl1
      real*8 sumcalc,sumcalcp,tcalc,dt

      common /wedgedata/summeas,summeasp,tmeas,sumcalc,sumcalcp,ampl1,
     :     tcalc,dt,iflag1,nfreq

      real*8 freq(nfreq),filter(nfreq)

      external u 
      real*8 xin1,xin2,x,x1,x2,x3,y1,y2,y3,u,delta 
      real*8 dx,x2mx1,xtemp,ytemp,small,deltxx 
      integer ilim,itrial,ierr,ntrial 
 
c---- Initialise the search values. 
 
      small = 1.0d-300
      x1 = xin1 
      x2 = xin2 
      ierr = 0 

c---- Compute the initial function values.  

      y1 = u(x1,freq,filter) 
      y2 = u(x2,freq,filter) 

c---- Check that the initial values straddle the root. 
 
      if(y1 * y2.gt.0.0d0) then 
 
        ierr = 3 
        return 
 
        else if(y1.eq.0.0d0) then 
 
        x = x1 
        return 
 
        else if(y2.eq.0.0d0) then 
 
        x = x2 
        return 
 
      endif 
 
      x2mx1 = x2 - x1 
 
c---- Increment the number of trials by one. 
 
      do ntrial = 1,ilim 
 
c---- If y2 equals 0.0d0 then return with the current value of x = x2  
 
        if(dabs(y2).le.small) goto 1
 
c---- 
c     Compute by linear interpolation the distance, dx  from x2 where a 
c     straight line would cross y = 0. 
c----
 
        dx = y2 * (x2mx1) / (y2 - y1) 
 
c----
c     If the dx is insignificant compared to x2 then determine a new 
c     point half way between x1 and x2. 
c---- 
 
        if(dabs(x2).gt.small.and.dabs(dx / x2).lt.1.0d-16) then 
 
          xtemp = (x2 + x1) / 2.0d0 
          ytemp = u(xtemp,freq,filter) 

c---- Check that the intermediate  values straddle the root. 
 
          if(ytemp * y2.gt.0.0d0) then 
 
            x2 = xtemp 
            y2 = ytemp 
 
            else 
 
            x1 = xtemp 
            y1 = ytemp 
 
          endif 
 
          x2mx1 = x2 - x1 
 
        endif 
 
c---- 
c     Determine the point x3 between x1 and x2 where a straight line 
c     would cross y = 0. 
c---- 
 
        x3 = x2 - dx 
        y3 = u(x3,freq,filter) 
 
c---- Check that the new values straddle the root.  
 
        if(y3 * y2.lt.0.0d0) then 
 
          x1 = x2 
          y1 = y2 
 
          else 
 
          y1 = y1 / 2.0d0 
 
        endif 
 
        x2 = x3 
        y2 = y3 
        x2mx1 = x2 - x1 
  
c---- If the amount of change in the values is less than delta return. 
 
        if(dabs(x1).gt.small) then 
 
          deltxx = dabs(x2mx1 / x1) 
 
          else 
 
          deltxx = dabs(x2mx1) 
 
        endif 
 
        if(deltxx.le.delta) goto 1 
 
      enddo 

c---- 
c     if number of iterations exceeds the limit set error code to 2 and 
c     return 
c---- 
 
    1 if(ntrial.ge.ilim) ierr = 2 
      itrial = ntrial 
      x = x2 
      return 
      end
