C***********************************************************************
C                 copyright 2001, Amoco Production Company             *
C                             All Rights Reserved                      *
C                     an affiliate of BP America Inc.                  *
C***********************************************************************
      subroutine eqcons(n,m,meq,a,b,xu,iact,meql,info,z,u,relacc,am)

c-----
c     This subroutine checks for consistency of equality constraints
c     during minimisation of a differentiable function of several
c     variables, subject to linear constraints on the values of the
c     variables.
c
c     See, Powell, M.J.D.,1989, "Tolmin: A Fortran Package for Linearly
c     Constrained Optimization Calculations" University of Cambridge,
c     Dept. of Applied Mathematics and Theoretical Physics report NA2.
c
c---- Subroutine inputs
c
c     n = The number of variables
c     m = The number of linear constraints (excluding simple bounds)
c     meq = The number of constraints that are equalities (0 <= meq <= m)
c     a = The n * m array of coefficients of equality constraints
c         such that  transpose[a] * [x] = [b]
c     b = The length m vector of constraints.
c     xu = The vector of upper bounds on x
c
c---- Subroutine outputs
c
c     iact = The m + 2 * n vector of active constraints
c     meql = The number of independent inequalities.
c     info = Termination code.
c            info = 1  The final X is feasible and the solution satisfies
c                      the condition, acc.
c            info = 2  The final X is feasible and termination occurs
c                      because rounding errors seem to prevent higher
c                      accuracy.
c            info = 3  The final X is feasible and line search fails to
c                      find a better solution of the objective function,
c                      although it is predicted by the gradient vector.
c                      If the final rate of convergence is slow, the
c                      coding of the gradient should be suspected.
c            info = 4  The calculation cannot start because the bounds
c                      are inconsistent
c            info = 5  The error indicates that the equality constraints
c                      are inconsistent including freezing of variables
c                      by setting xl(i) = xu(i)
c            info = 6  The equality constraints and the bounds are
c                      incompatible.
c            info = 7  The bounds and equality constraints can be
c                      satisfied but the general inequality constraints
c                      cannot.
c            info = 8  The limit on the number of function calls has been
c                      reached.
c     z = Working matrix such that z * z(transpose) = inverse of the
c         second derivatives of the active gradients.
c     u = The diagonal elements of the Goldfarb - Idnani upper triangular
c         factorization matrix.
c     xbig = The array of largest absolute values of the variables.
c     relacc = The relative accuracy of the computation. It is determined
c              by the program to be close to the relative precision of
c              the computer arithmetic.
c     am = workin space of (2 * n) elements
c
c---- Subroutines Required
c
c     addcon
c     dmmove
c     ddot
c
c-----

      integer n,m
      real*8 a(n * m),b(*),xu(*),z(n * n),u(*),am(2 * n)
      real*8 relacc
      integer iact(*)
      integer meq,meql,info

      real*8 sum,sumabs,vmult,rhs
      integer keq,np,i,k,j,jm,ikeq,ik,ij

c-----
c     am is doubled in size to act as working space for addcon
c     Try to add the next equality constraint to the active set
c-----

      do 50 keq = 1,meq

        if(meql.lt.n) then

          np = meql + 1
          iact(np) = keq
          call addcon(n,m,a,iact,meql,z,u,relacc,np,am)
          if(meql.eq.np) goto 50

        endif

c-----
c     If linear dependence occurs then find the multipliers of the
c     dependence relation and apply them to the right hand sides
c-----

        sum = b(keq)
        sumabs = dabs(b(keq))

        if(meql.gt.0) then

          ikeq = (keq - 1) * n + 1
          call dmmove(n,a(ikeq),am)
          k = meql
   20     ik = (k - 1) * n + 1
          call ddot(n,z(ik),am,vmult)
          vmult = vmult * u(k)
          j = iact(k)

          if(j.le.m) then

            ij = (j - 1) * n + 1

            do i = 1,n

              am(i) = am(i) - vmult * a(ij)
              ij = ij + 1

            enddo

            rhs = b(j)

          else

            jm = j - m - n
            am(jm) = am(jm) - vmult
            rhs = xu(jm)

          endif

          sum = sum - rhs * vmult
          sumabs = sumabs + dabs(rhs * vmult)
          k = k - 1
          if(k.ge.1) goto 20

        endif

c-----
c     Error return if the constraints are inconsistent
c-----

        if(dabs(sum).gt.relacc * sumabs) then

          info = 5
          return

        endif

   50 continue

      return
      end
