C***********************************************************************
C                 copyright 2001, Amoco Production Company             *
C                             All Rights Reserved                      *
C                     an affiliate of BP America Inc.                  *
C***********************************************************************
      subroutine getfes(n,m,a,b,xl,xu,x,iact,nact,info,z,u,xbig,
     :relacc,tol,meql,msat,mtot,space)

c-----
c     This subroutine gets a feasible point x, within the limits of
c     the constraints and residual tolerance, tol. It is used by the
c     subroutine TOLMIN, which carries out minimisation of a
c     differentiable function of several variables, subject to linear
c     constraints on the values of the variables.
c
c     See, Powell, M.J.D.,1989, "Tolmin: A Fortran Package for Linearly
c     Constrained Optimization Calculations" University of Cambridge,
c     Dept. of Applied Mathematics and Theoretical Physics report NA2.
c
c---- Subroutine inputs
c
c     n = The number of variables
c     m = The number of linear constraints (excluding simple bounds)
c     a = The n * m array of coefficients of equality constraints
c         such that  transpose[a] * [x] = [b]
c     b = The length m vector of constraints.
c     xl = The vector of lower bounds on x
c     xu = The vector of upper bounds on x
c     x = The vector of variables to be minimised.
c     relacc = The relative accuracy of the computation. It is determined
c              by the program to be close to the relative precision of
c              the computer arithmetic.
c
c---- Subroutine outputs
c
c     iact = The m + 2 * n vector of active constraints
c     nact = The final number of active constraints
c     info = Termination code.
c            info = 1  The final X is feasible and the solution satisfies
c                      the condition, acc.
c            info = 2  The final X is feasible and termination occurs
c                      because rounding errors seem to prevent higher
c                      accuracy.
c            info = 3  The final X is feasible and line search fails to
c                      find a better solution of the objective function,
c                      although it is predicted by the gradient vector.
c                      If the final rate of convergence is slow, the
c                      coding of the gradient should be suspected.
c            info = 4  The calculation cannot start because the bounds
c                      are inconsistent
c            info = 5  The error indicates that the equality constraints
c                      are inconsistent including freezing of variables
c                      by setting xl(i) = xu(i)
c            info = 6  The equality constraints and the bounds are
c                      incompatible.
c            info = 7  The bounds and equality constraints can be
c                      satisfied but the general inequality constraints
c                      cannot.
c            info = 8  The limit on the number of function calls has been
c                      reached.
c     z = Working matrix such that z * z(transpose) = inverse of the
c         second derivatives of the active gradients.
c     u = The diagonal elements of the Goldfarb - Idnani upper triangular
c         factorization matrix.
c     xbig = The array of largest absolute values of the variables.
c     tol = The relative tolerance of constraint residuals.  This is
c           reduced during the search until tol <= relacc
c     meql = The number of independent inequalities.
c     msat = The number of satisfied constraints
c     mtot = The total number of constraints and bounds
c     space = working space of (10 * n + m) elements
c
c---- Subroutines Required
c
c     addcon
c     adjtol
c     conres
c     delcon
c     getd
c     newcon
c     satact
c     sdegen
c     sdirn
c     stepbd
c     ddot
c     dzero
c     dmmove
c     dmxadd
c
c----

      integer n,m
      real*8 a(n * m),b(*),xl(*),xu(*),x(*),z(n * n)
      real*8 u(*),xbig(*),space(m + 10 * n)
      real*8 relacc,tol
      integer iact(*)
      integer nact,info,meql,msat,mtot

      real*8 sumrsk,stepcb,sumres
      integer msatk,indxbd,itest,i,ig,id,iztg,ispace,ibres

c-----
c     Define the entry points into working space
c-----

      ig = 1
      id = ig + n
      iztg = id + n
      ispace = iztg + n
      ibres = ispace + 5 * n

c-----
c     Make the correction to x for the active constraints
c-----

      info = 0
   10 call satact (n,m,a,b,xl,xu,x,iact,nact,info,z,u,xbig,relacc,
     :tol,meql)
      if(info.gt.0) msat = nact
      if(msat.eq.mtot) return

c-----
c     Try to correct the infeasibility
c-----

   20 msatk = msat
      sumrsk = 0.0d0
   30 call conres(n,m,a,b,xl,xu,x,iact,nact,space(ig),z,u,xbig,
     :space(ibres),space(id),space(iztg),relacc,tol,stepcb,sumres,
     :meql,msat,mtot,indxbd,space(ispace))

c-----
c     Include the new constraint in the active set
c-----

      if(stepcb.gt.0.0d0) then

        do i = 1,n

          x(i) = x(i) + stepcb * space(id + i - 1)
          xbig(i) = dmax1(xbig(i),dabs(x(i)))

        enddo

        call addcon(n,m,a,iact,nact,z,u,relacc,indxbd,space(ispace))

      endif

c-----
c     Test whether to continue the search for feasibility
c-----

      if(msat.lt.mtot) then

        if(stepcb.eq.0.0d0) goto 50
        if(msatk.lt.msat) goto 20

        if(sumrsk.eq.0.0d0.or.sumres.lt.sumrsk) then

          sumrsk = sumres
          itest = 0

        endif

        itest = itest + 1
        if(itest.le.2) goto 30

c-----
c     Reduce tol if it may be too large to allow feasibility
c-----

   50   if(tol.gt.relacc) then

          call adjtol(n,m,a,b,xl,xu,x,iact,nact,xbig,relacc,
     :tol,meql)
          goto 10

        endif

      endif

      return
      end
