C***********************************************************************
C                 copyright 2001, Amoco Production Company             *
C                             All Rights Reserved                      *
C                     an affiliate of BP America Inc.                  *
C***********************************************************************
      subroutine ktvec(n,m,a,iact,nact,par,g,reskt,z,u,bres,relaxf,
     :meql,ssqkt,space)

c-----
c     This subroutine calculates the Kuhn-Tucker residual vector during
c     during minimisation of a differentiable function of several
c     variables, subject to linear constraints on the values of the
c     variables.
c
c     See, Powell, M.J.D.,1989, "Tolmin: A Fortran Package for Linearly
c     Constrained Optimization Calculations" University of Cambridge,
c     Dept. of Applied Mathematics and Theoretical Physics report NA2.
c
c---- Subroutine inputs
c
c     n = The number of variables
c     m = The number of linear constraints (excluding simple bounds)
c     a = The n * m array of coefficients of equality constraints
c
c---- Subroutine outputs
c
c     iact = The m + 2 * n vector of active constraints
c     nact = The final number of active constraints
c     par = The array of estimates of the Lagrange multipliers of the
c           active constraints.
c     g = The array of function gradients at X.
c     reskt = The Kuhn-Tucker residual vector.
c     z = Working matrix such that z * z(transpose) = inverse of the
c         second derivatives of the active gradients.
c     u = The diagonal elements of the Goldfarb - Idnani upper triangular
c         factorization matrix.
c     bres = The array oc constraint residuals.
c     relaxf = The total change in the function value caused by applying
c              the Lagrange multipliers when X is at a constraint
c              boundary.
c     meql = The number of independent inequalities.
c     ssqkt = The sum of squares at of the vector reskt.
c     space = workin space of (2 * n) elements
c
c---- Subroutines Required
c
c     ddot
c     dmmove
c
c-----

      integer n,m
      real*8 a(m * n),par(*),g(*),reskt(*),z(n * n),u(*),bres(*)
      real*8 space(3 * n + m)
      real*8 relaxf
      integer iact(*)
      integer nact,meql

      integer i,icase,kk,k,j,jm,kl,iparw,iresktw,ik,ij
      real*8 temp,ssqkt,ssqktw

c-----
c     Define entry points into working space
c     space 1 to n is parw
c     space n + 1 to 2 * n is resktw
c-----

      iparw = 1
      iresktw = m + 2 * n + 1

c-----
c     Calculate the Lagrange parameters and the residual vector
c-----

      call dmmove(n,g,reskt)

      if(nact.gt.0) then

        icase = 0

   20   do kk = 1,nact

          k = nact + 1 - kk
          j = iact(k)
          ik = (k - 1) * n + 1
          call ddot(n,z(ik),reskt,temp)
          temp = temp * u(k)
          if(icase.eq.0) par(k) = 0.0d0

          if(k.le.meql.or.par(k) + temp.lt.0.0d0) then

            par(k) = par(k) + temp

            else

            temp = -par(k)
            par(k) = 0.0d0

          endif

          if(temp.ne.0.0d0) then

            if(j.le.m) then

              ij = (j - 1) * n + 1

              do i = 1,n

                reskt(i) = reskt(i) - temp * a(ij)
                ij = ij + 1

              enddo

            else

              jm = j - m

              if(jm.le.n) then

                reskt(jm) = reskt(jm) + temp

                else

                reskt(jm - n) = reskt(jm - n) - temp

              endif

            endif

          endif

        enddo

c-----
c     Calculate the sum of squares of the kt residual vector
c-----

        ssqkt = 0.0d0
        if(nact.eq.n) goto 130
        call ddot(n,reskt,reskt,ssqkt)

c-----
c     Apply iterative refinement to the residual vector
c-----

        if(icase.eq.0) then

          icase = 1
          call dmmove(nact,par,space(iparw))
          call dmmove(n,reskt,space(iresktw))
          ssqktw = ssqkt
          goto 20

        endif

c-----
c     Undo the iterative refinement if it does not reduce ssqkt
c-----

        if(ssqktw.lt.ssqkt) then

          call dmmove(nact,space(iparw),par)
          call dmmove(n,space(iresktw),reskt)
          ssqkt = ssqktw

        endif

c-----
c     Calculate ssqkt when there are no active constraints
c-----

        else

        call ddot(n,g,g,ssqkt)

      endif

c-----
c     Predict the reduction in f if one corrects any positive residuals
c     of active inequality constraints
c-----

      relaxf = 0.0d0

      if(meql.lt.nact) then

        kl = meql + 1

        do k = kl,nact

          j = iact(k)

          if(bres(j).gt.0.0d0) then

            relaxf= relaxf - par(k) * bres(j)

          endif

        enddo

      endif

  130 return
      end
