C***********************************************************************
C                 copyright 2001, Amoco Production Company             *
C                             All Rights Reserved                      *
C                     an affiliate of BP America Inc.                  *
C***********************************************************************
      subroutine lsrch(n,x,funx,gradx,itype,g,d,xs,gs,relacc,stepcb,
     :ddotg,f,step,nfvals,nfmax,gopt,freq,filter)

c-----
c     This subroutine carries out line search for the minimum point during
c     minimisation of a differentiable function of several variables,
c     subject to linear constraints on the values of the variables.
c
c     See, Powell, M.J.D.,1989, "Tolmin: A Fortran Package for Linearly
c     Constrained Optimization Calculations" University of Cambridge,
c     Dept. of Applied Mathematics and Theoretical Physics report NA2.
c
c---- Subroutine inputs
c
c     n = The number of variables
c     x = The vector of variables to be minimised.
c     g = The array of function gradients at X.
c     funx = The function to be optimised. this is an external procedure
c            name in the main program.
c     gradx = The name of the subroutine that computes the gradient of u.
c             this is an external procedure name in the main program.
c     itype = type of optimisation required :-
c             itype = 1 for maximising
c             itype = -1 for minimising
c     relacc = The relative accuracy of the computation. It is determined
c              by the program to be close to the relative precision of
c              the computer arithmetic.
c
c---- Subroutine outputs
c
c     d = The array of the components of the stepl length of the line
c         search in the directions with respect to X.
c     xs = The vector of variables to be minimised on entry to the
c          subroutine
c     gs = The array of function gradients at X on entry to the
c          subroutine
c     stepcb = The step length of the line search
c     ddotg = Sum of residuals at constraint violations.
c     f = The function value on output
c     step = The search step length.
c     nfvals = The number of function evaluations.
c     nfmax = The bound on the number of function evaluations
c             if nfmax > 0
c     gopt = Working space for the gradient vector at the optimum point.
c
c---- Subroutines Required
c
c     ddot
c     dmmove
c     dscalm
c
c-----

      implicit none

      integer nfreq,iflag1
      real*8 summeas,summeasp,tmeas,ampl1
      real*8 sumcalc,sumcalcp,tcalc,dt

      common /wedgedata/summeas,summeasp,tmeas,sumcalc,sumcalcp,ampl1,
     :                  tcalc,dt,iflag1,nfreq

      real*8 freq(nfreq),filter(nfreq)

      integer n
      real*8 x(*),g(*),d(*),xs(*),gs(*),gopt(n)
      real*8 relacc,stepcb,ddotg,f,step,funx
      integer nfvals,nfmax,itype
      external funx,gradx

      real*8 relint,ratio,temp,stpmin,sbase,ddotgb,stplow,stphgh
      real*8 stpopt,fopt,dgopt,fhgh,dghgh,fbase,dgmid,flow,dglow,dgknot
      integer icount,i

c-----
c     Initialisation
c-----

      relint = 0.9d0
      icount = 0
      ratio = -1.0d0
      call dmmove(n,x,xs)
      call dmmove(n,g,gs)
      call dmmove(n,g,gopt)

      do i = 1,n

        if(d(i).ne.0.0d0) then

          temp = dabs(x(i) / d(i))
          if(ratio.lt.0.0d0.or.temp.lt.ratio) ratio = temp

        endif

      enddo

      step = dmin1(1.0d0,stepcb)
      stpmin = dmax1(relacc * ratio,1.0d-12 * step)
      step = dmax1(stpmin,step)
      sbase = 0.0d0
      fbase = f
      ddotgb = ddotg
      stplow = 0.0d0
      flow = f
      dglow = ddotg
      stphgh = 0.0d0
      stpopt = 0.0d0
      fopt = f
      dgopt = dabs(ddotg)

c-----
c     Calculate another function and gradient value
c-----

   20 do i = 1,n

        x(i) = xs(i) + step * d(i)

      enddo

      f = funx(n,x,freq,filter)
      call gradx(n,x,g,freq,filter)

      if(itype.gt.0) then

        f = -f
        call dscalm(g,n,-1.0d0)

      endif

      icount = icount + 1
      call ddot(n,d,g,dgmid)

      if(f.le.fopt) then

        if(f.lt.fopt.or.dabs(dgmid).lt.dgopt) then

          stpopt = step
          fopt = f
          call dmmove(n,g,gopt)
          dgopt = dabs(dgmid)

        endif

      endif

      if(nfvals + icount.eq.nfmax) goto 70

c-----
c     Modify the bounds on the steplength or convergence
c-----

      if(f.ge.fbase + 0.1d0 * (step - sbase) * ddotgb) then

        if(stphgh.gt.0.0d0.or.f.gt.fbase.or.dgmid.gt.0.5d0 * ddotg) then

          stphgh = step
          fhgh = f
          dghgh = dgmid
          goto 60

        endif

        sbase = step
        fbase = f
        ddotgb = dgmid

      endif

      if(dgmid.ge.0.7d0 * ddotgb) goto 70
      stplow = step
      flow = f
      dglow = dgmid
   60 if(stphgh.gt.0.0d0.and.stplow.ge.relint * stphgh) goto 70

c-----
c     Calculate the next step length or end the iterations
c-----

      if(stphgh.eq.0.0d0) then

        if(step.eq.stepcb) goto 70
        temp = 10.0d0
        if(dgmid.gt.0.9d0 * ddotg) temp = ddotg / (ddotg - dgmid)
        step = dmin1(temp * step,stepcb)
        goto 20

        else if(icount.eq.1.or.stplow.gt.0.0d0) then

        dgknot = 2.0d0 * (fhgh - flow) / (stphgh - stplow) - 0.5d0 *
     : (dglow + dghgh)

        if(dgknot.ge.0.0d0) then

          ratio = dmax1(0.1d0,0.5d0 * dglow / (dglow - dgknot))

          else

          ratio = (0.5d0 * dghgh - dgknot) / (dghgh - dgknot)

        endif

        step = stplow + ratio * (stphgh - stplow)
        goto 20

        else

        step = 0.1d0 * step
        if(step.ge.stpmin) goto 20

      endif

c-----
c     Return from subroutine
c-----

   70 if(step.ne.stpopt) then

        step = stpopt
        f = fopt
        call dmmove(n,gopt,g)

        do i = 1,n

          x(i) = xs(i) + step * d(i)

        enddo

      endif

      nfvals = nfvals + icount
      return
      end
