C***********************************************************************
C                 copyright 2001, Amoco Production Company             *
C                             All Rights Reserved                      *
C                     an affiliate of BP America Inc.                  *
C***********************************************************************
      subroutine minfun(n,m,a,b,xl,xu,x,funx,gradx,itype,acc,iact,nact,
     :par,iprint,info,z,u,xbig,relacc,zznorm,tol,meql,mtot,iterc,nfvals,
     :nfmax,space,f,freq,filter)

c-----
c     This subroutine carries out minimisation of a differentiable
c     function of several variables, subject to linear constraints on
c     the values of the variables.
c
c     See, Powell, M.J.D.,1989, "Tolmin: A Fortran Package for Linearly
c     Constrained Optimization Calculations" University of Cambridge,
c     Dept. of Applied Mathematics and Theoretical Physics report NA2.
c
c---- Subroutine inputs
c
c     n = The number of variables
c     m = The number of linear constraints (excluding simple bounds)
c     a = The n * m array of coefficients of equality constraints
c         such that  transpose[a] * [x] = [b]
c     b = The length m vector of constraints.
c     xl = The vector of lower bounds on x
c     xu = The vector of upper bounds on x
c     x = The vector of variables to be minimised.
c     funx = The function to be optimised. this is an external procedure
c            name in the main program.
c     gradx = The name of the subroutine that computes the gradient of u.
c             this is an external procedure name in the main program.
c     itype = type of optimisation required :-
c             itype = 1 for maximising
c             itype = -1 for minimising
c     acc = The required accuracy of the solution (length of the
c           Kuhn - Tucker residual vector).
c           It is convenient to set acc = 0.0
c     iprint = The amount of diagnostic output.
c              iprint = 0  no output
c              iprint < 0  diagnostic output
c     relacc = The relative accuracy of the computation. It is determined
c              by the program to be close to the relative precision of
c              the computer arithmetic.
c
c---- Subroutine outputs
c
c     iact = The m + 2 * n vector of active constraints
c     nact = The final number of active constraints
c     par = The array of estimates of the Lagrange multipliers of the
c           active constraints.
c     info = Termination code.
c            info = 1  The final X is feasible and the solution satisfies
c                      the condition, acc.
c            info = 2  The final X is feasible and termination occurs
c                      because rounding errors seem to prevent higher
c                      accuracy.
c            info = 3  The final X is feasible and line search fails to
c                      find a better solution of the objective function,
c                      although it is predicted by the gradient vector.
c                      If the final rate of convergence is slow, the
c                      coding of the gradient should be suspected.
c            info = 4  The calculation cannot start because the bounds
c                      are inconsistent
c            info = 5  The error indicates that the equality constraints
c                      are inconsistent including freezing of variables
c                      by setting xl(i) = xu(i)
c            info = 6  The equality constraints and the bounds are
c                      incompatible.
c            info = 7  The bounds and equality constraints can be
c                      satisfied but the general inequality constraints
c                      cannot.
c            info = 8  The limit on the number of function calls has been
c                      reached.
c     z = Working matrix such that z * z(transpose) = inverse of the
c         second derivatives of the active gradients.
c     u = The diagonal elements of the Goldfarb - Idnani upper triangular
c         factorization matrix.
c     xbig = The array of largest absolute values of the variables.
c     zznorm = Scalar to the columns of the matrix z.
c     tol = The relative tolerance of constraint residuals.  This is
c           reduced during the search until tol <= relacc
c     meql = The number of independent inequalities.
c     mtot = The total number of constraints and bounds
c     iterc = The number of iterations of the main calculation
c     nfvals = The number of function evaluations.
c     nfmax = The bound on the number of function evaluations
c             if nfmax > 0
c     space = workin space of (13 * n + m) elements
c     f = The function value on output
c
c---- Subroutines Required
c
c     addcon
c     conres
c     delcon
c     getd
c     ktvec
c     lsrch
c     newcon
c     sdegen
c     sdirn
c     stepbd
c     zbfgs
c     ddot
c     dzero
c     dmmove
c     dmxadd
c     dmxsub
c     dscalm
c     dscald
c
c-----

      implicit none

      integer nfreq,iflag1
      real*8 summeas,summeasp,tmeas,ampl1
      real*8 sumcalc,sumcalcp,tcalc,dt

      common /wedgedata/summeas,summeasp,tmeas,sumcalc,sumcalcp,ampl1,
     :     tcalc,dt,iflag1,nfreq

      real*8 freq(nfreq),filter(nfreq)

      integer n,m
      real*8 a(n * m),b(*),xl(*),xu(*),x(*),par(*),z(n * n)
      real*8 u(*),xbig(*),space(2 * m + 14 * n)
      real*8 relacc,zznorm,tol,acc,funx,f
      integer iact(*)
      integer nact,iprint,info,meql,mtot,iterc,nfvals,nfmax,itype
      external funx,gradx

      real*8 fprev,stepcb,ddotg,relaxf,ssqkt,diff,step,sum
      integer msat,iterk,nfvalk,iterp,indxbd,i,k
      integer ig,ireskt,id,iztg,ixs,igs,ispace,ibres

c-----
c     Define entry points into working space (14 * n + 2 * m)
c-----

      ig = 1
      ireskt = ig + n
      id = ireskt + n
      iztg = id + n
      ixs = iztg + n
      igs = ixs + n
      ispace = igs + n
      ibres = ispace + 6 * n + m

c-----
c     Initialise the minimization calculation
c-----

      msat = mtot
      iterk = iterc
      nfvalk = nfvals

      if(nfvals.eq.0.or.info.eq.1) then

        f = funx(n,x,freq,filter)

        call gradx(n,x,space(ig),freq,filter)

        if(itype.gt.0) then

          f = -f
          call dscalm(space(ig),n,-1.0d0)

        endif

        nfvals = nfvals + 1

      endif

      fprev = dabs(f + f + 1.0d0)
      iterp = -1

      if(iprint.ne.0) then

        print 1000,tol
 1000   format(/5x,'new value of tol =',1pd13.5)
        iterp = iterc

      endif

c-----
c     Calculate the next search direction
c-----

   10 call conres(n,m,a,b,xl,xu,x,iact,nact,space(ig),z,u,xbig,
     :space(ibres),space(id),space(iztg),
     :relacc,tol,stepcb,ddotg,meql,msat,mtot,indxbd,space(ispace))

c-----
c     Calculate the Kuhn Tucker residual vector
c-----

      call ktvec(n,m,a,iact,nact,par,space(ig),space(ireskt),z,u,
     :space(ibres),relaxf,meql,ssqkt,space(ispace))

c-----
c     Test for convergence
c-----

      if(ssqkt.le.acc * acc) then

        info = 1
        goto 70

      endif

      if(ddotg.ge.0.0d0) then

        info = 2
        goto 70

      endif

c-----
c     Test for termination due to no decrease in f.
c-----

      if(f.ge.fprev) then

        if(tol.eq.relacc.or.nact.eq.0) then

          if(diff.gt.0.0d0) goto 20

        endif

        info = 3
        goto 70

      endif

   20 diff = fprev - f
      fprev = f

c-----
c     Test that more calls of fgcalc are allowed
c-----

      if(nfvals.eq.nfmax) then

        info = 8
        goto 70

      endif

c-----
c     Test whether to reduce tol and to provide printing
c-----

      if(tol.gt.relacc.and.iterc.gt.iterk.and.
     : 0.1d0 * relaxf.ge.dmax1(diff,-0.5d0 * ddotg)) goto 70

      if(iterp.eq.iterc) then

        iterp = iterc + iabs(iprint)
        goto 80

      endif

c-----
c     Calculate the step along the search direction
c-----

   40 iterc = iterc + 1

      call lsrch(n,x,funx,gradx,itype,space(ig),space(id),space(ixs),
     :space(igs),relacc,stepcb,ddotg,f,step,nfvals,nfmax,space(ispace),
     :freq,filter)

      if(step.eq.0.0d0) then

        info = 3
        sum = 0.0d0

        do i = 1,n

          sum = sum + dabs(space(id + i - 1) * space(igs + i - 1))

        enddo

        if(ddotg + relacc * sum.ge.0.0d0) info = 2
        goto 70

      endif

c-----
c     Revise xbig
c-----

      do i = 1,n

        xbig(i) = dmax1(xbig(i),dabs(x(i)))

      enddo

c-----
c     Revise the second derivative approximation
c-----

      call zbfgs(n,x,nact,space(ig),z,space(iztg),space(ixs),
     :space(igs),zznorm)

c-----
c     Add a constraint to the active set if it restricts the step
c-----

      if(step.eq.stepcb) then

        k = iact(indxbd)

        if(k.gt.m) then

          k = k - m

          if(k.le.n) then

            x(k) = xl(k)

            else

            x(k - n) = xu(k - n)

          endif

        endif

        call addcon(n,m,a,iact,nact,z,u,relacc,indxbd,space(ispace))

      endif

      goto 10

c-----
c     printing from the subroutine
c-----

   70 if(iprint.eq.0) goto 90

      iterp = -1

   80 print 1010, iterc,nfvals,f
 1010 format(/5x,'iters =',i4,5x,'f.vals =',i4,5x,'f =',1pe15.7)

      if(nfvals.gt.nfvalk) then

        print 1020, (x(i),i = 1,n)
 1020 format('    x =',(1p5d14.5))
        print 1030, (space(ig + i - 1),i = 1,n)
 1030 format('    g =',(1p5d14.5))

        else

        print 1040
 1040 format(5x,'no change to x and g since previous output')

      endif

      if(iprint.lt.0) then

        if(nact.eq.0) then

           print 1050
 1050 format(5x,'no active constraints')

           else

           print 1060, (iact(i),i = 1,nact)
 1060 format('    ia =',(14i5))
          print 1070, (par(i),i = 1,nact)
 1070 format('    lp =',(1p5d14.5))

        endif

        if(nact.eq.n) then

          print 1080
 1080 format(5x,'kt residual vector is zero')

          else

          print 1090, (space(ireskt + i - 1),i = 1,n)
 1090 format('    kt =',(1p5d14.5))

        endif

      endif

      if(iterp.ge.0) goto 40

   90 return
      end
