C***********************************************************************
C                 copyright 2001, Amoco Production Company             *
C                             All Rights Reserved                      *
C                     an affiliate of BP America Inc.                  *
C***********************************************************************
      subroutine newcon(n,m,a,iact,nact,z,u,d,relacc,mdeg,space)

c-----
c     This subroutine adds a new constraint to the active set during
c     minimisation of a differentiable function of several variables,
c     subject to linear constraints on the values of the variables.
c
c     See, Powell, M.J.D.,1989, "Tolmin: A Fortran Package for Linearly
c     Constrained Optimization Calculations" University of Cambridge,
c     Dept. of Applied Mathematics and Theoretical Physics report NA2.
c
c---- Subroutine inputs
c
c     n = The number of variables
c     m = The number of linear constraints (excluding simple bounds)
c     a = The n * m array of coefficients of equality constraints
c         such that  transpose[a] * [x] = [b]
c     relacc = The relative accuracy of the computation. It is determined
c              by the program to be close to the relative precision of
c              the computer arithmetic.
c
c---- Subroutine outputs
c
c     iact = The m + 2 * n vector of active constraints
c     nact = The final number of active constraints
c     z = Working matrix such that z * z(transpose) = inverse of the
c         second derivatives of the active gradients.
c     u = The diagonal elements of the Goldfarb - Idnani upper triangular
c         factorization matrix.
c     d = The array of the components of the stepl length of the line
c         search in the directions with respect to X.
c     mdeg = The number of constraints with very small residuals that
c            are considered to be in the set of satisfied constraints.
c     space = working space of (3 * n) elements
c
c---- Subroutines Required
c
c     addcon
c     sdegen
c     ddot
c     dmmove
c     dzero
c
c-----

      integer n,m
      real*8 a(n * m),z(n * n),u(*),d(*),space(3 * n)
      real*8 relacc
      integer iact(*)
      integer nact,mdeg

      real*8 cvmax,sum,sumabs,sumd,cviol,savsum,savabs,temp
      integer np,khigh,i,j,k,jm,iadd,jmv,izzdiag,ij,ik

c-----
c     Define entry points to workin space
c     space 1 to n is used by gmnew
c     space 1 to 2 * n is used a working space for addcon
c     space 2 * n to 3 * n is used for zzdiag
c-----

      izzdiag = 0

c-----
c     Initialization
c-----

      np = nact + 1
      khigh = mdeg

      do i = 1,n

        space(izzdiag + i) = 0.0d0

        do j = np,n

          ij = (j - 1) * n + i
          space(izzdiag + i) = space(izzdiag + i) + z(ij) * z(ij)

        enddo

      enddo

c-----
c     Calculate the scalar products of d with its constraints
c-----

   20 cvmax = 0.0d0

      do k = np,khigh

        j = iact(k)

        if(j.le.m) then

          sum = 0.0d0
          sumabs = 0.0d0
          sumd = 0.0d0
          ij = (j - 1) * n + 1

          do i = 1,n

            temp = d(i) * a(ij)
            sum = sum + temp
            sumabs = sumabs + dabs(temp)
            sumd = sumd + space(izzdiag + i) * a(ij) * a(ij)
            ij = ij + 1

          enddo

        else

          jm = j - m

          if(jm.le.n) then

            sum = -d(jm)

            else

            jm = jm - n
            sum = d(jm)

          endif

          sumabs = dabs(sum)
          sumd = space(izzdiag + jm)

        endif

c-----
c     Pick out the most violated constraint, or return if the
c     violation is negligible
c-----

        if(sum.gt.relacc * sumabs) then

          cviol = sum * sum / sumd

          if(cviol.gt.cvmax) then

            cvmax = cviol
            iadd = k
            savsum = sum
            savabs = sumabs

          endif

        endif

      enddo

      if(cvmax.le.0.0d0) return

      if(nact.gt.0) then

c-----
c     Set space to the gradient of the most violated constraint
c-----

        j = iact(iadd)

        if(j.le.m) then

          ij = (j - 1) * n + 1
          jmv = 0
          call dmmove(n,a(ij),space)

          else

          jmv = j - m
          call dzero(n,space)

          if(jmv.le.n) then

            space(jmv) = -1.0d0

            else

            jmv = jmv - n
            space(jmv) = 1.0d0

          endif

        endif

c-----
c     Modify space for the next active constraint
c-----

        k = nact

        do while (k.gt.0)

          ik = (k - 1) * n + 1
          call ddot(n,z(ik),space,temp)
          temp = temp * u(k)
          j = iact(k)

          if(j.le.m) then

            ij = (j - 1) * n + 1

            do i = 1,n

              space(i) = space(i) - temp * a(ij)
              ij = ij + 1

            enddo

          else

            jm = j - m

            if(jm.le.n) then

              space(jm) = space(jm) + temp

              else

              space(jm - n) = space(jm - n) - temp

            endif

          endif

c-----
c     Revise the values of savsum and savabs
c-----

          sum = 0.0d0
          sumabs = 0.0d0

          do i = 1,n

            temp = d(i) * space(i)
            sum = sum + temp
            sumabs = sumabs + dabs(temp)

          enddo

          savsum = dmin1(savsum,sum)
          savabs = dmax1(savabs,sumabs)
          k = k - 1

        enddo

c-----
c     Add the new constraint to the active set if the constraint
c     violation is still significant
c-----

        if(jmv.gt.0) d(jmv) = 0.0d0
        if(savsum.le.relacc * savabs) goto 120

      endif

      k = nact
      call addcon(n,m,a,iact,nact,z,u,relacc,iadd,space)
      if(nact.gt.k) return

c-----
c     Seek another constraint violation
c-----

      iadd = np

  120 if(np.lt.khigh) then

        k = iact(khigh)
        iact(khigh) = iact(iadd)
        iact(iadd) = k
        khigh = khigh - 1
        goto 20

      endif

      return
      end
