C***********************************************************************
C                 copyright 2001, Amoco Production Company             *
C                             All Rights Reserved                      *
C                     an affiliate of BP America Inc.                  *
C***********************************************************************
      subroutine satact(n,m,a,b,xl,xu,x,iact,nact,info,z,u,xbig,
     :relacc,tol,meql)

c-----
c     This subroutine adjusts the current point x and/or removes
c     constraints from the active set until the residual of every active
c     constraint is acceptably small at x.  It is used during
c     minimisation of a differentiable function of several variables,
c     subject to linear constraints on the values of the variables.
c
c     See, Powell, M.J.D.,1989, "Tolmin: A Fortran Package for Linearly
c     Constrained Optimization Calculations" University of Cambridge,
c     Dept. of Applied Mathematics and Theoretical Physics report NA2.
c
c---- Subroutine inputs
c
c
c     n = The number of variables
c     m = The number of linear constraints (excluding simple bounds)
c     a = The n * m array of coefficients of equality constraints
c         such that  transpose[a] * [x] = [b]
c     b = The length m vector of constraints.
c     xl = The vector of lower bounds on x
c     xu = The vector of upper bounds on x
c     x = The vector of variables to be minimised.
c
c---- Subroutine outputs
c
c     iact = The m + 2 * n vector of active constraints
c     nact = The final number of active constraints
c     info = Termination code.
c            info = 1  The final X is feasible and the solution satisfies
c                      the condition, acc.
c            info = 2  The final X is feasible and termination occurs
c                      because rounding errors seem to prevent higher
c                      accuracy.
c            info = 3  The final X is feasible and line search fails to
c                      find a better solution of the objective function,
c                      although it is predicted by the gradient vector.
c                      If the final rate of convergence is slow, the
c                      coding of the gradient should be suspected.
c            info = 4  The calculation cannot start because the bounds
c                      are inconsistent
c            info = 5  The error indicates that the equality constraints
c                      are inconsistent including freezing of variables
c                      by setting xl(i) = xu(i)
c            info = 6  The equality constraints and the bounds are
c                      incompatible.
c            info = 7  The bounds and equality constraints can be
c                      satisfied but the general inequality constraints
c                      cannot.
c            info = 8  The limit on the number of function calls has been
c                      reached.
c     z = Working matrix such that z * z(transpose) = inverse of the
c         second derivatives of the active gradients.
c     u = The diagonal elements of the Goldfarb - Idnani upper triangular
c         factorization matrix.
c     xbig = The array of largest absolute values of the variables.
c     relacc = The relative accuracy of the computation. It is determined
c              by the program to be close to the relative precision of
c              the computer arithmetic.
c     tol = The relative tolerance of constraint residuals.  This is
c           reduced during the search until tol <= relacc
c     meql = The number of independent inequalities.
c
c---- Subroutines required
c
c     delcon
c     ddot
c
c----

      integer n,m
      real*8 a(n * m),b(*),xl(*),xu(*),x(*),z(n * n),u(*),xbig(*)
      real*8 relacc,tol
      integer iact(*)
      integer nact,info,meql

      real*8 res,resabs,resbig,temp,tempa,savex,scale
      integer i,j,k,jx,idrop,ij,ik

      if(nact.eq.0) return

      do k = 1,nact

c-----
c     Calculate the next constraint residual
c-----

        j = iact(k)

        if(j.le.m) then

          res = b(j)
          resabs = dabs(b(j))
          resbig = resabs
          ij = (j - 1) * n + 1

          do i = 1,n

            tempa = a(ij)
            temp = tempa * x(i)
            res = res - temp
            resabs = resabs + dabs(temp)
            resbig = resbig + dabs(tempa) * xbig(i)
            ij = ij + 1

          enddo

        else

          jx = j - m

          if(jx.le.n) then

            res = x(jx) - xl(jx)
            resabs = dabs(x(jx)) + dabs(xl(jx))
            resbig = xbig(jx) + dabs(xl(jx))
            savex = xl(jx)

            else

            jx = jx - n
            res = xu(jx) - x(jx)
            resabs = dabs(x(jx)) + dabs(xu(jx))
            resbig = xbig(jx) + dabs(xu(jx))
            savex = xu(jx)

          endif

        endif

c-----
c     Shift X if necessary
c-----

        if(res.ne.0.0d0) then

          temp = res /resabs
          if(k.le.meql) temp = -dabs(temp)

          if(tol.eq.relacc.or.temp + relacc.lt.0.0d0) then

            info = 1
            scale = res * u(k)
            ik = (k - 1) * n + 1

            do i = 1,n

              x(i) = x(i) + scale * z(ik)
              ik = ik + 1
              xbig(i) = dmax1(xbig(i),dabs(x(i)))

            enddo

            if(j.gt.m) x(jx) = savex

c-----
c     Else flag a constraint deletion if necessary
c-----

            else if(res / resbig.gt.tol) then

            iact(k) = -iact(k)

          endif

        endif

      enddo

c-----
c     Delete any flagged constraints and then return
c-----

      idrop = nact

   40 if(iact(idrop).lt.0) then

        iact(idrop) = -iact(idrop)
        call delcon(n,m,a,iact,nact,z,u,relacc,idrop)

      endif

      idrop = idrop - 1
      if(idrop.gt.meql) goto 40

      return
      end
