C***********************************************************************
C                 copyright 2001, Amoco Production Company             *
C                             All Rights Reserved                      *
C                     an affiliate of BP America Inc.                  *
C***********************************************************************
      subroutine sdegen(n,m,a,iact,nact,par,z,u,d,ztg,gm,relacc,
     :ddotgm,meql,mdeg,space)

c-----
c     This subroutine revises the variables describing the constrained
c     search direction during minimisation of a differentiable function
c     of several variables, subject to linear constraints on the values
c     of the variables.
c
c     See, Powell, M.J.D.,1989, "Tolmin: A Fortran Package for Linearly
c     Constrained Optimization Calculations" University of Cambridge,
c     Dept. of Applied Mathematics and Theoretical Physics report NA2.
c
c---- Subroutine inputs
c
c     n = The number of variables
c     m = The number of linear constraints (excluding simple bounds)
c     a = The n * m array of coefficients of equality constraints
c         such that  transpose[a] * [x] = [b]
c     relacc = The relative accuracy of the computation. It is determined
c              by the program to be close to the relative precision of
c              the computer arithmetic.
c
c---- Subroutine outputs
c
c     iact = The m + 2 * n vector of active constraints
c     nact = The final number of active constraints
c     par = The array of estimates of the Lagrange multipliers of the
c           active constraints.
c     z = Working matrix such that z * z(transpose) = inverse of the
c         second derivatives of the active gradients.
c     u = The diagonal elements of the Goldfarb - Idnani upper triangular
c         factorization matrix.
c     d = The array of the components of the stepl length of the line
c         search in the directions with respect to X.
c     ztg = The vector obtained from z(transpose) * g
c     ddotgm = The scalar product of d and gm
c     meql = The number of independent inequalities.
c     mdeg = The number of constraints with very small residuals that
c            are considered to be in the set of satisfied constraints.
c     gm = The array of residual gradient values after the active
c          constraints are adjusted with the Lagrange parameter.
c     space = working space of (4 * n) elements
c
c---- Subroutines Required
c
c     addcon
c     delcon
c     newcon
c     sdegen
c     sdirn
c     ddot
c     dmmove
c     dzero
c
c-----

      integer n,m
      real*8 a(n * m),par(*),z(n * n),u(*),d(*),ztg(*),gm(*)
      real*8 space(3 * n)
      real*8 relacc,ddotgm
      integer iact(*)
      integer meql,mdeg,nact

      real*8 dtest,sum,temp,ratio
      integer mp,np,itest,i,j,k,jm,ku,idrop,theta
      integer igmnew,iparnew,ik,ij

c-----
c     Define entry points into workin space
c     space 1 to space 3 * n is used as working space by newcon
c     space 1 to space n is reused by gmnew
c     space n + 1 to space 2 * n is reused by parnew
c-----

      igmnew = 0
      iparnew = n

      mp = meql + 1
      dtest = 0.0d0

c-----
c     Calculate the search direction and branch if it is not downhill
c-----

   10 call sdirn(n,nact,z,d,ztg,gm,relacc,ddotgm)
      if(ddotgm.eq.0.0d0) return

c-----
c     Branch if there is no need to consider any degenerate constraints.
c     The test gives termination if two consecutive additions to the
c     active set fail to increase the predicted new value of F.
c-----

      if(nact.eq.mdeg) return
      np = nact + 1
      call ddot(n - np + 1,ztg(np),ztg(np),sum)

      if(dtest.gt.0.0d0.and.sum.ge.dtest) then

        if(itest.eq.1) return
        itest = 1

        else

        dtest = sum
        itest = 0

      endif

c-----
c     Add a constraint to the active set if there are any significant
c     violations of degenerate constraints
c-----

      k = nact
      call newcon(n,m,a,iact,nact,z,u,d,relacc,mdeg,space)
      if(nact.eq.k) return
      par(nact) = 0.0d0

c-----
c     Calculate the new reduced gradient and Lagrange parameters
c-----

   30 call dmmove(n,gm,space(igmnew + 1))
      k = nact
   50 ik = (k - 1) * n + 1
      call ddot(n,z(ik),space(igmnew + 1),temp)
      temp = temp * u(k)
      space(iparnew + k) = par(k) + temp
      if(k.eq.nact) space(iparnew + k) = dmin1(space(iparnew + k),0.0d0)
      j = iact(k)

      if(j.le.m) then

        ij = (j - 1) * n + 1

        do i = 1,n

          space(igmnew + i) = space(igmnew + i) - temp * a(ij)
          ij = ij + 1

        enddo

      else

        jm = j - m

        if(jm.le.n) then

          space(igmnew + jm) = space(igmnew + jm) + temp

          else

          space(igmnew + jm - n) = space(igmnew + jm - n) - temp

        endif

      endif

      k = k - 1
      if(k.gt.meql) goto 50

c-----
c     Set RATIO for linear interpolation between PAR and PARNEW
c-----

      ratio = 0.0d0

      if(mp.lt.nact) then

        ku = nact - 1

        do k = mp,ku

          if(space(iparnew + k).gt.0.0d0) then

            ratio = space(iparnew + k) / (space(iparnew + k) - par(k))
            idrop = k

          endif

        enddo

      endif

c-----
c     Apply the linear interpolation
c-----

      theta = 1.0d0 - ratio

      do k = mp,nact

        par(k) = dmin1(theta * space(iparnew + k) + ratio * par(k),
     :0.0d0)

      enddo

      do i = 1,n

        gm(i) = theta * space(igmnew + i) + ratio * gm(i)

      enddo

c-----
c     Drop a constraint if RATIO is positive
c-----

      if(ratio.gt.0.0d0) then

        call delcon(n,m,a,iact,nact,z,u,relacc,idrop)

        do k = idrop,nact

          par(k) = par(k + 1)

        enddo

        goto 30

      endif

c-----
c     Return if there is no freeedom for a new search direction
c-----

      if(nact.lt.n) goto 10
      ddotgm = 0.0d0
      return
      end
