C***********************************************************************
C                 copyright 2001, Amoco Production Company             *
C                             All Rights Reserved                      *
C                     an affiliate of BP America Inc.                  *
C***********************************************************************
      subroutine sdirn(n,nact,z,d,ztg,gm,relacc,ddotgm)

c-----
c     This subroutine determines the components in the search direction
c     during minimisation of a differentiable function of several
c     variables, subject to linear constraints on the values of the
c     variables.
c
c     See, Powell, M.J.D.,1989, "Tolmin: A Fortran Package for Linearly
c     Constrained Optimization Calculations" University of Cambridge,
c     Dept. of Applied Mathematics and Theoretical Physics report NA2.
c
c---- Subroutine inputs
c
c     n = The number of variables
c     relacc = The relative accuracy of the computation. It is determined
c              by the program to be close to the relative precision of
c              the computer arithmetic.
c
c---- Subroutine outputs
c
c     nact = The final number of active constraints
c     z = Working matrix such that z * z(transpose) = inverse of the
c         second derivatives of the active gradients.
c     d = The array of the components of the stepl length of the line
c         search in the directions with respect to X.
c     ztg = The vector obtained from z(transpose) * g
c     ddotgm = The scalar product of d and gm
c     gm = The array of residual gradient values after the active
c          constraints are adjusted with the Lagrange parameter.
c
c-----

      integer n
      real*8 z(n * n),d(*),ztg(*),gm(*)
      real*8 relacc,ddotgm
      integer nact

      real*8 sum,sumabs,temp
      integer np,i,j,ij

      ddotgm = 0.0d0
      if(nact.ge.n) goto 60

c-----
c     Premultiply GM by the transpose of Z.
c-----

      np = nact + 1

      do j = np,n

        sum = 0.0d0
        sumabs = 0.0d0
        ij = (j - 1) * n + 1

        do i = 1,n

          temp = z(ij) * gm(i)
          sum = sum + temp
          sumabs = sumabs + dabs(temp)
          ij = ij + 1

        enddo

        if(dabs(sum).le.relacc * sumabs) sum = 0.0d0
        ztg(j) = sum

      enddo

c-----
c     Form D by premultiplying ZTG by -Z.
c-----

      do i = 1,n

        sum = 0.0d0
        sumabs = 0.0d0

        do j = np,n

          ij = (j - 1) * n + i
          temp = z(ij) * ztg(j)
          sum = sum - temp
          sumabs = sumabs + dabs(temp)

        enddo

        if(dabs(sum).le.relacc * sumabs) sum = 0.0d0
        d(i) = sum

      enddo

c-----
c     Test that the search direction is downhill
c-----

      sumabs = 0.0d0

      do i = 1,n

        temp = d(i) * gm(i)
        ddotgm = ddotgm + temp
        sumabs = sumabs + dabs(temp)

      enddo

      if(ddotgm + relacc * sumabs.ge.0.0d0) ddotgm = 0.0d0

   60 return
      end
