C***********************************************************************
C                 copyright 2001, Amoco Production Company             *
C                             All Rights Reserved                      *
C                     an affiliate of BP America Inc.                  *
C***********************************************************************
      subroutine tolmin(n,m,meq,a,b,xl,xu,x,funx,gradx,itype,acc,iact,
     :nact,par,iprint,info,space,fout,freq,filter)

c-----
c     This subroutine carries out minimisation of a differentiable
c     function of several variables, subject to linear constraints on
c     the values of the variables.
c
c     See, Powell, M.J.D.,1989, "Tolmin: A Fortran Package for Linearly
c     Constrained Optimization Calculations" University of Cambridge,
c     Dept. of Applied Mathematics and Theoretical Physics report NA2.
c
c---- Subroutine inputs
c
c     n = The number of variables
c     m = The number of linear constraints (excluding simple bounds)
c     meq = The number of constraints that are equalities (0 <= meq <= m)
c     a = The n * m array of coefficients of equality constraints
c         such that  transpose[a] * [x] = [b]
c     b = The length m vector of constraints.
c     xl = The vector of lower bounds on x
c     xu = The vector of upper bounds on x
c     x = The vector of variables to be minimised.
c     funx = The function to be optimised. this is an external procedure
c            name in the main program.
c     gradx = The name of the subroutine that computes the gradient of u.
c             this is an external procedure name in the main program.
c     itype = type of optimisation required :-
c             itype = 1 for maximising
c             itype = -1 for minimising
c     acc = The required accuracy of the solution (length of the
c           Kuhn - Tucker residual vector).
c           It is convenient to set acc = 0.0
c     iprint = The amount of diagnostic output.
c              iprint = 0  no output
c              iprint < 0  diagnostic output
c
c---- Subroutine outputs
c
c     iact = The m + 2 * n vector of active constraints
c     nact = The final number of active constraints
c     par = The array of estimates of the Lagrange multipliers of the
c           active constraints.
c     info = Termination code.
c            info = 1  The final X is feasible and the solution satisfies
c                      the condition, acc.
c            info = 2  The final X is feasible and termination occurs
c                      because rounding errors seem to prevent higher
c                      accuracy.
c            info = 3  The final X is feasible and line search fails to
c                      find a better solution of the objective function,
c                      although it is predicted by the gradient vector.
c                      If the final rate of convergence is slow, the
c                      coding of the gradient should be suspected.
c            info = 4  The calculation cannot start because the bounds
c                      are inconsistent
c            info = 5  The error indicates that the equality constraints
c                      are inconsistent including freezing of variables
c                      by setting xl(i) = xu(i)
c            info = 6  The equality constraints and the bounds are
c                      incompatible.
c            info = 7  The bounds and equality constraints can be
c                      satisfied but the general inequality constraints
c                      cannot.
c            info = 8  The limit on the number of function calls has been
c                      reached.
c     space = workin space of (n * n + 16 * n + 2 * m) elements
c     fout = The function value on output
c
c---- Subroutines Required
c
c     addcon
c     adjtol
c     conres
c     delcon
c     eqcons
c     getd
c     getfes
c     initzu
c     ktvec
c     lsrch
c     minfun
c     newcon
c     satact
c     sdegen
c     sdirn
c     stepbd
c     tolmin
c     zbfgs
c     ddot
c     dzero
c     dmmove
c     dmxadd
c     dmxsub
c     dscald
c     dscalm
c
c-----

      implicit none

      integer nfreq,iflag1
      real*8 summeas,summeasp,tmeas,ampl1
      real*8 sumcalc,sumcalcp,tcalc,dt

      common /wedgedata/summeas,summeasp,tmeas,sumcalc,sumcalcp,ampl1,
     :     tcalc,dt,iflag1,nfreq

      real*8 freq(nfreq),filter(nfreq)

      integer n,m
      real*8 a(n * m),b(m),xl(n),xu(n),x(n),par(m + 2 * n)
      real*8 space(n * n + 16 * n + 2 * m)
      real*8 funx,fout,acc
      integer iact(m + 2 * n)
      integer meq,nact,iprint,info,meql,msat,mp,k,itype
      external funx,gradx

      real*8 zznorm,relacc,tol
      integer iterc,nfvals,nfmax,i,mtot,iz,iu,ixbig,ispace

c-----
c     Define pointers into working space
c-----

      iz = 1
      iu = iz + n * n
      ixbig = iu + n
      ispace = ixbig + n

c-----
c     Initialise ZZNORM, ITERC,NFVALS, and NFMAX
c-----

      zznorm = -1.0d0
      iterc = 0
      nfvals = 0
      nfmax = 0
      if(info.gt.0) nfmax = info

c-----
c     Check the bounds on N,M, and MEQ
c-----

      info = 4

      if(meq.gt.m) then

        if(iprint.ne.0) print 1010
 1010   format(/5x,'ERROR return from TOLMIN because  condition',
     :' MEQ is violated (MEQ > M)')
        goto 40

      endif

c-----
c     Initialize RELACC, Z, U, and TOL
c-----

      call initzu(n,m,xl,xu,x,iact,meql,info,space(iz),space(iu),
     :space(ixbig),relacc)
      tol = dmax1(0.01d0,10.0d0 * relacc)

      if(info.eq.4) then

        if(iprint.ne.0) print 1020
 1020   format(/5x,'ERROR return from TOLMIN because a lower',
     :' bound exceeds an upper bound')
        goto 40

      endif

c-----
c     Add any equality constraints to the active set.
c-----

      if(meq.gt.0) then

        call eqcons(n,m,meq,a,b,xu,iact,meql,info,space(iz),space(iu),
     :relacc,space(ispace))

        if(info.eq.5) then

          if(iprint.ne.0) print 1030
 1030     format(/5x,'ERROR return from TOLMIN because the',
     :' equality constraints are inconsistent')
          goto 40

        endif

      endif

      nact = meql
      msat = meql

c-----
c     add the bounds to the list of constraints
c-----

      mtot = nact

      do i = 1,n

        if(xl(i).lt.xu(i)) then

          mtot = mtot + 2
          iact(mtot - 1) = m + i
          iact(mtot) = m + n + i

        endif

      enddo

c-----
c     Try to satisfy the bound constraints
c-----

      call getfes(n,m,a,b,xl,xu,x,iact,nact,info,space(iz),space(iu),
     :space(ixbig),relacc,tol,meql,msat,mtot,space(ispace))

      if(msat.lt.mtot) then

          if(iprint.ne.0) print 1040
 1040     format(/5x,'ERROR return from TOLMIN because the',
     :' equalities and bounds are inconsistent')
          goto 40

      endif

c-----
c     Add the ordinary inequalities to the list of constraints
c-----

      if(m.gt.meq) then

        mp = meq + 1

        do k = mp,m

          mtot = mtot + 1
          iact(mtot) = k

        enddo

      endif

c-----
c     Correct any constraint violations
c-----

   30 call getfes(n,m,a,b,xl,xu,x,iact,nact,info,space(iz),space(iu),
     :space(ixbig),relacc,tol,meql,msat,mtot,space(ispace))

      if(msat.lt.mtot) then

          if(iprint.ne.0) print 1050
 1050     format(/5x,'ERROR return from TOLMIN because the',
     :' constraints are inconsistent')
          info = 7
          goto 40

          else if(meql.eq.n) then

          if(iprint.ne.0) print 1060
 1060     format(/5x,'TOLMIN finds that the variables are',
     :' determined by the equality constraints')
          goto 40

      endif

c-----
c     Minimise the objective function in the case when constraints are
c     treated as degenerate if their residuals are less than TOL.
c-----

      call minfun(n,m,a,b,xl,xu,x,funx,gradx,itype,acc,iact,nact,par,
     :iprint,info,space(iz),space(iu),space(ixbig),relacc,zznorm,tol,
     :meql,mtot,iterc,nfvals,nfmax,space(ispace),fout,freq,filter)

c-----
c     Reduce TOL if necessary
c-----

      if(tol.gt.relacc.and.nact.gt.0) then

        if(nfvals.ne.nfmax) then

          call adjtol(n,m,a,b,xl,xu,x,iact,nact,space(ixbig),relacc,tol,
     :meql)
          goto 30

          else

          info = 8

        endif

      endif

      if(iprint.ne.0) then

        if(info.eq.1) print 1070
 1070   format(/5x,'TOLMIN has achieved the required accuracy')
        if(info.eq.2) print 1080
 1080   format(/5x,'TOLMIN can make no further progress because',
     :' of rounding errors')
        if(info.eq.3) print 1090
 1090   format(/5x,'TOLMIN can make no further progress because',
     :' F will not decrease any more')
        if(info.eq.8) print 1100
 1100   format(/5x,'TOLMIN has reached the given limit on the',
     :' number of calls of fgcalc')

      endif

   40 return
      end
