C***********************************************************************
C                 copyright 2001, Amoco Production Company             *
C                             All Rights Reserved                      *
C                     an affiliate of BP America Inc.                  *
C***********************************************************************
      subroutine zbfgs(n,x,nact,g,z,ztg,xs,gs,zznorm)

c-----
c     This subroutine updates the elements of the matrix z by the
c     constrained case of the Broyden-Fletcher-Goldberg-Shanno method
c     during minimisation of differentiable function of several variables,
c     subject to linear constraints on the values of the variables.
c
c     See, Powell, M.J.D.,1989, "Tolmin: A Fortran Package for Linearly
c     Constrained Optimization Calculations" University of Cambridge,
c     Dept. of Applied Mathematics and Theoretical Physics report NA2.
c
c---- Subroutine inputs
c
c     n = The number of variables
c     x = The vector of variables to be minimised.
c     g = The array of function gradients at X.
c     nact = The final number of active constraints
c
c---- Subroutine outputs
c
c     z = Working matrix such that z * z(transpose) = inverse of the
c         second derivatives of the active gradients.
c     ztg = The vector obtained from z(transpose) * g
c     xs = The vector of variables to be minimised on entry to the
c          subroutine
c     gs = The array of function gradients at X on entry to the
c          subroutine
c     zznorm = Scalar to the columns of the matrix z.
c
c---- Subroutines Required
c
c     ddot
c     dmmove
c     dmxadd
c     dmxsub
c     dscald
c     dscalm
c
c-----

      integer n
      real*8 x(*),g(*),z(n * n),ztg(*),xs(*),gs(*)
      real*8 zznorm
      integer nact

      real*8 dd,dg,temp,wcos,wsin,sum
      integer i,k,kp,np,km,ik,ikp,inp

c-----
c     Test if there is sufficient convexity for the update.
c-----

      dd = 0.0d0
      dg = 0.0d0
      temp = 0.0d0
      call dmxsub(x,xs,xs,n,1)
      call ddot(n,xs,xs,dd)
      call ddot(n,gs,xs,temp)
      call dmxsub(g,gs,gs,n,1)
      call ddot(n,gs,xs,dg)

      if(dg.lt.0.1d0 * dabs(temp)) return

c-----
c     Transform the Z matrix
c-----

      k = n
   20 kp = k
      k = k - 1

      if(k.gt.nact) then

        if(ztg(kp).eq.0.0d0) goto 20
        temp = dabs(ztg(kp)) * dsqrt(1.0d0 + (ztg(k) / ztg(kp)) ** 2)
        wcos = ztg(k) / temp
        wsin = ztg(kp) / temp
        ztg(k) = temp
        ik = (k - 1) * n + 1
        ikp = (kp - 1) * n + 1

        do i = 1,n

          temp = wcos * z(ikp) - wsin * z(ik)
          z(ik) = wcos * z(ik) + wsin * z(ikp)
          z(ikp) = temp
          ik = ik + 1
          ikp = ikp + 1

        enddo

        goto 20

      endif

c-----
c     Update the value of ZZNORM
c-----

      if(zznorm.lt.0.0d0) then

        zznorm = dd / dg

        else

        temp = dsqrt(zznorm * dd / dg)
        zznorm = dmin1(zznorm,temp)
        zznorm = dmax1(zznorm,0.1d0 * temp)

      endif

c-----
c     Complete the updating of Z
c-----

      np = nact + 1
      temp = dsqrt(dg)
      inp = (np - 1) * n + 1
      call dmmove(n,xs,z(inp))
      call dscald(z(inp),n,temp)

      if(np.lt.n) then

        km = np + 1

        do k = km,n

          ik = (k - 1) * n + 1
          call ddot(n,gs,z(ik),temp)
          temp = temp / dg

          do i = 1,n

            ik = (k - 1) * n + i
            z(ik) = z(ik) - temp * xs(i)

          enddo

          ik = (k - 1) * n + 1
          call ddot(n,z(ik),z(ik),sum)

          if(sum.lt.zznorm) then

            temp = dsqrt(zznorm / sum)
            call dscalm(z(ik),n,temp)

          endif

        enddo

      endif

      return
      end
