/***********************************************************************
 *                copyright 2001, Amoco Production Company             *
 *                            All Rights Reserved                      *
 *                    an affiliate of BP America Inc.                  *
 ***********************************************************************/
/* This program was written to take care of trace green word damage on
   an otherwise valid dataset. Much of the code was stolen directly from
   the program sis_verify, which traverses a dataset without using USP
   I/O. 

   "Green words" are basically values preceding data in a dataset which
   tell how many bytes the data consists of. For USP disk datasets, they
   are always 4 bytes in length. Two basic types of green word problems 
   can be repaired. In the first type, the traces are all of the proper
   length ((no_of_samples * 4) + 256 bytes for the trace header ), but
   the green words have been clobbered. For the second type, the traces
   are actually of varying lengths, but the existing green word correctly
   match those lengths. If that existing value for a trace is shorter than
   the correct trace length, this program zeroes out the samples it is 
   forced to pad and, optionally, kills the trace.

				- written 12/7/95 - Joe M. Wade
*/

#include <stdio.h>
#include <localsys.h>
#include <save_defs.h>
#ifndef CONVEXSYSTEM
#include <string.h>
#endif
#include <fcntl.h>
#include <cu_defs.h>
#include <usp_headers.h>
#include <sys/unistd.h>

#ifdef CRAY
#define WORDSIZE 8
#else
#define WORDSIZE 4
#endif

main (argc, argv) 
int    argc;
char * argv[];
{
	FILE *fdin,*fdout;
	int c;
	int i,j;
	int allocated_bytes = 0;
	int pad;
	off_t tail,offset;
	long bad_gw_count = 0;
	int computed_green = 0;
	int in_place = 0;
	int verbose;
	int use_orig_gw,help;
	int format,start_rec,end_rec,start_trace,end_trace;
	int no_of_samples,no_of_records;
	long no_of_traces = 0;
	long expected_traces;
	int traces_per_record;
	int traces_to_skip,seq_rec,seq_trace;
	char ntap[256],otap[256];
	char *buffer[sizeof(usp_trace_header)];
	char *data = NULL;
	int C_ARGIS();
	FILE *prtout = NULL;
	char *name = "CLEANGREEN";
	char prt_name[32];
	char *ikp_pid,host[9];
        time_t current_time, start_time, estimated_time;
	int start_slice,time_est;
 
	union {
		int size;
		char dummy[WORDSIZE];
		} green_word ;

	help = C_ARGIS("-h",&argc,argv);
	help += C_ARGIS("-?",&argc,argv);
	if (help > 0) {
	  fprintf(stderr,"****************************************");
	  fprintf(stderr,"***************************************\n");
	  fprintf(stderr,"Usage: clean_green -N[ntap] -O[otap] [-use] [-V] [-VT]\n");
	  fprintf(stderr," -N[ntap]  (default=stdin)\t: Input data set");
	  fprintf(stderr," file name\n");
	  fprintf(stderr," -O[otap]  (default=stdout)\t: Output data set");
	  fprintf(stderr," file name\n");
	  fprintf(stderr," -use\t\t\t\t: use existing green");
	  fprintf(stderr," words for dataset traversal\n");
	  fprintf(stderr," -V\t\t\t\t: verbose printout with rec & trc\n");
	  fprintf(stderr," -VT\t\t\t\t: print estimates of time to completion\n");
	  fprintf(stderr," -h\t\t\t\t: print this help\n");
	  fprintf(stderr,"****************************************");
	  fprintf(stderr,"***************************************\n\n");
	  exit(0);
	  }

	if ((ikp_pid = (char *)getenv("IKP_PID")) == (char *)NULL)
	  sprintf(prt_name,"%s.%d",name,getpid());
	else {
	  gethostname(host,9);
	  sprintf(prt_name,"%s.%d.%s",name,getpid(),host);
	  }

	prtout = fopen(prt_name,"w");
	if (prtout == NULL) prtout = stderr;

	C_ARGSTR("-N",ntap," "," ",&argc,argv);
	C_ARGSTR("-O",otap," "," ",&argc,argv);

	if (strncmp(ntap," ",1) != 0) {
	  if (strcmp(ntap,otap) == 0) {
	    in_place = 1;
	    fdin = fopen(ntap, "r+");
	    setvbuf(fdin,NULL,_IONBF,0);        /* turn off buffering */
	    }
	  else
	    fdin = fopen(ntap, "r");
	  if (fdin == NULL) {
	    fprintf(prtout,"ERROR - %s: unable to open input dataset %s\n",
		argv[0],ntap);
	    fprintf(stderr,"ERROR - %s: unable to open input dataset %s\n",
		argv[0],ntap);
	    exit(1);
	    }
	  fprintf(prtout,"input dataset = %s",ntap);
	  }
	else {
	  fdin = stdin;
	  fprintf(prtout,"input dataset = stdin");
	  }

	if (strncmp(otap," ",1) != 0) {
	  if ( in_place )
	    fdout = fdin;
	  else
	    fdout = fopen(otap, "w");
	  if (fdout == NULL) {
	    fprintf(prtout,"%s: unable to open output dataset %s\n",
		argv[0],otap);
	    fprintf(stderr,"%s: unable to open output dataset %s\n",
		argv[0],otap);
	    exit(1);
	    }
	  if (in_place)
	    fprintf(prtout," ( changing data in place ) \n\n");
	  else
	    fprintf(prtout,"\n\noutput dataset = %s\n\n",otap);
	  }
	else {
	  fdout = stdout;
	  fprintf(prtout,"\n\noutput dataset = stdout\n\n");
	  }
	
	verbose = C_ARGIS("-V",&argc,argv);
	time_est = C_ARGIS("-VT",&argc,argv);

	use_orig_gw = C_ARGIS("-use",&argc,argv);
	if (in_place && use_orig_gw) {
	  fprintf(prtout,"\n\nERROR (clean_green):\n");
	  fprintf(prtout,"In-place processing is not valid with ");
	  fprintf(prtout,"traversal using original green words\n");
	  fprintf(stderr,"\n\nERROR (clean_green):\n");
	  fprintf(stderr,"In-place processing is not valid with ");
	  fprintf(stderr,"traversal using original green words\n");
	  exit(1);
	  }

	green_word.size = 0;
	for (i=0,j=WORDSIZE-4; i<4; i++, j++) {
		c = getc(fdin);
		green_word.dummy[j] = (char) c;
 		}

	if ( ! in_place ) {
	  if ((fwrite(&green_word.dummy[WORDSIZE-4],4,1,fdout)) != 1) {
	     perror("clean_green (green word)");
	     exit(1);
	     }
	  }

	data = ( char *) malloc(green_word.size);

	if (data == NULL) {
	   fprintf(prtout,"ERROR (clean_green): unable to allocate %d bytes\n",
		green_word.size);
	   fprintf(stderr,"ERROR (clean_green): unable to allocate %d bytes\n",
		green_word.size);
	   exit(1);
	   }
	else
	   allocated_bytes = green_word.size;

	if ((fread(data,green_word.size,1,fdin)) == EOF) {
	   perror("clean_green (read)");
	   exit(1);
	   }

	if ( ! in_place ) {
	  if ((fwrite(data,green_word.size,1,fdout)) != 1) {
	     perror("clean_green (write)");
	     exit(1);
	     }
	  }

	get_hw_val(data, "NumSmp", LINEHEADER, &no_of_samples);
	get_hw_val(data, "NumRec", LINEHEADER, &no_of_records);
	get_hw_val(data, "NumTrc", LINEHEADER, &traces_per_record);
	expected_traces = no_of_records * traces_per_record;
/*
	no_of_samples = ((usp_line_header *)data)->NumSmp;
*/

	computed_green = sizeof(usp_trace_header)+(no_of_samples*4);

	fprintf(prtout,"expected trace green word = %d\n\n",computed_green);
	if ( verbose )
	  fprintf(prtout,"Traces found to have illegal green words:\n\n");

	if ( time_est != 0 ) {
	    start_time = time(NULL);
	    fprintf(stderr,"CLEAN_GREEN - data processing begins - time: %s",
		(char *)ctime(&start_time));
	    }

	while(1) {
		for (i=0,j=WORDSIZE-4; i<4; i++, j++) {
	 		if ((c=getc(fdin)) == EOF) {
		  	  fprintf(prtout,"\ntraces processed = %ld\n\n",
				no_of_traces);
			  fprintf(prtout,"count of illegal green words = %ld\n",
				bad_gw_count);
			  exit(0);
			  }
			green_word.dummy[j] = (char) c;
 			}
	
		if ( in_place ) {
#ifdef FSEEKO
		  if (fseeko(fdin,(off_t) -4,SEEK_CUR) == -1) {
#else
		  if (fseek(fdin,(off_t) -4,SEEK_CUR) == -1) {
#endif
		     perror("clean_green (in-place positioning)");
		     exit(1);
		      }
		  }

/* if in-place, we've already got enough room for the trace headers */

		  if ( ! in_place ) {
		    if ((allocated_bytes < computed_green) &&
			(allocated_bytes < green_word.size)) {

		      if ((green_word.size > computed_green) && use_orig_gw) {
		        allocated_bytes = green_word.size;
		        }
		      else {
		        allocated_bytes = computed_green;
		        }
	 	      data = ( char *) realloc(data,allocated_bytes);

		      if (data == NULL) {
	   	        fprintf(prtout,
			 "ERROR (clean_green): unable to reallocate %d bytes\n",
			  green_word.size);
	   	        fprintf(stderr,
			 "ERROR (clean_green): unable to reallocate %d bytes\n",
		  	  green_word.size);
		        exit(1);
		        }
		      }
		    }

	   	if ((fwrite(&computed_green,4,1,fdout)) != 1) {
		     perror("clean_green (green word)");
		     exit(1);
		     }

/* we only need the trace header if we're printing rec,trc values */

		if ( verbose ) {
	  	  if ((fread(data,sizeof(usp_trace_header),1,fdin)) == EOF) {
		    perror("clean_green (trace header read)");
		    exit(1);
		    }
		  }

		if (green_word.size != computed_green) {
		   bad_gw_count++;
		   if (verbose) {
		     fprintf(prtout,"rec %d trace %d (%d)\n",
			((usp_trace_header *)data)->RecNum,
			((usp_trace_header *)data)->TrcNum,
			green_word.size);
		     }

		   if ( use_orig_gw ) 
		     pad = computed_green - green_word.size;
		   else {
		     pad = 0;
		     green_word.size = computed_green;
		     }
		   }
		else
		   pad = 0;

		if ( verbose )
		  offset = sizeof(usp_trace_header);
		else
		  offset = 0;
		  
		tail = green_word.size - offset;

		if (tail <= 0) {
		  fprintf(prtout, "\nERROR (clean_green): ");
		  fprintf(prtout,
			"something's gone wrong; green word size = %d,",
			green_word.size);
		  fprintf(prtout," trace header size = %d\n",
			(int)sizeof(usp_trace_header));
		  fprintf(stderr, "\nERROR (clean_green): ");
		  fprintf(stderr,
			"something's gone wrong; green word size = %d,",
			green_word.size);
		  fprintf(stderr," trace header size = %d\n",
			(int)sizeof(usp_trace_header));
		  exit(1);
		  }

		no_of_traces++;
		if ( in_place ) {
#ifdef FSEEKO
		  if (fseeko(fdin,(off_t)tail, SEEK_CUR) == -1) {
#else
		  if (fseek(fdin,(off_t)tail, SEEK_CUR) == -1) {
#endif
			perror("clean_green (in-place data skip)");
			exit(1);
			}
		  }
		else {
		  if ( pad != 0 )
		      memset((char *)data+offset+tail,(char)0,pad);

	  	  if ((fread(data+offset,(off_t)tail,1,fdin)) == EOF) {
		    perror("clean_green (trace data read)");
		    exit(1);
		    }

		  if ((fwrite(data,(off_t)computed_green,1,fdout)) != 1) {
		    perror("clean_green (write)");
		    exit(1);
	     	    }
		  }

	        if ((no_of_traces%traces_per_record == 0)
		    && (time_est != 0)) {
	          current_time = time(NULL);
	          fprintf(stderr,"CLEAN_GREEN - %d of %d: ",
		    (int)(no_of_traces/traces_per_record),no_of_records);
	          estimated_time = (time_t) (((current_time - start_time) *
		    expected_traces) / no_of_traces);
	          estimated_time += start_time;
	          fprintf(stderr,"est. time of completion: %s",
	            (char *)ctime(&estimated_time));
	          }
		}
}
