C***********************************************************************
C                 copyright 2001, Amoco Production Company             *
C                             All Rights Reserved                      *
C                     an affiliate of BP America Inc.                  *
C***********************************************************************
      program clust3dm 
		
      implicit none
		
c-----------------------------------------------------------------------
c Bertram Kaufhold 	07/01/98	
c
c The clust3dm routine represents a modified K-means cluster algorithm
c which allows the user to mask a specified (reduced) set of feature 
c vectors for clustering purposes. 
c From the whole set of complex traces (real part has to be supplied as
c usp-dataset, imaginary part is optional) of the input usp-data files
c the algorithm extracts an "initial subset" of feature vectors. 
c The "initial subset" extracted depends on: First, the argument
c specifications made on the command line (-tfstart, -tfskip, and -tfend), 
c i.e. wether we want to use a subsampled version of the input data or not
c and second, if the traces are masked as "active" or "passive". If a maskfile 
c exists "passive" traces are marked by a value of -1, all other traces
c are considered to be "active". All traces of output files corresponding
c to "passive" traces in the maskfile are marked as dead traces 
c (Sta_Cor=30000). If no maskfile exists all traces are considered to be
c "active" traces.  The initial subset" is given in output files outr.usp 
c (real part) and outi.usp (imaginary part) if not specified otherwise 
c using the command line arguments -OR and -OI, respectively. 
c This assumes the user has selected to output the "initial subset" 
c by specifying the feature output file flag (-fout). 
c From the "initial subset" we then extract a "start subset". 
c This "start subset" consists of all the live traces in the "initial subset".
c Using this "start subset" we then perform clustering (one cycle) based 
c on the K-means algorithm (see J.T. Tou, and R.C. Gonzales; Pattern 
c Recognition Principles, Massachusetts, Addison-Wesley, 1974). 
c The performance index that is minimized using the K-means algorithm is 
c defined as the sum of the squared distances from all points in a cluster
c domain to the cluster center (which can be thought of as a minimization
c of the overall noise power contained in the "start subset").
c The number of cluster centers and the cluster center initialization 
c method used for the "start subset" can be selected through the command
c line arguments -nclust and -sclustini, respectively.
c Furthermore we can modify the outcome of converged K-means cycles
c (the convergence threshold for the K-means cycles can be set through   
c usage of -kmthres). By selecting a threshold (-sdelthresl and/or 
c sdelthresu) and a modification method (sdelini) we are able to delete
c feature vectors, remove clusters, relocate cluster centers or delete
c feature vectors and remove the corresponding cluster. 
c Once the output of a converged K-means cycles produces no clusters 
c whose membership number falls within the boundaries specified by 
c -sdelthresl and -sdelthresu we write the cluster assignments in output
c file outc.usp if not specified otherwise (using the command line 
c argument -OC). Using -maxcycle the user can also specify how many iteration
c cycles (K-means cycles) will at most occur. 
c Note: Cluster assignments for dead and masked traces are set to zero
c       Cluster assignments for deleted traces are set to "number of 
c       clusters + 1"
c-----------------------------------------------------------------------

c Get machine dependent parameters 

#include <f77/iounit.h>
#include <f77/lhdrsz.h>
#include <f77/sisdef.h>
#include <save_defs.h> 

c Dimension standard USP variables 
c SZLNHD is defined in lhdrsz.h. It provides room to store a trace 
c (including both trace header and trace data). SZLNHD can also be
c used to store the line header (big enough to store line header).
c SZLNHD => line file size = (SZTRHD + SZSAMP + MAXSMP)/2
c SZTRHD => size of trace header in pipes in byte
c SZSAMP => size of floating point sample in byte
c MAXSMP => ?????

      integer     itrr(SZLNHD), itri(SZLNHD), itrm(SZLNHD)
      integer     luinr, luini, luinm, luoutr, luouti, luoutc     
      integer     lbytesr, lbytesi, lbytesm, lbyoutlh, lbyouttr, lbyoutc
      integer     lenhed, argis, jerr, nclust, sclustini, seed, range
      integer     sdelthresl,sdelthresu,sdelini,maxcycle,kmthres, index
      real        tfstart, tfend, tfskip
      integer     istart, iend, iskip, isamp
      character   name*(8)
      character   ntapr*(255), ntapi*(255), ntapm*(255)
      character   otapr*(255), otapi*(255), otapc*(255)
      logical     query, ikp, fout, verbos

      real cputim(10), waltim(10)
      real v10, vtot10, w10, wtot10
      integer SmpIntr, SmpInti, NumSmpr, NumSmpi, junitsr
      integer NumTrcr, NumTrci, NumTrcm, NumRecr, NumReci, NumRecm
      integer errcd, errcds, abort, nvect 
      integer in_ikp, pipei(3), pipeo(3), status(6), imagfile, maskfile
      real TmMsFSr, TmMsFSi, MutVelr, MutVeli
      real IlClInr, IlCLIni, ClClInr, ClClIni
      real ildm, cldm, dtmsecHz

c Program Specific - dynamic memory variables

      integer vect_size, centers_size, u_size
      integer uhdr_size, live_size, iclustasg_size, iclustasgloc_size
      integer ncenters_size, index_size, total_size
      integer vect_size_bytes, centers_size_bytes, u_size_bytes
      integer uhdr_size_bytes, live_size_bytes, iclustasg_size_bytes
      integer iclustasgloc_size_bytes, ncenters_size_bytes
      integer index_size_bytes, total_size_bytes

c ----------------------------------------------------------------------

c Uncomment the following line and adjust the array sizes in the else
c portion of the ifndef block below if you want to do bounds checking.
c #define STATIC_ALLOCATION_VERSION

#ifndef STATIC_ALLOCATION_VERSION

      complex vect, centers
      real u, uhdr
      integer live, iclustasg, iclustasgloc, ncenters
      pointer (memadr_vect,    	 vect(2))
      pointer (memadr_centers, 	 centers(2))
      pointer (memadr_u,       	 u(2))
      pointer (memadr_uhdr,    	 uhdr(2))
      pointer (memadr_live,    	 live(2))   
      pointer (memadr_iclustasg, iclustasg(2))   
      pointer (memadr_iclustasgloc, iclustasgloc(2))   
      pointer (memadr_ncenters,  ncenters(2))   
      pointer (memadr_index,  index(2))   

#else

c     This is just for bounds checking.  Adjust these arrays to fit the
c     test dataset before running.  Note: These numbers are currently
c     random!

      complex vect(3214), centers(6452)
      real u(7878), uhdr(1170)
      integer live(1000),iclustasg(245),iclustasgloc(5245)
      integer ncenters(523), index(6356)
      
#endif

c Program Specific static memory variables

      integer ifmt_StaCor, l_StaCor, ln_StaCor

c Initialize variables

      data abort/0/
      data name/'CLUST3DM'/ 
      data pipei/0,3,6/, pipeo/1,4,5/
      data status/0,0,0,0,0,0/
      data imagfile/0/, maskfile/0/
      data verbos/.false./

c Start the overall timer

      call timstr(v10,w10)

c Give command line help if requested

      query = ( argis ('-?')    .gt. 0 .or. 
     1          argis ('-h')    .gt. 0 .or.
     2          argis ('-help') .gt. 0 )
      if (query) then
         call help()
         stop
      endif

c Let us know someone is using this thing

      call tattle('clust3dm')

c Open printout file

#include <f77/open.h>

c-----------------------------------------------------------------------
c Get command line input parameters

      call cmdln(name, ntapr, ntapi, ntapm, otapr, otapi, otapc, 
     1           tfstart, tfskip, tfend, ildm, cldm, nclust,
     2           sclustini, seed, sdelthresl, sdelthresu, sdelini, 
     3           maxcycle, kmthres, fout, verbos)

c-----------------------------------------------------------------------
c Check if we are in IKP or not

      ikp = .false.
      if (in_ikp() .eq. 1) ikp = .true.     

c-----------------------------------------------------------------------
c If we are in IKP
c If ntapm is not specified -> no mask given, i.e., all data will be used
c If ntapr is not specified -> abort
c If otapc is not specified -> abort
c If fout is not specified no output is written on neither otapr nor otapi
c If otapr is not specified and fout is specified -> abort
c If otapi is not specified and fout is specified -> abort
c If ntapi is not specified -> imaginary part of feature vectors will
c			        be set to zero (see initialization of 
c			        dynamic memory later in this routine).
c-----------------------------------------------------------------------
c imagfile = 0  => ntapi was not specified
c imagfile = 1  => ntapi was specified
c maskfile = 0  => ntapm was not specified
c maskfile = 1  => ntapm was specified
c-----------------------------------------------------------------------
 
      if (ikp) then

c-----------------------------------------------------------------------
c If no pipe is conected "status" changes to one, otherwise it stays zero

         call pipchk(pipei(1),status(1))
         call pipchk(pipei(2),status(2))
         call pipchk(pipei(3),status(3))
         call pipchk(pipeo(1),status(4))
         call pipchk(pipeo(2),status(5))
         call pipchk(pipeo(3),status(6))

         if (status(1) .eq. 1) then
            write(lerr,*)'CLUST3DM: Input 0 must be a named file'
            write(lerr,*)'CLUST3DM: FATAL'
            write(ler,*)'CLUST3DM: Input 0 must be a named file'
            write(ler,*)'CLUST3DM: FATAL'
            stop
         endif

         if (status(4) .eq. 1) then 
            write(lerr,*)'CLUST3DM: Output 1 must be a named file'
            write(lerr,*)'CLUST3DM: FATAL'
            write(ler,*)'CLUST3DM: Output 1 must be a named file'
            write(ler,*)'CLUST3DM: FATAL'
            stop
         endif

         if (status(5) .eq. 1 .and. fout) then 
            write(lerr,*)'CLUST3DM: Output 4 must be a named file'
            write(lerr,*)'CLUST3DM: FATAL'
            write(ler,*)'CLUST3DM: Output 4 must be a named file'
            write(ler,*)'CLUST3DM: FATAL'
            stop
         endif

         if (status(6) .eq. 1 .and. status(2) .eq. 0 .and. fout) then 
            write(lerr,'(a)')'CLUST3DM: Output 5 must be a named file'
            write(lerr,'(a)')'CLUST3DM: FATAL'
            write(ler,'(a)')'CLUST3DM: Output 5 must be a named file'
            write(ler,'(a)')'CLUST3DM: FATAL'
            stop
         endif
         
         if (status(3) .eq. 1) then 

            if (status(2) .eq. 1 .and. fout) then
               call sisfdfit(luinr,pipei(1))
               call sisfdfit(luoutc,pipeo(1))
               call sisfdfit(luoutr,pipeo(2))
            elseif (status(2) .eq. 1 .and. .not.fout) then
               call sisfdfit(luinr,pipei(1))
               call sisfdfit(luoutc,pipeo(1))
            elseif (status(2) .eq. 0 .and. fout) then
               call sisfdfit(luinr,pipei(1))
               call sisfdfit(luini,pipei(2))
               call sisfdfit(luoutc,pipeo(1))
               call sisfdfit(luoutr,pipeo(2))
               call sisfdfit(luouti,pipeo(3))
               imagfile = 1
            else
               call sisfdfit(luinr,pipei(1))
               call sisfdfit(luini,pipei(2))
               call sisfdfit(luoutc,pipeo(1))
               imagfile = 1
            endif
         else
            maskfile = 1
            if (status(2) .eq. 1 .and. fout) then
               call sisfdfit(luinr,pipei(1))
               call sisfdfit(luinm,pipei(3))
               call sisfdfit(luoutc,pipeo(1))
               call sisfdfit(luoutr,pipeo(2))
            elseif (status(2) .eq. 1 .and. .not.fout) then
               call sisfdfit(luinr,pipei(1))
               call sisfdfit(luinm,pipei(3))
               call sisfdfit(luoutc,pipeo(1))
            elseif (status(2) .eq. 0 .and. fout) then
               call sisfdfit(luinr,pipei(1))
               call sisfdfit(luini,pipei(2))
               call sisfdfit(luinm,pipei(3))
               call sisfdfit(luoutc,pipeo(1))
               call sisfdfit(luoutr,pipeo(2))
               call sisfdfit(luouti,pipeo(3))
               imagfile = 1
            else
               call sisfdfit(luinr,pipei(1))
               call sisfdfit(luini,pipei(2))
               call sisfdfit(luinm,pipei(3))
               call sisfdfit(luoutc,pipeo(1))
               imagfile = 1
            endif
         endif
      endif

c-----------------------------------------------------------------------
c If we are not in IKP
c If ntapm is not specified -> no mask given, i.e., all data will be used
c If ntapr is not specified -> abort
c If fout is not specified no output is written on neither otapr nor otapi
c If otapr is not specified and fout is specified -> set it to outr.usp
c If otapi is not specified and fout is specified -> set it to outi.usp
c If otapc is not specified -> set it to outc.usp
c If ntapi is not specified -> imaginary part of feature vectors will
c			       be set to zero (see initialization of 
c			       dynamic memory later in this routine).

      if (.not.ikp) then

         if (ntapr .ne. ' ') then      
            call getln(luinr, ntapr,'r', 0)
         else
            write(lerr,*)'CLUST3DM: Input data set (real part of featur 
     1e vectors) must be a named file'
            write(lerr,*)'CLUST3DM: FATAL'
            write(ler,*) 'CLUST3DM: Input data set (real part of featur 
     2e vectors) must be a named file'
            write(ler,*)'CLUST3DM: FATAL'
            stop      
         endif      

         if (ntapi .ne. ' ') then
            call getln(luini, ntapi, 'r', 0)
            imagfile = 1
         endif

         if (ntapm .ne. ' ') then
            call getln(luinm, ntapm, 'r', 0)
            maskfile = 1
         endif

         if (otapc .eq. ' ') then
            otapc = 'outc.usp'
            call getln(luoutc, otapc,'w', 1) 
         else
            call getln(luoutc, otapc,'w', 1) 
         endif     

         if (otapr .eq. ' ' .and. fout) then
            otapr = 'outr.usp'
            call getln(luoutr, otapr,'w', 1) 
         elseif (otapr .ne. ' ' .and. fout) then
            call getln(luoutr, otapr, 'w', 1)
         endif

         if (otapi .eq. ' ' .and. ntapi .ne. ' ' .and. fout) then
            otapi = 'outi.usp'
            call getln(luouti, otapi,'w', 1) 
         elseif (otapi .ne. ' ' .and. ntapi .ne. ' ' .and. fout) then
            call getln(luouti, otapi,'w', 1) 
         endif         
      endif

c-----------------------------------------------------------------------
c If the number of clusters (nclust) is not defined as an argument in 
c the command line or it is but it is <= 0 -> assign it to 5

      if (nclust .lt. 0) then 
         write(ler,*)
     1   'CLUST3DM: nclust is set to 5 -> specified < 0 in command line' 
         write(lerr,*)
     1   'CLUST3DM: nclust is set to 5 -> specified < 0 in command line' 
      endif
      if (nclust .le. 0) nclust = 5

c-----------------------------------------------------------------------
c If select cluster method (sclustini) is not defined as an argument in 
c the command line or it defined but it is <= 0 or > 4 
c -> assign it to 4, i.e., use the "maxmin"-distance algorithm to find 
c initial cluster centers

      if (sclustini .lt. 0 .or. sclustini .gt. 4) then 
         write(ler,*)
     1   'CLUST3DM: sclustini is set to 4 -> specified < 0  or > 4 in co
     2mmand line' 
         write(lerr,*)
     1   'CLUST3DM: sclustini is set to 4 -> specified < 0  or > 4 in co
     2mmand line' 
      endif
      if (sclustini .le. 0 .or. sclustini .gt. 4) sclustini = 4 
 
c-----------------------------------------------------------------------
c If the seed value for the random generator used in subroutine clustinit
c is not defined as an argument in the command line or it is but it 
c is <= 0 -> assign it to 1

      if (seed .lt. 0) then
         write(ler,*)
     1   'CLUST3DM: seed is set to 1 -> specified < 0 in command line' 
         write(lerr,*)
     1   'CLUST3DM: seed is set to 1 -> specified < 0 in command line' 
      endif
      if (seed .le. 0) seed = 1
 
c-----------------------------------------------------------------------
c If the lower modification threshold (sdelthresl) is not defined as an
c argument in the command line or it defined but it is <= 0  -> assign
c it to 0

      if (sdelthresl .lt. 0) then
         write(ler,*)
     1'CLUST3DM: sdelthresl is set to 0 -> specified < 0 in command line
     2' 
         write(lerr,*)
     1'CLUST3DM: sdelthresl is set to 0 -> specified < 0 in command line
     2' 
      endif
      if (sdelthresl .le. 0) sdelthresl = 0  
 
c-----------------------------------------------------------------------
c If the lower modification threshold (sdelthresl) is not defined as an
c argument in the command line or it defined but it is <= 0  -> assign
c it to 0

      if (sdelthresl .lt. 0) then  
         write(ler,*)
     1'CLUST3DM: sdelthresl is set to 0 -> specified < 0 in command line
     2' 
         write(lerr,*)
     1'CLUST3DM: sdelthresl is set to 0 -> specified < 0 in command line
     2' 
      endif
      if (sdelthresl .le. 0) sdelthresl = 0  

c-----------------------------------------------------------------------
c If the upper  modification threshold (sdelthresu) is not defined as an
c argument in the command line or it defined but it is <= 0  -> assign
c it to 0

      if (sdelthresu .lt. 0) then  
         write(ler,*)
     1'CLUST3DM: sdelthresu is set to 0 -> specified < 0 in command line
     2' 
         write(lerr,*)
     1'CLUST3DM: sdelthresu is set to 0 -> specified < 0 in command line
     2' 
      endif
      if (sdelthresu .le. 0) sdelthresu = 0  

c-----------------------------------------------------------------------
c Specify the boundaries selected by sdelthresl and sdelthresu and 
c and store them into variable range

      if ((sdelthresl .eq. 0) .and. (sdelthresu .eq. 0))then
         range = 0
      elseif ((sdelthresl .ne. 0).and.(sdelthresu .eq. 0)) then
         range = 1 
      elseif ((sdelthresl .lt. sdelthresu).and.(sdelthresl .ne. 0)) then
         range = 2
      elseif ((sdelthresl .eq. 0).and.(sdelthresu .ne. 0)) then
         range = 3
      elseif ((sdelthresl .ge. sdelthresu).and.(sdelthresu .ne. 0)) then
         write(ler,*)'CLUST3DM: -sdelthresl can not be >= -sdelthresu' 
         write(ler,*)'CLUST3DM: FATAL'
         write(lerr,*)'CLUST3DM: -sdelthresl can not be >= -sdelthresu' 
         write(lerr,*)'CLUST3DM: FATAL'
         stop
      endif

c-----------------------------------------------------------------------
c If the modification method (sdelini) is not defined as an argument in the
c command line or it defined but it is <= 0 or > 4 -> assign it to 4 

      if (sdelini .lt. 0 .or. sdelini .gt. 4) then 
         write(ler,*)
     1   'CLUST3DM: sdelini is set to 4 -> specified < 0  or > 4 in comm 
     2and line' 
         write(lerr,*)
     1   'CLUST3DM: sdelini is set to 4 -> specified < 0  or > 4 in comm 
     2and line' 
      endif
      if (sdelini .le. 0 .or. sdelini .gt. 4) sdelini = 4 
                   
c-----------------------------------------------------------------------
c If the maximal cycle number (maxcycle) is not defined as an argument in 
c the command line or it defined but it is <= 0  -> assign it to 0

      if (maxcycle .lt. 0) then
         write(ler,*)
     1'CLUST3DM: maxcycle is set to 0 -> specified < 0 in command line' 
         write(lerr,*)
     1'CLUST3DM: maxcycle is set to 0 -> specified < 0 in command line' 
      endif
      if (maxcycle .le. 0) maxcycle = 0  
  

c-----------------------------------------------------------------------
c If the K-means convergence threshold (kmthres) is not defined
c as an argument in the command line or it defined but it
c is <= 0  -> assign it to 0

      if (kmthres .lt. 0) then
         write(ler,*)
     1'CLUST3DM: kmthres is set to 0 -> specified < 0 in command line' 
         write(lerr,*)
     1'CLUST3DM: kmthres is set to 0 -> specified < 0 in command line' 
      endif
      if (kmthres .le. 0) kmthres = 0  

c-----------------------------------------------------------------------
c Read input line headers and save certain parameters
c Synopsis: rtape(unit, buffer, length)
c unit   => unit number to read from
c buffer => buffer to read data into
c length => size of record read in bytes

      call rtape(luinr,itrr,lbytesr)
      if(lbytesr .eq. 0) then
         write(ler,*)'CLUST3DM: no header read from unit ',luinr
         write(ler,*)'CLUST3DM: FATAL'
         write(lerr,*)'CLUST3DM: no header read from unit ',luinr
         write(lerr,*)'CLUST3DM: FATAL'
         stop
      endif

      if (imagfile .eq. 1) then 
         call rtape(luini,itri,lbytesi)
         if(lbytesi .eq. 0) then
            write(ler,*)'CLUST3DM: no header read from unit ',luini
            write(ler,*)'CLUST3DM: FATAL'
            write(lerr,*)'CLUST3DM: no header read from unit ',luini
            write(lerr,*)'CLUST3DM: FATAL'
            stop
         endif
      endif

      if (maskfile .eq. 1) then 
         call rtape(luinm,itrm,lbytesm)
         if(lbytesm .eq. 0) then
            write(ler,*)'CLUST3DM: no header read from unit ',luinm
            write(ler,*)'CLUST3DM: FATAL'
            write(lerr,*)'CLUST3DM: no header read from unit ',luinm
            write(lerr,*)'CLUST3DM: FATAL'
            stop
         endif
      endif

c-----------------------------------------------------------------------
c Obtain input file line header information 
c 'NumSmp' => Number of samples per trace
c 'SmpInt' => Sample interval 
c 'NumTrc' => Number of traces per record
c 'NumRec' => Number of records per line
c 'ILClIn' => In line cell increment (in line spacing)
c 'CLClIn' => Cross line cell increment (cross line spacing)
c 'TmMsFS' => originally intended for "Time in miliseconds of first sample"
c             only, here also used for frequency in Hertz of first sample"
c             depending on wether -NI is specified or not
c 'MutVel' => originally intended for "Muting Velocity", here used for 
c             frequency grid spacing if _NI is specified 

      call saver(itrr, 'NumSmp', NumSmpr, LINEHEADER)
      call saver(itrr, 'SmpInt', SmpIntr, LINEHEADER)
      call saver(itrr, 'NumTrc', NumTrcr, LINEHEADER)
      call saver(itrr, 'NumRec', NumRecr, LINEHEADER)
      call saver(itrr, 'ILClIn', ILClInr, LINEHEADER)
      call saver(itrr, 'CLClIn', CLClInr, LINEHEADER)
      call saver(itrr, 'TmMsFS', TmMsFSr, LINEHEADER)
      call saver(itrr, 'MutVel', MutVelr, LINEHEADER)

      if (maskfile .eq. 1) then
         call saver(itrm, 'NumTrc', NumTrcm, LINEHEADER)
         call saver(itrm, 'NumRec', NumRecm, LINEHEADER)
      endif

      if (imagfile .eq. 1) then

         call saver(itri, 'NumSmp', NumSmpi, LINEHEADER)
         call saver(itri, 'SmpInt', SmpInti, LINEHEADER)
         call saver(itri, 'NumTrc', NumTrci, LINEHEADER)
         call saver(itri, 'NumRec', NumReci, LINEHEADER)
         call saver(itri, 'ILClIn', ILClIni, LINEHEADER)
         call saver(itri, 'CLClIn', CLClIni, LINEHEADER)
         call saver(itri, 'TmMsFS', TmMsFSi, LINEHEADER)
         call saver(itri, 'MutVel', MutVeli, LINEHEADER)

c-----------------------------------------------------------------------
c Check input file line header information for consistency

         errcds = 0

         if (NumSmpr .ne. NumSmpi) then
            write(ler,*)'CLUST3DM: Number of samples (NumSmp) in input f
     1iles are not the same'
            write(ler,*)'CLUST3DM: FATAL'
            write(lerr,*)'CLUST3DM: Number of samples (NumSmp) in input 
     1files are not the same'
            write(lerr,*)'CLUST3DM: FATAL'
            errcds = 1
         endif
         if (SmpIntr .ne. SmpInti) then
            write(ler,*)'CLUST3DM: Sample interval (SmpInt) used in inpu
     1t files are not the same'
            write(ler,*)'CLUST3DM: FATAL'
            write(lerr,*)'CLUST3DM: Sample interval (SmpInt) used in inp
     1ut files are not the same'
            write(lerr,*)'CLUST3DM: FATAL'
            errcds = 1
         endif
         if (maskfile .eq. 1) then 
            if ((NumTrcr .ne. NumTrci) .or. (NumTrcr .ne. NumTrcm) .or.
     1          (NumTrci .ne. NumTrcm)) then
               write(ler,*)'CLUST3DM: Number of traces (NumTrc) in input 
     1 files are not the same'
               write(ler,*)'CLUST3DM: FATAL'
               write(lerr,*)'CLUST3DM: Number of traces (NumTrc) in inpu
     1t files are not the same'
               write(lerr,*)'CLUST3DM: FATAL'
               errcds = 1
            endif
         else
            if (NumTrcr .ne. NumTrci) then 
               write(ler,*)'CLUST3DM: Number of traces (NumTrc) in input
     1 files are not the same'
               write(ler,*)'CLUST3DM: FATAL'
               write(lerr,*)'CLUST3DM: Number of traces (NumTrc) in inpu
     1t files are not the same'
               write(lerr,*)'CLUST3DM: FATAL'
               errcds = 1
            endif
         endif
         if (maskfile .eq. 1) then 
            if ((NumRecr .ne. NumReci) .or. (NumRecr .ne. NumRecm) .or.
     1          (NumReci .ne. NumRecm)) then
               write(ler,*)'CLUST3DM: Number of records (NumRec) in inpu
     1t files are not the same'
               write(ler,*)'CLUST3DM: FATAL'
               write(lerr,*)'CLUST3DM: Number of records (NumRec) in inp
     1ut files are not the same'
               write(lerr,*)'CLUST3DM: FATAL'
               errcds = 1
            endif
         else
            if (NumRecr .ne. NumReci) then
               write(ler,*)'CLUST3DM: Number of records (NumRec) in inpu
     1t files are not the same'
               write(ler,*)'CLUST3DM: FATAL'
               write(lerr,*)'CLUST3DM: Number of records (NumRec) in inp
     1ut files are not the same'
               write(lerr,*)'CLUST3DM: FATAL'
               errcds = 1
            endif
         endif
         if (IlClInr .ne. IlClIni) then
            write(ler,*)'CLUST3DM: In-line cell increments (IlClIn) in i
     1nput files are not the same'
            write(ler,*)'CLUST3DM: FATAL'
            write(lerr,*)'CLUST3DM: In-line cell increments (IlClIn) in 
     1input files are not the same'
            write(ler,*)'CLUST3DM: FATAL'
            errcds = 1
         endif
         if (ClClInr .ne. ClClIni) then
            write(ler,*)'CLUST3DM: Cross-line cell increments (ClClIn) i
     1n input files are not the same'
            write(ler,*)'CLUST3DM: FATAL'
            write(lerr,*)'CLUST3DM: Cross-line cell increments (ClClIn) 
     1in input files are not the same'
            write(lerr,*)'CLUST3DM: FATAL'
            errcds = 1
         endif 
         if (TmMsFSr .ne. TmMsFSi) then
            write(ler,*)'CLUST3DM: Time of first sample (TmMsFS) in inpu
     1t files are not the same'
            write(ler,*)'CLUST3DM: FATAL'
            write(lerr,*)'CLUST3DM: Time of first sample (TmMsFS) in inp
     1ut files are not the same'
            write(lerr,*)'CLUST3DM: FATAL'
            errcds = 1
         endif
         if (MutVelr .ne. MutVeli) then
            write(ler,*)'NPD: Frequency grid spacing (MutVel) in input f
     1iles are not the same'
            write(ler,*)'NPD: FATAL'
            write(lerr,*)'NPD: Frequency grid spacing (MutVel) in input 
     1 files are not the same'
            write(lerr,*)'NPD: FATAL'
            errcds = 1
         endif
      else
         if (maskfile .eq. 1) then 
            if (NumTrcr .ne. NumTrcm) then
               write(ler,*)'CLUST3DM: Number of traces (NumTrc) in input 
     1 files are not the same'
               write(ler,*)'CLUST3DM: FATAL'
               write(lerr,*)'CLUST3DM: Number of traces (NumTrc) in inpu
     1t files are not the same'
               write(lerr,*)'CLUST3DM: FATAL'
               errcds = 1
            endif
            if (NumRecr .ne. NumRecm) then
               write(ler,*)'CLUST3DM: Number of records (NumRec) in inpu
     1t files are not the same'
               write(ler,*)'CLUST3DM: FATAL'
               write(lerr,*)'CLUST3DM: Number of records (NumRec) in inp
     1ut files are not the same'
               write(lerr,*)'CLUST3DM: FATAL'
               errcds = 1
            endif
         endif
      endif
      if (errcds .ne. 0) stop 

c Check if the in line spacing and the cross line spacing are set to
c their default values, i.e., check if they are not one of the arguments
c listed in the command line. If they were specified in the command line
c then use this specifications to overwrite the header in line- and
c cross line spacing information

      if (ildm .ne. 0) then
         ILClInr = ildm 
         call savew(itrr, 'ILClIn', ILClInr, LINEHEADER)
      endif
      if (cldm .ne. 0) then
         CLClInr = cldm
         call savew(itrr, 'CLClIn', CLClInr, LINEHEADER)
      endif
       

c Check if the in line- and/or cross line spacing are set to zero. If
c they are then they have to be specified in the command line
c Note: ler specifies the terminal as output unit (unbuffered)
c       lerr specifies the print file as output unit (buffered)

      if (ILClInr .eq. 0 .or. CLClInr .eq. 0) then
         write(ler,*)'CLUST3DM: Error -- -ildm and -cldm values'
         write(ler,*)'were not in the line header.  Please'
         write(ler,*)'specify them explicitly on the command line'
         write(lerr,*)'CLUST3DM: Error -- -ildm and -cldm values'
         write(lerr,*)'were not in the line header.  Please'
         write(lerr,*)'specify them explicitly on the command line'
         call exitfu(-1)
      endif

c-----------------------------------------------------------------------
c Calculate the time/frequency sample increment dtmsecHz in msec/Hz 
c depending on wether the input files contain SWT or FC-vectors.
c 'T_Unit' => Time Units (0=milliseconds, 1=microseconds)
c USP convention is that if SmpInt > 16, it is microseconds
 
      if (imagfile .eq. 1) then
         dtmsecHz = MutVelr
      else
         call saver(itrr, 'T_Unit', junitsr, LINEHEADER)
         if(junitsr .eq. 1 .or. SmpIntr .gt. 16) then
            junitsr = 1
            call savew(itrr, 'T_Unit', junitsr, LINEHEADER)
            dtmsecHz = .001 * SmpIntr
         else
            dtmsecHz = SmpIntr
         endif
      endif

c-----------------------------------------------------------------------
c Check tfstart, tfend, tfskip specified in command line to derive 
c sample ranges 
c If no command arguments for tfstart, tfskip and tfend have been defined
c then the default values for istart, iskip and iend are set to

      istart = 1
      iskip  = 1		
c     iend   = NumSmp

c Note: for negative entries of tfstart, tfskip or tfend we also assign
c the same default values for istart, iskip and iend as mentioned above 


      if (tfstart .gt. TmMsFSr) then
         istart = ((tfstart-TmMsFSr) / dtmsecHz) + 1
         if ((istart-1)*dtmsecHz+TmMsFSr .lt. tfstart) istart=istart+1
         if (istart .gt. NumSmpr) istart = NumSmpr
      endif
      if (tfskip .gt. 0) then
         iskip = (tfskip / dtmsecHz) 
         if (iskip .lt. 1) iskip = 1
         if (iskip .gt. NumSmpr) iskip = NumSmpr
      endif
      if (tfend .gt. TmMsFSr) then      
         iend = ((tfend-TmMsFSr) / dtmsecHz) +1 
         if (iend .lt. 1) iend = 1
         if (iend .gt. NumSmpr) iend = NumSmpr
         iend = istart + iskip * ((iend-istart)/iskip)
      else
         iend = NumSmpr
         iend = istart + iskip * ((iend-istart)/iskip)
      endif	

      isamp = (iend - istart)/iskip + 1 
            
c-----------------------------------------------------------------------
c Recalculate tfstart, tfend and tfskip so that they match the actual 
c sampling grid of the time traces used in the application
c 'TmMsFS' 	=> 	Time in ms of first sample
c tfstart 	= 	time start point
c tfskip  	= 	time skip
c tfend   	= 	time end point 

      tfstart = TmMsFSr + (istart-1) * dtmsecHz
      tfskip  = dtmsecHz * iskip            
      tfend   = TmMsFSr + (iend-1) * dtmsecHz 

c Check if tfstart <= tfend, otherwise abort and print error message 

      if (tfstart .gt. tfend) then
         write(ler,*)'CLUST3DM: !!!!!!!!!!  Error !!!!!!!!!!!'
         write(ler,*)'-tfstart argument greater than -tfend argument'
         write(ler,*)'     Please modify them on command line'
         write(lerr,*)'CLUST3DM: !!!!!!!!!!  Error !!!!!!!!!!!'
         write(lerr,*)'-tfstart argument greater than -tfend argument'
         write(lerr,*)'     Please modify them on command line'
         call exitfu(-1)
      endif

c-----------------------------------------------------------------------
c Update the historical part of the line header  

      call hlhprt(itrr, lbytesr, name, 7, lerr)

c-----------------------------------------------------------------------
c If we have constrained the sample ranges, set header values accordingly

      if (istart .gt. 1 .or. iend .lt. NumSmpr) then
         call savew(itrr, 'NumSmp', isamp, LINEHEADER)
         if (istart .gt. 1) then
            call savew(itrr, 'TmMsFS', tfstart, LINEHEADER)
         endif
      endif
      if (iskip .gt. 1) then 
         if (imagfile .eq. 0) then       
            call savew(itrr, 'SmpInt', SmpIntr*iskip, LINEHEADER)
         else
            call savew(itrr, 'MutVel', MutVelr*iskip, LINEHEADER)
         endif
      endif

c Save out hlh and line header for feature vector output files

      call savhlh(itrr, lbytesr, lbyoutlh)
      if (fout) then
         call wrtape(luoutr, itrr, lbyoutlh)
      endif
      if (imagfile .eq. 1 .and. fout) then
         call wrtape(luouti, itrr, lbyoutlh)
      endif
      
c Set up pointers to header mnemonic StaCor

      call savelu('StaCor', ifmt_StaCor, l_StaCor, ln_StaCor,
     1     TRACEHEADER)

c-----------------------------------------------------------------------
c Dump some stats to the print file

      write(lerr,*)
      write(lerr,'(a,t60,i5,t70,a)')
     1   'input file name: real part of initial vectors', 
     2   luinr, ntapr
      if (maskfile .eq. 1) then
         write(lerr,'(a,t60,i5,t70,a)')
     1   'mask file name: ', 
     2   luinm, ntapm 
      endif
      if (imagfile .eq. 1) then
         write(lerr,'(a,t60,i5,t70,a)')
     1   'input file name: imag. part of initial vectors',
     2   luini, ntapi
      else
         write(lerr,'(a)')
     1   'input file name: imag. part of initial vectors not specified'           
         write(lerr,*)
      endif
      write(lerr,'(a,t60,i5,t70,a)')
     1   'cluster assignment output file',
     2   luoutc, otapc
      if (fout) then
         write(lerr,'(a,t60,i5,t70,a)')
     1   'cluster output file: real part of feature vectors', 
     2   luoutr, otapr 
      endif
      if (imagfile .eq. 1 .and. fout) then
         write(lerr,'(a,t60,i5,t70,a)')
     1   'cluster output file: imag. part of feature vectors',
     2   luouti, otapi
      endif
      write(lerr,'(a,t40,i5)') 'input sample interval',SmpIntr
      write(lerr,'(a,t40,i5)') 'input # of samples',NumSmpr
      if (imagfile .eq. 0) then
         write(lerr,'(a,t40,f12.6)') 'input:time of first sample',
     1                                TmMsFSr
      else
         write(lerr,'(a,t40,f12.6)') 'input:freq. of first sample',
     1                                TmMsFSr
      endif
      write(lerr,'(a,t40,i5)') '# of traces',NumTrcr
      write(lerr,'(a,t40,i5)') '# of records',NumRecr
      write(lerr,'(a,t40,f12.6)') 'inline trace spacing', ILClInr
      write(lerr,'(a,t40,f12.6)') 'crossline trace spacing', CLClInr
      write(lerr,'(a,t40,i5)') '# of initial cluster centers ', nclust
      write(lerr,'(a,t40,i5)') 'cluster initilization selection', 
     1                          sclustini
      write(lerr,'(a,t40,i5)') 'seed for random generator choosen', seed 
      write(lerr,'(a,t40,i5)') 'lower modification threshold', 
     1                          sdelthresl
      write(lerr,'(a,t40,i5)') 'upper modification threshold', 
     1                          sdelthresu
      write(lerr,'(a,t40,i5)') 'delete initilization selection', 
     1                          sdelini
      write(lerr,'(a,t40,i5)') 'maximal cycle number', maxcycle
      write(lerr,'(a,t40,i5)') 'K-means convergence threshold', kmthres 
      if ( verbos )  write(lerr,*) ' verbose printout requested'    
      write(lerr,'(a,t40,i5)') 'feature output # of samples',isamp
      write(lerr,'(a,t40,i5)') 'cluster output # of samples',1
      write(lerr,'(a,t40,f12.6)') 'tfstart',tfstart
      write(lerr,'(a,t40,f12.6)') 'tfskip',tfskip 
      write(lerr,'(a,t40,f12.6)') 'tfend',tfend

c-----------------------------------------------------------------------
c Dynamic memory allocation: look in /home/usp/include/c/lhdrsz.h
c SZSMPD => size of sample in pipe (in bytes) in bytes
c ITRWRD => number of full words (SZSMPD-bytes) in trace header

      vect_size			= NumSmpr * NumTrcr * NumRecr
      vect_size_bytes		= vect_size * SZSMPD * 2

      centers_size		= NumSmpr * nclust
      centers_size_bytes	= centers_size * SZSMPD * 2

      u_size             	= NumSmpr + ITRWRD
      u_size_bytes       	= u_size * SZSMPD

      uhdr_size          	= NumTrcr * NumRecr * ITRWRD
      uhdr_size_bytes    	= uhdr_size * SZSMPD

      live_size			= NumTrcr * NumRecr
      live_size_bytes		= live_size * SZSMPD

      iclustasg_size		= NumTrcr * NumRecr
      iclustasg_size_bytes	= iclustasg_size * SZSMPD 

      iclustasgloc_size		= NumTrcr * NumRecr
      iclustasgloc_size_bytes	= iclustasgloc_size * SZSMPD 

      ncenters_size		= nclust
      ncenters_size_bytes	= ncenters_size * SZSMPD 

      index_size		= nclust
      index_size_bytes	        = index_size * SZSMPD 

      total_size       = vect_size + centers_size + u_size + 
     1                   uhdr_size + live_size + iclustasg_size + 
     2                   iclustasgloc_size + ncenters_size +
     3                   index_size
      total_size_bytes = vect_size_bytes + centers_size_bytes +
     1                   u_size_bytes + uhdr_size_bytes +
     2                   live_size_bytes + iclustasg_size_bytes +
     3                   iclustasgloc_size_bytes + ncenters_size_bytes +
     4                   index_size_bytes

      errcds = 0
      
#ifndef STATIC_ALLOCATION_VERSION

      call galloc(memadr_vect,    vect_size_bytes,    	 errcd, abort)
      errcds = errcds + errcd
      call galloc(memadr_centers, centers_size_bytes, 	 errcd, abort)
      errcds = errcds + errcd
      call galloc(memadr_u,       u_size_bytes,       	 errcd, abort)
      errcds = errcds + errcd
      call galloc(memadr_uhdr,    uhdr_size_bytes,    	 errcd, abort)
      errcds = errcds + errcd
      call galloc(memadr_live,    live_size_bytes,    	 errcd, abort)
      errcds = errcds + errcd
      call galloc(memadr_iclustasg,iclustasg_size_bytes, errcd, abort)
      errcds = errcds + errcd
      call galloc(memadr_iclustasgloc,iclustasgloc_size_bytes,
     1errcd, abort)
      errcds = errcds + errcd
      call galloc(memadr_ncenters, ncenters_size_bytes,  errcd, abort)
      errcds = errcds + errcd
      call galloc(memadr_index, index_size_bytes,
     1errcd, abort)
      errcds = errcds + errcd
      
#endif

c-----------------------------------------------------------------------
      if ( errcds .ne. 0 ) then
         write(lerr,*)' '
         write(lerr,*)'Unable to allocate workspace:'
         write(lerr,*) total_size,       '  words'
         write(lerr,*) total_size_bytes, '  bytes'
         write(lerr,*)' '
         write(ler,*)' '
         write(ler,*)'Unable to allocate workspace:'
         write(ler,*) total_size,       '  words'
         write(ler,*) total_size_bytes, '  bytes'
         write(ler,*)' '

c-----------------------------------------------------------------------
c Close the units with unit number luin and luout

         call lbclos(luinr)
         if (imagfile .eq. 1) then
            call lbclos(luini)
         endif
         call lbclos(luoutc) 
         if (fout) then
            call lbclos(luoutr)
         endif
         if (imagfile .eq. 1 .and. fout) then
            call lbclos(luouti)  
         endif
         if (maskfile .eq. 1) then
            call lbclos(luinm)
         endif
 
         write(lerr,*)'CLUST3DM: ABNORMAL Termination'
         write(ler,*)'CLUST3DM: ABNORMAL Termination'
         stop
      else
         write(lerr,*)' '
         write(lerr,*)'Allocating workspace:'
         write(lerr,*) 'vect            = ',vect_size_bytes
         write(lerr,*) 'centers         = ',centers_size_bytes
         write(lerr,*) 'u               = ',u_size_bytes
         write(lerr,*) 'uhdr            = ',uhdr_size_bytes
         write(lerr,*) 'live            = ',live_size_bytes
         write(lerr,*) 'iclustasg       = ',iclustasg_size_bytes         
         write(lerr,*) 'iclustasgloc    = ',iclustasgloc_size_bytes         
         write(lerr,*) 'ncenters        = ',ncenters_size_bytes       
         write(lerr,*) 'index           = ',index_size_bytes       
         write(lerr,*) 'total           = ',total_size,       ' words'
         write(lerr,*) 'total           = ',total_size_bytes, ' bytes'
         write(lerr,*)' '
      endif

c-----------------------------------------------------------------------
c Initialization of dynamic memory -> set to zero 
c Synopsis vclr(c, ic, n) look in /home/usp/Tomography/src/cmd/femodel/vclr.
c c  => vector which will be set to zero (0.0)
c n  => integer input element countf
c ic => integer input stride

      call vclr(vect,       	1, vect_size)
      call vclr(centers,       	1, centers_size)
      call vclr(u,       	1, u_size)
      call vclr(uhdr,    	1, uhdr_size)
      call vclr(live,		1, live_size)
      call vclr(iclustasg,	1, iclustasg_size)
      call vclr(iclustasgloc,	1, iclustasgloc_size)
      call vclr(ncenters,	1, ncenters_size)
      call vclr(index,      	1, index_size)
                 
c Length of a trace header
     
      lenhed = ITRWRD

c Calculate the number of bytes/trace for feature vector output files  
c SZTRHD => size of trace header in pipe in bytes
c SZSMPD => size of sample in pipe in bytes

      lbyouttr = SZTRHD + (isamp * SZSMPD)

c----------------------------------------------------------------------- 
c Call subroutine that reads/writes feature vectors of input/output

      write(ler,'(a)')'CLUST3DM: Please wait...data sets are read/writte
     1n'
      call featrdwrm(vect, u, uhdr, live, 
     1              isamp, istart, iskip, iend, nclust,
     2              nvect, NumSmpr, NumTrcr, NumRecr, 
     3              lenhed, lbyouttr, luinr, luini,luinm,luoutr,luouti, 
     4              ler, lerr, ntapi, u_size, 
     5              ifmt_StaCor, l_StaCor, ln_StaCor,
     6              cputim, waltim, imagfile, maskfile, fout, verbos)	
      
c-----------------------------------------------------------------------
c Close the input and output files which contain the feature vectors 

      call lbclos(luinr)
      if (imagfile .eq. 1) then
         call lbclos(luini)
      endif
      if (maskfile .eq. 1) then
         call lbclos(luinm)
      endif
      if (fout) then
         call lbclos(luoutr)
      endif
      if (imagfile .eq. 1 .and. fout) then
         call lbclos(luouti) 
      endif
     
c-----------------------------------------------------------------------
c Call cluster subroutine

      call cluster(iclustasgloc, iclustasg, ncenters, centers,
     1             nclust, isamp, NumSmpr, nvect, vect, sclustini,
     2             seed, ler, lerr, range, sdelthresl, sdelthresu,
     3             sdelini, maxcycle, kmthres, index,
     4             cputim, waltim, verbos)

c-----------------------------------------------------------------------
c Dump some more stats to the print file

      write(lerr,*)
      write(lerr,'(a,t40,i5)') '# of cluster centers after K-means',
     1                          nclust

c----------------------------------------------------------------------- 
c Set the number of samples to one (which corresponds to the cluster 
c assignment of corresponding trace/record) and record it in the 
c lineheader

      call savew(itrr, 'NumSmp', 1, LINEHEADER)

c----------------------------------------------------------------------- 
c Record the number of clusters in the lineheader, use dummy header
c keyword MnDpIn 

      call savew(itrr, 'MnDpIn', nclust, LINEHEADER)

c----------------------------------------------------------------------- 
c Save out hlh and line header for cluster result output file

      call wrtape(luoutc, itrr, lbyoutlh)

c-----------------------------------------------------------------------   
c Calculate the number of bytes/trace for cluster result output file 
c SZTRHD => size of trace header in pipe in bytes
c SZSMPD => size of sample in pipe in bytes

      lbyoutc = SZTRHD + SZSMPD

c----------------------------------------------------------------------- 
c Write out trace headers and the cluster assignments

      write(ler,'(a)')
     1'CLUST3DM: Please wait...cluster results are written'

      call clustwrm(u, uhdr, live, iclustasgloc, iclustasg, nvect,
     1             nclust, isamp, NumSmpr, NumTrcr, NumRecr, 
     2             lenhed, lbyoutc, luoutc, u_size,
     3             cputim, waltim, ler, verbos) 
      
c-----------------------------------------------------------------------
c Write out timing statistics

      call timend(cputim(10),v10,vtot10,waltim(10),w10,wtot10) 
      write(lerr,*)
      write(lerr,'(a35,2a15,/)') 'routine','cpu time','wall time'
      write(lerr,'(a35,2f15.3)')
     1         'read/write feature input/output',cputim(1),waltim(1),
     1         'clustering',cputim(2),waltim(2),
     1         'write cluster assignment output',cputim(3),waltim(3),
     1         'total',cputim(10),waltim(10)

c-----------------------------------------------------------------------
c Close the output file which contains the cluster result 

      call lbclos(luoutc)

c-----------------------------------------------------------------------
      write(lerr,*)'CLUST3DM: Normal Termination'
      write(ler,*) 'CLUST3DM: Normal Termination'
      stop

      end
      
c-----------------------------------------------------------------------
      subroutine help()

c Provide terse online help

#include <f77/iounit.h>

      write(ler,*)' '
      write(ler,*)'===================================================='
      write(ler,*)' '
      write(ler,*)' Command Line Arguments for clust3dm:'
      write(ler,*)' '
      write(ler,*)' For a more detailed description of these parameters'
      write(ler,*)' see the online man page using uman, xman or xuspman'
      write(ler,*)' '
      write(ler,*)'-NR[]       -- input file                  (stdin=0)'
      write(ler,*)'-NI[]       -- input file                        (3)'
      write(ler,*)'-NM[]       -- mask file                         (6)'
      write(ler,*)'-OC[]       -- output file                (stdout=1)'
      write(ler,*)'-OR[]       -- output file                       (4)'
      write(ler,*)'-OI[]       -- output file                       (5)'
      write(ler,*)'-tfstart[]  -- start time (ms)        (first sample)'
      write(ler,*)'-tfend[]    -- end time (ms)           (last sample)'
      write(ler,*)'-tfskip[]   -- incremental time skip (ms)   (SmpInt)'
      write(ler,*)'-ildm[]     -- inline trace spacing         (ILClIn)'
      write(ler,*)'-cldm[]     -- crossline trace spacing      (CLClIn)'  
      write(ler,*)'-nclust[]   -- number of clusters'
      write(ler,*)'-sclustini[]-- select cluster center initilization'        
      write(ler,*)'-seed[]     -- select seed for random generator' 
      write(ler,*)'-sdelthresl[]-- select lower modification threshold' 
      write(ler,*)'-sdelthresu[]-- select upper modification threshold' 
      write(ler,*)'-sdelini[]  -- select delete initilization method' 
      write(ler,*)'-maxcycle[] -- select maximal cycle number' 
      write(ler,*)'-kmthres[]  -- select K-means convergence threshold' 
      write(LER,*)'-fout       -- feature output file flag'
      write(LER,*)'-V          -- verbos printout'
      write(ler,*)' '
      write(ler,*)'Usage:'
      write(ler,*)'     clust3dm -NR[] -NI[] -NM[] -OR[] -OI[] -OC[]'
      write(ler,*)'              -tfstart[] -tfend[] -tfskip[]'
      write(ler,*)'              -ildm[] -cldm[] -nclust[]'
      write(ler,*)'              -sclustini[] -seed[]'      
      write(ler,*)'              -sdelthresl[] -sdelthresu[]'      
      write(ler,*)'              -sdelini[] -maxcycle[] -kmthres[]'      
      write(ler,*)'              -fout -V'      
      write(ler,*)'===================================================='
      
      return
      end

c-----------------------------------------------------------------------
c Pick up command line arguments 

      subroutine cmdln(name, ntapr, ntapi, ntapm, otapr, otapi, otapc, 
     1                 tfstart, tfskip, tfend, ildm, cldm, nclust,
     2                 sclustini, seed, sdelthresl, sdelthresu,
     3                 sdelini, maxcycle, kmthres, fout, verbos)
     
#include <f77/iounit.h>

      integer    nclust, sclustini, seed, sdelthresl
      integer    sdelthresu, sdelini, maxcycle, kmthres, argis
      character  name*(*), ntapr*(*), ntapi*(*), ntapm*(*)
      character  otapr*(*), otapi*(*), otapc*(*) 
      real       tfstart, tfend, tfskip
      real       ildm, cldm
      logical    fout, verbos

c     Documented options

      call argstr('-NR', ntapr, ' ', ' ') 
      call argstr('-NI', ntapi, ' ', ' ')       
      call argstr('-NM', ntapm, ' ', ' ')       
      call argstr('-OR', otapr, ' ', ' ') 
      call argstr('-OI', otapi, ' ', ' ')       
      call argstr('-OC', otapc, ' ', ' ')       
      
      call argr4('-tfstart', tfstart, 0.0, 0.0)
      call argr4('-tfend',   tfend,   0.0, 0.0)
      call argr4('-tfskip',  tfskip,  0.0, 0.0)
      call argr4('-ildm', ildm, 0.0, 0.0)
      call argr4('-cldm', cldm, 0.0, 0.0)

      call argi4('-nclust', nclust, 0, 0)
      call argi4('-sclustini', sclustini, 0, 0)
      call argi4('-seed', seed, 0, 0)
      call argi4('-sdelthresl', sdelthresl, 0, 0)
      call argi4('-sdelthresu', sdelthresu, 0, 0)
      call argi4('-sdelini', sdelini, 0, 0)
      call argi4('-maxcycle', maxcycle, 0, 0)
      call argi4('-kmthres', kmthres, 0, 0)
  
      fout   = ( argis( '-fout' ) .gt. 0 )
      verbos = ( argis( '-V' ) .gt. 0 )
                        
c Check for extraneous arguments and abort if found

      call xtrarg(name, ler, .FALSE., .FALSE.)
      call xtrarg(name, lerr, .FALSE., .TRUE.)

      return
      end
