/***********************************************************************
 *                copyright 2001, Amoco Production Company             *
 *                            All Rights Reserved                      *
 *                    an affiliate of BP America Inc.                  *
 ***********************************************************************/
/*
  Author:  Cory Hoelting
  Date last modified:  August 13, 1996
  Purpose:  Compute a measure of the coherence between
            neighboring label sites that is used as a
            threshold in bethe.
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>

/* USP stuff */
/* Used for command line processing. */
/* This code may be converted to work with the */
/* USP file format in the future. */
#include <iounit.h>
#include <lhdrsz.h>
#include <save_defs.h>
#include <cu_defs.h>
#include <sisio.h>

#define      Q4      4   /* no. of near neighbors */
#define BACKGROUND -999999999.0

#ifdef _USE_NINT_MACRO
/* KDC - 12/18/96 - Couldn't find nint on the sgi. */
#define NINT(x) ((int)((x)+.5))
#else
#define NINT(x) nint(x)
#endif

/* The constant BORDER should become a variable.  Its value */
/* should be the same as the value assigned to the variable 'step_size' */
/* in the function  find_neigh_pair_values() */
#define BORDER 15        /* amount, on each side, by which the */ 
                         /* interiorlabel array is larger than */
                         /* the label lattice */

void cogeom2d_help();
int getcmdline();
void allocation();
void read_coherency();
void find_neigh_pair_values();
double estimate_coherency();
void write_coherency();

/* global variables */
int **interiorlabel;      /* array that is slightly larger than the label */
                          /* lattice which contains nonzero entries at */
                          /* positions that are inside the label lattice */
int idelta[Q4]={1,0,-1,0};   /* used in finding near neighbors */ 
int jdelta[Q4]={0,1,0,-1};   /* used in finding near neighbors */ 

/* global variable for program name, used in error messages */
char prog_name[] = "cogeom2d";


void main(argc, argv)
 int argc;
 char **argv;
 {
  char file_name[256];
  int ifd, ofd;           /* input file and output file descriptors */
  int ipipe, opipe;
  int one;
  int ierr = 0;
  char host[10], blank9[10];
  int ikp;                  /* flag set to 1 if running under ikp */

  int ROW,      /* number of rows in the data lattice with background */
      COL,      /* number of columns in the data lattice with background */
      COARSE,    /* inverse of the resolution */
      HWW,       /* half window width */
      LROW,      /* number of rows in the label lattice with background */
      LCOL,      /* number of columns in the label lattice with background */
      WW,        /* window width */
      h1_flag,   /* flag set if extracting horizon 1 */
      h2_flag,   /* flag set if extracting horizon 2 */
      h3_flag,   /* flag set if extracting horizon 3 */
      h4_flag,   /* flag set if extracting horizon 4 */
      tdatum_flag; /* flag set if extracting a time slice */

  int num_smp;     /* number of samples in the input file */
  int smp_int;     /* sample interval of the input file */
  float hrznul;    /* the null value for a horizon */

  float sltime;    /* the time of the slice to extract, if tdatum_flag is */
                   /* set on the command line */

  /* USP data */
  int usp_lineheader[SZLNHD];
  int usp_lineheader_bytes, usp_input_trace_bytes, usp_output_trace_bytes;

  double **coh_dat;   /* the input coherency values */
  double ***out_dat;  /* the output minimum coherency values for */
                      /* near neighbor pairs */

  /* Help? */
  if (C_ARGIS("-?", &argc, argv) ||
      C_ARGIS("-h", &argc, argv) ||
      C_ARGIS("-help", &argc, argv))
    {
     cogeom2d_help();
     exit(0);
    }

  /* check to see if this program is being run under ikp */
  strcpy(host, "         ");
  strcpy(blank9, "         ");
  ikpchk_(host, strlen(host));
  if (strcmp(host, blank9))
     ikp = 1;
  else
     ikp = 0;

  /* display usage part of help if there was */
  /* only one argument on the command line */
  /* and we are not running under ikp */
  if (argc == 1 && !ikp)
    {
     fprintf(stderr,
          "Usage:  cogeom2d [-N <coh_file>] [-O <out_file>]\n\
                 [-coarse <coarseness>] [-halfwin <half_win_size>]\n\
                 [-h1 | -h2 | -h3 | -h4 | -tdatum <sltime>]\n\
                 [-h -? -help]\n");
     exit(0);
    }

  /* set up the default file descriptors for ikp */
  ipipe = 0;
  opipe = 1;
  one = 1;

  /* Open the files and read some values out of the lineheader. */
  C_ARGSTR("-N", file_name, "", "", &argc, argv);
  if (strlen(file_name) >= 1)
     lbopen(&ifd, file_name, "r");
  else
    {
     if (ikp)
       {
        if (pipcnt_(&ipipe, &one) > 0)
           sisfdfit_(&ifd, &ipipe);
       }
     else
       {
        ifd = ipipe;
       }
    }

  C_ARGSTR("-O", file_name, "", "", &argc, argv);
  if (strlen(file_name) >= 1)
     lbopen(&ofd, file_name, "w");
  else
    {
     if (ikp)
       {
        if (pipcnt_(&opipe, &one) > 0)
           sisfdfit_(&ofd, &opipe);
       }
     else
       {
        ofd = opipe;
       }
    }

  C_RTAPE(ifd, usp_lineheader, &usp_lineheader_bytes);

  /* Grab values from the USP lineheader */
  /* the input file to 'texture' has 1 record, ROW traces, */
  /* and COL samples.  However, it was produced by extracting */
  /* a horizon or time slice from a file with the same dimensions */
  /* as the input file to this program and then running qdslice.  So here */
  /* we have ROW records, COL traces, and some number of samples */ 
  C_SAVER(usp_lineheader, "NumRec", &ROW, LINEHEADER);
  C_SAVER(usp_lineheader, "NumTrc", &COL, LINEHEADER);
  C_SAVER(usp_lineheader, "NumSmp", &num_smp, LINEHEADER);
  C_SAVER(usp_lineheader, "SmpInt", &smp_int, LINEHEADER);
  C_SAVER(usp_lineheader, "HrzNul", &hrznul, LINEHEADER);

  /* process the command line arguments */
  ierr = getcmdline(&argc, argv, &COARSE, &HWW, &h1_flag, &h2_flag, &h3_flag,
                    &h4_flag, &tdatum_flag, &sltime);

  /* exit if the error condition has been set */
  if (ierr) exit(1);

  WW = (2 * HWW) + 1;
  LROW=(ROW-WW)/COARSE+1;
  LCOL=(COL-WW)/COARSE+1;

  usp_input_trace_bytes = SZDTHD + (num_smp * SZSAMP);
  usp_output_trace_bytes = SZDTHD + (LCOL * SZSAMP);

  /* update the line header so it is correct for the cogeom2d (output) */
  /* file.  We extract a time slice or horizon and then rotate this */
  /* slice in the same manner qdslice does.  The input file to this */
  /* program has ROW records, COL traces per record, and num_smp samples */
  /* per trace.  Therefore, after the extraction and 'qdslice'-like */
  /* rotation, this program operates on a data structure that is equivalent */
  /* to a 1 record data set with ROW traces and COL samples.  For each of the */
  /* LROW X LCOL points in the label lattice we find the coherence */
  /* between this point and each of its Q4 near neighbors.  Therefore, we end */
  /* up filling an array of size LROW X LCOL X Q4.  Note:  arrays in C */
  /* are stored in row major order.  So the Q4 coherence values between */
  /* a point and each of its near neighbors are stored contiguously.  However,*/
  /* the cogeom2d (output) file will have LROW records, Q4 traces per */
  /* record, and LCOL samples per trace.  This ordering is used because */
  /* Q4 will be less than LROW and LCOL in most cases.  If we did use */
  /* Q4 as the number of samples per trace, then the size of */
  /* the cogeom2d (output) file would be much larger because we would have */
  /* many more trace headers. */
  C_SAVEW(usp_lineheader, "NumSmp", LCOL, LINEHEADER);
  C_SAVEW(usp_lineheader, "NumTrc", Q4, LINEHEADER);
  C_SAVEW(usp_lineheader, "NumRec", LROW, LINEHEADER);

  /* Update the historical line header */
  /* savhlh_(usp_lineheader, &usp_lineheader_bytes, &usp_lineheader_bytes); */

  /* write out the line header */
  C_WRTAPE(ofd, usp_lineheader, usp_lineheader_bytes);

  allocation(ROW, COL, LROW, LCOL, &coh_dat, &out_dat);

  read_coherency(ifd, ROW, COL, smp_int, hrznul, h1_flag, h2_flag,
                            h3_flag, h4_flag, tdatum_flag, sltime,
                                    usp_input_trace_bytes, coh_dat);

  find_neigh_pair_values(ROW, COL, LROW, LCOL, HWW, WW, COARSE,
                                              coh_dat, out_dat);

  write_coherency(LROW, LCOL, COARSE, HWW, ofd, out_dat, 
                                  usp_output_trace_bytes); 

  lbclos(ifd);
  lbclos(ofd);

  /* Normal completion */
  fprintf(stderr, "%s:  normal completion\n", prog_name);
 }


/* Process the parameters given on the command line. */
int getcmdline(pargc, argv, pCOARSE, pHWW, ph1_flag, ph2_flag, ph3_flag,
                                      ph4_flag, ptdatum_flag, psltime)
 int *pargc;  
 char **argv;
 int *pCOARSE, *pHWW, *ph1_flag, *ph2_flag, *ph3_flag, *ph4_flag;
 int *ptdatum_flag;
 float *psltime;
 {
  int ierr;

  C_ARGI4("-coarse", pCOARSE, 9, 9, pargc, argv);
  if (*pCOARSE < 1)
    {
     fprintf(stderr, "%s: error:  the command line argument -coarse ",
             prog_name);
     fprintf(stderr, "<coarseness> must be positive.\n");
     ierr = 1; 
    }

  C_ARGI4("-halfwin", pHWW, 7, 7, pargc, argv);
  if (*pHWW < 1)
    {
     fprintf(stderr, "%s: error:  the command line argument -halfwin ",
             prog_name);
     fprintf(stderr, "<half_win_size> must be positive.\n");
     ierr = 1; 
    }

  *ph1_flag = C_ARGIS("-h1", pargc, argv);
  *ph2_flag = C_ARGIS("-h2", pargc, argv);
  *ph3_flag = C_ARGIS("-h3", pargc, argv);
  *ph4_flag = C_ARGIS("-h4", pargc, argv);
  C_ARGR4("-tdatum", psltime, -1.0, -1.0, pargc, argv);
  if (*psltime == -1.0)
     *ptdatum_flag = 0;
  else
     *ptdatum_flag = 1;

  if (*ph1_flag + *ph2_flag + *ph3_flag + *ph4_flag + *ptdatum_flag < 1)
    {
     fprintf(stderr, "%s: warning: No horizon/time slice specified,\n",
             prog_name);
     fprintf(stderr, "        extracting slice at time 0.\n");
     *ptdatum_flag = 1;
     *psltime = 0.0;
    }
  else if (*ph1_flag + *ph2_flag + *ph3_flag + *ph4_flag + *ptdatum_flag > 1)
    {
     fprintf(stderr, 
             "%s: warning: More than one horizon/time slice specified,\n",
             prog_name);
     fprintf(stderr, "        extracting slice at time 0.\n");
     *ph1_flag = 0;
     *ph2_flag = 0;
     *ph3_flag = 0;
     *ph4_flag = 0;
     *ptdatum_flag = 1;
     *psltime = 0.0;
    }
 return(ierr);
 }


/* allocate memory for the arrays */
void allocation(ROW, COL, LROW, LCOL, ptr_coh_dat, ptr_out_dat)
 int ROW, COL, LROW, LCOL;
 double ***ptr_coh_dat;
 double ****ptr_out_dat;
 {
  double **pcc, *pc;
  int *ipc;
  int i, j, nn; 

  /* build the interiorlabel array so it will be easy to test */
  /* which near neighbors are inside the label lattice */
  interiorlabel =(int **)malloc((LROW+2*BORDER)*sizeof(int *));
  interiorlabel += BORDER;
  ipc=(int *)malloc((LROW+2*BORDER)*(LCOL+2*BORDER)*sizeof(int));
  ipc += BORDER*(LCOL+2*BORDER)+BORDER;
  for (i=-BORDER; i<LROW+BORDER; i++)
    interiorlabel[i]=ipc+i*(LCOL+2*BORDER);
  for (i=-BORDER;i<LROW+BORDER;i++)  /*initialize interior[][]*/
    for (j=-BORDER;j<LCOL+BORDER;j++) {
      if ( (i >0) && (i < LROW-1) && (j >0) && (j < LCOL-1) )
        interiorlabel[i][j]=4;
      else if (i<0 || j<0 || i>=LROW || j>=LCOL)
        interiorlabel[i][j]=0;
      else if ((i==0 && (j==0 || j==LCOL-1)) || (i==LROW-1 && (j==0 || j==LCOL-1)))
        interiorlabel[i][j]=2;
      else
        interiorlabel[i][j]=3;
    }/*for j*/

  /* 2D array for coherency data */
  (*ptr_coh_dat)=(double **)malloc(ROW*sizeof(double *));
  pc=(double *)malloc(ROW*COL*sizeof(double));
  for (i=0; i<ROW; i++)
    (*ptr_coh_dat)[i]=pc+i*COL;

  /* 3D array for minimum coherency b/t near neighbor */ 
  (*ptr_out_dat) = (double ***) malloc(LROW * sizeof(double **));
  pcc = (double **) malloc(LROW * LCOL * sizeof(double *));
  for(i = 0; i < LROW; i++)
     (*ptr_out_dat)[i] = pcc + i * LCOL; 
  pc = (double *) malloc(LROW * LCOL * Q4 * sizeof(double));
  for(i = 0; i < LROW; i++)
     for(j = 0; j < LCOL; j++)
        (*ptr_out_dat)[i][j] = pc + ((i * LCOL) + j) * Q4;

  /* initialize the arrays */ 
  for(i = 0; i < ROW; i++)
     for(j = 0; j < COL; j++)
        (*ptr_coh_dat)[i][j] = BACKGROUND;
  for(i = 0; i < LROW; i++)
     for(j = 0; j < LCOL; j++)
        for(nn = 0; nn < Q4; nn++)
           (*ptr_out_dat)[i][j][nn] = 0.0;
 }


/* Read the coherency values a trace at a time.  From each trace */
/* extract one sample.  This sample will be placed into the coh_dat */
/* array at element coh_dat[i][j] where i is the record number minus 1 */
/* (b/c of C) and j is the trace number minus 1 (b/c of C).  The coh_dat */
/* array is the 1 record data structure mentioned in the comment before the */
/* C_SAVEW calls in main.  This sample is either from a time slice or a */
/* horizon.  If the trace is dead (i.e. StaCor = 30000) or the horizon */ 
/* is null for the current input trace, the value placed into coh_dat */
/* will be BACKGROUND. */ 
void read_coherency(ifd, ROW, COL, smp_int, hrznul, h1_flag, h2_flag,
                               h3_flag, h4_flag, tdatum_flag, sltime,
                                       usp_input_trace_bytes, coh_dat)
 int ifd;          /* input file descriptor */
 int ROW, COL, smp_int; 
 float hrznul; 
 int h1_flag, h2_flag, h3_flag, 
     h4_flag, tdatum_flag;       /* flags that tell us to extract the */
                                 /* samples from a time slice or a horizon */
 float sltime;               /* the time of the slice */
 int usp_input_trace_bytes;
 double **coh_dat;
 {
  float *usp_input_trace;

  int   sta_cor;         /* Static Correction */
  float hor_val;         /* time of the horizon for the current trace */

  int i, j, index_of_smp;

  usp_input_trace = (float *) malloc(usp_input_trace_bytes);

  /* if extracting a time slice, calculate the index of the sample */
  index_of_smp = NINT(sltime / ((float) smp_int)); 

  /* loop over records in the input data set */
  for(i = 0; i < ROW; i++)
    {
     /* begin debug */
     /*
     fprintf(stderr, "in read -- i = %d\n", i);
     */
     /* end debug */

     /* loop over traces in the input data set */
     for(j = 0; j < COL; j++)
       {
        /* read in a trace and extract the sample, determine if the */
        /* sample is good (i.e. it's not in a dead trace and in */
        /* the horizon, if a horizon is being extracted).  If it is */
        /* a good sample, copy its value into coh_dat[i][j].  Otherwise, */
        /* copy BACKGROUND into coh_dat[i][j]. */ 
        C_RTAPE(ifd, usp_input_trace, &usp_input_trace_bytes);
        C_SAVER(usp_input_trace, "StaCor", &sta_cor, TRACEHEADER);
        if (sta_cor != 30000)
          {
           /* the trace is not dead */
           if (tdatum_flag)
             {
              /* extracting from a time slice */
              coh_dat[i][j] = (double) usp_input_trace[ITRWRD + index_of_smp]; 
             }
           else if (h1_flag)
             {
              C_SAVER(usp_input_trace, "Horz01", &hor_val, TRACEHEADER);
              if (hor_val != hrznul)
                {
                 /* horizon was picked in current trace */
                 index_of_smp = NINT(hor_val / ((float) smp_int));
                 coh_dat[i][j] = (double) usp_input_trace[ITRWRD+index_of_smp]; 
                }
              else
                 coh_dat[i][j] = BACKGROUND;
             }
           else if (h2_flag)
             {
              C_SAVER(usp_input_trace, "Horz02", &hor_val, TRACEHEADER);
              if (hor_val != hrznul)
                {
                 /* horizon was picked in current trace */
                 index_of_smp = NINT(hor_val / ((float) smp_int));
                 coh_dat[i][j] = (double) usp_input_trace[ITRWRD+index_of_smp]; 
                }
              else
                 coh_dat[i][j] = BACKGROUND;
             }
           else if (h3_flag)
             {
              C_SAVER(usp_input_trace, "Horz03", &hor_val, TRACEHEADER);
              if (hor_val != hrznul)
                {
                 /* horizon was picked in current trace */
                 index_of_smp = NINT(hor_val / ((float) smp_int));
                 coh_dat[i][j] = (double) usp_input_trace[ITRWRD+index_of_smp]; 
                }
              else
                 coh_dat[i][j] = BACKGROUND;
             }
           else if (h4_flag)
             {
              C_SAVER(usp_input_trace, "Horz04", &hor_val, TRACEHEADER);
              if (hor_val != hrznul)
                {
                 /* horizon was picked in current trace */
                 index_of_smp = NINT(hor_val / ((float) smp_int));
                 coh_dat[i][j] = (double) usp_input_trace[ITRWRD+index_of_smp]; 
                }
              else
                 coh_dat[i][j] = BACKGROUND;
             }
          }
        else
           /* trace is dead */
           coh_dat[i][j] = BACKGROUND;
       }
    }


  free(usp_input_trace);
 }


/* This function determines a measure of coherency between a label */
/* position and its near neighbors.  This measure is the minimum */
/* coherency value on the line that connects and includes the */
/* positions corresponding to the label positions in the data lattice. */
void find_neigh_pair_values(ROW, COL, LROW, LCOL, HWW, WW, COARSE, 
                                                  coh_dat, out_dat)
 int ROW, COL, LROW, LCOL, HWW, WW, COARSE;
 double **coh_dat;
 double ***out_dat;
 {
  int i, j;       /* label indices */
  int m, n;       /* near neighbor indices */
  int nn;         /* index for the near neighbors */
  int step_size;  /* the hor. or vert. distance a near neighbor */

  step_size = ((int) ceil( ((double) WW) / ((double) COARSE) ));
  for(i = 0; i < LROW; i++)
     for(j = 0; j < LCOL; j++)
       {
        fprintf(stderr, "%s: processing trace %d of %d\n", i+1, LROW);
        for(nn = 0; nn < Q4; nn++)
          {
           m = i + idelta[nn]*step_size;
           n = j + jdelta[nn]*step_size;
           /* check to see if the neighbor is inside the label lattice */
           if (interiorlabel[m][n])
              out_dat[i][j][nn] = estimate_coherency(i, j, m, n, HWW, 
                                             COARSE, nn, coh_dat);

           /* begin debug */
           /*
           if (out_dat[i][j][nn] == BACKGROUND)
             {
              fprintf(stderr, "WARNING:  BACKGROUND returned for i = %d ", i);
              fprintf(stderr, "j = %d\n", j);
             } 
           */
           /* end debug */
          }
       }
 }


/* This function maps the label positions into their corresponding */
/* data positions and finds the minimum coherency between the two points */
/* if both data positions have coherency values.  If both corresponding */
/* data positions are in the background, a value of 1.0 is returned. */
/* If one data position is in the background and the other has a coherency */
/* value, a value of 0.0 is returned. */ 
double estimate_coherency(i, j, m, n, HWW, COARSE, nn, coh_dat)
 int i, j, m, n, HWW, COARSE, nn;
 double **coh_dat;
 {
  double min_coh;       /* the minimum coherency */
  int pi, pj, pm, pn;   /* the indices of the label position and one of */
                        /* its near neighbors in the data lattice */
  int x, y;             /* the current position in the data lattice */

  /* map the label indices to the data indices */
  pi = HWW + i*COARSE;
  pj = HWW + j*COARSE;
  pm = HWW + m*COARSE;
  pn = HWW + n*COARSE;

  if (coh_dat[pi][pj] != BACKGROUND && coh_dat[pm][pn] != BACKGROUND)
    {
     /* both data positions have coherency values */
     min_coh = coh_dat[pi][pj];
     /* move to the next data position  on the line connected the two sites */
     x = pi + idelta[nn];
     y = pj + jdelta[nn];
     while( (x != pm) || (y != pn) )
       {
        if (coh_dat[x][y] < min_coh)
           min_coh = coh_dat[x][y]; 
        /* move to the next data position */
        x = x + idelta[nn];
        y = y + jdelta[nn];
       }
     /* check the last position */
     if (coh_dat[x][y] < min_coh)
        min_coh = coh_dat[x][y]; 

     /* Check to see if background was encountered b/t the two data */
     /* positions.  If so, set min_coh to 0.0. */
     if (min_coh == BACKGROUND)
        min_coh = 0.0;
    }
  else if (coh_dat[pi][pj] == BACKGROUND && coh_dat[pm][pn] == BACKGROUND)
    {
     /* both data positions are in the background */
     min_coh = 1.0;
    }
  else    /* one data position is in the background, the other has a value */
     min_coh = 0.0;

  return (min_coh);
 }


/* write out_dat to ofd */
/* Note:  The cogeom2d (output) file will have LROW records, */
/* Q4 traces per record, and LCOL samples per trace.  See the */
/* large comment in main before the first C_SAVEW calls for the reason. */
void write_coherency(LROW, LCOL, COARSE, HWW, ofd, out_dat, 
                                     usp_output_trace_bytes)
 int LROW, LCOL, COARSE, HWW;
 int ofd;                     /* output file */
 double ***out_dat;
 int usp_output_trace_bytes;
 {
  float *usp_output_trace;
  int i, j, k, nn;

  usp_output_trace = (float *) malloc(usp_output_trace_bytes);

  /* Set the Static Correction field in the traceheader */
  C_SAVEW(usp_output_trace, "StaCor", 0, TRACEHEADER);
 
  /* loop over records */
  for(i = 0; i < LROW; i++)
    {
     /* begin debug */
     /*
     fprintf(stderr, "in write -- i = %d\n", i);
     */
     /* end debug */

     /* set the RecNum Field in the traceheader */
     C_SAVEW(usp_output_trace, "RecNum", i * COARSE + HWW, TRACEHEADER);
     /* loop over traces */
     for(nn = 0; nn < Q4; nn++)
       {
        /* set the TrcNum Field in the traceheader */
        C_SAVEW(usp_output_trace, "TrcNum", nn, TRACEHEADER);
        /* loop over samples */
        for(j = 0, k = ITRWRD; j < LCOL; j++, k++)
           usp_output_trace[k] = (float) out_dat[i][j][nn];
        C_WRTAPE(ofd, usp_output_trace, usp_output_trace_bytes);
       }
    }

  free(usp_output_trace);
 }


void cogeom2d_help()
 {
  fprintf(stderr, "\
  This utility extracts a horizon or a time slice from a 3D coherency\n\
  data set.  Then it produces a measure of the coherence between label\n\
  sites which are near neighbors.  The measure of the coherence between\n\
  the near neighbors is the minimum value found on the line connecting\n\
  and including the data sites which correspond to the two label sites.\n\
  Then these values are written to a file which can be used by any\n\
  segmentation optimizer, such as bethe, as a threshold. \n\
  \n\
  Usage:  cogeom2d [-N <coh_file>] [-O <out_file>]\n\
                   [-coarse <coarseness>] [-halfwin <half_win_size>]\n\
                   [-h1 | -h2 | -h3 | -h4 | -tdatum <sltime>]\n\
                   [-h -? -help]\n\
  \n\
  -N <coh_file> - input file containing coherency for a 3D\n\
                  data set.  If the file is not in the current\n\
                  directory, the full path should be specified.\n\
                  (Default:  stdin)\n\
  \n\
  -O <out_file> - output file containing an estimate of the coherence\n\
                  between each label position and its near neighbors\n\
                  in a horizon or a time slice that was extracted\n\
                  from the 3D coherency data.\n\
                  (Default:  stdout)\n\
  \n\
  -coarse <coarseness> - inverse of the resolution at which the\n\
                         segmentation is performed.  That is, the ratio\n\
                         of the number of samples/traces in the data\n\
                         lattice to the number of samples/traces in the\n\
                         label lattice.  The data lattice is the input to\n\
                         the program texture, and the label lattice is the\n\
                         output from any segmentation optimizer, such as\n\
                         bethe.  For highest resolution set the coarseness\n\
                         to 1.\n\
                         (Default: 9)\n\
  \n\
  -halfwin <half_win_size> - half window width used to determine\n\
                             the width of the texture analysis window\n\
                             over which the cdf's are computed in the\n\
                             program disparity.  The width of the texture\n\
                             analysis window is given by\n\
                             2 * <half_win_size> + 1.\n\
                             (Default: 7)\n\
  \n\
  -h1 | -h2 | -h3 | -h4 | -tdatum <sltime> - only one of these\n\
                  options can be specified on the command line.  If\n\
                  the -h1 flag is set, this program will extract the\n\
                  first horizon from the coherency file and operate\n\
                  on it.  If the -h2 flag is set, this program will\n\
                  extract the second horizon and operate on it, etc.\n\
                  If the -tdatum option is used, the program will\n\
                  operate on the slice at time <sltime>.\n\
                  (Default:  -tdatum 0.0)\n\
  \n\
  -h, -?, -help - Help.\n\
");
}
