/***********************************************************************
 *                copyright 2001, Amoco Production Company             *
 *                            All Rights Reserved                      *
 *                    an affiliate of BP America Inc.                  *
 ***********************************************************************/
/*
  Author:  Cory Hoelting
  Date last modified:  August 12, 1996
  Purpose:  Combine the disparity values for near neighbor pairs
            into a single value.  Used in texture segmentation
            before the segmentation step.
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/uio.h>
#include <math.h>
 
/* USP stuff */
/* Used for command line processing. */
/* This code may be converted to work with the */
/* USP file format in the future. */
#include <iounit.h>
#include <lhdrsz.h>
#include <save_defs.h>
#include <cu_defs.h>
#include <sisio.h>

#define      Q4      4   /* no. of near neighbors */
#define      NUM_OF_INPUT_FILES 60

void combine2d_help();
int getcmdline();
void allocation();
void read_disparity();
void merge();
void write_disparity();

/* global variable for program name, used in error messages */
char prog_name[] = "combine2d";

int main(argc, argv)
 int argc;
 char **argv;
 {
  char file_name[256];

  int ifd[NUM_OF_INPUT_FILES], ofd;       /* file descriptors for input */
                                          /* files and output file */
  int ipipe[NUM_OF_INPUT_FILES], opipe; 
  int one;
  int ierr = 0;
  char host[10], blank9[10];
  int ikp;                  /* flag set to 1 if running under ikp */
  int had_file;             /* set to 1 when a file was just processed */
  int sum_flag;             /* set to 1 if the output is the sum of */
                            /* the inputs */
  int max_flag;             /* set to 1 if the output is the max of the */
                            /* input values */

  /* USP data */
  int usp_lineheader[SZLNHD];
  int usp_lineheader_bytes, usp_trace_bytes;

  int ***final_dispar, /* array containing final, combined disparity values */
      ***comp_dispar;  /* array used to hold disparity values of 1 component */

  int  NN;             /* number of neighbors */ 
  int LROW, LCOL;      /* numb. of rows and cols in the label lattice */
  int COARSE;          /* inverse of the resolution */
  int  HWW;            /* half window width */
  int i, j, nn, count;


  /* Help? */
  if (C_ARGIS("-?", &argc, argv) ||
      C_ARGIS("-h", &argc, argv) ||
      C_ARGIS("-help", &argc, argv))
    {
     combine2d_help();
     exit(0);
    }

  /* check to see if this program is being run under ikp */
  strcpy(host, "         ");
  strcpy(blank9, "         ");
  ikpchk_(host, strlen(host));
  if (strcmp(host, blank9))
     ikp = 1;
  else
     ikp = 0;

  /* display usage part of help if there was */
  /* only one argument on the command line */
  if (argc == 1 && !ikp)
    {
     fprintf(stderr,
             "Usage:  combine2d -N <dis_file_1> [... -N <dis_file_i>]\n\
                  [-O <out_dis_file]> [-coarse <coarseness>]\n\
                  [-halfwin <half_win_size>]\n\
                  [-sum | -max]\n\
                  [-h -? -help]\n");
     exit(0);
    }

  /* process the command line arguments */
  ierr = getcmdline(&argc, argv, &sum_flag, &max_flag, &COARSE, &HWW);

  /* set up the default file descriptors for ikp */
  ipipe[0] = 0;
  for(i = 1; i < NUM_OF_INPUT_FILES; i++)
     ipipe[i] = i + 2;
  opipe = 1;
  one = 1;

  count = 0;

  /* open the first input file */
  C_ARGSTR("-N", file_name, "", "", &argc, argv);
  if ( strlen(file_name) >= 1 )
     lbopen(&(ifd[count]), file_name, "r");
  else
    {
     if (ikp)
       {
        if (pipcnt_(&(ipipe[count]), &one) > 0)
           sisfdfit_(&(ifd[count]), &(ipipe[count]));
       }
     else
       {
        fprintf(stderr, "%s: error: at least one input file must be ",
                prog_name);
        fprintf(stderr, "specified on the command line\n");
        ierr = 1; 
       }
    }

  /* exit if the error condition has been set */
  if (ierr) exit(1);

  /* open the output file */
  C_ARGSTR("-O", file_name, "", "", &argc, argv);
  if ( strlen(file_name) >= 1 )
     lbopen(&ofd, file_name, "w");
  else
    {
     if (ikp)
       {
        if (pipcnt_(&opipe, &one) > 0)
           sisfdfit_(ofd, &opipe);
       }
     else
        ofd = fileno(stdout);
    }

  /* Read in the USP lineheader from the disparity file */
  rtape(ifd[count], usp_lineheader, &usp_lineheader_bytes);

  /* Grab values from the USP lineheader */
  /* The disparity file has LCOL samples per trace, NN traces */
  /* per record, and LCOL records.  So we can get these values */
  /* from the header. */
  saver(usp_lineheader, "NumRec", &LROW, LINEHEADER);
  saver(usp_lineheader, "NumTrc", &NN, LINEHEADER);
  saver(usp_lineheader, "NumSmp", &LCOL, LINEHEADER);

  usp_trace_bytes = SZDTHD + (LCOL * SZSAMP);

  /* Update the historical line header */
  /* savhlh_(usp_lineheader, &usp_lineheader_bytes, &usp_lineheader_bytes); */

  /* write out the line header */
  wrtape(ofd, usp_lineheader, usp_lineheader_bytes);

  allocation(LROW, LCOL, NN, &final_dispar, &comp_dispar);

  read_disparity(ifd[count], LROW, LCOL, NN, usp_trace_bytes, comp_dispar);
  if (!ikp)
     lbclos(ifd[count]);
  for(i = 0; i < LROW; i++)
     for(j = 0; j < LCOL; j++)
        for(nn = 0; nn < NN; nn++)
           final_dispar[i][j][nn] = comp_dispar[i][j][nn];

  count++;
  do {
     had_file = 1;
     C_ARGSTR("-N", file_name, "", "", &argc, argv);
     if ( strlen(file_name) >= 1 )
        lbopen(&(ifd[count]), file_name, "r");
     else
       {
        if (ikp)
          {
           if (pipcnt_(&(ipipe[count]), &one) > 0)
              sisfdfit_(&(ifd[count]), &(ipipe[count]));
          }
        else
           had_file = 0;   /* no file will be processed */
       }
     if (had_file) 
       {
        /* Read in the USP lineheader from the disparity file */
        rtape(ifd[count], usp_lineheader, &usp_lineheader_bytes);

        read_disparity(ifd[count], LROW, LCOL, NN, usp_trace_bytes,comp_dispar);
        if (!ikp)
           lbclos(ifd[count]);
        merge(comp_dispar, final_dispar, LROW, LCOL, NN, sum_flag);
        count++;
       }
  } while( had_file && count < NUM_OF_INPUT_FILES );

  /* print out status*/
  fprintf(stderr, "%s:  %d files merged.\n", prog_name, count);

  write_disparity(LROW, LCOL, NN, COARSE, HWW, ofd, final_dispar,
                                                 usp_trace_bytes);
  if (!ikp)
     lbclos(ofd);

  /* Normal completion */
  fprintf(stderr, "%s:  normal completion\n", prog_name);
 }


int getcmdline(pargc, argv, psum_flag, pmax_flag, pCOARSE, pHWW)
 int *pargc;
 char **argv;
 int *psum_flag, *pmax_flag, *pCOARSE, *pHWW;
 {
  int ierr = 0;

  *psum_flag = C_ARGIS("-sum", pargc, argv);
  *pmax_flag = C_ARGIS("-max", pargc, argv);
  if (*psum_flag && *pmax_flag)
    {
     fprintf(stderr, "%s: error: the flags -sum and -max cannot both be set ",
             prog_name);
     fprintf(stderr, "on the same command line.\n");
     ierr = 1; 
    }
  else if( (!(*psum_flag)) && (!(*pmax_flag)) )
     *psum_flag = 1;                /* sum the input vaules by default */

  C_ARGI4("-coarse", pCOARSE, 9, 9, pargc, argv);
  if (*pCOARSE < 1)
    {
     fprintf(stderr, "%s: error: the command line argument -coarse ",
             prog_name);
     fprintf(stderr, "<coarseness> must be positive.\n");
     ierr = 1; 
    }

  C_ARGI4("-halfwin", pHWW, 7, 7, pargc, argv);
  if (*pHWW < 1)
    {
     fprintf(stderr, "%s: error: the command line argument -halfwin ",
             prog_name);
     fprintf(stderr, "<half_win_size> must be positive.\n");
     ierr = 1; 
    }

  return (ierr);
 }


/* allocate memory for the arrays */
void allocation(LROW, LCOL, NN, ptr_final_dispar, ptr_comp_dispar)
 int LROW, LCOL, NN;
 int ****ptr_final_dispar, ****ptr_comp_dispar;
 {
  int **ipcc, *ipc;
  int i, j;

  /* 3D array for the final disparity values */
  (*ptr_final_dispar)=(int ***)malloc(LROW*sizeof(int **));
  ipcc=(int **)malloc(LROW*LCOL*sizeof(int *));
  for (i=0; i<LROW; i++)
    (*ptr_final_dispar)[i]=ipcc+i*LCOL;
  ipc=(int *)malloc(LROW*LCOL*NN*sizeof(int));
  for (i=0; i<LROW; i++)
    for (j=0; j<LCOL; j++)
      (*ptr_final_dispar)[i][j]=ipc+((i*LCOL)+j)*NN;

  /* 3D array for the component disparity values */ 
  (*ptr_comp_dispar)=(int ***)malloc(LROW*sizeof(int **));
  ipcc=(int **)malloc(LROW*LCOL*sizeof(int *));
  for (i=0; i<LROW; i++)
    (*ptr_comp_dispar)[i]=ipcc+i*LCOL;
  ipc=(int *)malloc(LROW*LCOL*NN*sizeof(int));
  for (i=0; i<LROW; i++)
    for (j=0; j<LCOL; j++)
      (*ptr_comp_dispar)[i][j]=ipc+((i*LCOL)+j)*NN;
 }


/* read the disparity values from the file fd into the array img */
/* the USP file has LROW records, NN traces per record, */
/* and LCOL samples per trace.  The array img is a 3d array */
/* with dimensions LROWxLCOLxNN */
void read_disparity(fd, LROW, LCOL, NN, usp_trace_bytes, img)
 int fd;               /* input file descriptor */
 int LROW, LCOL, NN;
 int usp_trace_bytes;
 int ***img;
 {
  float *usp_trace;
  int i, j, k, nn;
 
  usp_trace = (float *) malloc(usp_trace_bytes);

  /* loop over records */
  for(i = 0; i < LROW; i++)
    {
     /* loop over traces */
     for(nn = 0; nn < NN; nn++)
       {
        rtape(fd, usp_trace, &usp_trace_bytes);
        /* loop over samples */
        for(j = 0, k = ITRWRD; j < LCOL; j++, k++)
           img[i][j][nn] = (int) usp_trace[k];
       }
    }

  free(usp_trace);
 }


/* merge a component disparity array into the final disparity array */
void merge(comp_dispar, final_dispar, LROW, LCOL, NN, sum_flag)
 int ***comp_dispar, ***final_dispar;
 int LROW, LCOL, NN;
 int sum_flag;       /* set to 1 if the output is the sum of the inputs */
 {
  int i, j, nn;

  if (sum_flag)
    {
     for(i = 0; i < LROW; i++)
       for(j = 0; j < LCOL; j++)
         for(nn = 0; nn < NN; nn++)
            final_dispar[i][j][nn] += comp_dispar[i][j][nn];
    }
  else  
    {
     /* the max_flag must be set b/c of the */
     /* logic in the command line processing */
     for(i = 0; i < LROW; i++)
       for(j = 0; j < LCOL; j++)
         for(nn = 0; nn < NN; nn++)
            if (comp_dispar[i][j][nn] > final_dispar[i][j][nn]) 
               final_dispar[i][j][nn] = comp_dispar[i][j][nn];
    }
 }


/* write the img array to the file fd */
/* Note:  The disparity (output) file will have LROW records, */
/* NN traces per record, and LCOL samples per trace. */
void write_disparity(LROW, LCOL, NN, COARSE, HWW, fd, img, usp_trace_bytes)
 int LROW, LCOL, NN, COARSE, HWW;
 int fd;               /* output file descriptor */
 int ***img;
 int usp_trace_bytes;
 {
  float *usp_trace;
  int i, j, k, nn;
 
  usp_trace = (float *) malloc(usp_trace_bytes);
 
  /* Set the Static Correction field in the traceheader */
  savew(usp_trace, "StaCor", 0, TRACEHEADER);
 
  /* loop over records */
  for(i = 0; i < LROW; i++)
    {
     /* set the RecNum Field in the traceheader */
     savew(usp_trace, "RecNum", i * COARSE + HWW, TRACEHEADER);
     /* loop over traces */
     for(nn = 0; nn < NN; nn++)
       {
        /* set the TrcNum Field in the traceheader */
        savew(usp_trace, "TrcNum", nn, TRACEHEADER);
        /* loop over samples */
        for(j = 0, k = ITRWRD; j < LCOL; j++, k++)
           usp_trace[k] = (float) img[i][j][nn];
        wrtape(fd, usp_trace, usp_trace_bytes);
       }
    }
 
  free(usp_trace);
 }


void combine2d_help()
 {
  fprintf(stderr, "\
  This utility reads in several files containing disparity values which were\n\
  calculated from several different data files.  The data files could, for\n\
  instance, have been a set frequencies from the spectral decomposition of\n\
  a horizon.  Each disparity file was produced using the program disparity.\n\
  Each disparity file contains disparity values for each neighboring pair of\n\
  label sites.  The output of this utility is one disparity file which is\n\
  the same size as the input files, i.e. it contains one value for each pair\n\
  of neighbors in the label lattice.  The value for each pair is either the\n\
  maximum disparity value over all the input disparity values for the pair\n\
  or the sum of the input disparity values for the pair.\n\
  NOTE:  At the present time, a maximum of 60 input files can be processed\n\
         by combine2d.\n\
  \n\
  Usage:  combine2d -N <dis_file_1> [... -N <dis_file_i>] [-O <out_dis_file>]\n\
                    [-coarse <coarseness>] [-halfwin <half_win_size>]\n\
                    [-sum | -max]\n\
                    [-h -? -help]\n\
  \n\
  -N <dis_file_1> - input file containing the disparity values\n\
                    calculated from the first data file.  This\n\
                    file was produced by the program disparity.\n\
  \n\
  [... -N <dis_file_i>] - input file containing the disparity values\n\
                          calculated from the i-th data file.  This\n\
                          file was produced by the program disparity.\n\
  \n\
  -O <out_dis_file> - the output file which contains one disparity\n\
                      value for each neighboring pair of labels\n\
                      (Default: stdout)\n\
  \n\
  -coarse <coarseness> - inverse of the resolution at which the\n\
                         segmentation is performed.  That is, the ratio\n\
                         of the number of samples/traces in the data\n\
                         lattice to the number of samples/traces in the\n\
                         label lattice.  The data lattice is the input to\n\
                         the program texture, and the label lattice is the\n\
                         output from any segmentation optimizer, such as\n\
                         bethe.  For highest resolution set the coarseness\n\
                         to 1.\n\
                         (Default: 9)\n\
  \n\
  -halfwin <half_win_size> - half window width used to determine\n\
                             the width of the texture analysis window\n\
                             over which the cdf's are computed in the\n\
                             program disparity.  The width of the texture\n\
                             analysis window is given by\n\
                             2 * <half_win_size> + 1.\n\
                             (Default: 7)\n\
  \n\
  [-sum | -max] - if the command line switch -sum is set the output\n\
                  disparity value for a pair of neighboring labels\n\
                  is the sum of the input disparity values for that\n\
                  pair.  If the command line switch -max is set\n\
                  the output disparity value for a pair of neighboring\n\
                  labels is the maximum of the input disparity values\n\
                  for that pair.\n\
                  NOTE:  both of these switches cannot be set on the\n\
                         same command line.  If neither switch is set\n\
                         the output value will be the sum of the input\n\
                         values. \n\
  \n\
  -h -? -help - Help.\n\
");
}
