C***********************************************************************
C                 copyright 2001, Amoco Production Company             *
C                             All Rights Reserved                      *
C                     an affiliate of BP America Inc.                  *
C***********************************************************************
C********************************************************************C
C NAME: FCUINT  CUBIC INTERPOLATION             REV 1.0     JUN 88   C
C********************************************************************C
C
C  PURPOSE:
C       PERFORMS A CUBIC INTERPOLATION.
C
C  LANGUAGE:
C       FORTRAN 77
C
C  HISTORY:
C       ORIGINAL                JUN 88          R.D. COLEMAN, QTC
C
C  CALLING FORMAT:
C       CALL FCUINT (X1, Y1, N1, X2, Y2, N2, IZ, ZZ, INIT)
C
C  PARAMETERS:
C       X1      REAL INPUT VECTOR OF LENGTH N1
C               SOURCE VECTOR CONTAINING X COORDINATES CORRESPONDING
C               TO Y1.
C
C       Y1      REAL INPUT VECTOR OF LENGTH N1
C               SOURCE VECTOR CONTAINING Y COORDINATES.
C
C       N1      INTEGER INPUT SCALAR
C               ELEMENT COUNT FOR X1 AND Y1.  N1 MUST BE >= 4.
C
C       X2      REAL INPUT VECTOR OF LENGTH N2
C               SOURCE VECTOR CONTAINING X COORDINATES CORRESPONDING
C               TO Y2.
C
C       Y2      REAL OUTPUT VECTOR OF LENGTH N2
C               RESULT VECTOR.
C
C       N2      INTEGER INPUT SCALAR
C               ELEMENT COUNT FOR X2 AND Y2.
C
C       IZ      INTEGER INPUT/OUTPUT VECTOR OF LENGTH N2
C               INTERMEDIATE INDEX VECTOR CALCULATED IF INIT <> 0.
C
C       ZZ      REAL INPUT/OUTPUT VECTOR OF LENGTH 4*N2
C               INTERMEDIATE COEFFICIENT VECTOR CALCULATED IF INIT <> 0.
C
C       INIT    INTEGER INPUT SCALAR
C               INITIALIZATION FLAG.. IF INIT <> 0, THEN THE INTER-
C               MEDIATE VECTORS IZ AND ZZ ARE CALCULATED; OTHERWISE,
C               THEY ARE ASSUMED TO HAVE BEEN CALCULATED IN A PREVIOUS
C               CALL.
C
C  DESCRIPTION:
C       GIVEN A SET OF (X,Y) COORDINATES (X1,Y1), FCUINT PERFORMS A
C       CUBIC INTERPOLATION TO OBTAIN AN OUTPUT SET OF (X,Y)
C       COORDINATES (X2,Y2).  THE VALUES OF BOTH X1 AND X2 MUST BE IN
C       ASCENDING ORDER AND MAY HAVE ARBITRARY SPACING.  IF X2(I) <
C       X1(1) FOR SOME I, THEN Y2(I) = Y1(1).  SIMILARLY, IF X2(I) >
C       X1(N1), THEN Y2(I) = Y1(N1).
C
C       THE INTERMEDIATE RESULT VECTORS IZ AND ZZ ARE CALCULATE
C       IF THE INITIALIZATION FLAG INIT <> 0.  THESE VECTORS ARE
C       DEPENDENT ON N1, N2, X1, AND X2; I.E., THEY ARE DEPENDENT ON
C       ALL INPUT ARGUMENTS EXCEPT Y1.  WHEN A SEQUENCE OF CALLS ARE
C       MADE IN WHICH ONLY Y1 CHANGES, IT IS MOST EFFICIENT TO SET
C       INIT TO 1 FOR THE FIRST CALL THEN SET IT TO 0 FOR THE
C       SUBSEQUENT CALLS.
C
C  SUBPROGRAMS CALLED:
C       NONE
C
C  ERROR CONDITIONS:
C       IF N1 < 4 OR N2 < 1, THE ROUTINE IS ABORTED.
C
C---------------------------------------------------------------------
C
      SUBROUTINE FCUINT (X1, Y1, N1, X2, Y2, N2, IZ, ZZ, INIT)
C
      INTEGER N1, N2, IZ(N2), INIT
      REAL    X1(N1), Y1(N1), X2(N2), Y2(N2), ZZ(N2,4), XX(64)
C
C-----------------------------------------------------------------------
C

      IF (N1 .LT. 4 .OR. N2 .LT. 1) GO TO 800
C
      IF (INIT .EQ. 0) GO TO 200
C
C     ----------------------
C     PERFORM INITIALIZATION
C     ----------------------
C
      X1LO = X1( 1)
      X1HI = X1(N1)
C
      J  = 3
      I2 = 0
  100 CONTINUE
      I1 = I2 + 1
      I2 = I2 + 64
      IF (I2 .GT. N2) I2 = N2
C
C     DO FIX UP FOR OUT-OF-RANGE VALUES OF X2
C
      II = 0
CDIR$ SHORTLOOP
      DO 110 I = I1, I2
         II = II + 1
C
         IF      (X2(I) .LT. X1LO) THEN
            XX(II) = X1LO
         ELSE IF (X2(I) .GE. X1HI) THEN
            XX(II) = X1HI
         ELSE
            XX(II) = X2(I)
         ENDIF
  110 CONTINUE
C
C     CALCULATE IZ
C
      II = 0
CDIR$ SHORTLOOP
      DO 130 I = I1, I2
         II = II + 1
C
  120    CONTINUE
         IF (XX(II) .GT. X1(J) .AND. J .LT. N1-1) THEN
            J = J + 1
            GO TO 120
         ENDIF
C
         IZ(I) = J - 2
  130 CONTINUE
C
C     CALCULATE ZZ
C
      II = 0
CDIR$ SHORTLOOP
      DO 140 I = I1, I2
         II = II + 1
C
         J1 = IZ(I)
         J2 = J1 + 1
         J3 = J2 + 1
         J4 = J3 + 1
C
         DX1 = XX(II) - X1(J1)
         DX2 = XX(II) - X1(J2)
         DX3 = XX(II) - X1(J3)
         DX4 = XX(II) - X1(J4)
         D12 = X1(J1) - X1(J2)
         D13 = X1(J1) - X1(J3)
         D14 = X1(J1) - X1(J4)
         D23 = X1(J2) - X1(J3)
         D34 = X1(J3) - X1(J4)
         D42 = X1(J4) - X1(J2)
C
         d1234 = D12 * D13 * D14
         d1224 = D12 * D23 * D42
         d1323 = D13 * D23 * D34
         d1424 = D14 * D42 * D34

            ZZ(I,1) = DX2 * DX3 * DX4 / d1234
            ZZ(I,2) = DX1 * DX3 * DX4 / d1224
            ZZ(I,3) = DX1 * DX2 * DX4 / d1323
            ZZ(I,4) = DX1 * DX2 * DX3 / d1424
  140 CONTINUE
      IF (I2 .LT. N2) GO TO 100
C
C     ---------------------
C     PERFORM INTERPOLATION
C     ---------------------
C
  200 CONTINUE
      DO 210 I = 1, N2
         J  = IZ(I)
         Y2(I) = ZZ(I,1) * Y1(J) + ZZ(I,2) * Y1(J+1) + ZZ(I,3) * Y1(J+2)
     &         + ZZ(I,4) * Y1(J+3)
  210 CONTINUE
C
C     ------------
C     EXIT ROUTINE
C     ------------
C
  800 CONTINUE
      RETURN
      END
