/***********************************************************************
 *                copyright 2001, Amoco Production Company             *
 *                            All Rights Reserved                      *
 *                    an affiliate of BP America Inc.                  *
 ***********************************************************************/
/*
  Modified by:  Cory Hoelting
  Original code by:  Chi-hsin Wu
  Date last modified:  August 12, 1996
  Purpose:  Calculate the difference of the textures between
            neighboring label sites.
*/

#include <stdio.h>
#include <math.h>
#include <string.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/uio.h>
#include <stdlib.h>

/* USP stuff */
/* Used for command line processing */
/* This code may be converted to work with the */
/* USP file format in the future. */
#include <iounit.h>
#include <lhdrsz.h>
#include <save_defs.h>
#include <cu_defs.h>
#include <sisio.h>

#define      Q4      4   /* no. of near neighbors */
#define      FR      5   /*no. of features/transformations                    */
 
 
/* The constant BORDER should become a variable.  Its value */
/* should be the same as the value assigned to the variable 'stp' */
/* in the function findneighbor() */
#define      BORDER  15  /* using wider matrix to store interior[][] values   */
#define      SQR(a) ((a)*(a))

void disparity2d_help();
void allocation();
void t_allocate();
void finddisparity();
int disparity();
static int compare();
void read_features();
void writedisparity();
void read_neighbor_sys();

int *H;                      /* used to calculate the KS distance */
double *v1,*v2;              /* used to calculate the KS distance */
int **interiorlabel;         /* mask over the label lattice */
int idelta[Q4]={1,0,-1,0};   /* four near neighbors that assoc. pixel block  */
int jdelta[Q4]={0,1,0,-1};   /* not overlap.  see Geman's PAMI'90  p.615     */
double C[FR]={0.71,0.77,0.59,0.35,0.31};/* thresholds for feature selection   */
double total[5];
int count[5];
double max[5]={0.,0.,0.,0.,0.};
double min[5]={2.,2.,2.,2.,2.};

/* global variable for program name, used in error messages */
char prog_name[] = "disparity2d";


int main(argc,argv)
int argc;
char *argv[];
{
 int tfp[5], ffp, bfp, ofp;
 int tpipe[5], fpipe, bpipe, opipe;
 int one;
 int ierr = 0;
 char t_name[10], t_num[2];  /* used to process the -t* flags */
 char t_file[256], for_file[256], bac_file[256], output_file[256];
 char host[10], blank9[10];
 int ikp;                  /* flag set to 1 if running under ikp */
 int ks_flag, ad_flag;
 double threshold[5]; /* values from the C[] array that correspond */
                      /* to features/attributes that are being used */
 int num_of_tfiles;   /* the number of attributes used, must be >= 1 */
 int i, temp;

 int COARSE;       /* coarseness, i.e., inverse resolution */
 int ROW,COL;      /* size of row and column of pixel sites */
 int LROW,LCOL;    /* size of row and column of label sites */
 int LN;           /* total number of label sites, i.e., LN=LROW*LCOL */
 int HWW;          /* half window width for the c.d.f. calculation */
 int WW;           /* window width for the c.d.f. calculation, including */
                   /* the center point */
 int WS;           /* the window width squared, i.e. size of the window */
 int nr;           /* number of random neighbors */
 int NN;           /* number of neighbors, near and random */

 /* USP data */
 int usp_lineheader[SZLNHD];
 int usp_lineheader_bytes, usp_input_trace_bytes, usp_output_trace_bytes;

 int *forward;     /* array that stores forward indices */
 int *backward;    /* array that stores backward indices */
 double ***t_attr; /* 3d array used to hold the features/attributes */
                   /* for each data site */
 int ***dispar;    /* the resulting disparity values */

  /* Help? */
  if (C_ARGIS("-?", &argc, argv) ||
      C_ARGIS("-h", &argc, argv) ||
      C_ARGIS("-help", &argc, argv))
    {
     disparity2d_help();
     exit(0);
    }
 
  /* check to see if this program is being run under ikp */
  strcpy(host, "         ");
  strcpy(blank9, "         ");
  ikpchk_(host, strlen(host));
  if (strcmp(host, blank9))
     ikp = 1;
  else
     ikp = 0;

  /* display usage part of help if there was */
  /* only one argument on the command line */ 
  /* and we are not running under ikp */
  if (argc == 1 && !ikp)
    {
     fprintf(stderr, 
            "Usage:  disparity2d [-t1 <input_file_1>] [-t2 <input_file_2>]\n\
                    [-t3 <input_file_3>] [-t4 <input_file_4>]\n\
                    [-t5 <input_file_5>] -fmat <fifn>\n\
                    -bmat <bifn> [-ks | -ad]\n\
                    [-coarse <coarseness>] [-nr <num_of_rand_neigh>]\n\
                    [-halfwin <half_win_size>] [-O <output_file>]\n\
                    [-h -? -help]\n");
     exit(0);
    }

  num_of_tfiles= 0;

  /* set up the default file descriptors for ikp */
  tpipe[0] = 0;
  tpipe[1] = 3;
  tpipe[2] = 4;
  tpipe[3] = 5;
  tpipe[4] = 6;
  fpipe = 7;
  bpipe = 8;
  opipe = 1;
  one = 1;

  /* process the command line arguments */
  strcpy(t_name, "-t");
  for(i = 0; i < FR; i++)
    {
     sprintf(t_num, "%d", i+1); 
     C_ARGSTR(strcat(t_name, t_num), t_file, "", "", &argc, argv);
     if (strlen(t_file) >= 1)
       {
        threshold[num_of_tfiles] = C[i];
        lbopen(&(tfp[num_of_tfiles]), t_file, "r");
        num_of_tfiles++;
        strcpy(t_name, "-t");
       }
     else
       {
        if (ikp)
          {
           if (pipcnt_(&(tpipe[num_of_tfiles]), &one) > 0)
             {
              sisfdfit_(&(tfp[num_of_tfiles]), &(tpipe[num_of_tfiles]));
              threshold[num_of_tfiles] = C[i];
              num_of_tfiles++;
             }
          }
        strcpy(t_name, "-t");
       }
    }

  if (num_of_tfiles == 0)
    {
     fprintf(stderr, "%s: error: at least one input file must be specified\n",
             prog_name);
     ierr = 1;
    }

  C_ARGSTR("-fmat", for_file, "", "", &argc, argv);
  if (strlen(for_file) >= 1)
    {
     ffp = open(for_file, O_RDONLY);
     if (ffp == -1)
       {
        fprintf(stderr, "%s error: cannot open the forward index file %s\n",
                prog_name, for_file);
        ierr = 1;
       }
    }
  else
    {
     if (!ikp)
       {
        fprintf(stderr, "%s: error: -fmat <fifn> is required, for help use ",
                prog_name);
        fprintf(stderr, "disparity2d -h\n");
        ierr = 1;
       }
     else   /* running in ikp */
        ffp = fpipe;
    }

  C_ARGSTR("-bmat", bac_file, "", "", &argc, argv);
  if (strlen(bac_file) >= 1)
    {
     bfp = open(bac_file, O_RDONLY);
     if (bfp == -1)
       {
        fprintf(stderr, "%s error: cannot open the backward index file %s\n",
                prog_name, bac_file);
        ierr = 1;
       }
    }
  else
    {
     if (!ikp)
       {
        fprintf(stderr, "%s: error: -bmat <bifn> is required, for help use ",
                prog_name);
        fprintf(stderr, "disparity2d -h\n");
        ierr = 1;
       }
     else   /* running in ikp */
        bfp = bpipe;
    }

  /* Read in the USP lineheaders from the texture files */
  for(i = 0; i < num_of_tfiles; i++)
     C_RTAPE(tfp[i], usp_lineheader, &usp_lineheader_bytes);

  /* Grab values from the USP lineheader */
  C_SAVER(usp_lineheader, "NumSmp", &COL, LINEHEADER);
  C_SAVER(usp_lineheader, "NumTrc", &ROW, LINEHEADER);

  ks_flag = C_ARGIS("-ks", &argc, argv);
  ad_flag = C_ARGIS("-ad", &argc, argv);
  if (ks_flag && ad_flag)
    {
     fprintf(stderr, "%s: error: the flags -ks and -ad cannot both be set ",
             prog_name);
     fprintf(stderr, "on the same commandline.\n");
     ierr = 1;
    }
  else if( (!ks_flag) && (!ad_flag) )
     ks_flag = 1;  /* use the KS statistic by default */
  else if (ad_flag)
    {
     /* temporary */
     fprintf(stderr, "%s: warning: the Anderson-Darling statistic is ",
             prog_name);
     fprintf(stderr, "still being tested.  The K-S statistic is being used.\n");
     /* ad_flag = 1; */
     ks_flag = 1;
    }

  C_ARGI4("-nr", &nr, 16, 16, &argc, argv);
  /* check to see that nr is a power of two */
  temp = 0; 
  while( temp < nr )
    {
     if (temp != 0)
        temp *= 2; 
     else
        temp = 2;
    }
  if (temp != nr)
    {
     fprintf(stderr, "%s: warning: the number of random neighbors, i.e. ",
             prog_name);
     fprintf(stderr, "is not a power of two.  ");
     fprintf(stderr, "The value of <num_of_rand_neigh> is being ");
     fprintf(stderr, "changed to %d\n", temp);
     NN = temp + Q4;      /* use the symbolic constant for generality */  
                          /* in 3d there would be 6 near neighbors */
    }
  else
     NN = nr + Q4;

  C_ARGI4("-coarse", &COARSE, 9, 9, &argc, argv);
  if (COARSE < 1)
    {
     fprintf(stderr, "%s: error: the command line argument -coarse ",
             prog_name);
     fprintf(stderr, "<coarseness> must be positive.\n");
     ierr = 1;
    }

  C_ARGI4("-halfwin", &HWW, 7, 7, &argc, argv);
  if (HWW < 1)
    {
     fprintf(stderr, "%s: error:  the command line argument -halfwin ",
             prog_name);
     fprintf(stderr, "<half_win_size> must be positive.\n");
     ierr = 1;
    }

  C_ARGSTR("-O", output_file, "", "", &argc, argv);
  if (strlen(output_file) < 1)
    {
     if (!ikp)
       {
        ofp = opipe;
       } 
     else   /* running in ikp */
        if (pipcnt_(&opipe, &one) > 0)
           sisfdfit_(&ofp, &opipe);
    }
  else
     lbopen(&ofp, output_file, "w");

  /* exit if the error condition has been set */
  if (ierr) exit(1);

  WW = (2 * HWW) + 1;
  WS=SQR(WW);
  LROW=(ROW-WW)/COARSE+1;
  LCOL=(COL-WW)/COARSE+1;
  LN=LROW*LCOL;

  usp_input_trace_bytes = SZDTHD + (COL * SZSAMP);
  usp_output_trace_bytes = SZDTHD + (LCOL * SZSAMP);

  /* update the line header so it is correct for the disparity2d (output) */
  /* file.  We take one or more 1 record data sets with ROW traces and */
  /* COL samples as input, i.e. the texture files.  For each of the */
  /* LROW X LCOL points in the label lattice we find the disparity */
  /* between this point and each of its NN neighbors.  Therefore, we end */
  /* up filling an array of size LROW X LCOL X NN.  Note:  arrays in C */
  /* are stored in row major order.  So the NN disparity values between */
  /* a point and each of its neighbors are stored contiguously.  However, */
  /* the disparity2d (output) file will have LROW records, NN traces per */
  /* record, and LCOL samples per trace.  This ordering is used because */
  /* NN will be less than LROW and LCOL in most cases.  If we did use */
  /* NN as the number of samples per trace, then the size of */
  /* the disparity2d (output) file would be much larger because we would have */
  /* many more trace headers. */ 
  C_SAVEW(usp_lineheader, "NumSmp", LCOL, LINEHEADER);
  C_SAVEW(usp_lineheader, "NumTrc", NN, LINEHEADER);
  C_SAVEW(usp_lineheader, "NumRec", LROW, LINEHEADER);

  /* Also we want to keep track of the original number of traces and */
  /* samples in the 1 record input data set.  The original number of */
  /* traces will be stored in the lineheader field OrNTRC and the */
  /* original number of samples will be stored in the lineheader field */
  /* OrNREC */
  C_SAVEW(usp_lineheader, "OrNTRC", ROW, LINEHEADER);
  C_SAVEW(usp_lineheader, "OrNREC", COL, LINEHEADER);
 
  /* Update the historical line header */
  /* savhlh_(usp_lineheader, &usp_lineheader_bytes, &usp_lineheader_bytes); */

  /* write out the line header */
  C_WRTAPE(ofp, usp_lineheader, usp_lineheader_bytes);

  /* begin debug */
  /*
  fprintf(stderr, "num_of_tfiles = %d\n", num_of_tfiles);
  for(i = 0; i < num_of_tfiles; i++)
     fprintf(stderr, "threshold[%d] = %f\n", i, threshold[i]);
  fprintf(stderr, "last t_file %s\n", t_file);
  fprintf(stderr, "for_file %s\n", for_file);
  fprintf(stderr, "bac_file %s\n", bac_file);
  fprintf(stderr, "ROW = %d  COL = %d\n", ROW, COL);
  fprintf(stderr, "ks_flag = %d  ad_flag = %d\n", ks_flag, ad_flag);
  fprintf(stderr, "nr = %d  NN = %d\n", nr, NN);
  fprintf(stderr, "COARSE = %d  HWW =%d\n", COARSE, HWW);
  fprintf(stderr, "output_file %s\n", output_file);
  fprintf(stderr, "WW = %d  WS = %d  LROW = %d  LCOL = %d  LN = %d\n",
          WW, WS, LROW, LCOL, LN);
  */
  /* end debug */

  allocation(LROW, LCOL, NN, LN, &forward, &backward, ROW, COL, WS,
             num_of_tfiles, &t_attr, &dispar);

  /* read in the texture features/attributes */
  for(i = 0; i < num_of_tfiles; i++)
    {
     read_features(ROW, COL, i, tfp[i], t_attr, usp_input_trace_bytes);
    }

  /* read in the forward index and the backward index */
  read_neighbor_sys(forward, ffp, LN);
  read_neighbor_sys(backward, bfp, LN);

  finddisparity(LN, NN, LCOL, WW, HWW, WS, COARSE, ks_flag, dispar,
                forward, backward, num_of_tfiles, t_attr);

  writedisparity(LROW, LCOL, NN, COARSE, HWW, ofp, dispar, 
                                    usp_output_trace_bytes);

  /*
  fprintf(stderr, "%f %f %f %f %f\n", total[0]/count[0],total[1]/count[1],
                    total[2]/count[2],total[3]/count[3],total[4]/count[4]);
  fprintf(stderr,"%d %d %d %d %d\n", count[0],count[1],count[2],
                                              count[3],count[4]);
  fprintf(stderr, "%f %f %f %f %f\n", max[0],max[1],max[2],max[3],max[4]);
  fprintf(stderr, "%f %f %f %f %f\n", min[0],min[1],min[2],min[3],min[4]);
  */

  /* close files */
  for(i = 0; i < num_of_tfiles; i++)
    {
     lbclos(tfp[i]);
    }
  close(ffp);
  close(bfp);
  lbclos(ofp);

  /* Normal completion */
  fprintf(stderr, "%s:  normal completion\n", prog_name);

  return (0);
}/*main*/
 

/* allocate memory */
void allocation(LROW, LCOL, NN, LN, forward, backward, ROW, COL, WS,
                                       num_of_tfiles, t_attr, dispar)
int LROW, LCOL, NN, LN, ROW, COL, WS;
int **forward, **backward;
int num_of_tfiles;
double ****t_attr;
int ****dispar;
{
  int i,j,nn,fr;
  double ***pccc, **pcc, *pc;
  int    ***ipccc, **ipcc, *ipc;
                                                
  interiorlabel =(int **)malloc((LROW+2*BORDER)*sizeof(int *));
  interiorlabel += BORDER;
  ipc=(int *)malloc((LROW+2*BORDER)*(LCOL+2*BORDER)*sizeof(int));
  ipc += BORDER*(LCOL+2*BORDER)+BORDER;
  for (i=-BORDER; i<LROW+BORDER; i++)
    interiorlabel[i]=ipc+i*(LCOL+2*BORDER);
  for (i=-BORDER;i<LROW+BORDER;i++)  /*initialize interior[][]*/
    for (j=-BORDER;j<LCOL+BORDER;j++) {
      if ( (i >0) && (i < LROW-1) && (j >0) && (j < LCOL-1) )
        interiorlabel[i][j]=4;
      else if (i<0 || j<0 || i>=LROW || j>=LCOL)
        interiorlabel[i][j]=0;
      else if ((i==0 && (j==0 || j==LCOL-1)) || (i==LROW-1 && (j==0 || j==LCOL-1)))
        interiorlabel[i][j]=2; 
      else 
        interiorlabel[i][j]=3;
    }/*for j*/

  *forward=  (int *)malloc(LN*sizeof(int));
  *backward= (int *)malloc(LN*sizeof(int));

  /* 3D array for the features/attributes */
  /* allocate space for only those features/attributes */
  /*  specified on the command line */ 
  (*t_attr) = (double ***) malloc(num_of_tfiles * sizeof(double **));
  pcc = (double **) malloc(num_of_tfiles * ROW * sizeof(double *));
  for(i = 0; i < num_of_tfiles; i++)
     (*t_attr)[i] = pcc + (i * ROW);
  pc = (double *) malloc(num_of_tfiles * ROW * COL * sizeof(double));
  for(i = 0; i < num_of_tfiles; i++)
     for(j = 0; j < ROW; j++)
        (*t_attr)[i][j] = pc + ((i * ROW) + j) * COL;

  /* 3D array for the resulting disparity values */
  (*dispar)=(int ***)malloc(LROW*sizeof(int **));
  ipcc=(int **)malloc(LROW*LCOL*sizeof(int *));
  for (i=0; i<LROW; i++) 
    (*dispar)[i]=ipcc+i*LCOL;
  ipc=(int *)malloc(LROW*LCOL*NN*sizeof(int));
  for (i=0; i<LROW; i++)
    for (j=0; j<LCOL; j++)
      (*dispar)[i][j]=ipc+((i*LCOL)+j)*NN;

  /* used when calculating the KS distance */
  H=(int *)malloc(2*WS*sizeof(int));
  v1=(double *)malloc(WS*sizeof(double));
  v2=(double *)malloc(WS*sizeof(double));

/* Do not need to initialize dispar with zeroes, this is done by malloc 
  for (i=0; i<LROW; i++)
    for (j=0; j<LCOL; j++)
      for (nn=0; nn<NN; nn++)
        dispar[i][j][nn]=0;
*/

}/*allocation*/


void t_allocate(ROW, COL, t_attr)
 int ROW, COL;
 double ***t_attr; 
 {
  int i;
  double *pc;

  *t_attr = (double **) malloc(ROW * sizeof(double *)); 
  pc = (double *) malloc(ROW * COL * sizeof(double));
  for(i = 0; i < ROW; i++)
     (*t_attr)[i] = pc + (i * COL);
 }


/* find the disparity values */
void finddisparity(LN, NN, LCOL, WW, HWW, WS, COARSE, ks_flag,
                   dispar, forward, backward, num_of_tfiles, t_attr)
int LN, NN, LCOL, WW, HWW, WS, COARSE, ks_flag;
int ***dispar;
int *forward, *backward;
int num_of_tfiles;
double ***t_attr;
{
  int ND;
  int ENN;    /*exponent of no. of random nbr. (exclude near neighbors) */
  int RN;     /*no. of random number*/
  int *twos;  /*representation in base 2*/

  int ln,i,j,m,n,nn,rn,pm,pn;
  int rm,qn;
  int hb,lb,ind,nbr;
  int nd;     /*# of division by 2*/
  int stp; /*step size for near nbr. s.t. assco. pixel blocks don't overlap   */
  int old_i;  /* used in status print out */

  /* ND=ceil(log2((double)LN)); */
  ND=ceil( log10((double)LN) / log10(2.0) );
  RN=NN-Q4;
  /* ENN=(int)log2((double)(RN)); */
  ENN=(int)( log10((double)(RN)) / log10(2.0) );
  twos=(int *)malloc(ND*sizeof(int));

  old_i = -1;
  for (ln=0; ln<LN; ln++) {
    i=ln/LCOL;
    j=ln%LCOL;

    if (i != old_i)
      {
       fprintf(stderr, "%s: processing output trace %d of %d\n", 
               prog_name, i+1, LN/LCOL);
       old_i = i;
      }

    /*near neighbors*/
    stp=((int)ceil((double)WW/COARSE));
    for (nn=0; nn<Q4; nn++) {
      m=i+idelta[nn]*stp; n=j+jdelta[nn]*stp;
      if (interiorlabel[m][n])
        dispar[i][j][nn]=disparity(HWW, WS, COARSE, ks_flag,
                                   num_of_tfiles, t_attr, i,j,m,n);
    }

   /*random neighbors*/
    
    rn=forward[ln];
    qn=rn;
    for(nd=0; nd<ND; nd++) {
      twos[nd]=qn%2;
      qn=qn/2;
    }

    /*high-bit random neighbor, i.e., xxxbbbbb*/
    for (hb=0;hb<RN/2;hb++) {
      ind=hb;
      for (nd=ND-ENN-1; nd>=0; nd--)
        ind=ind*2+twos[nd+ENN];
      nbr=backward[ind];
      m=nbr/LCOL; n=nbr%LCOL;
      dispar[i][j][nn]=disparity(HWW, WS, COARSE, ks_flag,
                                 num_of_tfiles, t_attr, i,j,m,n);
      nn++;
    }

    /*low-bit random neighbor, i.e., bbbbbxxx*/
    for (lb=0;lb<RN/2;lb++) {
      ind=0;
      for (nd=ND-ENN-1; nd>=0; nd--)
        ind=ind*2+twos[nd];
      ind=ind*RN/2+lb;
      nbr=backward[ind];
      m=nbr/LCOL; n=nbr%LCOL;
      dispar[i][j][nn]=disparity(HWW, WS, COARSE, ks_flag,
                                 num_of_tfiles, t_attr, i,j,m,n);
      nn++;
    }
    nn=0;
  
  }/*for ln*/

 free(twos);
}/*finddisparity*/


/* find the disparity value for pair (i,j)-(m,n) */
int disparity(HWW, WS, COARSE, ks_flag, num_of_tfiles, t_attr, i,j,m,n)
int HWW, WS, COARSE, ks_flag, num_of_tfiles;
double ***t_attr;
int i,j,m,n;
{
  int fr,maxpenalty;
  int pi,pj,pm,pn,k,l,ppi,ppj,ppm,ppn,penalty;
  double distance, d, dt, fn1, fn2, denom;
  int k1,k2,ht,h,maxval, d1, d2;

  pi=HWW+i*COARSE; pj=HWW+j*COARSE;
  pm=HWW+m*COARSE; pn=HWW+n*COARSE;

  maxpenalty=-1;

  for (fr=0; fr < num_of_tfiles; fr++) {

    for (h=0; h<WS; h++) {H[h]=0; v1[h]=0.0; v2[h]=0.0;}
    for (h=WS; h<2*WS; h++) H[h]=0;

    h=0;
    for (k=-HWW; k<=HWW; k++)
      for (l=-HWW; l<=HWW; l++) {
        ppi=pi+k; ppj=pj+l;
        ppm=pm+k; ppn=pn+l;
        v1[h]=t_attr[fr][ppi][ppj];
        v2[h]=t_attr[fr][ppm][ppn];
        h++;
      }

    qsort(v1,WS,sizeof(double),compare);
    qsort(v2,WS,sizeof(double),compare);   

    /*
    k1=k2=h=ht=0;
    do {
      while (v1[k1]==v1[k1+1]  && (k1+1) <WS) {k1++; ht++;}
      while (v2[k2]==v2[k2+1]  && (k2+1) <WS) {k2++; ht--;}
      if       (v1[k1]>v2[k2]) { H[h]=(--ht); k2++; h++; }
      else if  (v1[k1]<v2[k2]) { H[h]=(++ht); k1++; h++; }
      else                     { k1++; k2++; H[h]=ht; h++; }
    } while (k1<WS && k2<WS);
    while (k1<WS) { H[h]=(++ht); h++; k1++; }
    while (k2<WS) { H[h]=(--ht); h++; k2++; }
    */

    /*Compute the Kolmogorov-Smirnov distance*/
    /*
    maxval=0;
    for (h=0; h<2*WS; h++) 
      if (abs(H[h])>maxval) maxval=abs(H[h]);  
    */

    /*Kolmogorov-Smirnov distance or variant*/
    /*
    distance=(double)maxval/WS;
    */

    if (ks_flag)
      {
       d = 0.0;
       fn1 = 0.0;
       fn2 = 0.0;
       k1 = k2 = 0;
       while(k1 < WS && k2 < WS)
         {
          if ( (d1 = v1[k1]) <= (d2 = v2[k2]) )
             fn1 = (++k1)/((double) WS);
          if (d2 <= d1)
             fn2 = (++k2)/((double) WS);
          if ((dt = fabs(fn1 - fn2)) > d) d = dt;
         }
       distance = d;
      }
    else   /* the ad_flag is set -- use the Anderson-Darling statistic */
      {
       d = 0.0;
       fn1 = 0.0;
       fn2 = 0.0;
       k1 = k2 = 0;
       while(k1 < WS && k2 < WS)
         {
          if ( (d1 = v1[k1]) <= (d2 = v2[k2]) )
             fn1 = (++k1)/((double) WS);
          if (d2 <= d1)
             fn2 = (++k2)/((double) WS);

          /* Calculate the denominator of the Anderson-Darling distance */
          /* When calculating the denominator make sure its value is */
          /* finite.  If not just use a value of 1.0 */ 
          if (fn1 != 0.0 && fn1 != 1.0)
             denom = fn1 * (1.0 - fn1);
          else if (fn2 != 0.0 && fn2 != 1.0)
             denom = fn2 * (1.0 - fn2);
          else
             denom = 1.0;

          if ( (dt = (fabs(fn1 - fn2)/sqrt(denom))) > d) d = dt;
         }
       distance = d;
      }
 

    total[fr]=distance+total[fr];
    count[fr]=count[fr]+1;
    if(distance<min[fr]) min[fr]=distance;
    if(distance>max[fr]) max[fr]=distance;

    penalty=2*(distance>C[fr])-1; /* either +1 or -1 */
   
    if (penalty==1) return(1); /* this early return saves computation */

    if (penalty>maxpenalty) maxpenalty=penalty;
  }/*fr*/

  return(maxpenalty);

}/*disparity*/


static int compare(i,j)
double *i,*j;
{
 return (*i<*j) ? -1 : (*i>*j) ? 1 : 0;
}/*compare*/


/* read a neighborhood index into an array named img from fp */
void read_neighbor_sys(img, fp, LN)
int *img;
int fp; /* input file pointer */
int LN;   /* size of img */
{
  read(fp, (char *) img, sizeof(int) * LN);

}/*read_neighbor_sys*/


/* read image from the file tfp */
void read_features(ROW, COL, fr, tfp, t_attr, usp_input_trace_bytes)
 int ROW, COL, fr;
 int tfp;                     /* file descriptor */
 double ***t_attr; 
 int usp_input_trace_bytes;
 {
  float *usp_input_trace;
  int i, j, k;

  usp_input_trace = (float *) malloc(usp_input_trace_bytes);
  for (i=0; i<ROW; i++)
   {
    /* Read the next trace */
    C_RTAPE(tfp, usp_input_trace, &usp_input_trace_bytes);
    for(j=0, k = ITRWRD; j<COL; j++, k++) 
       t_attr[fr][i][j] = (double) usp_input_trace[k];
   }
  free(usp_input_trace);
 }


/* write image to the file fp */
/* Note:  The disparity (output) file will have LROW records, */
/* NN traces per record, and LCOL samples per trace.  See the */
/* large comment in main before the first C_SAVEW calls for the reason. */ 
void writedisparity(LROW, LCOL, NN, COARSE, HWW, fp, img,
                                   usp_output_trace_bytes)
int LROW, LCOL, NN, COARSE, HWW;
int fp;             /* file descriptor */
int ***img;
int usp_output_trace_bytes;
{
  float *usp_output_trace;
  int i, j, k, nn;

  usp_output_trace = (float *) malloc(usp_output_trace_bytes);

  /* Set the Static Correction field in the traceheader */
  C_SAVEW(usp_output_trace, "StaCor", 0, TRACEHEADER);

  /* loop over records */
  for(i = 0; i < LROW; i++)
    {
     /* set the RecNum Field in the traceheader */
     C_SAVEW(usp_output_trace, "RecNum", i * COARSE + HWW, TRACEHEADER);
     /* loop over traces */
     for(nn = 0; nn < NN; nn++)
       {
        /* set the TrcNum Field in the traceheader */
        C_SAVEW(usp_output_trace, "TrcNum", nn, TRACEHEADER);
        /* loop over samples */
        for(j = 0, k = ITRWRD; j < LCOL; j++, k++)
           usp_output_trace[k] = (float) img[i][j][nn];
        C_WRTAPE(fp, usp_output_trace, usp_output_trace_bytes); 
       }
    }

  free(usp_output_trace);
}


void disparity2d_help()
 {
  fprintf(stderr, "\
  This program computes the disparity between all pairs of\n\
  neighboring label sites.  The two label sites in a neighboring pair\n\
  can be near neighbors or random neighbors.  The disparity for one\n\
  of these pairs is computed by comparing the attribute values in the\n\
  texture analysis window of one of the label sites with the attribute\n\
  values in the texture analysis window of the other label site.\n\
  \n\
  This program uses the Kolmogorov-Smirnov statistic (or\n\
  some variant, such as the Anderson-Darling statistic)\n\
  to determine the similarity of the cdf's of the attribute values in\n\
  the two texture analysis windows.  If the textures (i.e. the cdf's) of\n\
  the two sites are different, then the disparity between these two sites\n\
  is 1.  If the textures of the two sites are similar, then the disparity\n\
  between the two sites is -1.\n\
  \n\
  disparity2d [-t1 <input_file_1>] [-t2 <input_file_2>]\n\
              [-t3 <input_file_3>] [-t4 <input_file_4>]\n\
              [-t5 <input_file_5>] -fmat <fifn>\n\
              -bmat <bifn> [-ks | -ad]\n\
              [-coarse <coarseness>] [-nr <num_of_rand_neigh>]\n\
              [-halfwin <half_win_size>] [-O <output_file>]\n\
              [-h -? -help]\n\
  \n\
  -t1 <input_file_1> - the input file containing the first attribute of\n\
                       all the sites in the data lattice.  The first\n\
                       attribute is the data value at the data site.\n\
                       (optional)\n\
  \n\
  -t2 <input_file_2> - the input file containing the second attribute of\n\
                       all the sites in the data lattice.  The second\n\
                       attribute is the intensity range, i.e. the\n\
                       maxval - minval in the attribute analysis window\n\
                       of the data site.\n\
                       (optional)\n\
  \n\
  -t3 <input_file_3> - the input file containing the third attribute of\n\
                       all the sites in the data lattice.  The third\n\
                       attribute is the boundary residual, i.e. the\n\
                       data value at the data site minus the average\n\
                       of the data values on the boundary of the\n\
                       the data site's attribute analysis window.\n\
                       (optional)\n\
  \n\
  -t4 <input_file_4> - the input file containing the fourth attribute of\n\
                       all the sites in the data lattice.  The fourth\n\
                       attribute is the horizontal directional\n\
                       residual, i.e. a gradient approximation in the\n\
                       horizontal direction.\n\
                       (optional)\n\
  \n\
  -t5 <input_file_5> - the input file containing the fifth attribute of\n\
                       all the sites in the data lattice.  The fifth\n\
                       attribute is the vertical directional\n\
                       residual, i.e. a gradient approximation in the\n\
                       vertical direction.\n\
                       (optional)\n\
  \n\
                       NOTE: at least one of the attributes\n\
                             must be used to calculate the\n\
                             disparity values, i.e. at least\n\
                             one of the options {t1, t2, ..., t5}\n\
                             must be specified on the command line.\n\
  \n\
  -fmat <fifn> - the input file containing the forward index used\n\
                 for locating random neighbors.  The forward index\n\
                 is generated by the program randomgraph\n\
  \n\
  -bmat <bifn> - the input file containing the backward index used\n\
                 for locating random neighbors. The backward index\n\
                 is generated by the program randomgraph\n\
  \n\
  [-ks | -ad] - a pair of optional flags used to specify which\n\
                statistical test is used to compare textures.  The\n\
                Kolmogorov-Smirnov test is used when the -ks flag\n\
                is specified.  The Anderson-Darling test is used\n\
                when the -ad flag is specified.  Specification of\n\
                a test is optional.  If neither one of these flags\n\
                is set the Kolmogorov-Smirnov test is used.\n\
                NOTE:  at most one of these flags can be set on\n\
                       a command line.\n\
                NOTE:  The Anderson-Darling test is still being refined.\n\
                       So, it is not implemented in the current version.\n\
  \n\
  -coarse <coarseness> - inverse of the resolution at which the\n\
                         segmentation is performed.  That is, the ratio\n\
                         of the number of samples/traces in the data\n\
                         lattice to the number of samples/traces in the\n\
                         label lattice.  The data lattice is the input to\n\
                         the program texture, and the label lattice is the\n\
                         output from any segmentation optimizer, such as\n\
                         bethe.  For highest resolution set the coarseness\n\
                         to 1.\n\
                         (Default: 9)\n\
  \n\
  -nr <num_of_rand_neigh> - the number of random neighbors for each\n\
                            label site.  The neighborhood for a\n\
                            label site also includes 4 near neighbors.\n\
                            Thus, each label site will have\n\
                            <num_of_rand_neigh> + 4 neighbors.  The number\n\
                            of random neighbors must be a power of two.\n\
                            (Default: 16)\n\
  \n\
  -halfwin <half_win_size> - half window width used to determine\n\
                             the width of the texture analysis window\n\
                             over which the cdf's are computed in the\n\
                             program disparity2d.  The width of the texture\n\
                             analysis window is given by\n\
                             2 * <half_win_size> + 1.\n\
                             (Default: 7)\n\
  \n\
  -O <output_file> - the output file which contains the disparity\n\
                     values.  For each label site there are\n\
                     <num_of_rand_neigh> + 4 disparity values, one\n\
                     for each neighbor of the label site.\n\
                     (Default: stdout)\n\
  \n\
  -h, -?, -help - Help.\n\
");
}
