c********************************************************************
c                                                                   *
c                 copyright 2001, Amoco Production Company             *
c                             All Rights Reserved                      *
c                     an affiliate of BP America Inc.                  *
c           proprietary  _  to be maintained in confidence          *
c                                                                   *
c********************************************************************
c  routine:       fxdc                                              *
c  routine type:  program                                           *
c  purpose:   perform f-x domain deconvolution                      *
c  entry points:                                                    *
c  arguments:                                                       *
c       +--------------------------------------------------------+  *
c       |                 development information                |  *
c       +--------------------------------------------------------+  *
c  defining geophysicist: gary donathan and richard crider          *
c                                                                   *
c  author:     susan manning                 origin date:  87/01/28 *
c  language:   fortran 77             date last compiled:  87/01/28 *
c                                                                   *
c  date last modified:  87/08/03 corrected defaults for overlap     *
c                                parameters.                        *
c                       87/10/27 changed lwndw input format from    *
c                                i4 to i5 to match documentation.   *
c                       87/11/04 changed method of determining taper*
c                                weights                            *
c                rlc    88/02/19 modified for record sequential mode*
c                rlc    88/04/28 modified to add back percentage    *
c                                of original data to output         *
c                rlc    88/11/15 modified to abend with format 2    *
c                                input.                             *
c                rlc    89/01/04 modified to correct error in rec   *
c                                sequential mode.                   *
c       +--------------------------------------------------------+  *
c       |                  external environment                  |  *
c       +--------------------------------------------------------+  *
c                                                                   *
c  routines called:                                                 *
c    rfftb,rfftsc vsq,vmma,vmul,vsin,vsub,dotpr,svesq,vramp,vsmul   *
c    cvma,cvmul,vsdiv,cvconj,cvmgsa,cvmov,cvsub,cvfill,cvsmul,vma   *
c    hlh, rtape,wrtape,gamoco                                       *
c                                                                   *
c  intrinsic functions called:                                      *
c       cabs                                                        *
c       cmplx                                                       *
c       mod                                                         *
c                                                                   *
c  files:                                                           *
c  common:                                                          *
c       +--------------------------------------------------------+  *
c       |             other documentaiton details                |  *
c       +--------------------------------------------------------+  *
c  error handling:                                                  *
c                                                                   *
c  general description:   f-x domain deconvolution                  *
c                                                                   *
c******************* end of documentation package *******************

#include <save_defs.h>
#include <f77/lhdrsz.h>
#include <f77/iounit.h>
#include <f77/sisdef.h>

      INTEGER MAXSMP, MAXTR,TRLEN,TRHLEN

      PARAMETER (MAXSMP=SZLNHD, MAXTR = 150)
      PARAMETER (TRLEN = MAXSMP)
      PARAMETER (MAXSM2= MAXSMP+MAXSMP)
      PARAMETER (TRHLEN = MAXSMP)
      PARAMETER (MAXHL = MAXSMP)

      INTEGER input(MAXSMP), i4hdr(MAXSMP)
      INTEGER i2hdr(MAXSMP)
      INTEGER argis

      real UnitSc

      CHARACTER card*80, title*66, adinfo*15, crdid*5, fxdc1*5, name*4
      CHARACTER tmode(2)*18
      CHARACTER ntap*256, otap*256, qctap*256, crdin*256

c variables used with dynamic memory allocation

      integer hdrsav
      integer error, abort

      REAL      x(1), xn(1), tx(1), txn(1), xhld(1)

      pointer (wkhdrsav, hdrsav(1))
      POINTER (px,x)
      pointer (pxn,xn)
      pointer (ptx,tx)
      pointer (ptxn,txn)
      POINTER (pxhld,xhld)
C
      LOGICAL   open, altape, there, verbose

      equivalence (input(1), i4hdr(1)), (input, i2hdr)

      DATA      fxdc1/'1FXDC'/, ntproc/0/, ic/0/,
     &          adinfo/' (noise output)'/, open/.false./
      DATA tmode/'line sequential   ','record sequential '/
      data ierr / 0 /

      common/fxlu/   lucrd, lud, lui, luo, luo2, lup
      common/fxparm/ ldsign, loper, lsovlp, lwndw, ltovlp,
     &               if1, if2, pw, imute, iisi, ifmt, altape,
     &               ntjb,mode,ntpr,pcnt

      name = 'FXDC'
      card(1:4) = '   '
c +----------------------------+
c | set LOGICAL unit numbers   |
c +----------------------------+
      lucrd = 22
      lup  = LERR
      lui  = 7
      luo  = 28
      luo2 = 29
      lud  = 11
c +-------------------------------------+
c | set up title and open input dataset |
c +-------------------------------------+
      title = '                    F-X DOMAIN DECONVOLUTION  '//
     &        '                    '
      if ( argis ('-?') .gt. 0 .or. 
     :     argis ('-H') .gt. 0 .or. 
     :     argis ('-help') .gt. 0 .or. 
     :     argis ('-h') .gt. 0 ) then
         call help()
         stop
      endif

#include <f77/open.h>
      call gamoco (title, 1, LERR)
      ntap = ' '
      otap = ' '
      call gcmdln(ntap,otap,qctap,crdin,verbose,lucrd,altape)
      if(ntap.ne. ' ')then
        there = .false.
        inquire(file = ntap, exist = there)
        if(.not.there) then
      write(LERR,'(a)')'  Input file does not exist. FXDC aborted'
          call ccexit(100)
        endif
          call lbopen (lui, ntap, 'r')
      else
C  set lui to stdin for pipe input.
        lui = 0
      endif
      call rtape (lui, input, nbyt)
      if (nbyt .eq. 0) then
         write (lup, 100)
  100    format (/,  1x, '*** m0100 *** error in fxdc main.',
     &           /, 14x, 'no line header on your input data set.  ',
     &                   'check your data ',
     &           /, 14x, 'before you resumit this job.')
         ic = 100
         go to 5000
      end if
c +--------------+
c | initiate hlh |
c +--------------+
      jlen=4
      call hlhprt (input, nbyt, name, jlen, LERR)
c +--------------------------------------+
c | get information from the line header |
c +--------------------------------------+

      call saver(input,'Format',ifmt,LINHED)

      call saver(input,'NumTrc',ntpr,LINHED)

      call saver(input,'NumRec',nrec,LINHED)

      call saver(input,'SmpInt',iisi,LINHED)

      call saver(input,'NumSmp',nsmp,LINHED)
      call saver(input,'UnitSc',UnitSc,LINHED)

      if (UnitSc .eq. 0.0) then
          write(LERR,*)'********************************************'
          write(LERR,*)'WARNING: sample unit scaler in LH = ',UnitSc
          write(LERR,*)'         will set to .001 (millisec default)'
          write(LERR,*)'********************************************'
          UnitSc = .001
          call savew(input,'UnitSc',UnitSc,LINHED)
      endif

      call savelu('RecNum',ifmt_RecNum,l_RecNum,ln_RecNum,TRACEHEADER)
      call savelu('StaCor',ifmt_StaCor,l_StaCor,ln_StaCor,TRACEHEADER)

      item = ITRWRD * MAXTR
      call galloc (wkhdrsav, item*SZSMPD, ierr, iab)

      do i = 1, item
         hdrsav(i) = 0
      enddo

      if (nsmp .gt. MAXSMP)then
         write (lup, 105) nsmp,MAXSMP
  105    format (/,  1x, '*** m0105 *** error in fxdc main.',
     &           /, 14x, 'the number of samples in the line header ',
     &                   i5, ' exceeds the program',
     &           /, 14x, 'maximum of ',i5,'.  fix your input data ',
     &                   'before resubmitting this job.')
         ierr = ierr + 1
      end if
      if (ifmt .ne. 3)then
         write (lup, 106) ifmt
  106    format (/,  1x, '*** m0106 *** error in fxdc main.',
     &           /, 14x, 'the format of the input data set ( ',
     &                   i5, ' ) is invalid.',
     &           /, 14x, 'convert to 3 before ',
     &                   'resubmitting this job.')
         ierr = ierr + 1
      end if
      ntjb = ntpr * nrec
c +---------------------------------------------------------+
c | open the output dataset and the noise tape if requested |
c +---------------------------------------------------------+
        if(otap.ne.' ')then
           if (.not. open) then
               call lbopen (luo, otap, 'w')
               if (altape) call getln(luo2,qctap,'w',1)
               open = .true.
           end if
        else
C  set luo to stdout for pipe output.
         luo = 1
        endif
c +-------------------+
c | read 1fxdc card   |
c +-------------------+
c +--------------------------------------------------------------+
c |                                                              |
c |   card contains entire card in c*80                          |
c |   crdid    = card identifier                                 |
c |   ldsign   = length in traces of the design window           |
c |   loper    = length in traces of the operator                |
c |   lsovlp   = percent of spatial partition overlap            |
c |   lwndw    = length in milliseconds of the temporal window   |
c |              (must be a power of 2)                          |
c |   ltovlp   = percent of temporal window overlap              |
c |   if1, if2 = low and high pass frequencies                   |
c |   pw       = pre-whitening percent (max 100%)                |
c |   imute    = preserve early mute flag (0=yes 1=no)           |
c |   mode     = processing mode flag                            |
c |              (0=line sequential 1= record sequential)        |
c |   pcnt     = percent of origanal data to add back to outpu   |
c +--------------------------------------------------------------+
      if (crdin(1:1) .ne. ' ') then
          read (lucrd, 1000, end=5000) card, crdid, ldsign, loper,
     &                           lsovlp, lwndw, ltovlp, if1, if2, pw,
     &                           imute,mode,pcnt
 1000     format (a80, t1, a5, 1x, i2, 1x, i2, 1x,
     &            i2, i5, 1x, i2, 1x, i3, 1x, i3, f4.0,
     &            1x, i1,i1,f3.0)
      else
          call cmdln1 (ldsign,loper,lsovlp,lwndw,
     1             ltovlp,if1,if2,pcnt,pw,imute,mode)
      endif
c +---------------------------------+
c | make sure it's the correct card |
c +---------------------------------+
      if (crdid .ne. fxdc1 .and. crdin(1:1) .ne. ' ') then
         write(LERR, '(a)')card
         write (lup, 110) crdid
  110    format (/,  1x, '*** m0110 *** error in fxdc main.',
     &           /, 14x, 'invalid card used as input.  input card id ',
     &                   'is ', a5, ' and should',
     &           /, 14x, 'be 1fxdc.  fix card input before ',
     &                   'resubmitting this job.')
         ierr = ierr + 1
      endif
      if (pcnt.lt.0.0) then
         write (lup, 112) pcnt
  112    format (/,  1x, '*** m0112 *** error in fxdc main.',
     &           /, 14x, 'invalid add-back percentage (',f6.2,') input',
     &           /, 14x, 'the value must be greater than or equal to ',
     &                   'zero.',
     &           /, 14x, 'fix card input before ',
     &                   'resubmitting this job.')
         ierr = ierr + 1
      endif
      if (mode.eq.1.and.ldsign.gt.ntpr)then
         write (lup, 113) ldsign,ntpr
  113    format (/,  1x, '*** m0113 *** error in fxdc main.',
     &           /, 14x, 'record sequential processing requested and ',
     &           /, 14x, 'design partition width (',i3,') is greater ',
     &           /, 14x, 'than number of traces per record (',i3,'). ',
     &           /, 14x, 'fix card input before ',
     &                   'resubmitting this job.')
         ierr = ierr + 1
      endif
c +------------------------------------------------------------+
c | make sure the card parameters are set to default values if |
c | necessary and make sure all are in range                   |
c +------------------------------------------------------------+
      call prmchk (card, nsmp, UnitSc, ierr)
c +--------------------------------------------+
c | cancel the job if we have errors up to now |
c +--------------------------------------------+
      if (ierr .ne. 0) then
         write (lup, 150) ierr
  150    format (/,  1x, '*** m0150 *** program fxdc abended due to',
     &          i3,' input error(s) ')
         ic = 100
         go to 5000
      endif
c +-----------------------------------------+
c | write the card input in legible fashion |
c +-----------------------------------------+
      write (LERR, 1001) ldsign, loper, lsovlp, lwndw, ltovlp,
     &                  if1, if2, pw, imute,mode,pcnt
 1001 format (/, 10x, '          fxdc input parameters:', /,
     &        /, 10x, '#traces in spatial design partition .....', i3,
     &        /, 10x, '#traces in spatial operator .............', i3,
     &        /, 10x, 'spatial partition overlap (%) ...........', i3,
     &        /, 10x, 'length of temporal window (ms.) .......',   i5,
     &        /, 10x, 'temporal window overlap (%) .............', i3,
     &        /, 10x, 'minimum acceptable frequency ...........',  i4,
     &        /, 10x, 'maximum acceptable frequency ...........',  i4,
     &        /, 10x, 'pre-whitening (%) ....................',  f6.2,
     &        /, 10x, 'preserve early mute flag (0=yes, 1=no) ...',i2,
     &        /, 10x, 'processing mode ..........................',i2,
     &        /, 10x, '                 0=line sequential        ',
     &        /, 10x, '                 1=record sequential      ',
     &        /, 10x, 'data add-back ...........................',f4.1,
     &                '%',/)
      pcnt=pcnt/100.
c +---------------------------------------------------+
c |      correct ntjb for mode                        |
c +---------------------------------------------------+
      if(mode.eq.1)ntjb=ntpr
c +---------------------------------------------+
c | write the line header to the output dataset |
c +---------------------------------------------+
      ialfmt=3
      call savew(input,'Format',ialfmt,LINHED)
      iby=nbyt
      call savhlh(input,iby,nbyt)
      call wrtape (luo, input, nbyt)
c +----------------------------------------------------------+
c | give additional information to the noise line header and |
c | output the noise dataset line header                     |
c +----------------------------------------------------------+
      if (altape) then
*        i2hdr(33) = 3
         ialfmt=3
         call savew(input,'Format',ialfmt,LINHED)
         i4=4
         call wrtape (luo2, input, nbyt)
      endif
c +---------------------------------------+
c | open disk lud for storing dead traces |
c +---------------------------------------+
      nbytes=nsmp*ISZBYT + SZTRHD

c +-----------------------------------------------+
c | find buffer lengths for above the line allocs |
c +-----------------------------------------------+
      lw    = lwndw / iisi
      call power2 (lw, n, nn, lup, ierr)
      if (nn .eq. 0) nn = n / 2 + 1
c
      lsramp = ldsign * lsovlp / 100
      lenx   = nsmp * (ldsign + (loper - 1))
      lenxn  = nsmp * ldsign + 50
      lentx  = (n + 2)    * (ldsign + (loper - 1))
      lentxn = (n + 2)    * (ldsign + (loper - 1))
      lenhld = nsmp * lsramp * 4.

      abort = 0
      error = 0
      call galloc(px,  lenx*ISZBYT,  error, abort)
      call galloc(pxn, lenxn*ISZBYT, error, abort)
      call galloc(ptx, lentx*ISZBYT, error, abort)
      call galloc(ptxn,lentxn*ISZBYT,error, abort)
      call galloc(pxhld,lenhld*ISZBYT,error,abort)

      if(abort.ne.0)then
         write(LERR,*)' Error trying to allocate memory'
         call ccexit(100)
      endif

      call vclr ( x, 1, lenx )
      call vclr ( xn, 1, lenxn )
      call vclr ( tx, 1, lentx )
      call vclr ( txn, 1, lentxn )
      call vclr ( xhld, 1, lenhld )

      ndxx   = 0
      ndxxn  = 0
      ndxtx  = 0
      ndxtxn = 0
      ndxhld = 0

c +-------------------------------+
c | call the main control routine |
c +-------------------------------+
      call fxmain (x(ndxx+1), xhld(ndxhld+1), xn(ndxxn+1), nsmp,
     &             tx(ndxtx+1), txn(ndxtxn+1), n, nn, ntproc,
     &             hdrsav,ITRWRD,ITHWP1,SZLNHD,SZSMPD,UnitSc )
 5000 continue

      ilast = 1
      call fxdcon (x, ldread, n, nn, txn, ilast,
     1             SZLNHD,ITHWP1,ITRWRD,SZSMPD,UnitSc)
      call riclr (LERR)
      call lbclos (lui)
      call lbclos(luo)
      if (altape) call lbclos(luo2)
      call ccexit (ic)
      stop
      end
c********************************************************************
c                                                                   *
c                 copyright 2001, Amoco Production Company             *
c                             All Rights Reserved                      *
c                     an affiliate of BP America Inc.                  *
c           proprietary  _  to be maintained in confidence          *
c                                                                   *
c********************************************************************
c  routine:       fxmain                                            *
c  routine type:  subroutine                                        *
c  purpose:   control spatial and temporal windowing for fxdc       *
c  entry points:                                                    *
c      none                                                         *
c  arguments:                                                       *
c      x        REAL      input        -two-dimensional buffer sent *
c                                       to fxmain for input trace   *
c                                       matrix  (nsmp, ldsign)      *
c      xhld     REAL      input        -two-dimensional buffer sent *
c                                       to fxmain for holding output*
c                                       from fxdcon until spatial   *
c                                       ramping can be done.        *
c                                       (nsmp, ldsign)              *
c      nsmp     INTEGER   input        -first dimension of x and    *
c                                       xhld buffers.               *
c      xn       REAL      input        -two-dimensional buffer sent *
c                                       to fxmain for output trace  *
c                                       matrix  (n2, ldsign)        *
c      n2       INTEGER   input        -first dimension of xn buffer*
c      tx       REAL      input        -two-dimensional buffer sent *
c                                       to fxmain for input portions*
c                                       of the x matrix to fxdcon.  *
c                                       (n, ldsign)                 *
c      txn      REAL      input        -two-dimensional buffer sent *
c                                       to fxmain for output of     *
c                                       tx matrix from fxdcon.      *
c                                       (n, ldsign)                 *
c      n        INTEGER   input        -first dimension of tx and   *
c                                       txn buffers.                *
c      nn       INTEGER   nnnput        -half of n plus one          *
c      ntproc   INTEGER   output       -number of records processed *
c       +--------------------------------------------------------+  *
c       |                 development information                |  *
c       +--------------------------------------------------------+  *
c  defining geophysicist:                                           *
c                                                                   *
c  author:     s. manning                    origin date:  87/02    *
c  language:   fortran 77             date last compiled:  87/02    *
c       +--------------------------------------------------------+  *
c       |                  external environment                  |  *
c       +--------------------------------------------------------+  *
c  routine called by:                                               *
c      fxdc main                                                    *
c  routines called:                                                 *
c      daread        - read from disk routine                       *
c      dawrte        - disk write routine                           *
c      fxdcon        - fxdc main algorithm                          *
c      fptoi         - floating point to INTEGER*2 conversion       *
c      itofp         - INTEGER*2 to floating point conversion       *
c      move          - move byte routine                            *
c      riprnt        - prints record of ri's processed              *
c      rtape         - reads trace from input tape                  *
c      wrtape        - writes trace to output tape                  *
c                                                                   *
c  intrinsic functions called:   none                               *
c  files:                                                           *
c      lud  (output sequential) - store of dead trace samples       *
c  common:                                                          *
c      /fxparm/   - contains pertinent parameters from the input    *
c                   line header and the input card                  *
c      /fxlu/     - contains logical unit numbers                   *
c                                                                   *
c       +--------------------------------------------------------+  *
c       |             other documentaiton details                |  *
c       +--------------------------------------------------------+  *
c  error handling:  returns to main                                 *
c                                                                   *
c  general description:   handles spatial and temporal windowing    *
c                         for fxdc.  prepares data for input to     *
c                         fxdc's main algorithm.                    *
c                                                                   *
c  revised by:                                                      *
c                                                                   *
c******************* end of documentation package *******************
c
      subroutine fxmain (x, xhld, xn, nsmp, tx, txn, n, nn,
     &     ntproc,hdrsav,ITRWRD,ITHWP1,SZLNHD,SZSMPD, UnitSc )

c
*
#include <f77/sisdef.h>
#include <save_defs.h>
#include <f77/iounit.h>

      INTEGER MAXSMP, MAXTR,TRLEN,TRHLEN,ITRWRD,ITHWP1,SZLNHD,SZSMPD

      PARAMETER (MAXSMP=10000, MAXTR = 150)
      PARAMETER (TRLEN = MAXSMP)
      PARAMETER (MAXSM2= MAXSMP+MAXSMP)
      PARAMETER (TRHLEN = MAXSMP)

      INTEGER     input(TRLEN),rcmp
      INTEGER     i2hdr(TRHLEN),i2data(MAXSM2)
      INTEGER     hdrsav(ITRWRD,MAXTR)
      INTEGER     ilive(MAXTR)

      REAL  x(nsmp,*), xn(nsmp,*), tx(n,*), txn(n,*), diff(4096)
      real  xhld(nsmp,*), tramp(2048), sramp(101), taper(48)
      real  ftrmp(512), etrmp(512),temp(TRLEN), UnitSc
 
      LOGICAL  altape, last, tpeeof, receof
      LOGICAL  rlast,rfrst

      equivalence (input(1), i2hdr)

      DATA  pihalf/1.570796327/, ioldir/0/, last/.false./,
     &      tpeeof/.false./, receof/.false./,ltfold/0/, lteold/0/
      DATA rlast/.false./,rfrst/.true./,rcmp/0/,istat/0/

      common/fxlu/   lucrd, lud, lui, luo, luo2, lup
c
      common/fxparm/ ldsign, loper, lsovlp, lwndw, ltovlp,
     &               if1, if2, pw, imute, iisi, ifmt, altape,
     &               ntjb,mode,ntpr,pcnt

      SAVE

      call savelu('RecNum',ifmt_RecNum,l_RecNum,ln_RecNum,TRACEHEADER)
      call savelu('StaCor',ifmt_StaCor,l_StaCor,ln_StaCor,TRACEHEADER)

c +-------------------------------------------------------------------+
c | set up spatial parameters                                         |
c | loper2 is half the input operator length                          |
c | lsramp is the length of the overlap zone                          |
c | incrsw is the # of traces to move over to the next partition      |
c | nswndo is the calculated number of spatial partitions             |
c | itotal and nleft are used for placing the calculated partitions   |
c | ld1st and ldlast tell how many are in the 1st and last partitions |
c +-------------------------------------------------------------------+
      loper2 = (loper - 1) / 2
      mxread = ldsign + 2 * loper2
      mxr=mxread
      lsramp = ldsign * lsovlp / 100
      lsr=lsramp
      wsl    = lsramp
      incrsw = ldsign - lsramp
      incrs=incrsw
      spwndo = float(ntjb - lsramp + incrsw - 1) / float(incrsw)
      nswndo = spwndo
      nswn=nswndo
      itotal = nswndo * incrsw + lsramp
      nleft  = itotal - ntjb
      if (nleft .ne. 0) then
         ihalf = nleft / 2
         irem  = mod(nleft, 2)
         if (irem .eq. 0) then
            ld1st  = ldsign - ihalf
            ldlast = ld1st
         else
            ld1st  = ldsign - ihalf
            ldlast = ld1st - 1
         end if
      else
         ld1st  = ldsign
         ldlast = ldsign
      end if
      ld1=ld1st
      ldl=ldlast
c +----------------------------------------------------------------+
c |                                                                |
c | set up temporal parameters                                     |
c | lw is the # of samples ing the temporal window                 |
c | ltramp is the length of the overlap zone                       |
c | incrtw is the # of samples to move up for the next window      |
c +----------------------------------------------------------------+
      lw     = lwndw / iisi
      llw=lw
      ltramp = lw * ltovlp / 100
      ltrmp=ltramp
      ltrmp2 = ltramp / 2
      ltrp2=ltrmp2
      wtl    = ltramp
      incrtw = lw - ltramp
      incrt=incrtw
c +----------------------------------------------------------------+
c | calculate the temporal and spatial overlap zone ramp functions |
c +----------------------------------------------------------------+

      if(ltramp.gt.0)then
         wtl1=pihalf/(wtl+1)
         one=1.
         call vramp(one,one,tramp,1,ltramp)
         call vsmul(tramp,1,wtl1,tramp,1,ltramp)
         call vsin(tramp,1,tramp,1,ltramp)
         call vsq(tramp,1,tramp,1,ltramp)
      endif

      if(lsramp.gt.0)then
         wsl1=pihalf/(wsl+1)
         one=1.
         call vramp(one,one,sramp,1,lsramp)
         call vsmul(sramp,1,wsl1,sramp,1,lsramp)
         call vsin(sramp,1,sramp,1,lsramp)
         call vsq(sramp,1,sramp,1,lsramp)
      endif
c +--------------------------------------------------------+
c | if a mute retore is requested set up a ramp 48 ms long |
c +--------------------------------------------------------+
      if (imute .eq. 0) then

         ltpr = 48 / iisi
         do 6 i = 1, ltpr
            taper(i) = pihalf * float(i) / float(ltpr)
            taper(i) = sin (taper(i)) ** 2
    6    continue

      endif
c +----------------------------+
c | set up parameters for loop |
c +----------------------------+
      nrec   = 0
      ntrc   = 0
  155 lstrec = -9999
      istrt  = 1
      lpart  = 1
      isread = 1
      isdsgn = 1
      iadjst = 0
c +-----------------------------------+
c | begin loop for spatial partitions |
c +-----------------------------------+

  160 if (last) go to 60

c +-----------------------------------+
c | set the length for this partition |
c +-----------------------------------+
         if (lpart .eq. 1) then
            ld     = ld1st
            ldread = ld + loper2 + iadjst
         else
            ld     = ldsign
            ldread = ld + 2 * loper2 + iadjst
         end if
         incrsw = ld - lsramp
c
      if(lstrec.eq.-9999)then
         minliv = nsmp
         do 7 j = 1, istrt - 1
    7       if (minliv .gt. ilive(j)) minliv = ilive(j)
      endif
c +-------------------------------------------+
c | reading in input traces to partition loop |
c +-------------------------------------------+
         j    = istrt
         isxr = isread
  110    if (j.gt.ldread .or. last) go to 10

            if (tpeeof.or.receof) then
               last   = .true.
               ldread = j - 1
               ld     = ldlast
               incrsw = ld - lsramp
               go to 10
            end if
            if(.not.rlast)then
             call rtape (lui, input, nbyt)
            else
             nsmp64=nsmp+ITRWRD

             call vmov(temp,1,input,1,nsmp64)
             rlast=.false.
            endif
             if(rfrst)then
               call saver2(input,ifmt_RecNum,l_RecNum, ln_RecNum,
     1                     rcmp    , TRACEHEADER)
               rfrst=.false.
              endif
c +----------------------------------------------------------------+
c | if no more traces set last flag and reset partition parameters |
c +----------------------------------------------------------------+
            if (nbyt .eq. 0) then
               tpeeof = .true.
               last   = .true.
               ldread = j - 1
               if (nswndo .eq. 1) then
                  ld = min (ldread, ldsign)
               else
                  ld = min (ldread - loper2, ldsign)
               end if
               if (ld.eq.ldsign .and. ldread.gt.ldsign + loper2)
     &            last = .false.
c +------------------------------------------------------------+
c | if calculated number of traces doesn't match what was read |
c | output warning messages                                    |
c +------------------------------------------------------------+
               if ((nswndo.eq.1 .and. ldread.ne.ntjb) .or.
     &             (lpart.ne.nswndo .and. ld.ne.ldsign) .or.
     &             (lpart.eq.nswndo .and. ldread.ne.ldlast+loper2 .and.
     &              nswndo.ne.1)) then
                  write (lup, 111) ntjb, nrec, ntrc
  111             format (/,  1x, '*** m0111 *** error in fxdc ',
     &                            'fxmain.',
     &                    /, 14x, 'the number of traces in the job,(',
     &                            i5, ') calculated from',
     &                    /, 14x, '# of records and # of traces per ',
     &                            'record in the line ',
     &                    /, 14x, 'header, does not match the ',
     &                            'number of traces read.',
     &                    /, 14x, 'the line header information must ',
     &                            'match the data for ',
     &                    /, 14x, 'correct partitioning.  if you',
     &                            'correct the line header ',
     &                    /, 14x, 'by using (', i5, ') # of records ',
     &                            'and (', i5, ') # of traces per ',
     &                    /, 14x, 'record, your output may improve.')
               end if
               go to 10
            else
             call saver2(input,ifmt_RecNum,l_RecNum, ln_RecNum,
     1                   i2hdr106, TRACEHEADER)
              if(i2hdr106.ne.rcmp.and.mode.eq.1)then
                 rlast=.true.
                 last=.true.
                 receof=.true.
                 rcmp=i2hdr106
             call saver2(input,ifmt_RecNum,l_RecNum, ln_RecNum,
     1                   rcmp    , TRACEHEADER)
                 ldread=j-1
                 if(nswndo.eq.1)then
                   ld=min(ldread,ldsign)
                 else
                   ld=min(ldread-loper2,ldsign)
                 endif
                   nsmp64 = nsmp + ITRWRD
                 call vmov(input,1,temp,1,nsmp64)

                 if (ld.eq.ldsign .and. ldread.gt.ldsign + loper2)
     &            last = .false.
                 go to 10
              endif
            end if
c +-------------------------------+
c | update record and trace count |
c +-------------------------------+
            call saver2(input,ifmt_RecNum,l_RecNum, ln_RecNum,
     1                  iirec   , TRACEHEADER)
            if (iirec .ne. lstrec) then
               nrec = nrec + 1
               lstrec = iirec
               ntrc = 1
            else
               ntrc = ntrc + 1
            end if
c +----------------------------------------------------------------+
c | copy input to x matrix and check for minimum first live sample |
c +----------------------------------------------------------------+
            call saver2(input,ifmt_StaCor,l_StaCor, ln_StaCor,
     1                  istat   , TRACEHEADER)

            if (istat .ne. 30000) then
               if (ifmt .eq. 3)then

                  call vmov(input(ITHWP1),1,x(1,isxr),1,nsmp)
                 endif

               do 8 i = 1, nsmp
                  if (x(i,isxr) .ne. 0.0) then
                     ilive(isxr) = i
                     go to 9
                  end if
    8          continue
c +------------------------------------+
c | if trace is all zeros mark as dead |
c +------------------------------------+
               ilive(isxr) = nsmp

               call savew2(input,ifmt_StaCor,l_StaCor, ln_StaCor,
     1              30000   , TRACEHEADER)
    9          if (minliv .gt. ilive(isxr)) minliv = ilive(isxr)
            else
c +------------------------------------------------------+
c | if dead trace save on disk and zero matrix postition |
c +------------------------------------------------------+
               ilive(isxr) = nsmp
               call vclr(x(1,isxr),1,nsmp)
            endif
c +------------------+
c | save header info |
c +------------------+

         call vmov(input,1,hdrsav(1,isxr),1,ITRWRD)
         j    = j + 1
         isxr = isxr + 1
         if (isxr .gt. mxread) isxr = 1
         go to 110
   10    continue
c +------------------------------------------------------------------+
c |                                                                  |
c | complete setting temporal parameters                             |
c | nsamp is the # of samples in the traces from minimum live sample |
c | ntwndo is # of temporal windows                                  |
c | itotal and nleft used to calculate size of 1st and last windows  |
c +------------------------------------------------------------------+
         incrtw = lw - ltramp
         nsamp  = nsmp - minliv + 1
         ntwndo = float(nsamp - ltramp + incrtw - 1) / float(incrtw)
         itotal = ntwndo * incrtw + ltramp
         nleft  = itotal - nsamp
         if (nleft .ne. 0) then
            ihalf = nleft / 2
            irem  = mod (nleft, 2)
            if (irem .eq. 0) then
               lw1st  = lw - ihalf
               lwlast = lw1st
            else
               lw1st  = lw - ihalf
               lwlast = lw1st - 1
            end if
         else
            lw1st  = lw
            lwlast = lw
         endif
         if (lw1st  .gt. nsamp) lw1st  = nsamp
         if (lwlast .gt. nsamp) lwlast = nsamp
c +----------------------------------------------+
c | set indexing for outputting partiton results |
c +----------------------------------------------+
         if (lpart.ne.1 .or. (last .and. lpart.ne.1)) then
            istxn = loper2 + 1 + iadjst
            ietxn = loper2 + ld + iadjst
         else
            istxn = 1
            ietxn = ld
         endif
c +----------------------+
c | temporal window loop |
c +----------------------+
c        call move (0, xn(1,1), 0.0, 4*nsmp*ld)
         call vclr(xn(1,1),1,nsmp*ld)
         if (ntwndo .eq. 0) ntwndo = 1
         do 35 k = 1, ntwndo
c +-------------------+
c | set window length |
c +-------------------+
            if (k .eq. 1) then
               lwn = lw1st
               l   = minliv
            else if (k .eq. ntwndo) then
               lwn = lwlast
               l   = l + incrtw
            else
               lwn = lw
               l   = l + incrtw
            end if
            incrtw = lwn - ltramp
c +---------------------------------------------------+
c | set lead and end to fill to frequency window size |
c +---------------------------------------------------+
            ihalf = (n - lwn) / 2
            irem  = mod (n - lwn, 2)
            if (irem .eq. 0) then
               ifront = ihalf
               iend   = ihalf
            else
               ifront = ihalf + 1
               iend   = ihalf
            endif
c +-------------------------------------------------------------+
c | set leading and end zeros and leading and ending ramp sizes |
c +-------------------------------------------------------------+
            ltfrmp = min (ltrmp2, ifront)
            nfzero = ifront - ltfrmp
            ltermp = min (ltrmp2, iend)
            nezero = iend - ltermp
            if (k .eq. 1) then
               ltfrmp = 0
               nfzero = ifront
            end if
            if (k .eq. ntwndo) then
               ltermp = 0
               nezero = iend
            endif
c +--------------------------------+
c | build leading and ending ramps |
c +--------------------------------+
            if (ltfrmp .ne. ltfold) then
                 wtpr=pihalf/float(ltfrmp+1)
                 one=1.
                 call vramp(one,one,ftrmp,1,ltfrmp)
                 call vsmul(ftrmp,1,wtpr,ftrmp,1,ltfrmp)
                 call vsin(ftrmp,1,ftrmp,1,ltfrmp)
                 call vsq(ftrmp,1,ftrmp,1,ltfrmp)
               ltfold = ltfrmp
            endif
c
            if (ltermp .ne. lteold) then
                 wtpr=pihalf/float(ltermp+1)
                 one=1.
                 call vramp(one,one,etrmp(ltermp),-1,ltermp)
               call vsmul(etrmp,1,wtpr,etrmp,1,ltermp)
               call vsin(etrmp,1,etrmp,1,ltermp)
               call vsq(etrmp,1,etrmp,1,ltermp)
               lteold = ltermp
            end if
c +-----------------------------------------+
c | set index for x matrix to window matrix |
c +-----------------------------------------+
            lend = l + lwn - 1
            if (lend .gt. nsmp) lend = nsmp
c +-----------------------------------+
c | loop for setting up window matrix |
c +-----------------------------------+
            isxr = isdsgn
            do 20 j = 1, ldread
               istx = 1
               ietx = nfzero
c +---------------------+
c | count leading zeros |
c +---------------------+
               call vclr(tx(istx,j),1,(ietx-istx+1))
               istx = ietx + 1
               ietx = istx + ltfrmp - 1
c +--------------+
c | leading ramp |
c +--------------+
               m = 0
            if(istx.le.ietx) then
               call vmul(x(l-ltfrmp+m,isxr),1,ftrmp(m+1),1,
     &          tx(istx,j),1,ietx-istx+1)
            endif
               istx = ietx + 1
               ietx = istx + lwn - 1
c +-------------+
c | actual data |
c +-------------+
               call vmov(x(l,isxr),1,tx(istx,j),1,(ietx-istx+1))
               istx = ietx + 1
               ietx = istx + ltermp - 1
c +-------------+
c | ending ramp |
c +-------------+
               m = 0
            if(istx.le.ietx) then
               call vmul(x(l+lwn+m,isxr),1,etrmp(m+1),1,
     &          tx(istx,j),1,ietx-istx+1)
            endif
               istx = ietx + 1
               ietx = istx + nezero - 1
c +--------------+
c | ending zeros |
c +--------------+
               call vclr(tx(istx,j),1,(ietx-istx+1))
               isxr = isxr + 1
               if (isxr .gt. mxread) isxr = 1
   20       continue
c +--------------------+
c | process the window |
c +--------------------+
            ilast = 0
            call fxdcon (tx, ldread, n, nn, txn, ilast,
     1                   SZLNHD,ITHWP1,ITRWRD,SZSMPD,UnitSc)
c +---------------------------------------------+
c | move the output window to the output matrix |
c +---------------------------------------------+
            do 30 j = istxn, ietxn
               if (k .eq. 1) then
                  if (ltramp .gt. 0)
     &                 call vmov (txn(ifront+1,j),1,xn(l,j-istxn+1),1,
     :                 ltramp)
               else
c +-------------------------------------+
c | apply temporal ramp in overlap zone |
c +-------------------------------------+
                  do 25 i = 1, ltramp
                     ki      = l + i - 1
                     m       = ltramp - i + 1
                     xn(ki,j-istxn+1) = xn(ki,j-istxn+1) * tramp(m) +
     &                                  txn(ifront+i,j)  * tramp(i)
   25             continue
               endif
c +------------------------------------------------+
c | move in remaining window data to output matrix |
c +------------------------------------------------+
               nval = lend - (l + ltramp) + 1
               if (nval .gt. 0)
     &call vmov(txn(ltramp+ifront+1,j),1,xn(ltramp+l,j-istxn+1),1,nval)
   30       continue
   35    continue
c +----------------------------------------------------------------+
c | apply gain correction at front and ending edges of the dataset |
c +----------------------------------------------------------------+
         if (lpart.eq.1 .or. iadjst.ne.0) then
            if (iadjst .ne. 0) then
               isxn = loper2 + 1 + iadjst
               m    = 0
            else
               isxn = 2
               m    = 1
            end if
c
            isxr = isdsgn + isxn - 1
            if (isxr .gt. mxread) isxr = isxr - mxread
              ncnt=nsmp-minliv+1
            do 44 j = isxn, loper2
               c1 = 0.
               c2 = 0.
               m  = m + 1
            call dotpr(x(minliv,isxr),1,xn(minliv,m),1,c1,ncnt)
            call svesq(xn(minliv,m),1,c2,ncnt)
               if (c2 .ne. 0.) then
                  c = c1 / c2
                  call vsmul(xn(minliv,m),1,c,xn(minliv,m),1,ncnt)
               end if
               isxr = isxr + 1
               if (isxr .gt. mxread) isxr = 1
   44       continue
         end if
         if (last) then
            m    = ld - loper2
            isxr = isdsgn + ldread - loper2
            if (isxr .gt. mxread) isxr = isxr - mxread
               ncnt=nsmp-minliv+1
            do 48 j = ldread - loper2 + 1, ldread - 1
               m = m + 1
               if (j .gt. 0) then
                  c1 = 0.
                  c2 = 0.
            call dotpr(x(minliv,isxr),1,xn(minliv,m),1,c1,ncnt)
            call svesq(xn(minliv,m),1,c2,ncnt)
                  if (c2 .ne. 0.) then
                     c = c1 / c2
                  call vsmul(xn(minliv,m),1,c,xn(minliv,m),1,ncnt)
                  end if
               end if
               isxr = isxr + 1
               if (isxr .gt. mxread) isxr = 1
   48       continue
         end if
  200 continue
c +--------------------------------------------+
c | apply spatial ramp to spatial overlap zone |
c +--------------------------------------------+
         if (lpart .ne. 1) then
               ncnt=nsmp-minliv+1
            do 41 j = 1, lsramp
               m = lsramp - j + 1
               call vmma(xhld(minliv,j),1,sramp(m),0,xn(minliv,j),1,
     &              sramp(j),0,xn(minliv,j),1,ncnt)
   41       continue
         end if
c +-------------------------------------------+
c | set indexing for generating noise dataset |
c +-------------------------------------------+
         if (last) incrsw = ld
         if (lpart.ne.1 .or. (last .and. lpart.ne.1)) then
            isx = loper2 + 1 + iadjst
            iex = loper2 + incrsw + iadjst
         else
            isx = 1
            iex = incrsw
         end if
c +-----------------------------------+
c | loop for output of partition data |
c +-----------------------------------+
         isxr = isdsgn + isx - 1
         if (isxr .gt. mxread) isxr = isxr - mxread
         do 55 j = isx, iex
            if (altape) then
c +-----------------------------+
c | generate noise dataset data |
c +-----------------------------+
               call vsub(x(1,isxr),1,xn(1,j-isx+1),1,diff,1,nsmp)
            endif
c +--------------------------------------+
c | add original data back, if requested |
c +--------------------------------------+
         if(pcnt.ne.0.0)then
            ksx=j-isx+1
            call vma(x(1,isxr),1,pcnt,0,xn(1,ksx),1,xn(1,ksx),1,nsmp)
         endif
c +----------------------------------------------+
c | move saved header to input buffer for output |
c +----------------------------------------------+
            call vmov(hdrsav(1,isxr),1,input,1,ITRWRD)
            call saver2(input,ifmt_StaCor,l_StaCor, ln_StaCor,
     1                  i2hdr125, TRACEHEADER)

            if (i2hdr125 .ne. 30000) then
               if (imute .eq. 0) then
c +---------------------------------------------------+
c | restore mute if requested (ramp into output data) |
c +---------------------------------------------------+
                  m = ilive(isxr) - (ltpr / 2)
                  if (m .le. 1) then
                      m = 2
                  endif
                  call vclr(xn(1,j-isx+1),1,(m-1))

                  do 51 i = 1, ltpr
                     xn(m,j-isx+1) = xn(m,j-isx+1) * taper(i)
                     m = m + 1
   51             continue

               end if
c +--------------------------------------+
c | move data to input buffer for output |
c +--------------------------------------+
               if (ifmt .eq. 3)
     &            call vmov (xn(1,j-isx+1),1,input(ITHWP1),1,nsmp)
               if (ifmt .eq. 1)
     &            call fptoi (i2data, xn(1,j-isx+1), nsmp, itt)
            else
c +-----------------------------------------------------+
c | pick up saved dead trace if header said it was dead |
c +-----------------------------------------------------+
               nbyt = nsmp * ISZBYT + SZTRHD
	       call vclr(input(ITHWP1), 1, nsmp)
            endif
            nbyt = (nsmp + ITRWRD) * SZSMPD
c +-------------------------------+
c | write trace to output dataset |
c +-------------------------------+
            call wrtape (luo, input, nbyt)
            call saver2(input,ifmt_RecNum,l_RecNum, ln_RecNum,
     1                  ir      , TRACEHEADER)
c +---------------------------+
c | update record information |
c +---------------------------+
            if (ir .ne. ioldir) then
               call riprnt (ir, LERR)
               ioldir = ir
               ntproc = ntproc + 1
            end if
c +--------------------------------------------+
c | move noise data to input buffer for output |
c +--------------------------------------------+
            call saver2(input,ifmt_StaCor,l_StaCor, ln_StaCor,
     1                  i2hdr125, TRACEHEADER)
            if (altape) then
               if (i2hdr125 .ne. 30000) then
                  call vmov(diff,1,input(ITHWP1),1,nsmp)
               else
                  call vclr(input(ITHWP1),1,nsmp)
               end if
               nbyt = (nsmp + ITRWRD) * SZSMPD
c +---------------------------------------+
c | write noise to special output dataset |
c +---------------------------------------+
               call wrtape (luo2, input, nbyt)
            end if
            isxr = isxr + 1
            if (isxr .gt. mxread) isxr = 1
   55    continue
c +-----------------------------------+
c | save data in spatial overlap zone |
c +-----------------------------------+
         if (.not. last) then
            do 57 j = incrsw + 1, ld
               call vmov(xn(1,j),1,xhld(1,j-incrsw),1,nsmp)
   57       continue
c
            isxr = incrsw + isx - loper2
            iex  = ldread
            if (isxr .lt. 1) then
               iadjst = incrsw + isx - loper2 - 1
               isxr   = 1
            else if (isxr .gt. 1) then
               iadjst = 0
            end if
c +--------------------------------------+
c | increment spatial looping parameters |
c +--------------------------------------+
            isread = isdsgn + ldread
            if (isread .gt. mxread) isread = isread - mxread
            isdsgn = isxr + isdsgn - 1
            if (isdsgn .gt. mxread) isdsgn = isdsgn - mxread
            istrt  = iex - isxr + 2
            lpart  = lpart + 1
         endif
c +--------------------------------------------------------------+
c | if record sequential, reset parameters if not at end of data |
c +--------------------------------------------------------------+
            if(last.and..not.tpeeof)then
               rlast=.true.
               receof=.false.
               last=.false.
               mxread=mxr
               lsramp=lsr
               wsl=lsramp
               nswndo=nswn
               incrsw=incrs
               ld1st=ld1
               ldlast=ldl
               lw=llw
               ltramp=ltrmp
               ltrmp2=ltrp2
               wtl=ltramp
               incrtw=incrt
               go to 155
            else
               go to 160
            endif
   60 continue
c
 5000 continue
      return
      end
c
cccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc
c
      subroutine fxdcon (x, ld, n, nn, xn, ilast,
     1                   SZLNHD,ITHWP1,ITRWRD,SZSMPD,UnitSc)
#include <f77/iounit.h>
      integer   ITRWRD,ITHWP1,SZLNHD,SZSMPD
      complex   y, g, z, ops, tmp
      complex   w1, w2
      REAL      x(n,*), xn(n,*), UnitSc
      LOGICAL   first, altape
      POINTER (py, y(1))
      pointer (pg,g(1))
      pointer (pz,z(1))
      POINTER (pops, ops(1))
      pointer (ptmp,tmp(1))
      POINTER (tw1, w1(1))
      pointer (tw2, w2(1))
      integer error, abort
      DATA      first /.true./
      common/fxparm/ ldsign, loper, lsovlp, lwndw, ltovlp,
     &               if1, if2, pw, imute, iisi, ifmt, altape,
     &               ntjb,mode,ntpr,pcnt
      SAVE first,py,pg,pz,pops,ptmp,tw1,tw2
c
      ISZBYT = SZSMPD

      if(ilast.eq.1)then
        call gfree(py)
        call gfree(pg)
        call gfree(pz)
        call gfree(pops)
        call gfree(ptmp)
        call gfree(tw1)
        call gfree(tw2)
        return
      endif
      if (first) then
         lds8   = (ldsign + (loper - 1)) * 2
         leny   = nn * lds8
         leng   = leny
         lenz   = leny
         lenops = leny
         lentmp = leny * 2

         error = 0
         abort = 0
         ierror = 0
         call galloc(py,leny*ISZBYT,error,abort)
         if (error.ne.0) ierror = 1
         call galloc(pg,leng*ISZBYT,error,abort)
         if (error.ne.0) ierror = 1
         call galloc(pz,lenz*ISZBYT,error,abort)
         if (error.ne.0) ierror = 1
         call galloc(pops,lenops*ISZBYT,error,abort)
         if (error.ne.0) ierror = 1
         call galloc(ptmp,lentmp*ISZBYT,error,abort)
         if (error.ne.0) ierror = 1
         call galloc(tw1,lenops*ISZBYT,error,abort)
         if (error.ne.0) ierror = 1
         call galloc(tw2,lenops*ISZBYT,error,abort)
         if (error.ne.0) ierror = 1
         if(ierror.ne.0)then
            write(LERR,*)' Error trying to allocate memory'
            call ccexit(100)
         endif
c
         first = .false.
      endif
c
      call fxdcex (x, ld, n, nn, xn, y(1), g(1),
     &             z(1), ops(1), tmp(1),
     &             w1(1), w2(1), UnitSc)
c
      return
      end
c
c*********************************************************************
c
      subroutine fxdcex (x, ld, n, nn, xn, y, g, z, ops, tmp,
     &                   peop, op, UnitSc)
c
#include <f77/lhdrsz.h>
#include <f77/iounit.h>
      complex   y(nn,*), g(nn,*), z(nn,*), ops(nn,*), peop(nn,*),
     &          op(nn,*), tmp(nn,*), cpw, czero, cone, chalf
      complex conepw
      REAL      x(n,*), xn(n,*), fener(2049), waat(150)
      LOGICAL   altape, first
      DATA      czero/(0.,0.)/, cone/(1.,0.)/, chalf/(.5,0.)/,
     &          first/.true./
      common/fxlu/   lucrd, lud, lui, luo, luo2, lup
      common/fxparm/ ldsign, loper, lsovlp, lwndw, ltovlp,
     &               if1, if2, pw, imute, iisi, ifmt, altape,
     &               ntjb,mode,ntpr,pcnt
c
      SAVE first,n1,n12,ntrms,ntrms1,fnyqst,nstrt,nend,nvals,
     &         nvals2,nbyte,cpw,conepw,czero,cone,chalf
 
      if (first) then
         n1     = n
         n12    = nn
c
         ntrms  = loper / 2
         ntrms1 = ntrms + 1
c
         dt = float(iisi) * UnitSc
         fnyqst = 1. / (2. * dt)
         nstrt  = (float(n12 - 1) * float(if1) / fnyqst) + 1
         nend   = (float(n12 - 1) * float(if2) / fnyqst) + 1
         nvals  = nend - nstrt + 1
         nvals2 = nvals+nvals
         nbyte  = nvals * 4
c
         cpw    = cmplx(pw/100.,0.0)
         conepw = cone+cpw
         first = .false.
      end if
c +---------------------------------------+
c | the transform used is REAL to complex |
c | transform each trace in design window |
c +---------------------------------------+
      do 12 j = 1, ld
         call rfftb (x(1,j), y(1,j), n1, 1)
         call rfftsc (y(1,j), n1, 3, 1)
   12 continue
c
      call vclr(fener(nstrt),1,nvals)
c +--------------------------------------+
c | auto correlation of the fft'd traces |
c +--------------------------------------+
      call cacor (y, ld, n12, nstrt, nend, tmp, lc)
      istr = lc / 2 + 1
      xlen = ld
      do 22 j = istr, lc
         xlenr=1./xlen
         call cvsmul(tmp(nstrt,j),2,xlenr,tmp(nstrt,j),2,nvals)
   22    xlen = xlen - 1
c
      istr1 = istr + 1
      last  = istr + ntrms - 1
      last1 = last + 1
      k = 0
      do 24 j = istr, last
         k    = k + 1
 24      call dcopy (nvals, tmp(nstrt,j), 1, z(nstrt,k), 1)
         call cvmul(z(nstrt,1),2,conepw,0,z(nstrt,1),2,
     &        nvals,1)
         k = 0
         do 105 L = istr1, last1
            k    = k + 1
 105        call dcopy (nvals, tmp(nstrt,L), 1, g(nstrt,k), 1)

      call vclr(op,1,n12*ld*2)

      call vclr(peop,1,n12*ld*2)
      call cwlev (z, peop, ntrms, op, g, n12, nstrt, nend)
      do 26 j = ntrms, 1, -1
         call cvmul(op(nstrt,j),2,chalf,0,ops(nstrt,ntrms1-j),2,
     &      nvals,1)
   26 continue
      do 27 L = loper, loper - ntrms + 1, -1
            call cvconj(ops(nstrt,loper-L+1),2,ops(nstrt,L),2,nvals)
            call cvmgsa(ops(nstrt,L),2,fener(nstrt),1,fener(nstrt),
     & 1,nvals)
   27 continue

      call vclr(ops(nstrt,ntrms1),1,nvals*2)
c +-----------------------------------------------------------------+
c | convolve each operator with respective f-components and inverse |
c | transform.  print output                                        |
c +-----------------------------------------------------------------+
      call cfoldm (ld, y, loper, ops, n12, nstrt, nend, lc, tmp)
c
      jtrms = lc - ntrms
      l     = 0
      do 52 k = ntrms1, jtrms
         L = L + 1
         call dcopy (nvals, tmp(nstrt,k), 1, y(nstrt,L), 1)
   52 continue
c
      do 58 i = nstrt, nend
         if (fener(i) .gt. 0.) then
            sum = 0.
            do 56 k = 1, ntrms
               waat(k) = 2. - sum / fener(i)
               y(i,k)  = y(i,k) * waat(k)
               sum     = sum + cabs(ops(i,ntrms1-k)) **2
   56       continue
            do 57 L = ld, ld - ntrms + 1, -1
               y(i,L) = y(i,L) * waat(ld-L+1)
   57       continue
         end if
   58 continue
      do 68 j = 1, ld
         call rfftsc (y(1,j), n1, -3, 0)
         call rfftb (y(1,j), xn(1,j), n1, -1)
   68 continue
      return
      end
c *******************************************************************
c |                                                                 |
c |   routine to do complex wiener levinson recursion               |
c |                                                                 |
c |   original 1-d by t scheuer                                     |
c |                                                                 |
c |               r(n) = the complex autocorrelation vector         |
c |               a(n) = the complex prediction error operator      |
c |               g(n) = the complex right hand side vector         |
c |               f(n) = the complex solution (prediction filter)   |
c |                 n  = the dimension of the system                |
c |                                                                 |
c |   complex levinson recursion routine                            |
c |                                                                 |
c *******************************************************************
      subroutine cwlev (r, a, n, f, g, n1, nstrt, nend)
#include <f77/lhdrsz.h>
      complex   r(n1,*), a(n1,*), f(n1,*), g(n1,*), c0(4096), c1(4096),
     &          c2(4096), store(4096), c(4096), v(4096), czero
      complex store2(4096),cpone,cnone
      real cabs,abstmp
      LOGICAL   zeros(4096), fals(4096)
      DATA      fals/4096*.false./
      common/fxlu/   lucrd, lud, lui, luo, luo2, lup
c
      SAVE fals
 
      czero = cmplx(0.,0.)
      cpone=cmplx(1.,0.)
      cnone=cmplx(-1.,0.)
      nval = nend - nstrt + 1
      nbyt = nval * 2
c     call move (1, zeros, fals, 4096)
      do 15 ij=1, 4096
        zeros(ij)=fals(ij)
   15 continue
c +-----------------------------------------------------------------+
c |   normalize by dividing both sides by r(1)                      |
c +-----------------------------------------------------------------+
      call dcopy (nval, r(nstrt,1), 1, c(nstrt), 1)
c
      do 10 i = 1, n
         do 110 k = nstrt, nend
            if (r(k,1) .eq. czero) then
               zeros(k) = .true.
            else
               g(k,i) = g(k,i) / c(k)
               r(k,i) = r(k,i) / c(k)
            end if
  110    continue
   10 continue
c +--------------------+
c | initialize buffers |
c +--------------------+
         call cvfill(cpone,a(nstrt,1),2,nval)
         call cvfill(cnone,c(nstrt),2,nval)
         call cvfill(cpone,v(nstrt),2,nval)
c +---------------------------------------------+
c |   start the iteration of levinson recursion |
c +---------------------------------------------+
      call dcopy (nval, g(nstrt,1), 1, f(nstrt,1), 1)
c
      do 50 j = 2, n

         call vclr(a(nstrt,j),1,nbyt)

         call vclr(f(nstrt,j),1,nbyt)

         call vclr(c0(nstrt),1,nbyt)

         call vclr(c1(nstrt),1,nbyt)
c
         do 20 i = 2, j
         call cvma(r(nstrt,i),2,a(nstrt,j-i+1),2,
     % c0(nstrt),2,c0(nstrt),2,nval,1)
         call cvma(r(nstrt,i),2,f(nstrt,j-i+1),2,
     % c1(nstrt),2,c1(nstrt),2,nval,1)
   20    continue
         do 220 k = nstrt, nend
            c(k)  = c0(k) / v(k)

        abstmp = cabs(c(k))
            if (abstmp .gt. 1.) c(k) = c(k) / abstmp
            v(k)  = v(k) - c0(k) * conjg(c(k))
            if (v(k) .eq. 0.) v(k) = cmplx(0.0001, 0.0)
            c2(k) = (c1(k) - g(k,j)) / v(k)

        abstmp = cabs(c2(k))
            if (abstmp .gt. 1.) c2(k) = c2(k) / abstmp
  220    continue
c
         jh = (j + 1) / 2
         do 30 i = 1, jh
         call cvmul(a(nstrt,i),2,c(nstrt),2,store(nstrt),2,nval,-1)
         call cvsub(a(nstrt,j-i+1),2,store(nstrt),2,store(nstrt),2,
     &    nval)
         call cvmul(a(nstrt,j-i+1),2,c(nstrt),2,store2(nstrt),2,
     &    nval,-1)
         call cvsub(a(nstrt,i),2,store2(nstrt),2,a(nstrt,i),2,
     &   nval)
         call cvmov(store(nstrt),2,a(nstrt,j-i+1),2,nval)
   30    continue
c
         do 40 i = 1, jh
         call cvmul(a(nstrt,i),2,c2(nstrt),2,store(nstrt),2,nval,-1)
         call cvsub(f(nstrt,j-i+1),2,store(nstrt),2,store(nstrt),2,
     &    nval)
         call cvmul(a(nstrt,j-i+1),2,c2(nstrt),2,store2(nstrt),2,
     &    nval,-1)
         call cvsub(f(nstrt,i),2,store2(nstrt),2,f(nstrt,i),2,
     &   nval)
         call cvmov(store(nstrt),2,f(nstrt,j-i+1),2,nval)
   40    continue
c
   50 continue
c
      do 60 i = nstrt, nend
         if (zeros(i)) then
            do 55 j = 1, n
   55          f(i,j) = czero
         end if
   60 continue
c
      return
      end
c
c ****************************************************************
c |       subroutine to perform complex convolution              |
c |  created as modification of robinson's fold subroutine       |
c ****************************************************************
c
      subroutine cfoldm (la, a, lb, b, n1, nstrt, nend, lc, c)
#include <f77/lhdrsz.h>
      complex a(n1,*), b(n1,*), c(n1,*)
      SAVE
 
      lc    = la + lb - 1
      nval=nend-nstrt+1
c     call move (0, c, 0.0, lc*8*n1)
      call vclr(c,1,lc*n1*2)
      do 1 i = 1, la
         do 2 j = 1, lb
            k = i + j - 1
            call cvma(a(nstrt,i),2,b(nstrt,j),2,c(nstrt,k),2,
     &           c(nstrt,k),2,nval,1)
    2 continue
    1 continue
      return
      end
c ****************************************************************
c |       subroutine to perform complex correlation              |
c |  created as modification of robinson's fold subroutine       |
c ****************************************************************
      subroutine cacor (a, len, n, nstrt, nend, c, lc)
#include <f77/lhdrsz.h>
      complex a(n,*), c(n,*)
      SAVE
 
      lc = len * 2 - 1
      nval=nend-nstrt-1

      call vclr(c,1,lc*2*n)
c
      do 1 i = 1, len
         do 2 j = len, 1, -1
            k = i + len - j
            call cvma(a(nstrt,i),2,a(nstrt,j),2,c(nstrt,k),2,
     &           c(nstrt,k),2,nval,-1)
    2    continue
    1 continue
c
      return
      end
c
      subroutine prmchk (card, nsmp, UnitSc, ierr)
c
#include <f77/lhdrsz.h>
      LOGICAL altape
      CHARACTER card*80,iblnk2*2,lsodef*2,ltodef*2
      common/fxlu/   lucrd, lud, lui, luo, luo2, lup
      common/fxparm/ ldsign, loper, lsovlp, lwndw, ltovlp,
     &               if1, if2, pw, imute, iisi, ifmt, altape,
     &               ntjb,mode,ntpr,pcnt
      DATA iblnk2/'  '/
c +--------------------------------------------------------------+
c |   read those default parameters for which a blank does not |
c |   default to zero.                                           |
c +--------------------------------------------------------------+
      if (card(1:1) .ne. ' ') then
          read (unit=card,fmt=100) lsodef,ltodef
  100     format(12x,a2,6x,a2,59x)
      endif
c
      if (ldsign .eq. 0) ldsign = 15
      if (ldsign.lt.5 .or. ldsign.gt.101) then
         write (lup, 115) ldsign
  115    format (/,  1x, '*** m0115 *** error in fxdc prmchk.',
     &           /, 14x, 'length of design partition', i2, ' ',
     &                   'exceeds minimum (5) or',
     &           /, 14x, 'maximum (101).  fix card input before ',
     &                   'resubmitting this job.')
         ierr = ierr + 1
      end if
c
      if (loper .eq. 0) then
         loper = ldsign / 2
         if (mod(loper, 2) .eq. 0) loper = loper + 1
      end if
      if (mod(loper, 2) .eq. 0) then
         write (lup, 119) loper
  119    format (/,  1x, '*** m0119 *** error in fxdc prmchk.',
     &           /, 14x, 'length of operator ', i2, ' must be ',
     &                   'an odd integer.',
     &           /, 14x, 'fix card input before resubmitting this ',
     &                   'job.')
         ierr = ierr + 1
      end if
      if (loper.lt.3 .or. loper.gt.ldsign) then
         write (lup, 120) loper, ldsign
  120    format (/,  1x, '*** m0120 *** error in fxdc prmchk.',
     &           /, 14x, 'length of operator ', i2, ' exceeds ',
     &                   'minimum (3) or maximum (length',
     &           /, 14x, 'of design=', i2, ').  fix card input ',
     &                   'before resubmitting this job.')
         ierr = ierr + 1
      end if
c
      if (lsodef .eq. iblnk2) lsovlp = 50
      if (lsovlp.lt.0 .or. lsovlp.gt.50) then
         write (lup, 122) lsovlp
  122    format (/,  1x, '*** m0122 *** error in fxdc prmchk.',
     &           /, 14x, 'spatial overlap percent', i3, ' exceeds ',
     &                   'minimum (0) or ',
     &           /, 14x, 'maximum (50).  fix card input before ',
     &                   'resubmitting this job.')
         ierr = ierr + 1
      end if
c
      call power2 (nsmp, n, nn, lup, ier)

      if (n .gt. 4096) then
        n = 4096
        nn = 2049
      endif
      lw = lwndw / iisi
      if (lw .eq. 0) then
         lw    = nsmp
         if (lw .gt. n) lw = n
         lwndw = lw * iisi
      end if
      if (lw.lt.65 .or. lw.gt.n) then
         write (lup, 123) lw, n
  123    format (/,  1x, '*** m0123 *** error in fxdc prmchk.',
     &           /, 14x, 'length of temporal window in samples ', i4,
     &                   ' is less than minimum (65) or',
     &           /, 14x, 'greater than maximum (minimum of 4096 '
     &                   'and power of 2 greater than trace',
     &           /, 14x, 'length=', i4, ').  fix card input before ',
     &                   'resubmitting this job.')
         ierr = ierr + 1
      end if
c
      if (ltodef .eq. iblnk2) ltovlp = 50
      if (ltovlp.lt.0 .or. ltovlp.gt.50) then
         write (lup, 124) ltovlp
  124    format (/,  1x, '*** m0124 *** error in fxdc prmchk.',
     &           /, 14x, 'temporal overlap percent', i3, ' exceeds ',
     &                   'minimum (0) or ',
     &           /, 14x, 'maximum (50).  fix card input before ',
     &                   'resubmitting this job.')
         ierr = ierr + 1
      end if
c
      ifnyq = 1. / (2. * float(iisi)*UnitSc)
      if (if2 .eq. 0) if2 = ifnyq
      if (if1.lt.0 .or. if1.gt.if2) then
         write (lup, 125) if1, if2
  125    format (/,  1x, '*** m0125 *** error in fxdc prmchk.',
     &           /, 14x, 'minimum frequency to use ', i4, ' exceeds ',
     &                   'minimum (0) or maximum',
     &           /, 14x, '(maximum frequency to use=', i4, ').  ',
     &                   'fix card input before ',
     &           /, 14x, 'resubmitting this job.')
         ierr = ierr + 1
      end if
c
      if (if2.lt.if1 .or. if2.gt.ifnyq) then
         write (lup, 130) if2, if1, ifnyq
  130    format (/,  1x, '*** m0130 *** error in fxdc prmchk.',
     &           /, 14x, 'maximum frequency to use ', i4, ' exceeds ',
     &                   'minimum (minimum frequency ',
     &           /, 14x, 'to use=', i4, ') or maximum (nyquist=', i4,
     &                   ').  fix card input before ',
     &           /, 14x, 'resubmitting this job.')
         ierr = ierr + 1
      end if
c
      if (pw.lt.0. .or. pw.gt.100.) then
         write (lup, 135) pw
  135    format (/,  1x, '*** m0135 *** error in fxdc prmchk.',
     &           /, 14x, 'pre-whitening percent ', f5.2, ' exceeds ',
     &                   'minimum (0) or maximum (100).',
     &           /, 14x, 'fix card input before ',
     &                   'resubmitting this job.')
         ierr = ierr + 1
      end if
c
      if (imute.ne.0 .and. imute.ne.1) then
         write (lup, 140) imute
  140    format (/,  1x, '*** m0140 *** error in fxdc prmchk.',
     &           /, 14x, 'preserve early mute flag ', i1, ' has an ',
     &                   'invalid value.  should a 0 or a 1.',
     &           /, 14x, 'fix card input before ',
     &                   'resubmitting this job.')
         ierr = ierr + 1
      end if
      if (mode.ne.0 .and. mode.ne.1) then
         write (lup, 145) mode
  145    format (/,  1x, '*** m0145 *** error in fxdc prmchk.',
     &           /, 14x, 'processing mode flag ', i1, ' has an ',
     &                   'invalid value.  should a 0 or a 1.',
     &           /, 14x, 'fix card input before ',
     &                   'resubmitting this job.')
         ierr = ierr + 1
      end if
c
      return
      end
