C***********************************************************************
C                 copyright 2001, Amoco Production Company             *
C                             All Rights Reserved                      *
C                     an affiliate of BP America Inc.                  *
C***********************************************************************



* ======================================================================
* NIST Guide to Available Math Software.
* Source for module FFT from package GO.
* Retrieved from NETLIB on Wed Jul  5 11:50:07 1995.
* ======================================================================
      subroutine mfft(a,b,ntot,n,nspan,isn)
c  multivariate complex fourier transform, computed in place
c    using mixed-radix fast fourier transform algorithm.
c  by r. c. singleton, stanford research institute, sept. 1968
c  arrays a and b originally hold the real and imaginary
c    components of the data, and return the real and
c    imaginary components of the resulting fourier coefficients.
c  multivariate data is indexed according to the fortran
c    array element successor function, without limit
c    on the number of implied multiple subscripts.
c    the subroutine is called once for each variate.
c    the calls for a multivariate transform may be in any order.
c  ntot is the total number of complex data values.
c  n is the dimension of the current variable.
c  nspan/n is the spacing of consecutive data values
c    while indexing the current variable.
c  the sign of isn determines the sign of the complex
c    exponential, and the magnitude of isn is normally one.
c  a tri-variate transform with a(n1,n2,n3), b(n1,n2,n3)
c    is computed by
c      call mfft(a,b,n1*n2*n3,n1,n1,1)
c      call mfft(a,b,n1*n2*n3,n2,n1*n2,1)
c      call mfft(a,b,n1*n2*n3,n3,n1*n2*n3,1)
c  for a single-variate transform,
c    ntot = n = nspan = (number of complex data values), e.g.
c      call mfft(a,b,n,n,n,1)
c  the data can alternatively be stored in a single complex array c
c    in standard fortran fashion, i.e. alternating real and imaginary
c    parts. then with most fortran compilers, the complex array c can
c    be equivalenced to a real array a, the magnitude of isn changed
c    to two to give correct indexing increment, and a(1) and a(2) used
c    to pass the initial addresses for the sequences of real and
c    imaginary values, e.g.
c       complex c(ntot)
c       real    a(2*ntot)
c       equivalence (c(1),a(1))
c       call mfft(a(1),a(2),ntot,n,nspan,2)
c  arrays at(maxf), ck(maxf), bt(maxf), sk(maxf), and np(maxp)
c    are used for temporary storage.  if the available storage
c    is insufficient, the program is terminated by a stop.
c    maxf must be .ge. the maximum prime factor of n.
c    maxp must be .gt. the number of prime factors of n.
c    in addition, if the square-free portion k of n has two or
c    more prime factors, then maxp must be .ge. k-1.
      dimension a(1),b(1)
c  array storage in nfac for a maximum of 15 prime factors of n.
c  if n has more than one square-free factor, the product of the
c    square-free factors must be .le. 210
      dimension nfac(11),np(209)
c  array storage for maximum prime factor of 23
      dimension at(23),ck(23),bt(23),sk(23)
      equivalence (i,ii)
c  the following two constants should agree with the array dimensions.
      maxp=209
      maxf=23
      if(n .lt. 2) return
      inc=isn
      c72=0.30901699437494742
      s72=0.95105651629515357
      s120=0.86602540378443865
      rad=6.2831853071796
      if(isn .ge. 0) go to 10
      s72=-s72
      s120=-s120
      rad=-rad
      inc=-inc
   10 nt=inc*ntot
      ks=inc*nspan
      kspan=ks
      nn=nt-inc
      jc=ks/n
      radf=rad*float(jc)*0.5
      i=0
      jf=0
c  determine the factors of n
      m=0
      k=n
      go to 20
   15 m=m+1
      nfac(m)=4
      k=k/16
   20 if(k-(k/16)*16 .eq. 0) go to 15
      j=3
      jj=9
      go to 30
   25 m=m+1
      nfac(m)=j
      k=k/jj
   30 if(mod(k,jj) .eq. 0) go to 25
      j=j+2
      jj=j**2
      if(jj .le. k) go to 30
      if(k .gt. 4) go to 40
      kt=m
      nfac(m+1)=k
      if(k .ne. 1) m=m+1
      go to 80
   40 if(k-(k/4)*4 .ne. 0) go to 50
      m=m+1
      nfac(m)=2
      k=k/4
   50 kt=m
      j=2
   60 if(mod(k,j) .ne. 0) go to 70
      m=m+1
      nfac(m)=j
      k=k/j
   70 j=((j+1)/2)*2+1
      if(j .le. k) go to 60
   80 if(kt .eq. 0) go to 100
      j=kt
   90 m=m+1
      nfac(m)=nfac(j)
      j=j-1
      if(j .ne. 0) go to 90
c  compute fourier transform
  100 sd=radf/float(kspan)
      cd=2.0*sin(sd)**2
      sd=sin(sd+sd)
      kk=1
      i=i+1
      if(nfac(i) .ne. 2) go to 400
c  transform for factor of 2 (including rotation factor)
      kspan=kspan/2
      k1=kspan+2
  210 k2=kk+kspan
      ak=a(k2)
      bk=b(k2)
      a(k2)=a(kk)-ak
      b(k2)=b(kk)-bk
      a(kk)=a(kk)+ak
      b(kk)=b(kk)+bk
      kk=k2+kspan
      if(kk .le. nn) go to 210
      kk=kk-nn
      if(kk .le. jc) go to 210
      if(kk .gt. kspan) go to 800
  220 c1=1.0-cd
      s1=sd
  230 k2=kk+kspan
      ak=a(kk)-a(k2)
      bk=b(kk)-b(k2)
      a(kk)=a(kk)+a(k2)
      b(kk)=b(kk)+b(k2)
      a(k2)=c1*ak-s1*bk
      b(k2)=s1*ak+c1*bk
      kk=k2+kspan
      if(kk .lt. nt) go to 230
      k2=kk-nt
      c1=-c1
      kk=k1-k2
      if(kk .gt. k2) go to 230
      ak=c1-(cd*c1+sd*s1)
      s1=(sd*c1-cd*s1)+s1
      c1=2.0-(ak**2+s1**2)
      s1=c1*s1
      c1=c1*ak
      kk=kk+jc
      if(kk .lt. k2) go to 230
      k1=k1+inc+inc
      kk=(k1-kspan)/2+jc
      if(kk .le. jc+jc) go to 220
      go to 100
c  transform for factor of 3 (optional code)
  320 k1=kk+kspan
      k2=k1+kspan
      ak=a(kk)
      bk=b(kk)
      aj=a(k1)+a(k2)
      bj=b(k1)+b(k2)
      a(kk)=ak+aj
      b(kk)=bk+bj
      ak=-0.5*aj+ak
      bk=-0.5*bj+bk
      aj=(a(k1)-a(k2))*s120
      bj=(b(k1)-b(k2))*s120
      a(k1)=ak-bj
      b(k1)=bk+aj
      a(k2)=ak+bj
      b(k2)=bk-aj
      kk=k2+kspan
      if(kk .lt. nn) go to 320
      kk=kk-nn
      if(kk .le. kspan) go to 320
      go to 700
c  transform for factor of 4
  400 if(nfac(i) .ne. 4) go to 600
      kspnn=kspan
      kspan=kspan/4
  410 c1=1.0
      s1=0
  420 k1=kk+kspan
      k2=k1+kspan
      k3=k2+kspan
      akp=a(kk)+a(k2)
      akm=a(kk)-a(k2)
      ajp=a(k1)+a(k3)
      ajm=a(k1)-a(k3)
      a(kk)=akp+ajp
      ajp=akp-ajp
      bkp=b(kk)+b(k2)
      bkm=b(kk)-b(k2)
      bjp=b(k1)+b(k3)
      bjm=b(k1)-b(k3)
      b(kk)=bkp+bjp
      bjp=bkp-bjp
      if(isn .lt. 0) go to 450
      akp=akm-bjm
      akm=akm+bjm
      bkp=bkm+ajm
      bkm=bkm-ajm
      if(s1 .eq. 0) go to 460
  430 a(k1)=akp*c1-bkp*s1
      b(k1)=akp*s1+bkp*c1
      a(k2)=ajp*c2-bjp*s2
      b(k2)=ajp*s2+bjp*c2
      a(k3)=akm*c3-bkm*s3
      b(k3)=akm*s3+bkm*c3
      kk=k3+kspan
      if(kk .le. nt) go to 420
  440 c2=c1-(cd*c1+sd*s1)
      s1=(sd*c1-cd*s1)+s1
      c1=2.0-(c2**2+s1**2)
      s1=c1*s1
      c1=c1*c2
      c2=c1**2-s1**2
      s2=2.0*c1*s1
      c3=c2*c1-s2*s1
      s3=c2*s1+s2*c1
      kk=kk-nt+jc
      if(kk .le. kspan) go to 420
      kk=kk-kspan+inc
      if(kk .le. jc) go to 410
      if(kspan .eq. jc) go to 800
      go to 100
  450 akp=akm+bjm
      akm=akm-bjm
      bkp=bkm-ajm
      bkm=bkm+ajm
      if(s1 .ne. 0) go to 430
  460 a(k1)=akp
      b(k1)=bkp
      a(k2)=ajp
      b(k2)=bjp
      a(k3)=akm
      b(k3)=bkm
      kk=k3+kspan
      if(kk .le. nt) go to 420
      go to 440
c  transform for factor of 5 (optional code)
  510 c2=c72**2-s72**2
      s2=2.0*c72*s72
  520 k1=kk+kspan
      k2=k1+kspan
      k3=k2+kspan
      k4=k3+kspan
      akp=a(k1)+a(k4)
      akm=a(k1)-a(k4)
      bkp=b(k1)+b(k4)
      bkm=b(k1)-b(k4)
      ajp=a(k2)+a(k3)
      ajm=a(k2)-a(k3)
      bjp=b(k2)+b(k3)
      bjm=b(k2)-b(k3)
      aa=a(kk)
      bb=b(kk)
      a(kk)=aa+akp+ajp
      b(kk)=bb+bkp+bjp
      ak=akp*c72+ajp*c2+aa
      bk=bkp*c72+bjp*c2+bb
      aj=akm*s72+ajm*s2
      bj=bkm*s72+bjm*s2
      a(k1)=ak-bj
      a(k4)=ak+bj
      b(k1)=bk+aj
      b(k4)=bk-aj
      ak=akp*c2+ajp*c72+aa
      bk=bkp*c2+bjp*c72+bb
      aj=akm*s2-ajm*s72
      bj=bkm*s2-bjm*s72
      a(k2)=ak-bj
      a(k3)=ak+bj
      b(k2)=bk+aj
      b(k3)=bk-aj
      kk=k4+kspan
      if(kk .lt. nn) go to 520
      kk=kk-nn
      if(kk .le. kspan) go to 520
      go to 700
c  transform for odd factors
  600 k=nfac(i)
      kspnn=kspan
      kspan=kspan/k
      if(k .eq. 3) go to 320
      if(k .eq. 5) go to 510
      if(k .eq. jf) go to 640
      jf=k
      s1=rad/float(k)
      c1=cos(s1)
      s1=sin(s1)
      if(jf .gt. maxf) go to 998
      ck(jf)=1.0
      sk(jf)=0.0
      j=1
  630 ck(j)=ck(k)*c1+sk(k)*s1
      sk(j)=ck(k)*s1-sk(k)*c1
      k=k-1
      ck(k)=ck(j)
      sk(k)=-sk(j)
      j=j+1
      if(j .lt. k) go to 630
  640 k1=kk
      k2=kk+kspnn
      aa=a(kk)
      bb=b(kk)
      ak=aa
      bk=bb
      j=1
      k1=k1+kspan
  650 k2=k2-kspan
      j=j+1
      at(j)=a(k1)+a(k2)
      ak=at(j)+ak
      bt(j)=b(k1)+b(k2)
      bk=bt(j)+bk
      j=j+1
      at(j)=a(k1)-a(k2)
      bt(j)=b(k1)-b(k2)
      k1=k1+kspan
      if(k1 .lt. k2) go to 650
      a(kk)=ak
      b(kk)=bk
      k1=kk
      k2=kk+kspnn
      j=1
  660 k1=k1+kspan
      k2=k2-kspan
      jj=j
      ak=aa
      bk=bb
      aj=0.0
      bj=0.0
      k=1
  670 k=k+1
      ak=at(k)*ck(jj)+ak
      bk=bt(k)*ck(jj)+bk
      k=k+1
      aj=at(k)*sk(jj)+aj
      bj=bt(k)*sk(jj)+bj
      jj=jj+j
      if(jj .gt. jf) jj=jj-jf
      if(k .lt. jf) go to 670
      k=jf-j
      a(k1)=ak-bj
      b(k1)=bk+aj
      a(k2)=ak+bj
      b(k2)=bk-aj
      j=j+1
      if(j .lt. k) go to 660
      kk=kk+kspnn
      if(kk .le. nn) go to 640
      kk=kk-nn
      if(kk .le. kspan) go to 640
c  multiply by rotation factor (except for factors of 2 and 4)
  700 if(i .eq. m) go to 800
      kk=jc+1
  710 c2=1.0-cd
      s1=sd
  720 c1=c2
      s2=s1
      kk=kk+kspan
  730 ak=a(kk)
      a(kk)=c2*ak-s2*b(kk)
      b(kk)=s2*ak+c2*b(kk)
      kk=kk+kspnn
      if(kk .le. nt) go to 730
      ak=s1*s2
      s2=s1*c2+c1*s2
      c2=c1*c2-ak
      kk=kk-nt+kspan
      if(kk .le. kspnn) go to 730
      c2=c1-(cd*c1+sd*s1)
      s1=s1+(sd*c1-cd*s1)
      c1=2.0-(c2**2+s1**2)
      s1=c1*s1
      c2=c1*c2
      kk=kk-kspnn+jc
      if(kk .le. kspan) go to 720
      kk=kk-kspan+jc+inc
      if(kk .le. jc+jc) go to 710
      go to 100
c  permute the results to normal order---done in two stages
c  permutation for square factors of n
  800 np(1)=ks
      if(kt .eq. 0) go to 890
      k=kt+kt+1
      if(m .lt. k) k=k-1
      j=1
      np(k+1)=jc
  810 np(j+1)=np(j)/nfac(j)
      np(k)=np(k+1)*nfac(j)
      j=j+1
      k=k-1
      if(j .lt. k) go to 810
      k3=np(k+1)
      kspan=np(2)
      kk=jc+1
      k2=kspan+1
      j=1
      if(n .ne. ntot) go to 850
c  permutation for single-variate transform (optional code)
  820 ak=a(kk)
      a(kk)=a(k2)
      a(k2)=ak
      bk=b(kk)
      b(kk)=b(k2)
      b(k2)=bk
      kk=kk+inc
      k2=kspan+k2
      if(k2 .lt. ks) go to 820
  830 k2=k2-np(j)
      j=j+1
      k2=np(j+1)+k2
      if(k2 .gt. np(j)) go to 830
      j=1
  840 if(kk .lt. k2) go to 820
      kk=kk+inc
      k2=kspan+k2
      if(k2 .lt. ks) go to 840
      if(kk .lt. ks) go to 830
      jc=k3
      go to 890
c  permutation for multivariate transform
  850 k=kk+jc
  860 ak=a(kk)
      a(kk)=a(k2)
      a(k2)=ak
      bk=b(kk)
      b(kk)=b(k2)
      b(k2)=bk
      kk=kk+inc
      k2=k2+inc
      if(kk .lt. k) go to 860
      kk=kk+ks-jc
      k2=k2+ks-jc
      if(kk .lt. nt) go to 850
      k2=k2-nt+kspan
      kk=kk-nt+jc
      if(k2 .lt. ks) go to 850
  870 k2=k2-np(j)
      j=j+1
      k2=np(j+1)+k2
      if(k2 .gt. np(j)) go to 870
      j=1
  880 if(kk .lt. k2) go to 850
      kk=kk+jc
      k2=kspan+k2
      if(k2 .lt. ks) go to 880
      if(kk .lt. ks) go to 870
      jc=k3
  890 if(2*kt+1 .ge. m) return
      kspnn=np(kt+1)
c  permutation for square-free factors of n
      j=m-kt
      nfac(j+1)=1
  900 nfac(j)=nfac(j)*nfac(j+1)
      j=j-1
      if(j .ne. kt) go to 900
      kt=kt+1
      nn=nfac(kt)-1
      if(nn .gt. maxp) go to 998
      jj=0
      j=0
      go to 906
  902 jj=jj-k2
      k2=kk
      k=k+1
      kk=nfac(k)
  904 jj=kk+jj
      if(jj .ge. k2) go to 902
      np(j)=jj
  906 k2=nfac(kt)
      k=kt+1
      kk=nfac(k)
      j=j+1
      if(j .le. nn) go to 904
c  determine the permutation cycles of length greater than 1
      j=0
      go to 914
  910 k=kk
      kk=np(k)
      np(k)=-kk
      if(kk .ne. j) go to 910
      k3=kk
  914 j=j+1
      kk=np(j)
      if(kk .lt. 0) go to 914
      if(kk .ne. j) go to 910
      np(j)=-j
      if(j .ne. nn) go to 914
      maxf=inc*maxf
c  reorder a and b, following the permutation cycles
      go to 950
  924 j=j-1
      if(np(j) .lt. 0) go to 924
      jj=jc
  926 kspan=jj
      if(jj .gt. maxf) kspan=maxf
      jj=jj-kspan
      k=np(j)
      kk=jc*k+ii+jj
      k1=kk+kspan
      k2=0
  928 k2=k2+1
      at(k2)=a(k1)
      bt(k2)=b(k1)
      k1=k1-inc
      if(k1 .ne. kk) go to 928
  932 k1=kk+kspan
      k2=k1-jc*(k+np(k))
      k=-np(k)
  936 a(k1)=a(k2)
      b(k1)=b(k2)
      k1=k1-inc
      k2=k2-inc
      if(k1 .ne. kk) go to 936
      kk=k2
      if(k .ne. j) go to 932
      k1=kk+kspan
      k2=0
  940 k2=k2+1
      a(k1)=at(k2)
      b(k1)=bt(k2)
      k1=k1-inc
      if(k1 .ne. kk) go to 940
      if(jj .ne. 0) go to 926
      if(j .ne. 1) go to 924
  950 j=k3+1
      nt=nt-kspnn
      ii=nt-inc+1
      if(nt .ge. 0) go to 924
      return
c  error finish, insufficient array storage
  998 isn=0
      print 999
      stop
  999 format(' array bounds exceeded within subroutine mfft')
      end

CTITLESAFFTL -- FFT LENGTH TABLE (BASED ON 2 3 5)                       00010001
CA                                                                      00020001
CA   DESIGNER     JAMES SUN                                             00030001
CA   AUTHOR       JAMES SUN                                             00040001
CA   LANGUAGE     FORTRAN                                               00050001
CA   SYSTEM       IBM AND CRAY                                          00060001
CA   WRITTEN      02/08/89                                              00070001
CA                                                                      00080001
C    REVISED      03/31/89 JCS   REAPPLY MUTE TO THE DEPTH SECTION      00090001
C    REVISED      01/20/90 JJC   MODIFIED TO MEET EDP STANDARDS.        00100001
C    REVISED      02/20/90 JJC   RENAMED SAZM2DV TO SAFFTL.             00110001
C    REVISED      07/23/90 CLJ   ALLOW PREP TO RUN ON THE IBM           00120001
C    REVISED      03/02/94 ESN   REVISED FOR UPDATED LIST OF LENGTHS.   00130001
C                                                                       00140001
CA                                                                      00150001
CA CALLING PROCEDURE:                                                   00160001
CA     SUBROUTINE SAFFTL(NIN,NOUT)                                      00170001
CA                                                                      00180001
C   CALLING ARGUMENTS                                                   00190001
CA                                                                      00200001
CA     IN/PUT     NIN      INPUT SAMPLE NUMBER                       I4 00210001
CA     OUTPUT     NOUT     OUTPUT FFT LENGTH                         I4 00220001
C                                                                       00230001
C     THIS SUBROUTINE COMPUTES THE FFT LENGTH BASED ON POWER OF 2, 3,   00240001
C     AND 5.  DUE TO SOME RESTRICTIONS IN ZM2D., THIS FFT LENGTH TABLE  00250001
C     IS DIFFERENT WITH THE SAFFLN SUBROUTINE.                          00260001
C                                                                       00270001
      SUBROUTINE SAFFTL(NIN,NOUT)                                       00280001
C                                                                       00290001
      IMPLICIT   INTEGER(A-Z)                                           00300001
C                                                                       00310001
      DIMENSION MAGICN(126)                                             00320001
C     DATA MAGICN/ 64, 72, 80,  90,  96, 100, 108,  120,  128,  144,    00330001
C    *  150,  160,  162,  180,  192,  200,  216,  240,  250,  256,      00340001
C    *  270,  288,  300,  320,  324,  360,  384,  400,  432,  450,      00350001
C    *  480,  486,  500,  512,  540,  576,  600,  640,  648,  720,      00360001
C    *  750,  768,  800,  810,  864,  900,  960,  972, 1000, 1024,      00370001
C    *  1080, 1152, 1200, 1250, 1280, 1296, 1350, 1440, 1458, 1500,     00380001
C    *  1536, 1600, 1620, 1728, 1800, 1920, 1944, 2000, 2048, 2160,     00390001
C    *  2250, 2304, 2400, 2430, 2500, 2560, 2592, 2700, 2880, 2916,     00400001
C    *  3000, 3072, 3200, 3240, 3456, 3600, 3750, 3840, 3888, 4000,     00410001
C    *  4050, 4096, 4320, 4374, 4500, 4608, 4800, 4860, 5000, 5120,     00420001
C    *  5184, 5400, 5760, 5832, 6000, 6144, 6250, 6400, 6480, 6750,     00430001
C    *  6912, 7200, 7290, 7500, 7680, 7776, 8000, 8100, 8192, 8640,     00440001
C    *  8748, 9000, 9216, 9600, 9720,10000/                             00450001
      DATA MAGICN/                                                      00460001
     *     64,  72,  80,  84,  96, 112, 120, 128, 140, 144, 160,        00470001
     *    168, 180, 192, 224, 240, 252, 256, 280, 288, 320, 336,        00480001
     *    360, 384, 420, 448, 480, 504, 512, 560, 576, 640, 672,        00490001
     *    720,  768,  840,  896,  960, 1008, 1024, 1120, 1152, 1260,    00500001
     *   1280, 1344, 1440, 1536, 1680, 1792, 1920, 2016, 2048, 2240,    00510001
     *   2304, 2520, 2560, 2688, 2880, 3072, 3360, 3584, 3840, 4032,    00520001
     *   4096, 4480, 4608, 5040, 5120, 5376, 5760, 6144, 6720, 7168,    00530001
     *   7680, 8064, 8192, 8960, 9216, 10240, 47*0 /                    00540001
      DATA NM/ 126/                                                     00550001
      NOUT = 0                                                          00560001
      DO 100 I=1,NM                                                     00570001
        IF (MAGICN(I) .GE. NIN) THEN                                    00580001
          NOUT = MAGICN(I)                                              00590001
          RETURN                                                        00600001
        ENDIF                                                           00610001
  100 CONTINUE                                                          00620001
      RETURN                                                            00630001
      END                                                               00640001

      subroutine sblas()
      i = isamax(n,sx,incx)
      r = sasum(n,sx,incx)
      call saxpy(n,sa,sx,incx,sy,incy)
      call scopy(n,sx,incx,sy,incy)
      r = sdot(n,sx,incx,sy,incy)
      r = smach(job)
      r = snrm2 ( n, sx, incx)
      call srot (n,sx,incx,sy,incy,c,s)
      call srotg(sa,sb,c,s)
      call sscal(n,sa,sx,incx)
      call sswap (n,sx,incx,sy,incy)
      stop
      end
      integer function isamax(n,sx,incx)
c
c     finds the index of element having max. absolute value.
c     jack dongarra, linpack, 3/11/78.
c     modified 3/93 to return if incx .le. 0.
c     modified 12/3/93, array(1) declarations changed to array(*)
c
      real sx(*),smax
      integer i,incx,ix,n
c
      isamax = 0
      if( n.lt.1 .or. incx.le.0 ) return
      isamax = 1
      if(n.eq.1)return
      if(incx.eq.1)go to 20
c
c        code for increment not equal to 1
c
      ix = 1
      smax = abs(sx(1))
      ix = ix + incx
      do 10 i = 2,n
         if(abs(sx(ix)).le.smax) go to 5
         isamax = i
         smax = abs(sx(ix))
    5    ix = ix + incx
   10 continue
      return
c
c        code for increment equal to 1
c
   20 smax = abs(sx(1))
      do 30 i = 2,n
         if(abs(sx(i)).le.smax) go to 30
         isamax = i
         smax = abs(sx(i))
   30 continue
      return
      end
      real function sasum(n,sx,incx)
c
c     takes the sum of the absolute values.
c     uses unrolled loops for increment equal to one.
c     jack dongarra, linpack, 3/11/78.
c     modified 3/93 to return if incx .le. 0.
c     modified 12/3/93, array(1) declarations changed to array(*)
c
      real sx(*),stemp
      integer i,incx,m,mp1,n,nincx
c
      sasum = 0.0e0
      stemp = 0.0e0
      if( n.le.0 .or. incx.le.0 )return
      if(incx.eq.1)go to 20
c
c        code for increment not equal to 1
c
      nincx = n*incx
      do 10 i = 1,nincx,incx
        stemp = stemp + abs(sx(i))
   10 continue
      sasum = stemp
      return
c
c        code for increment equal to 1
c
c
c        clean-up loop
c
   20 m = mod(n,6)
      if( m .eq. 0 ) go to 40
      do 30 i = 1,m
        stemp = stemp + abs(sx(i))
   30 continue
      if( n .lt. 6 ) go to 60
   40 mp1 = m + 1
      do 50 i = mp1,n,6
        stemp = stemp + abs(sx(i)) + abs(sx(i + 1)) + abs(sx(i + 2))
     *  + abs(sx(i + 3)) + abs(sx(i + 4)) + abs(sx(i + 5))
   50 continue
   60 sasum = stemp
      return
      end
      subroutine saxpy(n,sa,sx,incx,sy,incy)
c
c     constant times a vector plus a vector.
c     uses unrolled loop for increments equal to one.
c     jack dongarra, linpack, 3/11/78.
c     modified 12/3/93, array(1) declarations changed to array(*)
c
      real sx(*),sy(*),sa
      integer i,incx,incy,ix,iy,m,mp1,n
c
      if(n.le.0)return
      if (sa .eq. 0.0) return
      if(incx.eq.1.and.incy.eq.1)go to 20
c
c        code for unequal increments or equal increments
c          not equal to 1
c
      ix = 1
      iy = 1
      if(incx.lt.0)ix = (-n+1)*incx + 1
      if(incy.lt.0)iy = (-n+1)*incy + 1
      do 10 i = 1,n
        sy(iy) = sy(iy) + sa*sx(ix)
        ix = ix + incx
        iy = iy + incy
   10 continue
      return
c
c        code for both increments equal to 1
c
c
c        clean-up loop
c
   20 m = mod(n,4)
      if( m .eq. 0 ) go to 40
      do 30 i = 1,m
        sy(i) = sy(i) + sa*sx(i)
   30 continue
      if( n .lt. 4 ) return
   40 mp1 = m + 1
      do 50 i = mp1,n,4
        sy(i) = sy(i) + sa*sx(i)
        sy(i + 1) = sy(i + 1) + sa*sx(i + 1)
        sy(i + 2) = sy(i + 2) + sa*sx(i + 2)
        sy(i + 3) = sy(i + 3) + sa*sx(i + 3)
   50 continue
      return
      end
      subroutine scopy(n,sx,incx,sy,incy)
c
c     copies a vector, x, to a vector, y.
c     uses unrolled loops for increments equal to 1.
c     jack dongarra, linpack, 3/11/78.
c     modified 12/3/93, array(1) declarations changed to array(*)
c
      real sx(*),sy(*)
      integer i,incx,incy,ix,iy,m,mp1,n
c
      if(n.le.0)return
      if(incx.eq.1.and.incy.eq.1)go to 20
c
c        code for unequal increments or equal increments
c          not equal to 1
c
      ix = 1
      iy = 1
      if(incx.lt.0)ix = (-n+1)*incx + 1
      if(incy.lt.0)iy = (-n+1)*incy + 1
      do 10 i = 1,n
        sy(iy) = sx(ix)
        ix = ix + incx
        iy = iy + incy
   10 continue
      return
c
c        code for both increments equal to 1
c
c
c        clean-up loop
c
   20 m = mod(n,7)
      if( m .eq. 0 ) go to 40
      do 30 i = 1,m
        sy(i) = sx(i)
   30 continue
      if( n .lt. 7 ) return
   40 mp1 = m + 1
      do 50 i = mp1,n,7
        sy(i) = sx(i)
        sy(i + 1) = sx(i + 1)
        sy(i + 2) = sx(i + 2)
        sy(i + 3) = sx(i + 3)
        sy(i + 4) = sx(i + 4)
        sy(i + 5) = sx(i + 5)
        sy(i + 6) = sx(i + 6)
   50 continue
      return
      end
      real function sdot(n,sx,incx,sy,incy)
c
c     forms the dot product of two vectors.
c     uses unrolled loops for increments equal to one.
c     jack dongarra, linpack, 3/11/78.
c     modified 12/3/93, array(1) declarations changed to array(*)
c
      real sx(*),sy(*),stemp
      integer i,incx,incy,ix,iy,m,mp1,n
c
      stemp = 0.0e0
      sdot = 0.0e0
      if(n.le.0)return
      if(incx.eq.1.and.incy.eq.1)go to 20
c
c        code for unequal increments or equal increments
c          not equal to 1
c
      ix = 1
      iy = 1
      if(incx.lt.0)ix = (-n+1)*incx + 1
      if(incy.lt.0)iy = (-n+1)*incy + 1
      do 10 i = 1,n
        stemp = stemp + sx(ix)*sy(iy)
        ix = ix + incx
        iy = iy + incy
   10 continue
      sdot = stemp
      return
c
c        code for both increments equal to 1
c
c
c        clean-up loop
c
   20 m = mod(n,5)
      if( m .eq. 0 ) go to 40
      do 30 i = 1,m
        stemp = stemp + sx(i)*sy(i)
   30 continue
      if( n .lt. 5 ) go to 60
   40 mp1 = m + 1
      do 50 i = mp1,n,5
        stemp = stemp + sx(i)*sy(i) + sx(i + 1)*sy(i + 1) +
     *   sx(i + 2)*sy(i + 2) + sx(i + 3)*sy(i + 3) + sx(i + 4)*sy(i + 4)
   50 continue
   60 sdot = stemp
      return
      end
      real function smach(job)
      integer job
c
c     smach computes machine parameters of floating point
c     arithmetic for use in testing only.  not required by
c     linpack proper.
c
c     if trouble with automatic computation of these quantities,
c     they can be set by direct assignment statements.
c     assume the computer has
c
c        b = base of arithmetic
c        t = number of base  b  digits
c        l = smallest possible exponent
c        u = largest possible exponent
c
c     then
c
c        eps = b**(1-t)
c        tiny = 100.0*b**(-l+t)
c        huge = 0.01*b**(u-t)
c
c     dmach same as smach except t, l, u apply to
c     double precision.
c
c     cmach same as smach except if complex division
c     is done by
c
c        1/(x+i*y) = (x-i*y)/(x**2+y**2)
c
c     then
c
c        tiny = sqrt(tiny)
c        huge = sqrt(huge)
c
c
c     job is 1, 2 or 3 for epsilon, tiny and huge, respectively.
c
c
      real eps,tiny,huge,s
c
      eps = 1.0
   10 eps = eps/2.0
      s = 1.0 + eps
      if (s .gt. 1.0) go to 10
      eps = 2.0*eps
c
      s = 1.0
   20 tiny = s
      s = s/16.0
      if (s*100. .ne. 0.0) go to 20
      tiny = (tiny/eps)*100.0
      huge = 1.0/tiny
c
      if (job .eq. 1) smach = eps
      if (job .eq. 2) smach = tiny
      if (job .eq. 3) smach = huge
      return
      end
      REAL             FUNCTION SNRM2 ( N, X, INCX )
*     .. Scalar Arguments ..
      INTEGER                           INCX, N
*     .. Array Arguments ..
      REAL                              X( * )
*     ..
*
*  SNRM2 returns the euclidean norm of a vector via the function
*  name, so that
*
*     SNRM2 := sqrt( x'*x )
*
*
*
*  -- This version written on 25-October-1982.
*     Modified on 14-October-1993 to inline the call to SLASSQ.
*     Sven Hammarling, Nag Ltd.
*
*
*     .. Parameters ..
      REAL                  ONE         , ZERO
      PARAMETER           ( ONE = 1.0E+0, ZERO = 0.0E+0 )
*     .. Local Scalars ..
      INTEGER               IX
      REAL                  ABSXI, NORM, SCALE, SSQ
*     .. Intrinsic Functions ..
      INTRINSIC             ABS, SQRT
*     ..
*     .. Executable Statements ..
      IF( N.LT.1 .OR. INCX.LT.1 )THEN
         NORM  = ZERO
      ELSE IF( N.EQ.1 )THEN
         NORM  = ABS( X( 1 ) )
      ELSE
         SCALE = ZERO
         SSQ   = ONE
*        The following loop is equivalent to this call to the LAPACK
*        auxiliary routine:
*        CALL SLASSQ( N, X, INCX, SCALE, SSQ )
*
         DO 10, IX = 1, 1 + ( N - 1 )*INCX, INCX
            IF( X( IX ).NE.ZERO )THEN
               ABSXI = ABS( X( IX ) )
               IF( SCALE.LT.ABSXI )THEN
                  SSQ   = ONE   + SSQ*( SCALE/ABSXI )**2
                  SCALE = ABSXI
               ELSE
                  SSQ   = SSQ   +     ( ABSXI/SCALE )**2
               END IF
            END IF
   10    CONTINUE
         NORM  = SCALE * SQRT( SSQ )
      END IF
*
      SNRM2 = NORM
      RETURN
*
*     End of SNRM2.
*
      END
      subroutine srot (n,sx,incx,sy,incy,c,s)
c
c     applies a plane rotation.
c     jack dongarra, linpack, 3/11/78.
c     modified 12/3/93, array(1) declarations changed to array(*)
c
      real sx(*),sy(*),stemp,c,s
      integer i,incx,incy,ix,iy,n
c
      if(n.le.0)return
      if(incx.eq.1.and.incy.eq.1)go to 20
c
c       code for unequal increments or equal increments not equal
c         to 1
c
      ix = 1
      iy = 1
      if(incx.lt.0)ix = (-n+1)*incx + 1
      if(incy.lt.0)iy = (-n+1)*incy + 1
      do 10 i = 1,n
        stemp = c*sx(ix) + s*sy(iy)
        sy(iy) = c*sy(iy) - s*sx(ix)
        sx(ix) = stemp
        ix = ix + incx
        iy = iy + incy
   10 continue
      return
c
c       code for both increments equal to 1
c
   20 do 30 i = 1,n
        stemp = c*sx(i) + s*sy(i)
        sy(i) = c*sy(i) - s*sx(i)
        sx(i) = stemp
   30 continue
      return
      end
      subroutine srotg(sa,sb,c,s)
c
c     construct givens plane rotation.
c     jack dongarra, linpack, 3/11/78.
c
      real sa,sb,c,s,roe,scale,r,z
c
      roe = sb
      if( abs(sa) .gt. abs(sb) ) roe = sa
      scale = abs(sa) + abs(sb)
      if( scale .ne. 0.0 ) go to 10
         c = 1.0
         s = 0.0
         r = 0.0
         z = 0.0
         go to 20
   10 r = scale*sqrt((sa/scale)**2 + (sb/scale)**2)
      r = sign(1.0,roe)*r
      c = sa/r
      s = sb/r
      z = 1.0
      if( abs(sa) .gt. abs(sb) ) z = s
      if( abs(sb) .ge. abs(sa) .and. c .ne. 0.0 ) z = 1.0/c
   20 sa = r
      sb = z
      return
      end
      subroutine sscal(n,sa,sx,incx)
c
c     scales a vector by a constant.
c     uses unrolled loops for increment equal to 1.
c     jack dongarra, linpack, 3/11/78.
c     modified 3/93 to return if incx .le. 0.
c     modified 12/3/93, array(1) declarations changed to array(*)
c
      real sa,sx(*)
      integer i,incx,m,mp1,n,nincx
c
      if( n.le.0 .or. incx.le.0 )return
      if(incx.eq.1)go to 20
c
c        code for increment not equal to 1
c
      nincx = n*incx
      do 10 i = 1,nincx,incx
        sx(i) = sa*sx(i)
   10 continue
      return
c
c        code for increment equal to 1
c
c
c        clean-up loop
c
   20 m = mod(n,5)
      if( m .eq. 0 ) go to 40
      do 30 i = 1,m
        sx(i) = sa*sx(i)
   30 continue
      if( n .lt. 5 ) return
   40 mp1 = m + 1
      do 50 i = mp1,n,5
        sx(i) = sa*sx(i)
        sx(i + 1) = sa*sx(i + 1)
        sx(i + 2) = sa*sx(i + 2)
        sx(i + 3) = sa*sx(i + 3)
        sx(i + 4) = sa*sx(i + 4)
   50 continue
      return
      end
      subroutine sswap (n,sx,incx,sy,incy)
c
c     interchanges two vectors.
c     uses unrolled loops for increments equal to 1.
c     jack dongarra, linpack, 3/11/78.
c     modified 12/3/93, array(1) declarations changed to array(*)
c
      real sx(*),sy(*),stemp
      integer i,incx,incy,ix,iy,m,mp1,n
c
      if(n.le.0)return
      if(incx.eq.1.and.incy.eq.1)go to 20
c
c       code for unequal increments or equal increments not equal
c         to 1
c
      ix = 1
      iy = 1
      if(incx.lt.0)ix = (-n+1)*incx + 1
      if(incy.lt.0)iy = (-n+1)*incy + 1
      do 10 i = 1,n
        stemp = sx(ix)
        sx(ix) = sy(iy)
        sy(iy) = stemp
        ix = ix + incx
        iy = iy + incy
   10 continue
      return
c
c       code for both increments equal to 1
c
c
c       clean-up loop
c
   20 m = mod(n,3)
      if( m .eq. 0 ) go to 40
      do 30 i = 1,m
        stemp = sx(i)
        sx(i) = sy(i)
        sy(i) = stemp
   30 continue
      if( n .lt. 3 ) return
   40 mp1 = m + 1
      do 50 i = mp1,n,3
        stemp = sx(i)
        sx(i) = sy(i)
        sy(i) = stemp
        stemp = sx(i + 1)
        sx(i + 1) = sy(i + 1)
        sy(i + 1) = stemp
        stemp = sx(i + 2)
        sx(i + 2) = sy(i + 2)
        sy(i + 2) = stemp
   50 continue
      return
      end
C


      subroutine cmov(rr,n2,n3,  data)
        implicit none
      integer              n2,n3
      complex*8         data(n2,n3),          rr(n2,n3)
      integer  i2,i3
      do i3=1,n3
      do i2=1,n2
         data(i2,i3) = rr(i2,i3)
      enddo
      enddo
      return
      end
      



      

 


C
C PAGE 208-209: NUMERICAL MATHEMATICS AND COMPUTING, CHENEY/KINCAID, 1985
C
C FILE: NGAUSS.FOR
C
C NAIVE GAUSSIAN ELIMINATION TO SOLVE LINEAR SYSTEMS (NGAUSS)
C
      SUBROUTINE NGAUSS(N,A,IA,B,X)
c
c    Solve  B = A X  for X
c
c
c
c    n = number of columns in a
c    ia = number of physical rows in a, used rows must be n
c    b = left hand side
c    x = solution
c
 
      COMPLEX*8 A(IA,N),B(N),X(N)
      COMPLEX*8 XMULT,SUM

      DO 4 K = 1,N-1
        DO 3 I = K+1,N
          XMULT = A(I,K)/A(K,K)
          DO 2 J = K+1,N
            A(I,J) = A(I,J) - XMULT*A(K,J)
   2      CONTINUE
          A(I,K) = XMULT
          B(I) = B(I) - XMULT*B(K)
   3    CONTINUE
   4  CONTINUE
      X(N) = B(N)/A(N,N)
      DO 6 I = N-1,1,-1
        SUM = B(I)
        DO 5 J = I+1,N
          SUM = SUM - A(I,J)*X(J)
   5    CONTINUE
        X(I) = SUM/A(I,I)
   6  CONTINUE
      RETURN
      END
       subroutine boxconv( nb, nx, xx, yy, ierr)
	implicit none
C------# inputs:       nx,  xx(i), i=1,nx      the data
C------#               nb                      the box length
C------# output:       yy(i),i=1,nx+nb-1       smoothed data

#include <localsys.h>
#include <f77/iounit.h>
#include <f77/lhdrsz.h>

       integer nx, ny, nb, i, ierr
       integer jerr, jabort, jsz
       real*4 xx(nx)
       real*4 yy(1)
c      real*4 bb(nx+nb)
       real*4 bb(*)

       pointer(wkbb,bb)

       data jerr / 0 /, jabort / 0 /, jsz / 0 /

       save jsz, wkbb

       if (jsz .eq. 0) then
	 call sizefloat(jsz)
	 call galloc(wkbb, (nx+nb)*jsz, jerr, jabort)
	 if (jerr .ne. 0) then
	   write(LER,*) ' FXYDECON: Space allocation error',
     :       ' - ',(nx+nb)*jsz,' bytes requested; aborting'
	   write(LERR,*) ' FXYDECON: Space allocation error',
     :       ' - ',(nx+nb)*jsz,' bytes requested; aborting'
	   call abortfu()
	 endif
       endif

       if( nb .lt. 1 .or. nb .gt. nx)  then
         ierr = 765
         return
       endif

       ny = nx+nb-1
       do i= 1, ny
               bb(i) = 0.
       enddo

       bb(1) = xx(1)

       do i= 2, nx
                       bb(i) = bb(i-1) + xx(i) 
       enddo
       do i= nx+1, ny
                       bb(i) = bb(i-1)
       enddo
       do i= 1, nb
                       yy(i) = bb(i)
       enddo
       do i= nb+1, ny
                       yy(i) = bb(i) - bb(i-nb)  
       enddo
       do i= 1, ny
                       yy(i) = yy(i) / nb
       enddo

       return
       end

       subroutine trianglei( nr, m1, n12, uu, vv, ierr)
	implicit none
C --- integer in, integer out
C------# Convolve with triangle
C------# input:  nr   rectangle width (points) (Triangle base twice as wide.)
C------# input:        uu(m1,i2),i2=1,n12      is a vector of data.
C------# output:       vv(m1,i2),i2=1,n12      may be on top of uu

#include <f77/iounit.h>

       integer nr,m1,n12, i,np,nq
       integer uu( m1, n12)
       integer vv( m1, n12)
c      real*4 pp(n12+nr-1), qq(n12+nr+nr-2), tt(n12)
       real*4 pp(*), qq(*), tt(*)
       integer ierr
       integer jerr,jerrt,jabort,jsz

       pointer(wkpp,pp)
       pointer(wkqq,qq)
       pointer(wktt,tt)

       data jerr / 0 /, jerrt / 0 /, jabort / 0 /, jsz / 0 /

       save jsz, wkpp, wkqq, wktt

       if (jsz .eq. 0) then
	 call sizefloat(jsz)
         call galloc(wkpp, (n12+nr-1)*jsz, jerr, jabort)
	 jerrt = jerrt + jerr
         if (jerr .ne. 0) then
	   write(LER,100) (n12+nr-1)*jsz
  100      format(' FXYDECON ERROR: Space allocation error - '
     :      i,' bytes requested; aborting')
	   write(LERR,100) (n12+nr-1)*jsz
         endif

         call galloc(wkqq, (n12+2*nr-2)*jsz, jerr, jabort)
	 jerrt = jerrt + jerr
         if (jerr .ne. 0) then
	   write(LER,100) (n12+2*nr-2)*jsz
	   write(LERR,100) (n12+2*nr-2)*jsz
         endif

         call galloc(wktt, n12*jsz, jerr, jabort)
	 jerrt = jerrt + jerr
         if (jerr .ne. 0) then
	   write(LER,100) n12*jsz
	   write(LERR,100) n12*jsz
         endif

	 if (jerrt .ne. 0) then
	   call abortfu()
	 endif
       endif

       do i=1,n12 
          qq(i) = uu(1,i) 
       enddo

       if( n12 .eq.  1 ) then
          do i=1,n12 
            tt(i) = qq(i) 
          enddo
       else 
          call boxconv( nr, n12, qq, pp, ierr)
          np = nr+n12-1
          call boxconv( nr, np , pp, qq, ierr)
          nq = nr+np-1

          do i= 1, n12
           tt(i) = qq(i+nr-1)
          enddo

          do i= 1, nr-1
           tt(i) = tt(i) + qq(nr-i)
          enddo

          do i= 1, nr-1
           tt(n12-i+1) = tt(n12-i+1) + qq(n12+(nr-1)+i)
          enddo

       endif

       do i=1,n12 
         vv(1,i) = nint( tt(i) )
       enddo

       return
       end

C---------------------------------------------
       subroutine conjnull( conj, add, x, nx,  y, ny )
        implicit none
       integer ix, iy, conj, add, nx, ny
       real*4 x( nx), y( ny )
       if( add .eq. 0 ) then
	       if( conj .eq. 0 ) then
		       do iy= 1, ny 
                           y(iy) = 0.
                       enddo
	       else
		       do ix= 1, nx 
                           x(ix) = 0. 
                       enddo
              endif
       endif
       return
       end

C---------------------------------------------
       integer function pad2cnt( n )
       integer n
       integer pad2
       pad2 = 1
       pad2cnt = 0
100    continue
       if ( pad2 .lt. n) then
          pad2 = pad2 * 2
          pad2cnt = pad2cnt + 1
          go to 100
       endif
       return
       end
C
C
C---------------------------------------------
      real*4 function dot( vec1, vec2, nt )
C
C      implicit none
C
      integer i
C       loop counter
      integer nt
C       length of vector
C
      real*4 vec1(nt)
C       data trace
      real*4 vec2(nt)
C       data trace
C
      dot = 0.0
C
      do 100 i=1,nt
         dot = dot + vec1(i) * vec2(i)
  100 continue
C
      return
      end
      REAL             FUNCTION SCNRM2( N, X, INCX )
*     .. Scalar Arguments ..
      INTEGER                           INCX, N
*     .. Array Arguments ..
      COMPLEX                           X( * )
*     ..
*
*  SCNRM2 returns the euclidean norm of a vector via the function
*  name, so that
*
*     SCNRM2 := sqrt( conjg( x' )*x )
*
*
*
*  -- This version written on 25-October-1982.
*     Modified on 14-October-1993 to inline the call to CLASSQ.
*     Sven Hammarling, Nag Ltd.
*
*
*     .. Parameters ..
      REAL                  ONE         , ZERO
      PARAMETER           ( ONE = 1.0E+0, ZERO = 0.0E+0 )
*     .. Local Scalars ..
      INTEGER               IX
      REAL                  NORM, SCALE, SSQ, TEMP
*     .. Intrinsic Functions ..
      INTRINSIC             ABS, AIMAG, REAL, SQRT
*     ..
*     .. Executable Statements ..
      IF( N.LT.1 .OR. INCX.LT.1 )THEN
         NORM  = ZERO
      ELSE
         SCALE = ZERO
         SSQ   = ONE
*        The following loop is equivalent to this call to the LAPACK
*        auxiliary routine:
*        CALL CLASSQ( N, X, INCX, SCALE, SSQ )
*
         DO 10, IX = 1, 1 + ( N - 1 )*INCX, INCX
            IF( REAL( X( IX ) ).NE.ZERO )THEN
               TEMP = ABS( REAL( X( IX ) ) )
               IF( SCALE.LT.TEMP )THEN
                  SSQ   = ONE   + SSQ*( SCALE/TEMP )**2
                  SCALE = TEMP
               ELSE
                  SSQ   = SSQ   +     ( TEMP/SCALE )**2
               END IF
            END IF
            IF( AIMAG( X( IX ) ).NE.ZERO )THEN
               TEMP = ABS( AIMAG( X( IX ) ) )
               IF( SCALE.LT.TEMP )THEN
                  SSQ   = ONE   + SSQ*( SCALE/TEMP )**2
                  SCALE = TEMP
               ELSE
                  SSQ   = SSQ   +     ( TEMP/SCALE )**2
               END IF
            END IF
   10    CONTINUE
         NORM  = SCALE * SQRT( SSQ )
      END IF
*
      SCNRM2 = NORM
      RETURN
*
*     End of SCNRM2.
*
      END

C*----------------------------------------------------------------------
C       fwarn
C*----------------------------------------------------------------------
      subroutine fwarn(message)
      character*(*) message

      write(*,*) message

      end
C*----------------------------------------------------------------------
C       fmess
C*----------------------------------------------------------------------
      subroutine fmess(message)
      character*(*) message

      write(*,*) message

      end

      subroutine zero( n, xx)
      integer i, n
      real xx(n)
      do 23000 i= 1, n
      xx(i) = 0.
23000 continue
      return
      end
      subroutine copy( n, xx, yy)
      integer i, n
      real xx(n), yy(n)
      do 23000 i= 1, n
      yy(i) = xx(i)
23000 continue
      return
      end

      subroutine null( xx, n)
      integer i, n
      real xx( n)
      do 23000 i= 1, n
      xx(i) = 0.
23000 continue
      return
      end

      subroutine adjnull( adj, add, x, nx, y, ny )
      integer ix, iy, adj, add, nx, ny
      real x( nx), y( ny )
      if(.not.( add .eq. 0 ))goto 23000
      if(.not.( adj .eq. 0 ))goto 23002
      do 23004 iy= 1, ny
      y(iy) = 0.
23004 continue
      goto 23003
23002 continue
      do 23006 ix= 1, nx
      x(ix) = 0.
23006 continue
23003 continue
23000 continue
      return
      end
      subroutine erexit( carray )
       character carray*(*)
       write(*,*) ' ****************************************** '
       write(*,*) ' ',carray
       write(*,*) ' ****************************************** '
       CALL U_ERR_FATAL( ' error in sep subroutine' )
      return
      end
C*----------------------------------------------------------------------
C       fwarn
C*----------------------------------------------------------------------
      subroutine U_ERR_FATAL(message)
      character*(*) message

      write(*,*) message
      stop
      end




c
c   - threshold testing supplied by John E.
c
c
      subroutine nanchk(buff,nsamp,thresh)
       implicit none

       integer nsamp
       real buff(nsamp)
       real thresh

       integer count
       integer i



        count=0
c
          do i=1,nsamp
             if( .not.
     :           (-thresh .lt. buff(i)
     :            .and.
     :            buff(i) .lt. thresh)
     :          )then
                   count=count+1
             end if
          end do
c
          if(count .ge. 1)then
              write(0,*)'Found samples above threshhold at vector#',
     1                                             thresh
              write(0,*)'There were ',count,' of them'
              write(0,*)' zeroing trace'
c
c
          do i=1,nsamp
                  buff(i)=0.
          end do

          end if


      return
      end

      integer function len_trim(stringg)

      character  stringg*(*)
      integer k

	len_trim_result = 0
	do k=len(stringg),1,-1
	  if (stringg(k:k) .ne. " ") then
            len_trim = k
            return
          endif
       enddo
     
       return 
       end 
