C***********************************************************************
C                 copyright 2001, Amoco Production Company             *
C                             All Rights Reserved                      *
C                     an affiliate of BP America Inc.                  *
C***********************************************************************
C**********************************************************************C
C
C     PROGRAM MODULE GROUPV
C
C*********************************************************************C
C
C READS SEISMIC TRACE FROM AN INPUT FILE,
C PERFORMS MULTIPLE FILTER ANALYSIS, AND PLOTS
C SPECTRAL AMPLITUDES AND GROUP VELOCITIES VERSUS FREQUENCY
C
C SUBROUTINE CALLS: RTAPE, HLH, WRTAPE, SAVE, BEGPLT, ENDPLT
C
C**********************************************************************C
C
C     DECLARE VARIABLES
C

#include <save_defs.h>
#include <f77/iounit.h>
#include <f77/lhdrsz.h>
#include <f77/sisdef.h>

      INTEGER     ITR ( SZLNHD )
      INTEGER     LHED( SZLNHD ), LUIN, NBYTES
      INTEGER     NSAMP, NSI, NTRC, NREC, IFORM, LBYTES
      integer     msamp
      INTEGER     START, STOP, ISTART
      integer     ns, ne, JJ
      REAL        TRI ( 8192 )
      REAL        DIST, TSTART
      CHARACTER   NAME * 6,  ntap * 256, stdout * 6

      real        f
      pointer     (wkadr, f(1))

#include <f77/pid.h>
       logical     verbos
       logical     query
       common/pltcnt/plton
       common/errout/outon
       common/group/gpvel(4,100)
       logical   plton
       logical   outon
       integer   argis
       integer   nper
       real      AA, BB, CC
       real      alpha0, alpha1
       real      pumin,pumax
       logical   lafln,lufln,heap
       real      dmul, tmul
 
c     EQUIVALENCE ( ITR(129), TRI (1) )
      EQUIVALENCE ( ITR(  1), LHED(1) )
      DATA NAME /'GROUPV'/, NBYTES / 0 /, LBYTES / 0 /
      DATA stdout/'stdout'/

C**********************************************************************C
C     get online help if necessary
C**********************************************************************C
       query = (argis('-?').gt.0 .or. argis('-h').gt.0)
       if( query ) then
           call help ()
           stop
       endif

C**********************************************************************C
C     open printout file
C**********************************************************************C
#include <f77/open.h>

C**********************************************************************C
C     READ PROGRAM PARAMETERS FROM A CARD IMAGE FILE
C**********************************************************************C
      call cmdln (ntap,dmul,tmul,start,stop,
     1               ns,ne,irs,ire,nper,pumin,pumax,lafln,lufln,
     2               aa,alpha0,alpha1,bb,cc,plton,outon,verbos)

C**********************************************************************C
C     verbos printout of arguments
C**********************************************************************C
c      if ( verbos ) then
       write(LERR,*)' Values read from file plot.crd'
       write(LERR,*)' Input data set name                     =  ',ntap
       write(LERR,*)' Start time in ms (default = 0 ms)       =  ',start
       write(LERR,*)' Distance multiplied by a factor         =  ',dmul
       write(LERR,*)' Time multiplied by a factor             =  ',tmul
       write(LERR,*)' Stop time in ms (default = full trace)  =  ',stop
       write(LERR,*)' Minimum velocity for plots              =  ',pumin
       write(LERR,*)' Maximum velocity for plots              =  ',pumax
       write(LERR,*)' Linear frequency contour plot           =  ',lafln
       write(LERR,*)' Linear velocity  contour plot           =  ',lufln
       write(LERR,*)' First record to process                 =  ',irs
       write(LERR,*)' Last record to process                  =  ',ire
       write(LERR,*)' Starting trace number                   =  ',ns
       write(LERR,*)' Ending   trace number                   =  ',ne
       write(LERR,*)' Peak contour values on error output     =  ',outon
       write(LERR,*)' Plot output                             =  ',plton
       write(LERR,*)' Freq Selection aa                          =  ',aa
       write(LERR,*)' Freq Selection bb                          =  ',bb
       write(LERR,*)' Freq Selection cc                          =  ',cc
       write(LERR,*)' Alpha selection AA                  =  ',alpha0
       write(LERR,*)' Alpha selection BB                  =  ',alpha1
       write(LERR,*)' Number of frequencies nper          =  ',nper
       write(LERR,*)' Start/end recs                      =  ',irs,ire
       write(LERR,*)' Start/end trcs                      =  ',ns,ne
c      end if
C**********************************************************************C

C**********************************************************************C
C     READ AND UPDATE LINE HEADER,
C     WRITE LINE HEADER, SAVE KEY PARAMETERS.
C**********************************************************************C
      call getln ( luin, ntap, 'r', 0 )

      if (luin .eq. 0) then
         write(LERR,*)'Input can not be a pipe - sorry'
         write(LERR,*)'Re-run with -N[file name] on cmd line'
         write(LERR,*)'and use the -rs -re -ns -ne cmd line'
         write(LERR,*)'arguments to select the desired traces'
c        stop
      endif

      CALL RTAPE ( LUIN, itr, LBYTES           )
      if(lbytes .eq. 0) then
         write(LERR,*)'GROUPV: no header read on unit ',ntap
         write(LERR,*)'GROUPV: no header read on luin= ',luin
         write(LERR,*)'FATAL'
         write(LERR,*)'Check existence of file & rerun'
         stop
      endif
      CALL HLHprt ( ITR , LBYTES, name, 6, LERR         )

c----------------------------------
c  save key header values
#include <f77/saveh.h>

      call savelu('TrcNum',ifmt_TrcNum,l_TrcNum,ln_TrcNum,TRACEHEADER)
      call savelu('RecNum',ifmt_RecNum,l_RecNum,ln_RecNum,TRACEHEADER)
      call savelu('SrcLoc',ifmt_SrcLoc,l_SrcLoc,ln_SrcLoc,TRACEHEADER)
      call savelu('RecInd',ifmt_RecInd,l_RecInd,ln_RecInd,TRACEHEADER)
      call savelu('DphInd',ifmt_DphInd,l_DphInd,ln_DphInd,TRACEHEADER)
      call savelu('DstSgn',ifmt_DstSgn,l_DstSgn,ln_DstSgn,TRACEHEADER)
      call savelu('DstUsg',ifmt_DstUsg,l_DstUsg,ln_DstUsg,TRACEHEADER)
      call savelu('StaCor',ifmt_StaCor,l_StaCor,ln_StaCor,TRACEHEADER)

c----------------------------------

c------
c       set up number of points for FFT
c------
       NPTS = 1
 6000          continue
       NPTS = 2*NPTS
       if(NPTS.lt.NSAMP)goto 6000
       NPTS21 = NPTS/2 + 1
       if ( verbos ) then
              write(LERR,*)' '
              write(LERR,*)' FFT NPTS = ',NPTS
       endif

c---------------------------------------------------
c  malloc only space we're going to use
       heap = .true.
       item1 = npts * nper * SZSMPD

c  note also SZSMPD is the native
c  size of a float or int in bytes
c--------------------------

c--------
c  galloc - general allocation (machine independent since it uses C
c  malloc internally
c  inputs to galloc are pointer, number of bytes to allocate
c  outputs are error codes:
c     errcod = 1  (allocation succeeded)
c     errcod = 0  (allocation failed)
c--------

      call galloc (wkadr, item1, errcd1, abort1)

      if (errcd1 .ne. 0.) heap = .false.
      if (.not. heap) then
         write(LERR,*)' '
         write(LERR,*)'Unable to allocate workspace:'
         write(LERR,*) item1,'  bytes'
         write(LERR,*)' '
         go to 999
      else
         write(LERR,*)' '
         write(LERR,*)'Allocating workspace:'
         write(LERR,*) item1,'  bytes'
         write(LERR,*)' '
      endif
c---------------------------------------------------

c-----
c       hard limit on data set
c-----
       msamp = NSAMP
       if(msamp .gt. SZLNHD) msamp =SZLNHD
       nbyte = SZTRHD + SZSMPD * nsamp

c-------------------------------------------
c  set default rec & trc end points

      call cmdchk ( ns, ne, irs, ire, ntrc, nrec )

c-------------------------------------------
c  verbos printout of line header vealues
c      if ( verbos ) then
       write(LERR,*)' '
       write(LERR,*)' Values read from input data set lineheader'
       write(LERR,*)' # of Samples/Trace =  ',nsamp
       write(LERR,*)' Sample Interval    =  ',nsi
       write(LERR,*)' Traces per Record  =  ',ntrc
       write(LERR,*)' Records per Line   =  ',nrec
       write(LERR,*)' Format of Data     =  ',iform
       write(LERR,*)' '
       write(LERR,*)' Start/end rec #    =  ',irs,ire
       write(LERR,*)' Start/end trc #    =  ',ns,ne
       write(LERR,*)' '
c      end if
c-------------------------------------------

C**********************************************************************C
C     CHECK DEFAULT VALUES AND SET PARAMETERS
C**********************************************************************C
      TSTART =  REAL ( START / 1000 )
      ISTART =      ( START / NSI ) + 1
      ISTOP  =      ( STOP  / NSI ) + 1
        IF ( ISTART . LT. 1     .OR. ISTART.GT. NSAMP ) THEN
           ISTART = 1
        ENDIF
        IF ( ISTOP . LE. ISTART .OR. ISTOP .GT. NSAMP ) THEN
           ISTOP  = NSAMP
        ENDIF
c-----
c       hard limit of 8192 points
c-----
       IF(ISTOP .GT. 8192)ISTOP=8192

c------------------------------------------------
c  skip to start record
      call recskp(1,irs-1,luin,ntrc,itr)
c------------------------------------------------

C**********************************************************************C
C
C     READ TRACE, DO MULTIPLE FILTER ANALYSIS, EXTRACT GRP VEL,
C     OUTPUT PLOTS AND/OR MAXIMA 
C
C**********************************************************************C

         dt = real (nsi) * unitsc

       dt = dt * tmul
       k=0

c------------------------------
c  initialize plots
       if(plton)then
c             call pinit()
              call pinitf(stdout)
       endif
c------------------------------

      t0 = 0.
      DO 1000 JJ = IRS, IRE

c---------------------------------------------
c  skip to start trace
            call trcskp(jj,1,ns-1,luin,ntrc,itr)
c---------------------------------------------

            DO 1001 KK = NS, NE
                  nbytes = 0
                  call rtape(luin,itr,nbytes)
                  if(nbytes .eq. 0) then
                     write(LERR,*)'End of file on input:'
                     write(LERR,*)'  rec= ',jj,'  trace= ',kk
                     go to 999
                  endif
                  call vmov (itr(ITHWP1), 1, tri, 1, nsamp)
                  call saver2(lhed,ifmt_StaCor,l_StaCor, ln_StaCor,
     1                        istatic, TRACEHEADER)
                  call saver2(lhed,ifmt_RecNum,l_RecNum, ln_RecNum,
     1                        irec   , TRACEHEADER)
                  call saver2(lhed,ifmt_TrcNum,l_TrcNum, ln_TrcNum,
     1                        itrc   , TRACEHEADER)
                  call saver2(lhed,ifmt_DstSgn,l_DstSgn, ln_DstSgn,
     1                        idis   , TRACEHEADER)

                  IF(istatic .ne. 30000) THEN
                     dist = iabs( idis ) * dmul
                     write(LERR,*)'Rec= ',irec,'Trc= ',itrc,
     1                            '  dist= ',dist

                     call gpv(f,NPTS,dt,dist,tri,ntap,JJ,KK,NSI,NSAMP,
     1                 ISTART,ISTOP,nper,aa,bb,cc,alpha0,alpha1,
     2                  pumin,pumax,lafln,lufln,t0)

                     if(plton)then
                        if( JJ.ne.ire .or. KK.lt.ne) then
                            call frame()
                        endif
                     endif
                  ELSE
                     write(LERR,*)'Rec ',jj,' trace ',kk,
     1               ' is dead'
                  ENDIF

 1001       CONTINUE

c---------------------------------------------
c  skip to end of this record
            call trcskp(jj,ne+1,ntrc,luin,ntrc,itr)
c---------------------------------------------


 1000 CONTINUE
  999 continue

c----------------------------------
c  close plots; input unit
      if(plton)call pend()
      call lbclos (luin)
c----------------------------------
      END

       subroutine gpv(f,n,dt,dist,tri,ntap,nrec,ktrc,NSI,NSAMP,
     1             ISTART,ISTOP,mper,aa,bb,cc,alpha0,alpha1,
     2         pumin,pumax,lafln,lufln,t0)

#include <f77/iounit.h>

c-----
c       general subroutine for performing Gaussian Filter Group Velocity analysis
c
c       REFERENCES:
c              Herrmann, R. B. (1973). Some aspects of band-pass
c                     filtering of surface waves, Bull. Seism. Soc.
c                     Am. 63, 663-671 (spectral amplitudes)
c
c              Gaussian filters introduced by Landismann and Dziewonski
c              modified by others, e.g., filter parameter dependent
c              upon frequency
c
c-----
c       SUBROUTINE ARGUMENTS:
c
c       n       I*4       number of points for FFT, power of 2
c       d4       sampling interval in seconds
c       dist       R*4       distance from source to receiver
c       tri       R*4       array of original time series
c       nC*120       string giving file name
c       nrec       I*4       record number of trace
c       ktrc       I*4       trace number in record
c       NSI       I*4       sample interval in milliseconds
c       NSAMP       I*4       number of samples in trace <=n
c       ISTART       I*4       process time series starting at this sample
c       ISTOP       I*4       process time series ending at this sample
c       mper       I*4       number of filter center frequencies
c       aa       R*4       defines coefficients of center frequencies
c       bb       R*4       cc = 0.0 centfreq = aa + bb*I
c       cc       R*4       cc gt  0 centfreq = 10 **(aa+bb*I)
c       cc       R*4       cc lt  0 centfreq = ( aa + bb*i)*df
c                     I = 1,nper
c       alpha0       R*4       defines Gaussian filter parameter
c       alpha1       R*4              alpha = alpha0+alpha1*centfreq
c       pumin       R*4       minimum group velocioty for plots
c       pumax       R*4       maximum group velocity for plots
c       lafln       L       .true. linear frequency scale
c                     .false. logarithmic frequency scale
c       lufln       L       .true. linear group velocity scale
c                     .false logarithmic velocity scale
c       tt0       R*4       absolute time of first sample
c                     with respect to source origin time
c-----
c       CONTROL COMMON BLOCKS FROM MAIN PROGRAM
c
c       common/pltcnt/plton
c       
c       plton       L       .true. plot
c                     .false. no plot
c       common/errout/outon
c
c       outon       L       .true. peak amp, vel, freq output on error unit
c                     .false. no such output for later analysis
c-----
c
              complex  datas(8192)
              complex  data(8192)
c             real     f(8192,80)
              real     f(n, mper)
              real     wn(80)
              real     tri(*)
              integer   nrec,ktrc,NSI
              integer   NSAMP
              integer   ISTART, ISTOP, nper, mper
              real*4    AA, BB, CC
              character ntap * 100
              common/pltcnt/plton
              logical   plton
              common/errout/outon
              logical   outon
              common/group/gpvel(4,100)
              real      pumin,pumax
              logical   lufln,lafln
       save data,datas,wn
              
c-----
c       open scratch file to store filtered envelopes
c-----
c      open(1,access='sequential',form='unformatted',status='scratch')
c      rewind 1
c-----
c       initialize array
c-----
              DO 5000 I = 1,80
                     DO 5001 J = 1,4
                            gpvel(J,I) = 0.0
 5001                      continue
 5000               continue
c-----
c       set up frequencies to be used
c-----
              fnyq = 1./(2.*dt)
              wmin = 1.0e+38
              wmax = -1.0e+38
              nper = 0
                     df = 1./(n*dt)
c-----
c       if cc .lt. 0.0 adjust the aa,bb valuees
c-----
       if(cc.lt.0.0)then
              aa = aa * df
              bb = bb * df
       endif
              DO 5002 II=1,mper
                     wn(II) = fmap(real(II),aa,bb,cc)
                     if(wn(II).le.fnyq .and. wn(II).gt. 0.0)then
                            nper=nper + 1
                            if(wn(II).gt.wmax)wmax=wn(II)
                            if(wn(II).lt.wmin)wmin=wn(II)
                     endif
 5002               continue
              if(nper.eq.0)then
                     write(LER,*)'GPV: nper = 0 check aa,bb,cc'
                     return
              endif
c-----
c       obtain Fourier Transform of original Time series
c       and save in the array datas()
c-----
              do 5003 LL=1,n
                     datas(LL)=cmplx(0.0,0.0)
 5003               continue
              do 5004 LL=ISTART,ISTOP
                     datas(LL)=cmplx(tri(LL),0.0)
 5004               continue
              call four(datas,n,-1,dt,df)
c-----
c       narrow band pass filter at each frequency
c       and store filtered time series in the array f(8192,80)
c-----
c-----
c       narrow band pass filter using gaussian filter, passing
c       rejecting frequencies for which the filter response is
c       less than exp(-3.1415927)
c-----
      do 999 jump = 1,nper
              alpha = alpha0 + alpha1*wn(jump)**2
              fac = sqrt(3.1415927/alpha)
              frequp = (1.0+fac)*wn(jump)
              freqlw = (1.0-fac)*wn(jump)
              n21 = n/2 + 1
             do 1002 i=1,n21
             xi = i - 1
             freq = xi *df
              if(freq.ge.freqlw .and. freq.le.frequp)then
                           fact = -alpha*((freq-wn(jump))/(wn(jump)))**2
                           filt = exp(fact)
                           data(i) = filt*datas(i)
       else
              data(i) = cmplx(0.0,0.0)
       endif
c-----
c       zero out negative frequencies
c-----
              if(i.gt.1)data(n+2-i)=cmplx(0.0,0.0)
 1002 continue
      call four(data,n,+1,dt,df)
       do 5005 KK = 1,n
              f(KK,jump) = cabs(data(KK))
 5005        continue
c      write(1)f
  999 continue
       call spcamp(f,datas,n,nper,wn,df,t0,dt,dist,ntap,nrec
     1     ,ktrc,nsi,NSAMP,alpha0,alpha1,wmin,wmax,lafln)
       if(plton)then
              call conplt(f,n,nper,df,t0,dt,dist,ntap,nrec
     1     ,ktrc,nsi,NSAMP,tri,aa,bb,cc,wmin,wmax,
     2         pumin,pumax,lafln,lufln,alpha0,alpha1)
       endif
c      close (1)
       return
       end

       subroutine spcamp(f,datas,n,nper,wn,df,t0,dt,dist,ntap,nrec
     1               ,ktrc,nsi,NSAMP,alpha0,alpha1,wmin,wmax,lafln)
       real         f(n,nper)
       complex      datas(*)
       real         wn(*)
       character    ntap * 100
       integer      nrec,ktrc,nsi,NSAMP
       character*20 titlx,titly
       common/pltcnt/plton
       logical      plton
       real         alpha0,alpha1,wmin,wmax
       logical      lafln
c-----
c       plot multiple filter analysis traces
c       as spectral amplitudes with group velocity
c       annotation
c-----
       if(plton)then
              call plot(2.0,1.0,-3)
              call title(0.0,6.1,ntap,nrec,ktrc,nsi,dist,NSAMP,
     1                alpha0,alpha1)
c-----
c       get limits of spectral amplitude
c       from original unfiltered spectra
c-----
       np1 = n/2 + 1
       ymax = 0.0
       do 805 j = 1,np1
              freq = (j-1)*df
              if(freq.ge.wmin .and. freq.le.wmax)then
                     dmm = cabs(datas(j))
                     if(dmm.gt.ymax)ymax = dmm
              endif
  805  continue
c-----
c       set up plot parameters
c       if lafln .eq. .true then linear frequency range
c-----
       if(lafln)then
              nocx = 0
              xaxlen = 6.0
              x1 = wmin
              deltax = (wmax - wmin)/xaxlen
       else
              fmin = alog10(wmin)
              lmin = fmin
              x1 = lmin
              if(fmin.lt.0.0)x1 = x1 - 1.0
              nocx = 2
              xaxlen = 6.0
              deltax = nocx/xaxlen
       endif
       mtx =14
       titlx = 'FREQUENCY (HZ)'
c-----
c       set up y-axis logarithmin
c-----
              if(ymax.eq.0.0)ymax = 1.0
              ymax = alog10(ymax)
              lmax = ymax
              if(ymax.gt.real(lmax))lmax = lmax + 1
              lmin = lmax - 3
              y1 = lmin
              yaxlen = 6.0
              nocy = 3
              deltay = nocy/yaxlen
              titly = 'AMPLITUDE (CM-SEC)'
              mty = 18
       call algax(xaxlen,yaxlen,nocx,nocy,titlx,titly,mtx,mty,
     1        x1,y1,deltax,deltay)
       endif
c-----
c       search filtered traces for maximum amplitudes,
c       determine group velocities, estimate spectral amplitude
c       plot if ploton = .true.
c-----
c      rewind 1
       DO 5006 KK=1,nper
c             read(1)f
              call maxval(f(1,kk),n,KK,wn(KK),x1,deltax,y1,deltay,
     1                 t0,dt,dist,alpha0,alpha1,lafln)
 5006        continue
       if(plton)then
              call plot(-2.0,-1.0,-3)
              call frame()
       endif
       return
       end

       subroutine maxval(f,n,nper,w0,x1,deltax,y1,deltay,t0,dt,dist,
     1                 alpha0,alpha1,lafln)
#include <f77/iounit.h>
       real*4 f(8192)
              common/group/gpvel(4,100)
       common/pltcnt/plton
       logical plton
       common/errout/outon
       logical outon
       logical lafln
c-----
c       search the envelop of the filtered trace for maxima
c       the index of the maximum yields the group velocity
c       For a cleaner plot, no more than 4 maxima are plotted
c       at a given frequency
c-----
       real*4 u(4),a(4),ti,amp1,amp2,amp3,amp4
       integer*4 MAX4, kk
c-----
c       MAX4 = 4 to plot out only the four largest amplitudes
c-----
       MAX4 = 4
c-----
c       get proper factor to convert envelope maxima to spectral
c       amplitudes
c-----
       if (outon) open(LUN,file='GROUPV.o')
       alpha = alpha0 + alpha1*w0**2
       afac = sqrt(alpha/3.1415927)/w0
       if(plton)then
              if(lafln)then
                     x = (w0 - x1)/deltax
              else
                     x = (alog10(w0) - x1)/deltax
              endif
       endif
       kk = 0
       do  388  i = 1, 4
                a(i) = 0.
                u(i) = 0.
388    continue
       amp1 = f(1)
       amp2 = f(2)
       amp3 = 0.
       amp4 = 0.
      nm1 = n - 1
c     print*,'nper= ',nper
      do 500 i = 3,nm1
      ti = t0 + float(i-2) * dt
      amp3 = f(i)
      if( (amp2-amp1).gt.1.0e-7*amp1.and.(amp2-amp3).ge.
     1 1.0e-7*amp3)go to 400
      go to 401
  400 amp4 = f(i+1)
      if((amp2-amp4).lt.1.0e-7*amp4) go to 401
  402  continue
c     print*,'ti= ',i,dt,ti,amp1,amp2,amp3,amp4
       if(ti.gt.0.0 .and. amp2.gt.0.0)then
              kk = kk + 1
              vel = dist/ti
              ampl = amp2 * afac
c      print*,'kk= ',kk,vel,ampl
              call srt(a,u,vel,ampl,kk,MAX4)
c     print*,(a(ii),ii=1,4)
c     print*,(u(ii),ii=1,4)
      
       endif
  401 amp1 = amp2
      amp2 = amp3
c     print*,'401: ',i,dt,ti,amp1,amp2,amp3,amp4
      
  500 continue
c-----
c       plot the four largest amplitudes
c-----
       DO 5008 i=1,MAX4
c     print*,'5008: i= ',i,u(i),a(i)
            if(a(i).gt.0.0)then
              if(plton)call plt(u(i),a(i),x,y1,deltay)
              gpvel(i,nper) = u(i)
              if(outon)then
              write(LUN,'(f19.5,f19.5,f19.5,e19.5)')w0,dist,u(i),a(i)
              endif
       endif
 5008        continue
      return
      end

       subroutine srt(a,u,vel,ampl,k,MAX4)
c-----
c       subroutine to maintain a list of the MAX4
c       largest values of ampl in a(i)
c       also saving the corresponding vel in v(i)
c
c       This is being done to avoid have two very large
c       arrays in overhead and because we do not need
c       to do a full sort
c-----
       real*4 a(1), u(1), vel, ampl
       integer*4 MAX4, k
       integer*4 MAX41
       integer*4 key(10)
       real*4 tmp(10)
       real*4 tamp(10), tvel(10)
       MAX41 = MAX4 + 1
       if(k .eq. 1)then
              do 100 i=1,MAX4
                     a(i) = 0.0
                     u(i) = 0.0
  100               continue
       endif
       if(k.lt.MAX4)then
              kup = k
       else
              kup = MAX4
       endif
c-----
c       assume amplitudes arranged in decreasing order
c-----
c-----
c       we now know that the value will replace one of the amplitudes,
c       at least the lowest
c-----
       do 200 i=1,MAX4
              tmp(i) = a(i)
              tamp(i) = a(i)
              tvel(i) = u(i)
  200        continue
              tmp(MAX41) = ampl
              tamp(MAX41) = ampl
              tvel(MAX41) = vel
              call sort(tmp,key,MAX41)
              do 250 i= 1,MAX4
                     kk = key(MAX41 +1 - i)
                     a(i) = tamp(kk)
                     u(i) = tvel(kk)
  250               continue
       return
       end
       subroutine sort(x,key,no)
       dimension x(1),key(1)
       do 1 i=1,no
    1   key(i) = i
       mo = no
2       if(mo-15)21,21,23
21       if(mo-1)29,29,22
22       mo=2*(mo/4)+1
       go to 24
23       mo=2*(mo/8)+1
24       ko=no-mo
       jo=1
25       i=jo
26       if(x(i)-x(i+mo))28,28,27
27      temp=x(i)
       x(i)=x(i+mo)
       x(i+mo)=temp
       kemp = key(i)
       key(i) = key(i+mo)
       key(i+mo) = kemp
       i=i-mo
       if(i-1)28,26,26
28       jo=jo+1
       if(jo-ko)25,25,2
29       return
       end

      subroutine plt(u,amp,x,y1,deltay)
       g = u
      amp = abs(amp)
      if(amp.le.0) go to 300
      y = alog10(amp)
      if(y.lt.y1) go to 300
      y = (y-y1)/deltay
      call symbol(x,y,0.07,char(3),0.0,-1)
c-----
c       do make clear output, do not plot group velocity adjacent to
c       spectral amplitude
c      call number(x,y+0.07,0.07,g,90.0,-1)
c-----
  300 continue
      return
      end

       subroutine conplt(f,n,nper,df,t0,dt,dist,ntap,nrec,
     1              ktrc,nsi,NSAMP,tri,aa,bb,cc,wmin,wmax,
     2              pumin,pumax,lafln,lufln,alpha0,alpha1)
              real         f(n,nper)
              integer      n
              integer      nper
              real         df
              real         t0,dt,dist
              integer      nrec,ktrc,NSI
              integer      NSAMP
              real         aa,bb,cc
              real         tri(*)
              logical      lufln,lafln
              character*25 titlx,titly
              common/group/gpvel(4,100)
              character    ntap * 100
              common/pptcon/pfmin,pfmax,pvmin,pvmax,x1,deltax,nocx
     1          ,xaxlen,y1,deltay,nocy,yaxlen,kafln,kufln
              logical      kafln,kufln
c      call pinitf("stdout")
       call plot(0.75,1.0,-3)
c-----
c       normalize each trace individually
c-----
c-----
c       set up plot parameters
c-----
c       if lafln = .true. linear frequency
c              else       logarithmic
c       if lufln = .true. linear group velocity
c              else logarithmic
c-----
       if(lafln)then
              nocx = 0
              xaxlen = 6.0
              x1 = wmin
              deltax = (wmax - wmin)/xaxlen
              pfmin = wmin
              pfmax = wmax
       else
              fmin = alog10(wmin)
              lmin = fmin
              x1 = lmin
              if(fmin.lt.0.0)x1 = x1 - 1.0
              nocx = 2
              pfmin = 10.0**x1
              pfmax = pfmin * 10.0**nocx
              xaxlen = 6.0
              deltax = nocx/xaxlen
       endif
       kafln = lafln
       kufln = lufln
       mtx = 14
       titlx = 'FREQUENCY (HZ)'
       titly = 'GROUP VELOCITY (FT/SEC)'
       mty = 23
       if(lufln)then
              y1 = pumin
              nocy = 0
              yaxlen = 6.0
              deltay = (pumax - pumin)/yaxlen
              pvmin = pumin
              pvmax = pumax
       else
              vmin = alog10(pumin)
              lmin = vmin
              y1 = lmin
              if(vmin.lt.0.0)then y1 = y1 - 1.0
              nocy = 2
              pvmin = 10.0**y1
              pvmax = pvmin * 10.0**nocy
              yaxlen = 6.0
              deltay = nocy/yaxlen
       endif
       call algax(xaxlen,yaxlen,nocx,nocy,titlx,titly,mtx,mty,
     1     x1,y1,deltax,deltay)
       call title(0.0,6.1,ntap,nrec,ktrc,nsi,dist,NSAMP,
     1         alpha0,alpha1)
       call plot(0.0,0.0,3)
c-----
c       PLOT THE GROUP VELOCITIES OF THE FOUR LARGEST SPECTRAL
c       AMPLITUDES ON TOP OF THE CONTOUR PLOT. WE USE THE
c       ARRAY SET UP BY maxval AND STORED IN common/group/gpval
c-----
       fold = 0.0
       vold = 0.0
       DO 5009 jump = 1,nper
              DO 5010 KK = 1,4
c     print*,'JUMP= ',jump,' kk= ',kk,gpvel(kk,jump),t0,dist,dt
                     if(gpvel(KK,jump).ne.0.0)then
                            xx = jump
                            yy = 1 + (dist/gpvel(KK,jump)-t0)/dt
                            call pplot(xx,yy,dist,dt,x,y,aa,bb,cc,
     1                       fold,vold,iret,t0)
                            call symbol(x,y,0.02,char(3),0.0,-1)
                     endif
 5010               continue
 5009        continue
c     print*,'end 5009'
       dx = 1.0
       dy = 1.0
c-----
c       DRAW CONTOURS USING A LOGARITHMIC SCALE
c-----
       DO 5011 KVAL = 50,100,10
              val = KVAL/100.0
              val = 0.10* 10.0**val
c     print*,'KVAL= ',n,nper,val,dx,dy,dist,dt
              call contur(f,n,nper,val,dx,dy,dist,dt,aa,bb,cc,t0)
 5011        continue
              trimax = 0.0
       DO 5012 KK=1,NSAMP
              if(abs(tri(KK)).gt.trimax)trimax = abs(tri(KK))
 5012        continue
       ipen = 3
       fold  = 0.0
       vold =  0.0
c-----
c       plot trace on velocity scaled time axis to right of contours
c-----
       DO 5013 KK=1,NSAMP
              x = xaxlen + 0.25 + 0.25*tri(KK)/trimax
              call pplot(xx,real(KK),dist,dt,xxx,y,aa,bb,cc,
     1                       fold,vold,iret,t0)
              call plot(x,y,ipen)
              ipen = 2
 5013        continue
       x1 = xaxlen
       x2 = x1 + 0.5
       y1 = 0.0
       y2 = yaxlen
       call box(x1,x2,y1,y2)
c-----
c       plot trace to right on a linear scale
c-----
       x1 = x1 + 0.6
       x2 = x2 + 0.6
       call box(x1,x2,y1,y2)
       ipen = 3
       DO 5014 KK=1,NSAMP
              x = xaxlen + 0.6 + 0.25 + 0.25*tri(KK)/trimax
              y = yaxlen -(KK-1)*yaxlen/NSAMP
              call plot(x,y,ipen)
              ipen = 2
 5014        continue
c-----
c       put maximum amplitude at base of trace
c-----
              call number(x1,y1-0.20,0.07,trimax,0.0,1003)
              call plot(0.0,0.0,3)
       call plot(-2.0,-1.0,-3)
       return
       end


       subroutine contur(f,imax,jmax,val,dx,dy,dist,dt,aa,bb,cc,t0)
       dimension f(imax,jmax)
       logical interp
c-----
c       contouring program
c              plot a contour of level val from array f(imax,jmax)
c-----
       logical inarea
       real*4 aa,bb,cc
       ipen = 3
c   normalize matrix f
       call fnorm(imax,jmax,f)
       do 1000 jj = 2,jmax
c             call getf(f,jj,imax)
       do 1000 i = 2,imax
c-------
c
c       F(i-1,j-1)      F(i-1,j)
c
c       F(i,j-1)       F(i,j)
c
c-----
c-----
c       to save memory and to permit this to
c       work in a non-virtual machine, the  envelopes f(time,freq)
c       are stored sequentially on UNIT 01. At any one time we
c       need only access 2 columns, so we use j = 2
c       If the entire array can be stored in virtual memory, then
c       just redo the dimension to be f(8192,80) and make
c              do 1000 jj=2,jmax
c       be      do 1000  j=2,jmax
c-----
c             j = 2
              j = jj
              ipen = 3
              fold = 0.0
              vold = 0.0
              inarea = interp(F(i,j),F(i-1,j),val,pos)
              if(inarea)then
                     xx = (jj)*dx
                     yy = (i-pos)*dy
                     call pplot(xx,yy,dist,dt,x,y,aa,bb,cc,
     1                       fold,vold,iret,t0)
                     if(iret.gt.0)then
                                   call plot(x,y,ipen)
                                   ipen = 2
                     endif
              endif
              inarea = interp(F(i-1,j),F(i-1,j-1),val,pos)
              if(inarea)then
                     yy = (i-1)*dy
                     xx = (jj-pos)*dx
                     call pplot(xx,yy,dist,dt,x,y,aa,bb,cc,
     1                       fold,vold,iret,t0)
                     if(iret.gt.0)then
                                   call plot(x,y,ipen)
                                   ipen = 2
                     endif
              endif
              inarea = interp(F(i-1,j-1),F(i,j-1),val,pos)
              if(inarea)then
                     xx = (jj-1)*dx
                     yy = (i-1+pos)*dy
                     call pplot(xx,yy,dist,dt,x,y,aa,bb,cc,
     1                       fold,vold,iret,t0)
                     if(iret.gt.0)then
                                   call plot(x,y,ipen)
                                   ipen = 2
                     endif
              endif
              inarea = interp(F(i,j-1),F(i,j),val,pos)
              if(inarea)then
                     xx = (jj-1+pos)*dx
                     yy = (i)*dy
                     call pplot(xx,yy,dist,dt,x,y,aa,bb,cc,
     1                       fold,vold,iret,t0)
                     if(iret.gt.0)then
                                   call plot(x,y,ipen)
                                   ipen = 2
                     endif
              endif
              inarea = interp(F(i,j),F(i-1,j),val,pos)
              if(inarea)then
                     xx = (jj)*dx
                     yy = (i-pos)*dy
                     call pplot(xx,yy,dist,dt,x,y,aa,bb,cc,
     1                       fold,vold,iret,t0)
                     if(iret.gt.0)then
                                   call plot(x,y,ipen)
                     endif
                     ipen = 3
              endif
 1000         continue
       return
       end

       function interp(y0,y1,val,x)
       logical interp
       real y1,y0,val,x
       denom = y1-y0
       if(denom.eq.0.0 .and. val.ne.y0)then
              interp = .false.
       elseif(denom.eq.0.0 .and. val.eq.y0)then
              interp = .true.
              x = 0.0
       else
              x = (val-y0)/denom
              if(x.lt.0.0 .or. x.gt.1.0)then
                     interp = .false.
              else
                     interp = .true.
              endif
       endif
       return
       end

       subroutine getf(f,jj,n)
       dimension f(8192,2),g(8192)
c-----
c       get the filtered envelopes from UNIT 01
c       normalize the envelopes
c       then update the array f(8192,2) so that column 1 refers
c       always to the leftmost column, even after updating
c-----
c-----
c       use special care for the first entry, jj = 2
c-----
       if(jj.eq.2)then
              rewind 1
              read(1)g
              fmax = 0.0
              DO 5015 KK=1,n       
                     if(g(KK).gt.fmax)fmax=g(KK)
 5015               continue
              if(fmax .eq. 0.0)fmax = 1.0
              DO 5016 KK = 1,n
                     f(KK,2) = g(KK)/fmax
 5016               continue
       endif
c-----
c       normal processing, no check for EOF on read
c
c       1. read in new column
c       2. determine maximum amplitude of new column
c       3. normalize new column, put in f(KK,2)
c          move old f(KK,2) to f(KK,1)
c-----
              read(1)g
              fmax = 0.0
              DO 5017 KK=1,n       
                     if(g(KK).gt.fmax)fmax=g(KK)
 5017               continue
              if(fmax .eq. 0.0)fmax = 1.0
              DO 5018 KK = 1,n
                     f(KK,1) = f(KK,2)
                     f(KK,2) = g(KK)/fmax
 5018               continue
       return
       end

       subroutine pplot(xx,yy,dist,dt,x,y,aa,bb,cc,
     1                       fold,vold,iret,t0)
       real*4 xx,yy,dist,dt,x,y
       real*4 aa,bb,cc
       real*4 fmap,ti
              common/pptcon/pfmin,pfmax,pvmin,pvmax,x1,deltax,nocx
     1          ,xaxlen,y1,deltay,nocy,yaxlen,kafln,kufln
              logical kafln,kufln
c-----
c       maps xx index to frequency
c            yy index to time and thus velocity
c       then maps
c            frequency to log-log x coordinate
c            velocity  to log-log y coordinate
c
c-----
c       velocity goes from pvmin to pvmax
c       frequency goes from pfmin to pfmax
c
c       all other points are not plotted
c-----
c
c       E.g. in array f(8192,80), this is f(i,1)
c
c       IT IS ASSUMED THAT yy = 0.0 CORRESPONDS TO TIME 0.0 sec
c       E.g. in array f(8192,80), this is f(1,j)
c
c-----
c
c       assume (yy-1)*dt  + t0 = time
c-----
       iret = 0.0
       ti = (yy-1)*dt + t0
       freq = fmap(xx,aa,bb,cc)
       if(ti.lt.1.0e-10)ti=1.0e-10
       vel = dist/ti
       if(vel.lt.0.0)vel = abs(vel)
       if(fold.ge.pfmin .and. fold.le.pfmax .and. vold.ge.pvmin
     1         .and. vold.le.pvmax)iret = iret +1
       if(freq.ge.pfmin .and. freq.le.pfmax .and. vel.ge.pvmin
     1         .and. vel.le.pvmax)iret = iret +1
       fold = freq
       vold = vel
       if(freq.lt.pfmin)freq=pfmin
       if(freq.gt.pfmax)freq=pfmax
       if(vel.lt.pvmin)vel = pvmin
       if(vel.gt.pvmax)vel = pvmax
       if(kafln)then
              x = (freq - pfmin)/deltax
       else
              x = alog10(freq/pfmin)/deltax
       endif
       if(kufln)then
              y = (vel - pvmin)/deltay
       else
              y = alog10(vel/pvmin)/deltay
       endif
       return
       end


       subroutine box(x1,x2,y1,y2)
              call plot(x1,y1,3)
              call plot(x2,y1,2)
              call plot(x2,y2,2)
              call plot(x1,y2,2)
              call plot(x1,y1,2)
       return
       end

      subroutine algax(xaxlen,yaxlen,nocx,nocy,ttlx,ttly,mtx,mty,x1,
     1y1,deltax,deltay)
      character*(*) ttlx,ttly
      slt = -0.02*yaxlen
      sst = -0.01 * yaxlen
      sp = -0.06*yaxlen
      ss = 0.035*yaxlen
      ssp = sp + ss - 0.06
      ttlp = -0.11*yaxlen - 0.1
      sttl = 0.035*yaxlen
      xnum = 1
      yl = y1
c - changed anocy -> nocy - apparent typo - j.m.wade 8/21/92
c     yu = y1 + anocy
      yu = y1 + nocy
      if(abs(yl).ge.10. .or. abs(yu).ge.10. )xnum = xnum + 1.
      if(abs(yl).ge.100..or. abs(yu).ge.100.)xnum = xnum + 1.
      if(y1.lt.0) xnum = xnum + 1.0
       call plot(0.0,0.,3)
      call plot(-slt,0.0,2)
      call plot(0.0,-slt,3)
      call plot(0.0,0.0,2)
      xpo = x1
      ypo = y1
      if(nocx.eq.0) go to 4
      anocx = nocx
      factx = xaxlen/anocx
      call symbol(-.6*ss,sp,ss,'10',0.0,2)
      call number(999.,ssp,0.5*ss,x1,0.0,-1)
      call plot(0.0,0.0,3)
      do 3 j = 1,nocx
      do 2 i = 1,10
      x = i
      x = alog10(x) *factx + (j-1)*factx
      if(i.eq.1)go to 2
      call plot(x,0.0,2)
      call plot(x,-sst,2)
    2 call plot(x,0.0,3)
      call plot(x,-slt,2)
      call symbol(x-.6*ss,sp,ss,'10',0.0,2)
      xpo = xpo + 1.0
      call number(999.,ssp,0.5*ss,xpo,0.0,-1)
    3 call plot(x,0.0,3)
      xtl = mtx
      xtl = (xaxlen-xtl*sttl)/2.0
      call symbol(xtl,ttlp,sttl,ttlx,0.0,mtx)
c-----
c       put tic marks at the top of the plot
c-----
      do 30 j = 1,nocx
      do 20 i = 1,10
      x = i
      x = alog10(x) *factx + (j-1)*factx
      if(i.eq.1)go to 20
      call plot(x,yaxlen,2)
      call plot(x,yaxlen+sst,2)
   20 call plot(x,yaxlen,3)
      call plot(x,yaxlen+slt,2)
   30 call plot(x,yaxlen,3)
      go to 6
    4 call axis(0.0,0.0,ttlx,-mtx,xaxlen,0.0,x1,deltax)
c-----
c       put in scale at top of the plot
c-----
       ntic = xaxlen + 1.0
       xx = 0.0
       yy = yaxlen
       call plot(xx,yy,3)
       do 101 i=1,ntic
              call plot(xx,yy,2)
              call plot(xx,yy+0.07,2)
              call plot(xx,yy,2)
              xx = xx + 1.0
  101         continue
    6 call plot(0.0,0.0,3)
      if(nocy.eq.0) go to 10
      anocy = nocy
      sp = sp - (xnum - 1.5) * 0.5 * ss
      ttlp = ttlp - (xnum-1.)*0.5*ss
      facty = yaxlen/anocy
      call symbol(sp-0.4,-0.5*ss,ss,'10',0.0,2)
      call number(999.,.5*ss-.06,.5*ss,y1,0.0,-1)
      call plot(0.0,0.0,3)
      do 9 j = 1,nocy
      do 8 i = 1,10
      y = i
      y = alog10(y) * facty + (j-1)*facty
      if(i.eq.1)go to 8
      call plot(0.0,y,2)
      call plot(-sst,y,2)
    8 call plot(0.0,y,3)
      call plot(-slt,y,2)
      call symbol(sp-.4,y-.5*ss,ss,'10',0.0,2)
      ypo = ypo + 1
      call number(999.,y+.5*ss-.06,.5*ss,ypo,0.0,-1)
    9 call plot(0.0,y,3)
      ytl=mty
      ytl = (yaxlen-ytl*sttl)/2.0
      call symbol(ttlp-.2,ytl,sttl,ttly,90.,mty)
      do 90 j = 1,nocy
      do 80 i = 1,10
      y = i
      y = alog10(y) * facty + (j-1)*facty
      if(i.eq.1)go to 80
      call plot(xaxlen,y,2)
      call plot(xaxlen+sst,y,2)
   80 call plot(xaxlen,y,3)
      call plot(xaxlen+slt,y,2)
   90 call plot(xaxlen,y,3)
       call plot(0.0,0.0,3)
      return
   10 call axis(0.0,0.0,ttly,mty,yaxlen,90.,y1,deltay)
c-----
c       put in scale at right of the plot
c-----
       ntic = yaxlen + 1.0
       yy = 0.0
       xx = xaxlen
       call plot(xx,yy,3)
       do 201 i=1,ntic
              call plot(xx,yy,2)
              call plot(xx+0.07,yy,2)
              call plot(xx,yy,2)
              yy = yy + 1.0
  201         continue
       call plot(0.0,0.0,3)
      return
      end

       subroutine title(x,y,ntap,nrec,ktrc,nsi,dist,n,alpha0,alpha1)
       character ntap*100
       integer*4 nrec,ktrc,nsi
       real*4 x,y,dist
c-----
c       put up title at top of plot
c-----
       integer*4 lgstr
              nchar = lgstr(ntap,100)
	      if (nchar .ne. 0)
     1          call symbol(x,y+0.40,0.10,ntap,0.0,nchar)
              call symbol(x,y+0.20,0.10,'REC=',0.0,4)
              call number(999.,999.,0.10,real(nrec),0.0,-1)
              call symbol(999.,999.,0.10,'  TRC=',0.0,6)
              call number(999.,999.,0.10,real(ktrc),0.0,-1)
              call symbol(999.,999.,0.10,'  DT =',0.0,6)
              call number(999.,999.,0.10,real(nsi),0.0,-1)
              call symbol(999.,999.,0.10,' MS',0.0,3)
              call symbol(999.,999.,0.10,'  DIST=',0.0,7)
              call number(999.,999.,0.10,dist,0.0,-1)
              call symbol(999.,999.,0.10,'  NSAMP=',0.0,8)
              call number(999.,999.,0.10,real(n),0.0,-1)
              call symbol(x,y,0.10,'ALPHA0 =',0.0,8)
              call number(999.,999.,0.10,alpha0,0.0,3)
              call symbol(999.,999.,0.10,' ALPHA1 =',0.0,8)
              call number(999.,999.,0.10,alpha1,0.0,3)
       return
       end

       function fmap(x,aa,bb,cc)
              real*4 fmap
              real*4 x
              real*4 aa,bb,cc
c-----
c       this general function defines the mapping between 
c       array index I and the center frequency of the 
c gaussian filter
c
c       changing this routine will change the frequency sampling
c       throughout the program
c-----
       if(cc .gt. 0.0)then
              fmap = 10.0**(aa + bb*x)
       else if(cc .eq. 0.0)then
              fmap = aa + bb*x
       else if(cc.lt.0.0)then
              fmap = (aa +bb*x)
       endif
       return
       end

       function lgstr(str,n)
c-----
c       function to find the length of a string
c       this will only be used with file system path names
c       thus the first blank 
c       indicates the end of the string
c-----
       character*(*) str
       integer*4 lgstr
       lgstr = 0
       do 1000 i=1,n
              if(str(i:i).eq.' ')goto 100
              lgstr = i
 1000        continue
  100   continue
        return
       end

      subroutine fnorm(imax,jmax,f)
c
c     normalize values of matrix columnwise
c
c   imax - column dimension
c   jmax - row dimension
c      f - matrix
c
      dimension f(imax,jmax)
c
      do 10 j = 1, jmax
         fmax = 0.
         do 5 i = 1,imax
            if(f(i,j) .gt. fmax) fmax = f(i,j)
    5    continue
         do 6 i = 1,imax
            if(fmax .eq. 0.0) fmax = 1.0
            f(i,j) = f(i,j)/fmax
    6    continue
   10 continue
      return
      end

c-------------------------------------------------
c  online help section
c-------------------------------------------------
      subroutine help
#include <f77/iounit.h>

       write(LER,*)
     :'***************************************************************'
       write(LER,*)' '
       write(LER,*)
     :'Execute groupv by typing groupv and the following arguments'
       write(LER,*)' '
       write(LER,*)
     :' -N [ntap]     (no default)               : Input data file name'
       write(LER,*)
     :' -tmin [start] (default = 0 ms)           :  Start time in ms'
       write(LER,*)
     :' -tmax [stop]  (default = full trace)     : Stop time in ms'
       write(LER,*)
     :' -vmin [vmin]  (default = 100)   : minimum group velocity'
       write(LER,*)
     :' -vmax [vmax]  (default = 10000) : maximum group velocity'
       write(LER,*)
     :' -nper         (default = 80)    : number frequencies'
       write(LER,*)
     :' -rs [irs],    (default=1)       : beginning record number'
       write(LER,*)
     :' -re [ire],    (default=last)    : ending record number'
       write(LER,*)
     :' -ns[ns]  (default=1)            : first trace to be processed'
       write(LER,*)
     :' -ne[ne]  (default=1)            : last  trace to be processed'
       write(LER,*)
     :' -dmul [dmul]  (default = 1.0)   : distance multiplier'
       write(LER,*)
     ;' -tmul [tmul]  (default = 1.0)   : sampling interval multiplier'
       write(LER,*)' '
       write(LER,*)
     :' -plt     (default no plot)      : plot spectral amp and grp vel'
       write(LER,*)
     :' -o       (default no file)      : output dispersion curves in'
       write(LER,*)
     :'                                   ascii file called GROUPV.o'
       write(LER,*)' '
       write(LER,*)
     :' -aa      (default = 1.5)'
       write(LER,*)
     :' -bb      (default = 0.5)'
       write(LER,*)
     :' -cc      (default = 0.0)'
       write(LER,*)
     :' -AA       (default 50.27)'
       write(LER,*)
     :' -BB       (default 0.0)'
       write(LER,*)' freq = aa + bb * I if cc = 0.0'
       write(LER,*)' freq = 10.0**(aa + bb * I) if cc .gt. 0.0'
       write(LER,*)' freq = (aa + bb * i ) * df if cc .lt. 0.0'
           write(LER,*)' '
     :,' -V verbose output'
           write(LER,*)' '
          write(LER,*)
     :'Usage: groupv -N[ntap] -ns[ns] -tmin[start] -tmax[stop] -ne[ne]'
     :,' -rs[irs], -re[ire] -aa[aa] -bb[bb] -cc[cc] -nper[nper]'
     :,' -AA[alpha0] -BB[alpha1]'
     :,' -dmul[dmul] -tmul[tmul] -plt -o -V > [plot file name]'
          write(LER,*)' '
          write(LER,*)
     :'To display plots enter:  plot_xview < [[plot file name]'
          write(LER,*)
     :'Then use right mouse button in plot window to get next page'
          write(LER,*)' '
          write(LER,*)
     :'***************************************************************'

      return
      end

c-----
c     get command arguments
c
c     ntap  - C*100  input file name
c      ns   - I      start trace
c      ne   - I      stop end trace
c     irs   - I      start record
c     ire   - I      stop end record
c    dmul   - R      distance multiplier
c    tmul   - R      time multiplier
c    nper   - I      number of periods
c   pumin   - R      min velocity to contour
c   pumax   - R      max velocity to contour
c      aa   - R      gaussian filter parameter
c      bb   - R      gaussian filter parameter
c      cc   - R      gaussian filter parameter
c   alpha0  - R      gaussian filter parameter
c   alpha1  - R      gaussian filter parameter
c   lafln   - L      linear freq scale on plots
c   lufln   - L      linear velocity scale on plots
c   plton   - L      plot output
c   outon   - L      output contour maxima
c    verbos - L      verbose output or not
c-----
      subroutine cmdln (ntap,dmul,tmul,start,stop,
     1               ns,ne,irs,ire,nper,pumin,pumax,lafln,lufln,
     2               aa,alpha0,alpha1,bb,cc,plton,outon,verbos)

#include <f77/iounit.h>
      character  ntap *(*)
      integer    start,stop,ns,ne,irs,ire,nper,argis
      real       aa,bb,cc,alpha0,alpha1,pumin,pumax,dmul,tmul
      logical    lafln,lufln,outon,plton,verbos

          call argstr('-N', ntap, ' ', ' ' )
          call argr4 ('-dmul', dmul, 1.0, 1.0 )
          call argr4 ('-tmul', tmul, 1.0, 1.0 )
          call argi4 ('-tmin', start, 0, 0 )
          call argi4 ('-tmax', stop , 0 , 0)
          call argi4 ('-rs',irs,1,1)
          call argi4 ('-re',ire,0,0)
          call argi4('-ns',ns,1,1)
          call argi4('-ne',ne,0,0)
          call argi4('-nper',nper,80,80)
          call argr4('-aa',aa,1.5,1.5)
          call argr4('-AA',alpha0,50.27,50.27)
          call argr4('-BB',alpha1,0.0,0.0)
          call argr4('-vmin',pumin,100.0,100.0)
          call argr4('-vmax',pumax,10000.0,10000.0)
          lafln = ( argis('-flin') .eq. 0 )
          lufln = ( argis('-vlin') .eq. 0 )
          call argr4('-bb',bb,0.5,0.5)
          call argr4('-cc',cc,0.0,0.0)
          if(nper.gt.80)nper=80
          verbos = ( argis ('-V') .gt. 0 )
          plton  = ( argis ('-plt') .gt. 0 )
          outon  = ( argis ('-o') .gt. 0 )
          if(.not.outon .and. .not.plton)then
                 write(LERR,*)' STOP: no output specified ',
     1                         ' use -o and/or -plt '
                 stop
          endif

      return
      end
