c -----------------  Main Routine -----------------------
c
c     copyright 2002, Amoco Production Company 
c              All rights reserved
c        an affiliate of BP America Inc.
c
c -----------------  ------------ -----------------------

c     Program Changes:

c      - original written: October 23, 2002
c      - author: Paul G.A. Garossino
c
c     Oct 28, 2002:  Added dead trace detector in subroutine to 
c                    prevent totally dead records from crashing
c                    the subroutine logic.
c     Garossino

c     Program Description:

c      - routine to determine wavelet coefficient threshold useful 
c        to discern noise from signal.  The initial routine was written
c        to support a processing project from Calgary [McKenzie Delta]
c        where S/N is very low.  The threshold required was a percent of
c        to running summation of the record wise amplitude histogram.  In
c        the preconditioned swt1d data that is input to this routine the
c        initial threshold required was the amplitude associated with
c        97.25 percent of the running sum.
c

      implicit none

c get machine dependent parameters 

#include <f77/iounit.h>
#include <f77/lhdrsz.h>
#include <f77/sisdef.h>
#include <save_defs.h> 

c dimension standard USP variables 

      integer     nsamp, nsi, ntrc, nrec, nreco, iform, obytes
      integer     luin , luout, lbytes, nbytes, lbyout
      integer     ist, iend, irs, ire, argis, jerr

      real        UnitSc, dt

      character   ntap*255, otap*255, name*7

      logical     verbos

c Program Specific _ dynamic memory variables

      integer itr, Headers
      integer RecordSize, HeaderSize, errcd1, errcd2, errcd3, errcd4
      integer alloc_size, abort, errcd5

      real    Record, histogram, lb, ub

      pointer (mem_Record, Record(2))
      pointer (mem_Headers, Headers(2))
      pointer (mem_histogram, histogram(2))
      pointer (mem_lb, lb(2))
      pointer (mem_ub, ub(2))
      pointer (mem_itr, itr(2))

c Program Specific _ static memory variables

      integer ifmt_HdrWrd,l_HdrWrd,ln_HdrWrd, HdrWrd
      integer ifmt_RecNum,l_RecNum,ln_RecNum, RecNum
      integer ifmt_StaCor,l_StaCor,ln_StaCor, StaCor
      integer hdr_index, tr_index, JJ, KK, number_bins, i

      real threshold, minv, maxv, r_HdrWrd
      real vmin_temp, vmax_temp, bin_size

      character c_HdrWrd*6

c Initialize variables

      data abort/0/
      data name/"HTHRESH"/

c give command line help if requested

      if ( argis ( '-?' ) .gt. 0 .or. 
     :     argis ( '-h' ) .gt. 0 .or. 
     :     argis ( '-help' ) .gt. 0 ) then
         call help()
         stop
      endif

c open printout file

#include <f77/open.h>

c get command line input parameters

      call cmdln ( ntap, otap, irs, ire, ist, iend, threshold, 
     :     minv, maxv, number_bins, c_HdrWrd, name, verbos )

c allocate memory for initial line header read

      alloc_size = SZLNHD * SZSMPD
      errcd1 = 0
      call galloc(mem_itr,alloc_size,errcd1,abort)
      if (errcd1 .ne. 0) then
	write(LERR,*) 'ERROR: Unable to allocate workspace '
	write(LERR,*) '       ',alloc_size,' bytes requested '
	write(LERR,*) '       FATAL'
	write(LER,*) 'HTHRESH: Unable to allocate workspace '
	write(LER,*) '       ',alloc_size,' bytes requested '
	write(LER,*) 'FATAL'
	stop 
      else
        write(LERR,*)'Allocating workspace for line header:'
	write(LERR,*) alloc_size,' bytes requested '
      endif

c open input and output files

      call getln(luin , ntap,'r', 0)
      call getln(luout, otap,'w', 1)

c  read input line header and save certain parameters

      call rtape(luin,itr,lbytes)
      if(lbytes.eq.0)then
         write(LERR,*)'HTHRESH: no line header on input dataset',
     :        ntap
         write(LER,*)' '
         write(LER,*)'HTHRESH: '
         write(LER,*)' no line header on input dataset',ntap
         write(LER,*)'FATAL'
         write(LER,*)' '
         stop
      endif

      call saver(itr, 'NumSmp', nsamp, LINHED)
      call saver(itr, 'SmpInt', nsi  , LINHED)
      call saver(itr, 'NumTrc', ntrc , LINHED)
      call saver(itr, 'NumRec', nrec , LINHED)
      call saver(itr, 'Format', iform, LINHED)
      call saver(itr, 'UnitSc', UnitSc, LINHED)

c print HLH to printout file 

      call hlhprt (itr, lbytes, name, 7, LERR)

c POLICEMAN: check header for units scaling.  Using UnitSc, remember
c that UnitSc default is milliseconds [i.e. 0.001] and UnitSc
c is a floating point variable.  A UnitSc entry of 1.0 would
c mean units are in seconds.  A UnitSc entry of 0 indicates that
c the unit was not defined.  In this case milliseconds are 
c assumed and loaded to the header for further processing.

      if ( UnitSc .eq. 0.0 ) then
         write(LERR,*)'********************************************'
         write(LERR,*)'WARNING: sample unit scaler in LH = ',UnitSc
         write(LERR,*)'         will set to .001 (millisec default)'
         write(LERR,*)'********************************************'
         UnitSc = 0.001
         call savew ( itr, 'UnitSc', UnitSc, LINHED)
      endif

c compute delta T in seconds

      dt = real (nsi) * UnitSc

c check user supplied boundary conditions and set defaults

      if ( irs .eq. 0 ) irs = 1
      if ( ire .eq. 0 .or. ire .gt. nrec ) ire = nrec

c this parameterization assumes input in units of the dataset

      ist = nint ( float(ist) / float(nsi) ) + 1
      if ( ist .eq. 0 ) ist = 1

      if ( iend .eq. -99999 ) then
         iend = nsamp
      else
         iend = nint ( float(iend) / float(nsi) ) + 1
         if ( iend .eq. 0 .or. iend .gt. nsamp ) iend = nsamp
      endif

      nreco = ire - irs + 1

c modify line header to reflect actual record configuration output
c NOTE: in this case the trace and sample limits are used to 
c       limit processing only.   All data within the selected record
c       range are actually passed.

      call savew(itr, 'NumRec', nreco, LINHED)

c number output bytes

      obytes = SZTRHD + SZSMPD * nsamp 

c save out hlh and line header

      call savhlh  ( itr, lbytes, lbyout )
      call wrtape ( luout, itr, lbyout )

c verbose output of all pertinent information before processing begins

      call verbal( ntap, otap, nsamp, nsi, ntrc, nrec, iform, ist, 
     :     iend, irs, ire, threshold, minv, maxv, number_bins, c_HdrWrd, 
     :     verbos )

c set up pointers to header mnemonic StaCor

      call savelu ( 'RecNum', ifmt_RecNum, l_RecNum, ln_RecNum, 
     :     TRACEHEADER )
      call savelu ( 'StaCor', ifmt_StaCor, l_StaCor, ln_StaCor, 
     :     TRACEHEADER )
      call savelu ( c_HdrWrd, ifmt_HdrWRd, l_HdrWRd, ln_HdrWRd, 
     :     TRACEHEADER ) 

c reallocate input buffer to size of input trace plus header

      alloc_size = SZTRHD + SZSMPD * nsamp
      errcd1 = 0
      call grealloc(mem_itr,alloc_size,errcd1,abort)
      if (errcd1 .ne. 0) then
	write(LERR,*) 'ERROR: Unable to allocate trace workspace '
	write(LERR,*) '       ',alloc_size,' bytes requested '
	write(LERR,*) 'FATAL'
	write(LER,*) 'HTHRESH: Unable to allocate trace workspace '
	write(LER,*) '       ',alloc_size,' bytes requested '
	write(LER,*) 'FATAL'
	stop 
      else
        write(LERR,*)'Allocating workspace:'
	write(LERR,*) alloc_size,' bytes requested '
      endif

c additional dynamic memory allocation:  

      RecordSize = ntrc * nsamp 
      HeaderSize = ntrc * ITRWRD 

      call galloc (mem_Record, RecordSize * SZSMPD, errcd1, abort)
      call galloc (mem_Headers, HeaderSize * SZSMPD, errcd2, abort)
      call galloc (mem_Histogram, (number_bins+2) * SZSMPD, errcd3, 
     :     abort)
      call galloc (mem_lb, (number_bins+2) * SZSMPD, errcd4, abort)
      call galloc (mem_ub, (number_bins+2) * SZSMPD, errcd5, abort)
    
      if ( errcd1 .ne. 0 .or.
     :     errcd2 .ne. 0 .or.
     :     errcd3 .ne. 0 .or.
     :     errcd4 .ne. 0 .or.
     :     errcd5 .ne. 0 )then
         write(LERR,*)' '
         write(LERR,*)'Unable to allocate workspace:'
         write(LERR,*) (RecordSize+HeaderSize)* SZSMPD, '  bytes'
         write(LERR,*) 3 * number_bins * SZSMPD, '  bytes'
         write(LERR,*)' '
         write(LER,*)' '
         write(LER,*)'HTHRESH: Unable to allocate workspace:'
         write(LER,*)  (RecordSize+HeaderSize)* SZSMPD, '  bytes'
         write(LER,*) 3 * number_bins * SZSMPD, '  bytes'
         write(LER,*)'FATAL '
         write(LER,*)' '
         go to 999
      else
         write(LERR,*)' '
         write(LERR,*)'Allocating workspace:'
         write(LERR,*) (RecordSize+HeaderSize)* SZSMPD, '  bytes'
         write(LERR,*) 3 * number_bins * SZSMPD, '  bytes'
         write(LERR,*)' '
      endif

c initialize memory
         
      call vclr ( Record, 1, RecordSize )
      call vclr ( Headers, 1, HeaderSize )
      call vclr ( histogram, 1, number_bins )
      call vclr ( lb, 1, number_bins )
      call vclr ( ub, 1, number_bins )

c set up histogram parameters

      vmin_temp=minv
      vmax_temp=maxv

      if (maxv .eq. -99999. ) maxv = 0.
      if (minv .eq. -99999. ) minv = 1.E30

      if ( ( abs( vmin_temp + 99999.) .gt. 1.e-32 ) .and.
     :     ( abs( vmax_temp + 99999.) .gt. 1.e-32 ) ) then

         bin_size = (maxv - minv) / float(number_bins)

         histogram(1) = 0.0
         lb(1) = minv - bin_size
         ub(1) = minv
         
         do i = 2, number_bins + 1
            histogram(i) = 0
            ub(i) = minv + (bin_size * float(i-1))
            lb(i) = ub(i-1)
         enddo

         histogram(number_bins+2) = 0
         lb(number_bins+2) = maxv
         ub(number_bins+2) = maxv + bin_size

      endif

c BEGIN PROCESSING 

      write(LERR,*)' '
      write(LERR,*)' " Threshold Amplitudes'


c skip unwanted input records

      call recskp ( 1, irs-1, luin, ntrc, itr )

      DO JJ = irs, ire

c load record to memory

         tr_index = 1 - nsamp
         hdr_index = 1 - ITRWRD

          DO KK = 1, ntrc

            nbytes = 0
            call rtape( luin, itr, nbytes)

c if end of data encountered (nbytes=0) then bail out

            if(nbytes .eq. 0) then
               write(LERR,*)'Premature EOF on input at:'
               write(LERR,*)'  rec= ',JJ,'  trace= ',KK
               go to 999
            endif

c set array load points for this trace 

           tr_index = tr_index + nsamp
           hdr_index = hdr_index + ITRWRD

c process only live traces and zero out dead traces

           call saver2 ( itr, ifmt_StaCor, l_StaCor, ln_StaCor, StaCor, 
     :          TRACEHEADER )

           if ( StaCor .ne. 30000 ) then

c load trace to array Record[]

              call vmov ( itr(ITHWP1), 1, Record(tr_index), 1, nsamp )

           else
              call vclr ( Record(tr_index), 1, nsamp )
           endif

c load trace header to array Headers[]

            call vmov ( itr, 1, Headers(hdr_index), 1, ITRWRD )

         ENDDO

c Put your subroutine here [remember to declare any arguments you need
c over and above those already declared]

         call hthresh_sub ( record, nsamp, ntrc, ist, iend, minv, 
     :        maxv, vmax_temp, vmin_temp, number_bins, bin_size, 
     :        histogram, lb, ub, threshold, r_HdrWrd, Headers,
     :        ifmt_StaCor, l_StaCor, ln_StaCor, verbos )

c reset array load points for this trace 

         tr_index = 1 - nsamp
         hdr_index = 1 - ITRWRD

c write output data

         DO KK = 1, ntrc

            tr_index = tr_index + nsamp
            hdr_index = hdr_index + ITRWRD
            
            if ( KK .eq. 1 ) then
               call saver2 ( Headers(hdr_index), ifmt_RecNum, 
     :              l_RecNum, ln_RecNum, RecNum, TRACEHEADER )
               write(LERR,*) float(RecNum), r_HdrWrd
            endif

c load threshold amplitude out to each trace header

            if ( ifmt_HdrWrd .eq. SAVE_FKFLT_DEF .or. 
     :           ifmt_HdrWrd .eq. SAVE_FLOAT_DEF ) then
               call savew2 ( Headers(hdr_index), ifmt_HdrWrd, l_HdrWrd, 
     :              ln_HdrWrd, r_HdrWrd, TRACEHEADER )
            else
               HdrWrd = nint(r_HdrWrd)
               call savew2 ( Headers(hdr_index), ifmt_HdrWrd, l_HdrWrd, 
     :              ln_HdrWrd, HdrWrd, TRACEHEADER )
            endif

            call vmov ( Record(tr_index), 1, itr(ITHWP1), 1, nsamp )
            call vmov ( Headers(hdr_index), 1, itr(1), 1, ITRWRD )
            call wrtape (luout, itr, obytes)
 
         ENDDO
      ENDDO

c close data files 

      call lbclos ( luin )
      call lbclos ( luout )
      write(LERR,*)'hthresh: Normal Termination'
      write(LER,*)'hthresh: Normal Termination'
      stop

 999  continue

      call lbclos ( luin )
      call lbclos ( luout )
      write(LERR,*)'hthresh: ABNORMAL Termination'
      write(LER,*)'hthresh: ABNORMAL Termination'
      stop
      end

c -----------------  Subroutine -----------------------

      subroutine help()

c provide terse online help [detailed help goes in man page]

#include <f77/iounit.h>

      write(LER,*)' '
      write(LER,*)'===================================================='
      write(LER,*)' '
      write(LER,*)' Command Line Arguments for HTHRESH: histogram '
      write(LER,*)'                                     thresholding'
      write(LER,*)' '
      write(LER,*)' '
      write(LER,*)' '
      write(LER,*)' For a more detailed description of these parameters'
      write(LER,*)' see the online man page using uman, xman or xuspman'
      write(LER,*)' '
      write(LER,*)'Input...................................... (def)'
      write(LER,*)' '
      write(LER,*)'-N[]   -- input data set                  (stdin)'
      write(LER,*)'-O[]   -- output data set                (stdout)'
      write(LER,*)'-s[]   -- process start time (ms)             (1)'
      write(LER,*)'-e[]   -- process end time (ms)     (last sample)'
      write(LER,*)'-rs[]  -- start record                        (1)'
      write(LER,*)'-re[]  -- end record                (last record)'
      write(LER,*)'-hw[]  -- scalar mnemonic                (Horz05)'
      write(LER,*)'-nbins[] -- number of histogram bins       (2500)'
      write(LER,*)'-min[] -- minimum histogram amplitude (from data)'
      write(LER,*)'-max[] -- maximum histogram amplitude (from data)'
      write(LER,*)'-pct[] -- % area under curve for thresh   (97.25)'
      write(LER,*)'-V     -- verbos printout'
      write(LER,*)' '
      write(LER,*)'Usage:'
      write(LER,*)'       hthresh -N[] -O[] -s[] -e[] -rs[] -re[] '
      write(LER,*)'          -hw[] -nbins[] -min[] -max[] -pct[] -V'
      write(LER,*)' '
      write(LER,*)'===================================================='
      
      return
      end

c -----------------  Subroutine -----------------------

c pick up command line arguments 

      subroutine cmdln ( ntap, otap, irs, ire, ist, iend, threshold, 
     :     minv, maxv, number_bins, c_HdrWrd, name, verbos )

      implicit none

#include <f77/iounit.h>

      integer    ist, iend, irs, ire, argis

      real       threshold, minv, maxv, number_bins

      character  ntap*(*), otap*(*), name*(*), c_HdrWrd*6

      logical    verbos

           call argi4 ( '-e', iend, -99999, -99999 )

           call argstr ( '-hw', c_HdrWrd, 'Horz05', 'Horz05' ) 

           call argr4 ( '-max', maxv, -99999., -99999. )
           call argr4 ( '-min', minv, -99999., -99999. )

           call argi4 ( '-nbins', number_bins, 2500, 2500 )
           call argstr ( '-N', ntap, ' ', ' ' ) 

           call argstr ( '-O', otap, ' ', ' ' ) 

           call argr4 ( '-pct', threshold, 97.25, 97.25 )

           call argi4 ( '-re', ire, 0, 0 )
           call argi4 ( '-rs', irs, 0, 0 )

           call argi4 ( '-s', ist, 1, 1 )


           verbos = (argis('-V') .gt. 0)

c check for extraneous arguments and abort if found to
c catch all manner of user typo's

      call xtrarg ( name, ler, .FALSE., .FALSE. )
      call xtrarg ( name, lerr, .FALSE., .TRUE. )

           
      return
      end

c -----------------  Subroutine -----------------------

c verbal printout of pertinent program particulars


      subroutine verbal ( ntap, otap, nsamp, nsi, ntrc, nrec, iform, 
     :     ist, iend, irs, ire, threshold, minv, maxv, number_bins, 
     :     c_HdrWrd, verbos )

#include <f77/iounit.h>

      integer    nsamp, ntrc, iform, ist, iend, irs, ire, nsi
      integer    number_bins

      real       threshold, minv, maxv

      character  ntap*(*), otap*(*), c_HdrWrd*6

      logical    verbos

      write(LERR,*)' '
      write(LERR,*)' Input Line Header Parameters'
      write(LERR,*)' '
      write(LERR,*) ' input data set name   =  ', ntap
      write(LERR,*) ' samples per trace     =  ', nsamp
      write(LERR,*) ' traces per record     =  ', ntrc
      write(LERR,*) ' number of records     =  ', nrec
      write(LERR,*) ' data format           =  ', iform
      write(LERR,*) ' sample interval       =  ', nsi
      write(LERR,*)' '
      write(LERR,*)' '
      write(LERR,*)' Command Line Parameters '
      write(LERR,*)' '
      write(LERR,*) ' output data set name    =  ', otap
      write(LERR,*) ' start record            =  ', irs 
      write(LERR,*) ' end record              =  ', ire 
      write(LERR,*) ' processing sample start = ', ist
      write(LERR,*) ' processing sample end   = ', iend
      write(LERR,*) ' scalar mnemonic         = ', c_HdrWrd
      write(LERR,*) ' percent running sum     = ', threshold
      write(LERR,*) ' minimum histogram amp   = ', minv
      write(LERR,*) ' maximum histogram amp   = ', maxv
      write(LERR,*) ' number of bins          = ', number_bins

      if ( verbos )  write(LERR,*) ' verbose printout requested'
      write(LERR,*)' '
      write(LERR,*)'========================================== '
      write(LERR,*)' '

      return
      end





