C***********************************************************************
C                 copyright 2001, Amoco Production Company             *
C                             All Rights Reserved                      *
C                     an affiliate of BP America Inc.                  *
C***********************************************************************
C
C
C
       SUBROUTINE ASIG2(R, Q, SAMP, NSAMP, NOP, DATA)
C
C John Bodine originally authored "ASIG" in July of 1981. This is
C a rewrite of Johns work to remove errors, and to speed up the
C code by removing some unnecessary calculations.
C (Dennis Frampton, March 11th, 1992).
C
C  RETURNS DESIRED COMPLEX TRACE ATTRIBUTE
C        R = SEISMIC TRACE
C     SAMP = SAMPLE INTERVAL
C    NSAMP = NO. SAMPLES PER TRACE
C      NOP = DESIRED ATTRIBUTE
C          = 1 SEISMIC TRACE
C          = 2 QUADRATURE
C          = 3 ENVELOPE
C          = 4 INST. PHASE
C          = 5 RESPONSE PHASE
C          = 6 INST. FREQUENCY
C          = 7 RESPONSE FREQUENCY
C          = 8 0-PHASE DECOMPOSITION
C          = 9 90-PHASE DECOMPOSITION
C          = 10 RESPONSE AMPLITUDE
C          = 11 RESPONSE LENGTH
C          = 12 ENVELOPE SKEWNESS
C          = 13 ENVELOPE RISE TIME
C
C     DATA = OUTPUT ATTRIBUTE
C
      REAL    R(*),Q(*),DUM(2000), DATA(*), SAMP, PI
      REAL    A(4096),FREQ(4096),PH(4096), RADDEG, WC
      INTEGER NSAMP, NOP, NF, NZ
C
      PI     = 3.14159265
      RADDEG = 360./(2.*PI)
      NF     = 101
      WC     = 3
      NZ     = NF + NSAMP
C
C Calculate Quadrature if needed.
C
      IF (NOP .GT. 1) THEN
         CALL HILBRT(NZ, NF, WC, DUM, SAMP, NSAMP, R, Q)
      ENDIF
C
C Seismic trace.
C
      IF (NOP .EQ. 1) THEN
         DO 10 I = 1, NSAMP
              DATA(I) = R(I)
   10    CONTINUE
         GO TO 9999
      ENDIF
C
C Quadrature.
C
      IF (NOP .EQ. 2) THEN
         DO 20 I = 1, NSAMP
              DATA(I) = Q(I)
   20    CONTINUE
         GO TO 9999
      ENDIF
C
C Envelope.
C
      IF (NOP .EQ. 3) THEN
         DO 30 I = 1, NSAMP
              A(I) = 0.0
              IF (R(I) .NE. 0.0 .AND. Q(I) .NE. 0.0) THEN
                   A(I) = (R(I)**2 + Q(I)**2) ** .5
              ENDIF
              DATA(I) = A(I)
   30    CONTINUE
         GO TO 9999
      ENDIF
C
C Inst. Phase.
C
      IF (NOP .EQ. 4) THEN
         DO 40 I = 1, NSAMP
              DATA(I) = PHAS(R(I), Q(I)) * RADDEG
              IF (DATA(I) .GT. 180.0) DATA(I) = DATA(I) - 360.0
   40    CONTINUE
         GO TO 9999
      ENDIF
C
C Response Phase.
C
      IF (NOP .EQ. 5) THEN
         DO 50 I = 1, NSAMP
              PH(I) = PHAS(R(I), Q(I)) * RADDEG
              IF (PH(I) .GT. 180.0) THEN
                   PH(I) = PH(I) - 360.0
              ENDIF
              A(I) = 0.0
              IF (R(I) .NE. 0.0 .AND. Q(I) .NE. 0.0) THEN
                   A(I) = (R(I)**2 + Q(I)**2) ** .5
              ENDIF
              DATA(I) = A(I)
   50    CONTINUE
         PH(1) = PH(2)
         CALL RPHS(PH, DATA, NSAMP)
         GO TO 9999
      ENDIF
C
C Inst. Frequency.
C
      IF (NOP .EQ. 6) THEN
         DO 60 I = 1, NSAMP
              FREQ(I) = XFREQ(R, Q, SAMP, NSAMP, I) / (2.0 * PI)
              DATA(I) = -1.0 * FREQ(I)
   60    CONTINUE
         GO TO 9999
      ENDIF
C
C Response Frequency.
C
      IF (NOP .EQ. 7) THEN
         DO 70 I = 1, NSAMP
              A(I) = 0.0
              IF (R(I) .NE. 0.0 .AND. Q(I) .NE. 0.0) THEN
                   A(I) = (R(I)**2 + Q(I)**2) ** .5
              ENDIF
              DATA(I) = A(I)
              FREQ(I) = XFREQ(R, Q, SAMP, NSAMP, I) / (2.0 * PI)
              FREQ(I) = -1.0 * FREQ(I)
   70    CONTINUE
         CALL PKDT(FREQ, DATA, NSAMP)
         GO TO 9999
      ENDIF
C
C Calculate 0 or 90(which ever asked for) Phase Decomposed Signal.
C
      IF (NOP .EQ. 8 .OR. NOP .EQ. 9) THEN
         DO 80 I = 1, NSAMP
              A(I) = 0.0
              IF (R(I) .NE. 0.0 .AND. Q(I) .NE. 0.0) THEN
                   A(I) = (R(I)**2 + Q(I)**2) ** .5
              ENDIF
              DATA(I) = A(I)
              PH(I) = PHAS(R(I), Q(I)) * RADDEG
              IF (PH(I) .GT. 180.0) THEN
                   PH(I) = PH(I) - 360.0
              ENDIF
   80    CONTINUE
         PH(1) = PH(2)
         CALL RPHS(PH, DATA, NSAMP)
         CALL QCON(R, Q, DATA, NSAMP)
         IF (NOP .EQ. 8) THEN
              DO 90 I = 1, NSAMP
                   DATA(I) = R(I)
   90         CONTINUE
         ENDIF
         IF (NOP .EQ. 9) THEN
              DO 100 I = 1, NSAMP
                   DATA(I) = Q(I)
  100         CONTINUE
         ENDIF
         GO TO 9999
      ENDIF
C
C Calculate Response Amplitude.
C
      IF (NOP .EQ. 10) THEN
         DO 110 I = 1, NSAMP
              A(I) = 0.0
              IF (R(I) .NE. 0.0 .AND. Q(I) .NE. 0.0) THEN
                   A(I) = (R(I)**2 + Q(I)**2) ** .5
              ENDIF
              DATA(I) = A(I)
  110    CONTINUE
         CALL PKDT(A, DATA, NSAMP)
         GO TO 9999
      ENDIF
C
C Calculate length of each envelope lobe.
C
      IF (NOP .EQ. 11) THEN
         DO 120 I = 1, NSAMP
              A(I) = 0.0
              IF (R(I) .NE. 0.0 .AND. Q(I) .NE. 0.0) THEN
                   A(I) = (R(I)**2 + Q(I)**2) ** .5
              ENDIF
              DATA(I) = A(I)
  120    CONTINUE
         CALL ALEN(DATA, NSAMP)
         GO TO 9999
      ENDIF
C
C Calculate skewness of envelope lobe.
C
      IF (NOP .EQ. 12) THEN
         DO 130 I = 1, NSAMP
              A(I) = 0.0
              IF (R(I) .NE. 0.0 .AND. Q(I) .NE. 0.0) THEN
                   A(I) = (R(I)**2 + Q(I)**2) ** .5
              ENDIF
              DATA(I) = A(I)
  130    CONTINUE
         CALL ASKW(DATA, NSAMP)
         GO TO 9999
      ENDIF
C
C Calculate max slope of envelope lobe.
C
      IF (NOP .EQ. 13) THEN
         DO 140 I = 1, NSAMP
              A(I) = 0.0
              IF (R(I) .NE. 0.0 .AND. Q(I) .NE. 0.0) THEN
                   A(I) = (R(I)**2 + Q(I)**2) ** .5
              ENDIF
              DATA(I) = A(I)
  140    CONTINUE
         CALL ASLP(DATA, NSAMP)
         GO TO 9999
      ENDIF
C
 9999 CONTINUE
C
      RETURN
      END
C
      SUBROUTINE HILBRT(NZ,NF,WC,F,DX,NX,X,Z)
C
C  PROGRAM TO PERFORM A HILBERT TRANSFORM ON A VECTOR X
C  OF LENGTH NX.
C
C  Z SERVES FIRST AS A DUMMY CONVOLUTION ARRAY
C  AND MUST BE DIMENSIONED NZ=NX+NF-1
C
C  TRANSFORM IS RETURNED AS Z OF LENGTH NX
C
C  NOTE: HILF OPERATOR RESULTS IN COS --> -SIN.....
C  BY JH BODINE   7/27/81
C
C
C      F = DUMMY ARRAY FOR THE HILBERT OPERATOR
C     NF = NO.POINTS IN HILBERT TRANSFER FUNCTION (ODD)
C     DX = SAMPLE INTERVAL
C   NX,X = NO. POINTS IN VECTOR X WHICH IS TO BE TRANSFORMED
C
      DIMENSION X(*),F(*),Z(*)
C     DIMENSION X(1),F(NF),Z(NZ)
C
      CALL HILF(DX,WC,NF,F)
C
      CALL FOLD(NF,F,NX,X,NZ,Z)
C
      DO 10 I=1,NX
C     WRITE(*,*) Z(I),I
   10 Z(I)= Z((NF-1)/2+I)*DX
C
      RETURN
      END
C***********************************************************************
C                                                                      *
c                 copyright 2001, Amoco Production Company             *
c                             All Rights Reserved                      *
c                     an affiliate of BP America Inc.                  *
C                                                                      *
C***********************************************************************
C  ROUTINE:       RPHS                                                 *
C  ROUTINE TYPE:  SUBROUTINE                                           *
C  PURPOSE:  TO CALCULATE RESPONSE PHASE.                              *
C  ENTRY POINTS:                                                       *
C      RPHS  (PHASE,A,NPT)                                             *
C  ARGUMENTS:                                                          *
C      PHASE   REAL         U*  (1) - WORKING ARRAY CONTAINS INST PHAS *
C      A       REAL         U*  (1) - COMPLEX ENVELOPE ARRAY ON INPUT  *
C                                   - RESPONSE PHASE ON OUTPUT         *
C      NPT     INTEGER      I*      - NUMBER OF SAMPLE POINTS          *
C       +------------------------------------------------------+       *
C       |               DEVELOPMENT INFORMATION                |       *
C       +------------------------------------------------------+       *
C  AUTHOR:   J. BODINE                          ORIGIN DATE: 81/09/02  *
C  LANGUAGE: FORTRAN 77/IV               DATE LAST COMPILED: 83/06/03  *
C       +------------------------------------------------------+       *
C       |                 EXTERNAL ENVIRONMENT                 |       *
C       +------------------------------------------------------+       *
C  ROUTINES CALLED:                                                    *
C      MNMX         - TO FIND RELATIVE MAXIMA/MINIMA IN AMPLITUDE ARRAY*
C      UNWRAP       - TO UNWRAP PHASE                                  *
C      FPEAK   REAL - TO FIND MAXIMUM VALUE ARRAY POSITION             *
C  INTRINSIC FUNCTIONS CALLED:                                         *
C      FLOAT   REAL -                                                  *
C  FILES:                                                              *
C      6  ( OUTPUT SEQUENTIAL ) - DEBUG OUTPUT                         *
C  COMMON:           NONE                                              *
C  STOP CODES:       NONE                                              *
C       +------------------------------------------------------+       *
C       |             OTHER DOCUMENTATION DETAILS              |       *
C       +------------------------------------------------------+       *
C  ERROR HANDLING:  NONE                                               *
C  GENERAL DESCRIPTION:                                                *
C       ROUTINE TO SET ANALYTIC SIGNAL ENVELOPE LOBES TO VALUE OF      *
C       THE INSTANTANEOUS PHASE AT THE POINT WHERE THE ENVELOPE        *
C       IS A MAXIMUM.                                                  *
C       THE VALUE OF THE ENVELOPE IS SET TO THAT OF THE INSTANTANEOUS P*
C       FROM TROUGH TO TROUGH. THE RESULTING "ENVELOPE" MAY BE         *
C       CONSIDERED TO BE A MEASURE OF THE PHASE OF AN ASSUMED NON-DISPE*
C       SEISMIC RESPONSE CONTAINED WITHIN THE ENVELOPE.                *
C       SEE TANER ET AL., 1979, GEOPHYSICS, V.44, #6,P.1041.           *
C                                                                      *
C       PROGRAM ASSUMES THE FIRST POINT IN THE SIGNAL IS A TROUGH      *
C       AND THE FIRST POINT OF ZERO SLOPE IS A PEAK. IF THIS POINT     *
C       IS A TROUGH IT WILL BE SKIPPED IN MNMX.                        *
C                                                                      *
C       NOTE FOR USERS:                                                *
C     THIS ROUTINE WILL FIND EVERY ENVELOPE PEAK REGARDLESS            *
C     OF RELATIVE AMPLITUDE. DISPLAY OF RESPONSE PHASE OF NOISY        *
C     DATA IS THUS MOST EFFECTIVE WITH A COLOR PLOT WHERE COLOR        *
C     INTENSITY IS AMPLITUDE MODULATED AND A COLOR WHEEL IS USED       *
C     FOR PHASE (0-360 DEGREES).                                       *
C                                                                      *
C  REVISED BY:  J. NORRIS                     REVISION DATE: 83/06/01  *
C       MODIFY MIN/MAX LOCATOR LOGIC TO BE SUBROUTINE CALL.            *
C       +------------------------------------------------------+       *
C       |                 ANALYSIS INFORMATION                 |       *
C       +------------------------------------------------------+       *
C  NON-STANDARD FEATURES:  NONE DETECTED                               *
C*******************   END OF DOCUMENTATION PACKAGE   ******************
C***********************************************************************
      SUBROUTINE RPHS (PHASE,A,NPT)
      DIMENSION A(*), PHASE(*), ITRF(2000), IPEAK(2000), P(3)
C***********************************************************************
C       FIND ARRAY POSITIONS OF MAXIMA AND MINIMA. THEN
C       SET IPEAK VALUES TO INTEGER VALUE OF INSTANTANEOUS
C       PHASE AT IPEAK(I) + FRACTION ARRAY POSITIONS.
C***********************************************************************
      CALL MNMX (A,NPT,IPEAK,JP,ITRF,JT)
C
      DO 20  I=1,JP
         K = IPEAK(I)
         DO 10 M=1,3
         P(M) = PHASE(K-2+M)
         IF( ABS(P(M)) .GT. 1.0E20 ) P(M)=0.0
         IF( 1/ABS(P(M)) .GT. 1.0E20 ) P(M)=0.0
C  50    P(M) = PHASE(K-2+M)
C   10    WRITE(*,*)M,P(M)
   10    CONTINUE
         CALL UNWRAP(P)
         FP = FPEAK(A,K)
         TK = FLOAT(K)
         IF(FP.LT.TK)IPEAK(I) = P(2)+(TK-FP)*(P(1)-P(2))
         IF(FP.GE.TK)IPEAK(I) = P(2)+(TK-FP)*(P(2)-P(3))
         IF(IPEAK(I).GT.180)IPEAK(I) = IPEAK(I)-360
C        WRITE (6,*) K,IPEAK(I),ITRF(I),PHASE(K)
   20 CONTINUE
C***********************************************************************
C       SET ENVELOPE LOBES TO IPEAK VALUES
C***********************************************************************
      J = 1
      ISTRT = 1
   30 IEND = ITRF(J)
      PEAK = FLOAT(IPEAK(J))
      DO 40  I=ISTRT,IEND
   40 A(I) = PEAK
C
      J = J+1
      IF (IEND.EQ.NPT) GO TO 50
      ISTRT = IEND
      GO TO 30
C
   50 CONTINUE
C
      RETURN
C
      END
C***********************************************************************
C                                                                      *
c                 copyright 2001, Amoco Production Company             *
c                             All Rights Reserved                      *
c                     an affiliate of BP America Inc.                  *
C                                                                      *
C***********************************************************************
C  ROUTINE:       FPEAK                                                *
C  ROUTINE TYPE:  FUNCTION  REAL                                       *
C  PURPOSE: ROUTINE TO FIND INFERRED FRACTIONAL ARRAY POSITION         *
C         (TK +/- FRACTION) OF A PEAK OF AN ARRAY.                     *
C  ENTRY POINTS:                                                       *
C      FPEAK  REAL  (A,K)                                              *
C  ARGUMENTS:                                                          *
C      A       REAL         I*  (1) - INPUT ARRAY OF VALUES            *
C      K       INTEGER      O*      - INDEX OF LOCAL MAXIMUM IN A      *
C       +------------------------------------------------------+       *
C       |               DEVELOPMENT INFORMATION                |       *
C       +------------------------------------------------------+       *
C  AUTHOR:   J.H. BODINE                        ORIGIN DATE: 82/10/17  *
C  LANGUAGE: FORTRAN 77                  DATE LAST COMPILED: 83/06/03  *
C       +------------------------------------------------------+       *
C       |                 EXTERNAL ENVIRONMENT                 |       *
C       +------------------------------------------------------+       *
C  ROUTINES CALLED:  NONE                                              *
C  INTRINSIC FUNCTIONS CALLED:                                         *
C      FLOAT   REAL    -                                               *
C      ABS     GENERIC -                                               *
C  FILES:            NONE                                              *
C  COMMON:           NONE                                              *
C  STOP CODES:       NONE                                              *
C       +------------------------------------------------------+       *
C       |             OTHER DOCUMENTATION DETAILS              |       *
C       +------------------------------------------------------+       *
C  ERROR HANDLING:  NONE                                               *
C  GENERAL DESCRIPTION:  LATERAL SYMMETRY ABOUT PEAK IS ASSUMED.       *
C  REVISED BY:  ???                           REVISION DATE: ?Y/?M/?D  *
C       +------------------------------------------------------+       *
C       |                 ANALYSIS INFORMATION                 |       *
C       +------------------------------------------------------+       *
C  NON-STANDARD FEATURES:  NONE DETECTED                               *
C*******************   END OF DOCUMENTATION PACKAGE   ******************
C***********************************************************************
      FUNCTION FPEAK (A,K)
      DIMENSION A(*)
C
      TK = FLOAT(K)
      DAL = ABS(A(K)-A(K-1))
      DAH = ABS(A(K)-A(K+1))
C
      TEST = DAL-DAH
      IF(TEST.GT.0.)FPEAK = TK+(1.-DAH/DAL)*.5
      IF(TEST.LT.0.)FPEAK = TK-(1.-DAL/DAH)*.5
      IF(TEST.EQ.0.)FPEAK = TK
C
      RETURN
      END
C***********************************************************************
C                                                                      *
c                 copyright 2001, Amoco Production Company             *
c                             All Rights Reserved                      *
c                     an affiliate of BP America Inc.                  *
C                                                                      *
C***********************************************************************
C  ROUTINE:       UNWRAP                                               *
C  ROUTINE TYPE:  SUBROUTINE                                           *
C  PURPOSE:                                                            *
C      ROUTINE TESTS FOR CONDITIONS REQUIRING UNWRAPPING OF PHASE      *
C      TO PERMIT INTERPOLATION ACROSS DISCONTINUITIES AT +/-180 DEG.   *
C  ENTRY POINTS:                                                       *
C      UNWRAP  (P)                                                     *
C  ARGUMENTS:                                                          *
C      P       REAL      U*  (1) - VALUES TO UNWRAP                    *
C       +------------------------------------------------------+       *
C       |               DEVELOPMENT INFORMATION                |       *
C       +------------------------------------------------------+       *
C  AUTHOR:   J.H. BODINE                        ORIGIN DATE: 82/10/17  *
C  LANGUAGE: FORTRAN 77                  DATE LAST COMPILED: 83/06/03  *
C       +------------------------------------------------------+       *
C       |                 EXTERNAL ENVIRONMENT                 |       *
C       +------------------------------------------------------+       *
C  ROUTINES CALLED:  NONE                                              *
C  INTRINSIC FUNCTIONS CALLED:                                         *
C      ABS     GENERIC -                                               *
C  FILES:            NONE                                              *
C  COMMON:           NONE                                              *
C  STOP CODES:       NONE                                              *
C       +------------------------------------------------------+       *
C       |             OTHER DOCUMENTATION DETAILS              |       *
C       +------------------------------------------------------+       *
C  ERROR HANDLING:  NONE                                               *
C  GENERAL DESCRIPTION:  NONE                                          *
C  REVISED BY:  ???                           REVISION DATE: ?Y/?M/?D  *
C       +------------------------------------------------------+       *
C       |                 ANALYSIS INFORMATION                 |       *
C       +------------------------------------------------------+       *
C  NON-STANDARD FEATURES:  NONE DETECTED                               *
C*******************   END OF DOCUMENTATION PACKAGE   ******************
C***********************************************************************
      SUBROUTINE UNWRAP(P)
      DIMENSION P(*)
C
      DIS = 0.
C
      DIFL = ABS(P(1)-P(2))
      DIFH = ABS(P(2)-P(3))
C
      IF (DIFL.LT.180..AND.DIFH.LT.180.) RETURN
C
      DO 10 I=1,3
   10 IF(P(I).LT.0.)P(I) = P(I)+360.
C
      RETURN
      END
       SUBROUTINE QCON(R,Q,A,N)
C
C  JH BODINE  10/19/82
C
C  ROUTINE TO OBTAIN COMPLIMENTARY FUNCTIONS FROM REAL
C  SEISMIC TRACE USING THE RESPONSE PHASE (DESCRIBED IN PROGRAM
C  CSIG). ONE FUNCTION IS ASSOCIATED WITH ZERO PHASE AND THE OTHER
C  WITH PI/2 PHASE.
C
C      R = REAL SEISMIC TRACE INPUT
C          RETURNED AS ZERO PHASE COMPONENT
C      Q = QUADRATURE FOR ANALYTIC SIGNAL AS INPUT
C          RETURNED AS PI/2 PHASE COMPONENT
C      A = RESPONSE PHASE ARRAY
C
C      N = INTEGER LENGTH OF ALL ARRAYS
C
C  DERIVATION OF MATH BY KEN HANSON.
C
C
       DIMENSION R(*),Q(*),A(*)
C
       PI = 3.14159265
       DEGRAD = PI/180.
C
       DO 10 I=1,N
       ANG = A(I) * DEGRAD
       RP = (COS(ANG)**2*R(I)+COS(ANG)*SIN(ANG)*Q(I))
       Q(I) = R(I) - RP
   10  R(I) = RP
       RETURN
       END
      SUBROUTINE ALEN(A,NPT)
C
C  JH BODINE 9/2/81
C
C  ROUTINE TO SET ANALYTIC SIGNAL ENVELOPE LOBES TO VALUE OF
C  THE TIME WIDTH OF THE ENVELOPE LOBES.
C  THE VALUE OF THE ENVELOPE IS SET FROM TROUGH
C  TO TROUGH. THE RESULTING "ENVELOPE" MAY BE CONSIDERED TO
C  REPRESENT A MODIFICATION OF
C     A = COMPLEX ENVELOPE
C   NPT = NUMBER OF SAMPLE POINTS
C
C  PROGRAM ASSUMES THE FIRST POINT IN THE SIGNAL IS A TROUGH
C  AND THE FIRST MAXIMA IS A PEAK. IF THE FIRST MAXIMA IS A TROUGH,
C  IT WILL BE SKIPPED (STATEMENT #50).
C
      DIMENSION A(*),ITRF(1000)
C
      JT = 1
      JP = 1
      SLOP = 0.
C
C  FIND ARRAY POSITIONS OF MAXIMA AND MINIMA
C
      NPT1 = NPT - 1
      DO 20  I = 2,NPT1
      SLAST = SLOP
      SLOP = A(I+1) - A(I)
      CHNG = SLAST * SLOP
      IF(CHNG.GE.0.) GO TO 20
      IF(CHNG.LT.0. .AND. SLAST.LT.0.) GO TO 10
      JP = JP+1
      GO TO 20
C
   10 IF(JP.EQ.1) GO TO 20
      ITRF(JT) = I
      JT = JT+1
C
   20 CONTINUE
      ITRF(JT)=NPT
      JP = JP-1
      IF(JP.EQ.0) JP=1
C
C  SET ENVELOPE LOBES TO VALUE OF ENVELOPE LOBE WIDTH
C
      J = 1
      ISTRT = 1
   30 IEND = ITRF(J)
      VALUE = IEND - ISTRT
      DO 40  I = ISTRT,IEND
   40 A(I) = VALUE
C
      J = J+1
      IF(IEND.EQ.NPT) GO TO 50
      ISTRT = IEND
      GO TO 30
C
   50 CONTINUE
C
      RETURN
      END
      SUBROUTINE ASLP(A,NPT)
C
C  JH BODINE 9/2/81
C
C  ROUTINE TO SET ANALYTIC SIGNAL ENVELOPE LOBES TO A VALUE
C  PROPORTIONAL TO THE MAX SLOPE OF THE ENVELOPE LOBES.
C
C    SLOPE = MAX(A(I+1)-A(I))/MAX(A(I)) * 1000.
C
C  THE VALUE OF THE ENVELOPE IS SET FROM TROUGH
C  TO TROUGH. THE RESULTING "ENVELOPE" MAY BE CONSIDERED TO
C  REPRESENT A MODIFICATION OF
C     A = COMPLEX ENVELOPE
C   NPT = NUMBER OF SAMPLE POINTS
C
C  PROGRAM ASSUMES THE FIRST POINT IN THE SIGNAL IS A TROUGH
C  AND THE FIRST MAXIMA IS A PEAK. IF THE FIRST MAXIMA IS A TROUGH,
C  IT WILL BE SKIPPED (STATEMENT #50).
C
      DIMENSION A(*),ITRF(1000),SLOPMX(1000),ENVMAX(1000)
C
      JT = 1
      JP = 1
      SLOP = 0.
C
C  FIND ARRAY POSITIONS OF MAXIMA AND MINIMA
C
      NPT1 = NPT - 1
      SMAX = 0.
      AMAX = 0.
      DO 20  I = 2,NPT1
      SLAST = SLOP
      SLOP = A(I+1) - A(I)
      CHNG = SLAST * SLOP
      IF(ABS(SLOP).GT.ABS(SMAX)) SMAX = SLOP
      IF(ABS(A(I)).GT.ABS(AMAX)) AMAX = A(I)
      IF(CHNG.GE.0.) GO TO 20
      IF(CHNG.LT.0. .AND. SLAST.LT.0.) GO TO 10
      JP = JP+1
      GO TO 20
C
   10 IF(JP.EQ.1) GO TO 20
      ENVMAX(JT) = AMAX
      SLOPMX(JT) = SMAX
      SMAX = 0.
      ITRF(JT) = I
      JT = JT+1
C
   20 CONTINUE
      ITRF(JT)=NPT
      JP = JP-1
      IF(JP.EQ.0) JP=1
C
C  SET ENVELOPE LOBES TO VALUE OF ENVELOPE LOBE WIDTH
C
      J = 1
      ISTRT = 1
   30 IEND = ITRF(J)
      DO 40  I = ISTRT,IEND
      IF (ENVMAX(J).EQ.0.) ENVMAX(J) = 1.
   40 A(I) = SLOPMX(J)/ENVMAX(J)*1000.
C
      J = J+1
      IF(IEND.EQ.NPT) GO TO 50
      ISTRT = IEND
      GO TO 30
C
   50 CONTINUE
C
      RETURN
      END
      SUBROUTINE ASKW(A,NPT)
C
C  JH BODINE 9/2/81
C
C  ROUTINE TO CALCULATE THE ASYMMETRY OF THE ENVELOPE
C
C  ROUTINE TO SET ANALYTIC SIGNAL ENVELOPE LOBES TO VALUE OF
C  THE DIFFERENCE BETWEEN THE ARRAY POSITIONS OF THE ENVELOPE LOBE
C  PEAK AND THE FIRST MOMENT OF THE LOBE.
C  THE VALUE OF THE ENVELOPE IS SET TO THAT OF THIS DIFFERENCE * 100
C  FROM TROUGH TO TROUGH. THE RESULTING "ENVELOPE" MAY BE
C  CONSIDERED TO BE A MEASURE OF THE DEGREE OF ASYMMETRY OF EACH
C  ENVELOPE LOBE.
C
C  SEE TANER ET AL., 1979, GEOPHYSICS, V.44, #6,P.1041.
C     A = COMPLEX ENVELOPE
C   NPT = NUMBER OF SAMPLE POINTS
C
C  PROGRAM ASSUMES THE FIRST POINT IN THE SIGNAL IS A TROUGH
C  AND THE FIRST POINT OF ZERO SLOPE IS A PEAK. IF THIS POINT
C  IS A TROUGH IT WILL BE SKIPPED (STATEMENT #50).
C
C  NOTE FOR USER'S;
C     THIS ROUTINE WILL FIND EVERY ENVELOPE PEAK REGARDLESS
C     OF RELATIVE AMPLITUDE. DISPLAY OF RESULTS FOR NOISY
C     DATA IS THUS MOST EFFECTIVE WITH A COLOR PLOT WHERE COLOR
C     INTENSITY IS AMPLITUDE MODULATED.
C
C
      DIMENSION A(*),ITRF(1000),IPEAK(1000)
C
      JT = 1
      JP = 1
      SLOP = 0.
C
C  FIND ARRAY POSITIONS OF MAXIMA AND MINIMA
C
      NPT1 = NPT - 1
      DO 20  I = 2,NPT1
      SLAST = SLOP
      SLOP = A(I+1) - A(I)
      CHNG = SLAST * SLOP
      IF(CHNG.GE.0.) GO TO 20
      IF(CHNG.LT.0. .AND. SLAST.LT.0.) GO TO 10
      IPEAK(JP) = I
      JP = JP+1
      GO TO 20
C
   10 IF(JP.EQ.1) GO TO 20
      ITRF(JT) = I
      JT = JT+1
C
   20 CONTINUE
      ITRF(JT)=NPT
      JP = JP-1
      IF(JP.EQ.0) JP=1
C
C  SET ENVELOPE LOBES TO DIFFERENCE IN ARRAY POSITION OF ENVELOPE
C  PEAK AND FIRST MOMENT
C
      J = 1
      ISTRT = 1
   30 IEND = ITRF(J)
      SUMM = 0.
      ASUM = 0.
C  GET FRACTIONAL ARRAY POSITION OF ENVELOPE PEAK
      K = IPEAK(J)
      FP = FPEAK(A,K)
C  GET POSITION OF FIRST MOMENT
      DO 40  II = ISTRT,IEND
      SUMM = SUMM + (FP-II)*A(II)
      ASUM = ASUM + A(II)
   40 CONTINUE
      IF(ASUM.EQ.0.) ASUM = 1.
      AMOM = SUMM/ASUM
      IF(ASUM.EQ.0.) AMOM = 0.
C  SET ENVELOPE ARRAY TO DIFFERENCE OF PEAK AND 1ST MOM POSITIONS
      DO 50  I = ISTRT,IEND
   50 A(I) = AMOM *100.
C
      J = J+1
      IF(IEND.EQ.NPT) GO TO 60
      ISTRT = IEND
      GO TO 30
C
   60 CONTINUE
C
      RETURN
      END
C***********************************************************************
C                                                                      *
c                 copyright 2001, Amoco Production Company             *
c                             All Rights Reserved                      *
c                     an affiliate of BP America Inc.                  *
C                                                                      *
C***********************************************************************
C  ROUTINE:       PKDT                                                 *
C  ROUTINE TYPE:  SUBROUTINE                                           *
C  PURPOSE:  TO SET ANALYTIC SIGNAL ENVELOPE LOBES TO VALUE OF         *
C            THE INPUT DATA AT THE POINT WHERE THE ENVELOPE            *
C            IS A MAXIMUM. VALUE OF ENVELOPE SET FROM TROUGH TO TROUGH *
C  ENTRY POINTS:                                                       *
C      PKDT  (DATA,A,NPT)                                              *
C  ARGUMENTS:                                                          *
C      DATA    REAL         I*  (1) - INSTANTANEOUS ATTRIBUTE ARRAY    *
C      A       REAL         U*  (1) - COMPLEX ENVELOPE ON INPUT        *
C                                   - RESPONSE ATTRIBUTE OUTPUT        *
C      NPT     INTEGER      I*      - NUMBER OF SAMPLES IN 'DATA'&'A'  *
C       +------------------------------------------------------+       *
C       |               DEVELOPMENT INFORMATION                |       *
C       +------------------------------------------------------+       *
C  AUTHOR:   J.H. BODINE                        ORIGIN DATE: 81/09/02  *
C  LANGUAGE: FORTRAN 77/IV               DATE LAST COMPILED: 83/06/03  *
C       +------------------------------------------------------+       *
C       |                 EXTERNAL ENVIRONMENT                 |       *
C       +------------------------------------------------------+       *
C  ROUTINES CALLED:                                                    *
C      MNMX - TO FIND RELATIVE MINIMA/RELATIVE MAXIMA OF AMPLITUDE FCN *
C  INTRINSIC FUNCTIONS CALLED:  NONE                                   *
C  FILES:                                                              *
C      6  ( OUTPUT SEQUENTIAL ) - DEBUG FILE                           *
C  COMMON:           NONE                                              *
C  STOP CODES:       NONE                                              *
C       +------------------------------------------------------+       *
C       |             OTHER DOCUMENTATION DETAILS              |       *
C       +------------------------------------------------------+       *
C  ERROR HANDLING:  NONE                                               *
C  GENERAL DESCRIPTION:  PROGRAM ASSUMES FIRST POINT IN SIGNAL IS TROUG*
C      AND THE FIRST MAXIMA IS A PEAK. IF THE FIRST MAXIMA IS A TROUGH,*
C      IT WILL BE SKIPPED IN MNMX.                                     *
C  REVISED BY:  J. NORRIS (PKDTA)             REVISION DATE: 83/06/02  *
C       ADD CALL TO MNMX TO SCAN FOR RELATIVE MINIMA/MAXIMA.           *
C       +------------------------------------------------------+       *
C       |                 ANALYSIS INFORMATION                 |       *
C       +------------------------------------------------------+       *
C  NON-STANDARD FEATURES:  NONE DETECTED                               *
C*******************   END OF DOCUMENTATION PACKAGE   ******************
C***********************************************************************
      SUBROUTINE PKDT(DATA,A,NPT)
      DIMENSION A(*),DATA(*),ITRF(1000),IPEAK(1000),PEAK(1000)
C***********************************************************************
C       FIND ARRAY POSITIONS OF MAXIMA AND MINIMA. THEN SET PEAK VALUES
C        OF INPUT DATA AT DATA AT IPEAK(I) ARRAY POSITIONS
C***********************************************************************
      CALL MNMX(A,NPT,IPEAK,JP,ITRF,JT)
C
      DO 10  I=1,JP
         K = IPEAK(I)
         PEAK(I) = DATA(K)
   10 CONTINUE
C***********************************************************************
C  SET ENVELOPE LOBES TO IPEAK VALUES
C***********************************************************************
      J = 1
      ISTRT = 1
   20 IEND = ITRF(J)
      DO 30  I = ISTRT,IEND
   30 A(I) = PEAK(J)
C
      J = J+1
      IF(IEND.EQ.NPT) GO TO 40
      ISTRT = IEND
      GO TO 20
C
   40 CONTINUE
C
      RETURN
      END
      FUNCTION XFREQ(R,Q,DX,NPT,M)
C
C  JH BODINE  7/28/81
C
C  CALCULATES INSTANTANEOUS FREQUENCY FOR ANALYTIC
C  TRACE ANALYSIS USING FINITE DIFFERENCE APPROXIMATIONS
C  (SEE TANER ET AL., 1979, GEOPHYSICS, V44 NO.6, P1041.).
C
C      R, Q = REAL AND QUADRATURE TRACES
C       NPT = NUMBER OF POINTS IN THE TRACE ARRAYS
C        DX = SAMPLE INTERVAL
C         M = CURRENT CENTRAL DIFFERENCE POSITION
C
      DIMENSION R(*),Q(*)
      PI = 3.14159265
C
      XFREQ = 0.
      IF(R(M).EQ.0. .AND. Q(M).EQ.0.) GO TO 10
C
      RLST = R(2)
      QLST = Q(2)
      RMST = R(NPT-1)
      QMST = Q(NPT-1)
C
      IF(M.LT.NPT) RMST = R(M+1)
      IF(M.LT.NPT) QMST = Q(M+1)
      IF(M.GT.1) RLST = R(M-1)
      IF(M.GT.1) QLST = Q(M-1)
C
      XFREQ = (R(M)*(QMST-QLST)/(2.*DX)
     : - Q(M)*(RMST-RLST)/(2.*DX))
     : /(R(M)**2 + Q(M)**2)
C
   10 RETURN
      END
      FUNCTION PHAS(FR,FI)
C
C  JH BODINE    7/28/81
C
C  PROGRAM CALCULATES PHASE IN RADIANS ZERO TO 2*PI
C  FROM ENTERED REAL AND IMAGINARY COMPONENTS.
C
      PHAS = 0.
      PI=3.14159265
      A = (FR**2 + FI**2)**.5
      IF(A.EQ.0.) GO TO 10
      ARG = ABS(FI)/A
      IF(ARG.GT.1.) ARG=1.
C
C  0 TO PI/2
      IF(FI.GT.0. .AND. FR.GE.0.) PHAS=ASIN(ARG)
C
C  PI/2 TO PI
      IF(FI.GE.0. .AND. FR.LT.0.) PHAS=PI - ASIN(ARG)
C
C  PI TO 3*PI/2
      IF(FI.LT.0. .AND. FR.LE.0.) PHAS=PI + ASIN(ARG)
C
C  3*PI/2 TO 2*PI
      IF(FI.LT.0. .AND. FR.GT.0.) PHAS=2*PI - ASIN(ARG)
C
   10 RETURN
      END
      SUBROUTINE FOLD (LA,A,LB,B,LC,C)
C***********************************************************************
C                                                                      *
c                 copyright 2001, Amoco Production Company             *
c                             All Rights Reserved                      *
c                     an affiliate of BP America Inc.                  *
C                                                                      *
C***********************************************************************
C  ROUTINE:       FOLD
C  ROUTINE TYPE:  SUBROUTINE  SINGLE_ENTRY
C  PURPOSE:
C      This subroutine finds the complete transient convolution of the
C      vectors A and B.
C  ARGUMENTS:
C      LA  I*4  I - LENGTH OF VECTOR A IN SAMPLES
C      A   R*4  I  ( 2 ) - THE VECTOR A A(1),A(2),.....,A(LA)
C      LB  I*4  I - THE LENGTH OF VECTOR B IN SAMPLES
C      B   R*4  I  ( 2 ) - THE VECTOR B B(1),B(2),.....,B(LA)
C      LC  I*4  O - THE LENGTH OF VECTOR C (LA + LB - 1)
C      C   R*4  O  ( 2 ) - THE COMPLETE TRANSIENT CONVOLUTION VECTOR C
C  CATEGORY:  UTILITY ARRAY CONVOLVE FILTER
C  KEYWORDS:  CONVOLUTION
C       +------------------------------------------------------+
C       |               DEVELOPMENT INFORMATION                |
C       +------------------------------------------------------+
C  AUTHOR:    ?                               ORIGIN DATE:     ?
C  LANGUAGE:  FORTRAN IV
C       +------------------------------------------------------+
C       |                 EXTERNAL ENVIRONMENT                 |
C       +------------------------------------------------------+
C  ROUTINES CALLED:
C      MOVE     - USED TO ZERO THE VECTOR C ARRAY BEFORE CONVOLUTION
C       +------------------------------------------------------+
C       |             OTHER DOCUMENTATION DETAILS              |
C       +------------------------------------------------------+
C  ERROR HANDLING:
C  GENERAL DESCRIPTION:
C            NOTE: THIS SUBROUTINE ZEROS THE VECTOR C PRIOR
C                  TO CONVOLUTION
C*******************   END OF DOCUMENTATION PACKAGE   ******************
C***********************************************************************
C     THIS IS A CONFIDENTIAL AMOCO PRODUCTION COMPANY PROGRAM
C
C     THIS SUBROUTINE FINDS THE COMPLETE TRANSIENT CONVOLUTION OF THE
C     VECTORS A AND B
C
C     INPUTS ARE
C        LA=LENGTH OF THE VECTOR A    INTEGER*4
C         A=THE VECTOR A, A(1),A(2),...A(LA)      REAL*4
C        LB=LENGTH OF THE VECTOR B      INTEGER*4
C         B=THE VECTOR B, B(1),B(2),...B(LB)    REAL*4
C     OUTPUTS ARE
C        LC=LA+LB-1=LENGTH OF THE VECTOR C     INTEGER*4
C         C=THE COMPLETE TRANSIENT CONVOLUTION VECTOR C     REAL*4
C
C     NOTE--THIS SUBROUTINE ZEROS THE VECTOR C PRIOR TO CONVOLUTION
C
C          THIS SUBROUTINE IS MAINTAINED BY A. DOWDY
C
      DIMENSION A(*),B(*),C(*)
      LC=LA+LB-1
C
C
C  ZERO SUBSTITUTED FOR IBM SYSTEM ROUTINE: MOVE
C
C
      CALL ZERO(C,LC)
C     CALL MOVE(0,C,0,LC*4)
      DO 10 I=1,LC
   10 C(I) = 0.
      DO 20 I=1,LA
      DO 20 J=1,LB
      K=I+J-1
   20 C(K)=A(I)*B(J)+C(K)
      RETURN
      END
C
      SUBROUTINE ZERO(ARRAY,NSAMP)
      REAL ARRAY(*)
      DO 10 I=1,NSAMP
          ARRAY(I) = 0.0
   10 CONTINUE
      RETURN
      END
C
C
      SUBROUTINE HILF(TDEL,WC,LF,F)
C                            FORTRAN BY KEN PEACOCK  11-11-76
C      SUBROUTINE HILF CONSTRUCTS THE OPERATOR FOR HILBERT
C      TRANSFORMATION. WEIGHTING IS AVAILABLE ON OPTION.
C         INPUTS ARE;
C           TDEL = SAMPLE INCREMENT IN SECONDS
C             WC = 0 OUTPUT RAW OPERATOR
C                NE 0 USE A ROSS WEIGHT FUNCTION WITH EXPONENT WC
C             LF = LENGTH OF FILTER IN SAMPLES (MUST BE ODD)
C        OUTPUT IS;
C              F = THE LF-LENGTH OUTPUT ARRAY
C      CODED FOR THE IBM 370/158 COMPUTER
C      VERSION AS OF 11-11-76
C
      DIMENSION F(*)
      N = LF/2
      ISTA = N+2
      KFACT = 1-ISTA
      JFACT = LF+1
      DO 10 I=1,LF
   10 F(I) = 0.
      FACT = -2./(TDEL*3.14159265)
      DO 20 I=1,N,2
      J = N+1+I
      K = N+1-I
      F(J) = FACT/I
   20 F(K) = -F(J)
      IF(WC.EQ.0.) GO TO 40
      DO 30 I=ISTA,LF
      AK = I+KFACT
      F(I) = F(I)*((1.-(AK/N)**2)**WC)
      J = JFACT-I
   30 F(J) = -F(I)
   40 RETURN
      END
C***********************************************************************
C                                                                      *
c                 copyright 2001, Amoco Production Company             *
c                             All Rights Reserved                      *
c                     an affiliate of BP America Inc.                  *
C                                                                      *
C***********************************************************************
C  ROUTINE:       MNMX                                                 *
C  ROUTINE TYPE:  SUBROUTINE                                           *
C  PURPOSE:  TO PICK RELATIVE MINIMA AND MAXIMA FOR INPUT FUNCTION.    *
C  ENTRY POINTS:                                                       *
C      MNMX  (A,NPT,IPEAK,JP,ITRF,JT)                                  *
C  ARGUMENTS:                                                          *
C      A       REAL         I*  (1) - INPUT ARRAY TO SCAN FOR MIN/MAX  *
C      NPT     INTEGER      I*      - NUMBER OF SAMPLES IN ARRAY 'A'   *
C      IPEAK   INTEGER      O*  (1) - INDEX  ARRAY OF RELATIVE PEAKS   *
C      JP      INTEGER      O*      - NUMBER OF RELATIVE PEAKS IN IPEAK*
C      ITRF    INTEGER      O*  (1) - INDEX  ARRAY OF RELATIVE TROUGHS *
C      JT      INTEGER      O*      - NUMBER OF REL TROUGHS IN ITRF    *
C       +------------------------------------------------------+       *
C       |               DEVELOPMENT INFORMATION                |       *
C       +------------------------------------------------------+       *
C  AUTHOR:   J. NORRIS                          ORIGIN DATE: 83/06/03  *
C  LANGUAGE: FORTRAN 77/IV               DATE LAST COMPILED: 83/06/03  *
C       +------------------------------------------------------+       *
C       |                 EXTERNAL ENVIRONMENT                 |       *
C       +------------------------------------------------------+       *
C  ROUTINES CALLED:  NONE                                              *
C  INTRINSIC FUNCTIONS CALLED:  NONE                                   *
C  FILES:            NONE                                              *
C  COMMON:           NONE                                              *
C  STOP CODES:       NONE                                              *
C       +------------------------------------------------------+       *
C       |             OTHER DOCUMENTATION DETAILS              |       *
C       +------------------------------------------------------+       *
C  ERROR HANDLING:  IF NPT <=1, SUBROUTINE IS NOT EXECUTED.            *
C  GENERAL DESCRIPTION:  EVERY PEAK OR TROUGH IS FOUND REGARDLESS OF   *
C       AMPLITUDE.  THIS CODE ADAPTED FROM PKDT AND RPHS BY J.H.BODINE.*
C       +------------------------------------------------------+       *
C  REVISED BY:  ???                           REVISION DATE: ?Y/?M/?D  *
C       +------------------------------------------------------+       *
C       |                 ANALYSIS INFORMATION                 |       *
C       +------------------------------------------------------+       *
C  NON-STANDARD FEATURES:  NONE DETECTED                               *
C*******************   END OF DOCUMENTATION PACKAGE   ******************
C***********************************************************************
      SUBROUTINE MNMX(A,NPT,IPEAK,JP,ITRF,JT)
      DIMENSION A(*), ITRF(*), IPEAK(*)
C
      JT = 0
      JP = 0
      SLOP = 0.
      ILAST=0
C***********************************************************************
C ***** FIND ARRAY POSITIONS OF MAXIMA AND MINIMA AFTER CHECKING VALID
C ***** LOOP INDEX.
C***********************************************************************
      NPT1 = NPT-1
      IF(NPT1.LE.0) GO TO 30
      DO 20  I=1,NPT1
         SLAST = SLOP
         SLOP = A(I+1)-A(I)
         CHNG = SLAST*SLOP
         IF (CHNG.GT.0.) GO TO 20
         IF (SLOP.EQ.0..AND.SLAST.EQ.0.) GO TO 20
         IF (SLAST.LE.0..AND.SLOP.GE.0.) GO TO 10
C***********************************************************************
C ******* HERE FOR A PEAK.
C ******* CHECK FOR TWO CONSECUTIVE PEAKS. IGNORE SUBSEQUENT PEAKS UNTIL
C ******* A TROUGH IS FOUND.  ILAST=1 FOR PEAK.
C***********************************************************************
         IF(ILAST.EQ.1)GO TO 20
         JP = JP+1
         IPEAK(JP) = I
         ILAST=1
         GO TO 20
C***********************************************************************
C ******* HERE FOR A TROUGH. SEE IF PEAK ALREADY PICKED FIRST.
C ******* CALLING ROUTINES EXPECT FIRST POINT OF ZERO SLOPE TO BE PEAK.
C ******* CHECK FOR TWO CONSECUTIVE TROUGHS. IGNORE SUBSEQUENT TROUGHS
C ******* UNTIL A PEAK IS FOUND.ILAST=-1 FOR TROUGH.
C***********************************************************************
   10    IF (JP.EQ.0) GO TO 20
         IF(ILAST.EQ.-1)GO TO 20
         JT = JT+1
         ITRF(JT) = I
         ILAST=-1
   20 CONTINUE
C***********************************************************************
C ******* ASSIGN LAST POINT OF TROUGH ARRAY TO BE LAST SAMPLE. IF NO
C ******* PEAKS FOUND, DEFAULT SET JP TO ONE POINT.
C***********************************************************************
      ITRF(JT+1) = NPT
C
   30 CONTINUE
      IF(JP.EQ.0)JP=1
      RETURN
      END
C
