C***********************************************************************
C                 copyright 2001, Amoco Production Company             *
C                             All Rights Reserved                      *
C                     an affiliate of BP America Inc.                  *
C***********************************************************************
c**********************************************************************c
c
c lsqinv reads seismic trace data from an input file,
c performs least squares inversion, and
c writes the results to an output file
c
c
c**********************************************************************c
c
c
c Changes
c
c April 03, 2003  -removed symmetric assumption from -A application
c Garossino
c
c     declare variables
c

      implicit none

#include <save_defs.h>
#include <f77/iounit.h>
#include <f77/lhdrsz.h>
#include <f77/sisdef.h>

c declare standard USP variables
 
      integer     itr ( SZLNHD )
      integer     nsamp, nsi, ntrc, nrec, iform, ist, iend
      integer     luin , luout, lbytes, nbytes, lbyout, obytes
      integer     argis, jerr, JJ, KK, nsampo

      real UnitSc, dt
      real tri ( SZLNHD )

#include <f77/pid.h>

c declare program dependent static variables

      integer nfilt, isp, ldsr
      integer nfilt2, i, ied, nfiltk, ldsrk, lc
      integer ifmt_RecNum, l_RecNum, ln_RecNum, recnum
      integer ifmt_TrcNum, l_TrcNum, ln_TrcNum, trcnum
      integer ifmt_StaCor, l_StaCor, ln_StaCor, static

      real prew, ang, xmax, sgn, ase

      real  trii ( SZLNHD ), xin ( SZLNHD )
      real  work ( SZLNHD ), space ( 4*SZLNHD )
      real  d(SZLNHD), weight(SZLNHD)

      character   ntap * 256, otap * 256, name*6

      logical verbos, actual, maxmum, bart, cost
      logical wcenter
 
c initialize variables

      data lbytes / 0 /, nbytes / 0 /, name/'LSQINV'/
 
c-----
c give command line help if requested
c-----

      if ( argis ( '-?' ) .gt. 0  .or. 
     :     argis ( '-h' ) .gt. 0  .or. 
     :     argis ( '-help' ) .gt. 0  ) then
            call help()
            stop
      endif
 
c-----
c     open printout files
c-----
#include <f77/open.h>
 
c-----
c     read program parameters from command line card image file
c-----

      call gcmdln(ntap,otap,ist,iend,prew,isp,actual,maxmum,verbos,
     1            ldsr,bart,cost,wcenter)
c-----
c     get logical unit numbers for input and output
c-----
      call getln(luin , ntap,'r', 0)
      call getln(luout, otap,'w', 1)

c-----
c     read line header of input
c     save certain parameters
c-----
      call rtape   ( luin, itr, lbytes)
      if(lbytes .eq. 0) then
         write(LER,*)'LSQINV: no header read from unit ',luin
         write(LER,*)'FATAL'
         stop
      endif

      call saver(itr, 'NumSmp', nsamp, LINHED)
      call saver(itr, 'SmpInt', nsi  , LINHED)
      call saver(itr, 'NumTrc', ntrc , LINHED)
      call saver(itr, 'NumRec', nrec , LINHED)
      call saver(itr, 'Format', iform, LINHED)
      call saver(itr, 'UnitSc', unitsc, LINHED)
      if (unitsc .eq. 0.0) then
          write(LERR,*)'********************************************'
          write(LERR,*)'WARNING: sample unit scaler in LH = ',unitsc
          write(LERR,*)'         will set to .001 (millisec default)'
          write(LERR,*)'********************************************'
          unitsc = .001
          call savew(itr, 'UnitSc', unitsc, LINHED)
      endif

c------
c     save certain parameters
 
c     For trace header values we take mnemonics and build a
c     set of pointers to an I*2 array equivalenced to the
c     RTAPE  Integer array (headers + data)
c     TRACEHEADER is a value in the include file <sisdef.h> that
c     refers to the trace header
 
      call savelu('TrcNum',ifmt_TrcNum,l_TrcNum,ln_TrcNum,TRACEHEADER)
      call savelu('RecNum',ifmt_RecNum,l_RecNum,ln_RecNum,TRACEHEADER)
      call savelu('StaCor',ifmt_StaCor,l_StaCor,ln_StaCor,TRACEHEADER)

      call hlhprt (itr, lbytes, name, 6, LERR)

c  figure out design window times

      dt = real (nsi) * unitsc

      ist = ist/nsi
      if (ist .lt. 1) ist = 1
      isp = isp/nsi
      if (isp .lt. 1) isp = 1
      iend = iend/nsi
      if (iend .lt. 1) iend = nsamp
      if (iend .gt. nsamp) iend = nsamp
      nfilt  = iend - ist + 1
      nfilt2 = nfilt / 2
      ldsr = ldsr/nsi
      if (ldsr .le. 0) ldsr = nfilt

      if (isp .gt. ldsr) then
         write(LERR,*)'WARNING: desired output spike is beyond end of'
         write(LERR,*)'design window.  Resetting it to = last sample'
         write(LERR,*)'in design window'
         isp = nfilt
      endif

      if (bart) then
         write(LERR,*)'Computing ',nfilt,' Bartlett weights'
         do  i = 1, nfilt
             weight(i) = float( nfilt - i + 1) / float(nfilt)
         enddo
      elseif (cost) then
         write(LERR,*)'Computing ',nfilt,' Cosine weights'
         do  i = 1, nfilt
             ang = 3.14159265 * float(i-1)/float(nfilt)
             weight(i) = .5 * (1. + cos ( ang ))
         enddo
      else
         call vfill (1.0, weight, 1, nfilt)
      endif

      nsampo = nsamp
      obytes = SZTRHD + nsampo * SZSMPD
      call savew (itr, 'NumSmp', nsampo, LINHED)

c-----
c     verbose output of all pertinent information before
c     processing begins
c-----

      call verbal(nsamp, nsi, ntrc, nrec, iform, nfilt,
     1     ist,iend,prew,isp,ntap,otap,maxmum,ldsr,
     2     bart,cost,wcenter)

      prew = prew/100.

c-----
c     update historical line header
c     write out line header
c-----

      call savhlh(itr,lbytes,lbyout)
      call wrtape ( luout, itr, lbyout )

c--------------------------
c  set up desired output, put spike at user defined position

      call vclr (d,1,ldsr)
      d(isp) = 1.0

c-----
c     BEGIN PROCESSING
c     read trace, do inverse, write to output file
c-----

      do 1000 JJ = 1, nrec
         
         do 1001 KK= 1, ntrc

            nbytes = 0
            call rtape( luin, itr, nbytes)
            if(nbytes .eq. 0) then
               write(LERR,*)'Premature End of file on input:'
               write(LERR,*)'  rec= ',JJ,'  trace= ',KK
               go to 999
            endif

            call vmov (itr(ITHWP1), 1, tri, 1, nsamp)

            call saver2(itr,ifmt_RecNum,l_RecNum, ln_RecNum,
     1           recnum , TRACEHEADER)
            call saver2(itr,ifmt_TrcNum,l_TrcNum, ln_TrcNum,
     1           trcnum , TRACEHEADER)
            call saver2(itr,ifmt_StaCor,l_StaCor, ln_StaCor,
     1           static, TRACEHEADER)
            
            IF (static .ne. 30000) THEN

c-------------------------
c  variable spike position

               if (maxmum) then
                  call vclr (d,1,nsamp)
                  call maxmgv (tri,1,xmax,isp,nsamp)
                  sgn = sign (1.0, tri(isp))
                  d(isp) = sgn
               else
                  sgn = sign (1.0, tri(isp))
                  d(isp) = sgn
               endif
c-------------------------

               call vclr (work , 1, max(nsamp,nfilt))
               call vclr (space, 1, SZLNHD)

c-------------------------
c  set up design window

               if (wcenter) then
                  ist = isp - nfilt2
                  ied = isp + nfilt2
                  if (ist .lt. 1) ist = 1
                  if (ied .gt. nsamp) ied = nsamp
                  nfiltk = ied - ist + 1
                  ldsrk = nfiltk
                  call vmov (d(ist), 1, work, 1, ldsr)
                  call vmov (work, 1, d, 1, ldsrk)
               else
                  nfiltk = nfilt
                  ldsrk  = ldsr
               endif
               call vmov (tri(ist), 1, xin, 1, nfiltk)
c-------------------------

               call vclr (trii, 1, nsampo)
               call shape1 (nfiltk,xin,ldsrk,d,nfiltk,trii,lc,
     1              work,ase,prew,space,weight)

            ELSE
               call vclr(trii,1,nsampo)
               trii (1) = 1.0
               call vclr(itr(ITHWP1),1,nsampo)
            ENDIF

            if (actual) then

c               ish = nfiltk / 2 - 1
c               if (ish .lt. 1 ) ish = 1
c hmmm  I do not know what this ish stuff was all about as the filter
c       designed by shape1 is single ended and minimum phase by default.
c       It seems the correct application requires output from the 
c       start of work in all cases....Garossino Apr3,2003

               call fold (nfiltk, trii, nsamp, tri, lc, work)

c               call vmov (work(ish),1,itr(ITHWP1),1,nsamp)

               call vmov (work,1,itr(ITHWP1),1,nsamp)

            else
               call vmov (trii,1,itr(ITHWP1),1,nsampo)
            endif

            call wrtape( luout, itr, obytes)

            if(verbos)write(LERR,*)'ri ',recnum,' trace ',trcnum,
     1           ' ase=  ',ase,' isp= ',isp
 1001    continue
 
 1000 continue
c-----
c     close data files
c-----
  999 continue

      call lbclos ( luin )
      call lbclos ( luout )

            write(LERR,*)'end of lsqinv, processed',nrec,' record(s)',
     :               ' with ',ntrc, ' traces'
      end
 
C***********************************************************************
      subroutine help
#include <f77/iounit.h>
         write(LER,*)
     :'***************************************************************'
        write(LER,*)
     :'                 lsqinv - least squares inverse'
        write(LER,*)
     :'execute lsqinv by typing lsqinv and a list of program parameters'
        write(LER,*)
     :'note that each parameter is proceeded by -a where "a" is '
        write(LER,*)
     :'a character(s) corresponding to some parameter.'
        write(LER,*)
     :'users enter the following parameters, or use the default values'
        write(LER,*) ' '
        write(LER,*)
     :' -N [ntap]    (no default)      : input data file name'
        write(LER,*)
     :' -O [otap]    (no default)      : output data file name'
        write(LER,*)' '
        write(LER,*)
     :' -s[ist] (default = 0 ms)          : window start time (ms)'
        write(LER,*)
     :' -e[iend] (default = end trace)    : window end time (ms)'
        write(LER,*)
     :' -p[prew] (default = .01%)         :  prewhitening percent'
        write(LER,*)
     :' -i[isp] (default = ist ms)        :  time of inverse spike (ms)'
        write(LER,*)
     :' -d[ldesr] (default = design wind) :  length desired output (ms)'
        write(LER,*)
     :' -M  put inverse spike at position of max abs of input trace'
        write(LER,*)
     :' -W  center design window on spike position'
        write(LER,*)
     :' -A  write out actual output of filter * input data'
        write(LER,*)
     :' -B  apply Bartlett weighting to autocorrelation functions'
        write(LER,*)
     :' -C  apply Cosine weighting to autocorrelation functions'
        write(LER,*)
     :' -V  verbose printout'
        write(LER,*) ' '
        write(LER,*)
     :'usage:   lsqinv -N[ntap] -O[otap] -s[ist] -e[iend] -p[prew] '
        write(LER,*)
     :'                -i[isp] -d[ldsr] [-M -W -A -C -B -V]'
         write(LER,*)
     :'***************************************************************'
      return
      end
 
C***********************************************************************
      subroutine gcmdln(ntap,otap,ist,iend,prew,isp,actual,maxmum,
     1                   verbos, ldsr,bart,cost,wcenter)
c-----
c     get command arguments
c
c     ntap  - c*100     input file name
c     otap  - c*100     output file name
c     ist   - I*4       window start time
c     iend  - I*4       window end time
c     isp   - I*4       time of inverse spike
c     ldsr  - I*4       length of desired output
c     prew  - R*4       prewhitening percent
c     maxmum      - L   put inverse spike at time of abs max of input
c     actual      - L   write out acual output of filter
c     verbos      - L   verbose output or not
c-----
#include <f77/iounit.h>
      character   ntap*(*), otap*(*)
      real        prew
      integer     ist, iend, isp, ldsr
      logical     verbos, actual, maxmum, bart, cost, wcenter
      integer     argis
 
            call argstr( '-N', ntap, ' ', ' ' )
            call argstr( '-O', otap, ' ', ' ' )
            call argr4( '-p', prew, .01, .01 )
            call argi4( '-i', isp, 0, 0 )
            call argi4( '-s', ist, 0, 0 )
            call argi4( '-e', iend, 0, 0 )
            call argi4( '-d', ldsr, 0, 0 )
            maxmum = (argis('-M') .gt. 0)
            wcenter= (argis('-W') .gt. 0)
            actual = (argis('-A') .gt. 0)
            cost   = (argis('-C') .gt. 0)
            bart   = (argis('-B') .gt. 0)
            verbos = (argis('-V') .gt. 0)

            if (wcenter) then
               cost = .false.
               bart = .false.
            endif
 
      return
      end
 
      subroutine verbal(nsamp, nsi, ntrc, nrec, iform, nfilt,
     1            ist,iend,prew,isp,ntap,otap,maxmum,ldsr,
     2            bart,cost,wcenter)
c-----
c     verbose output of processing parameters
c
c     nsamp - I*4       number of samples in trace
c     nsi   - I*4       sample interval in ms
c     ist   - I*4       window start time
c     iend  - I*4       window end time
c     ntrc  - I*4       traces per record
c     nrec  - I*4       Number of records per line
c     iform - I*4       format of data
c     prew  - R*4       prewhitening percent
c     isp   - I*4       time of inverse spike
c     nfilt - I*4       length of design window
c     ldsr  - I*4       length of desired output
c     maxmum      - L   put inverse spike at time of abs max of input
c     ntap  - C*100     input file name
c     otap  - C*100     output file name
c-----
#include <f77/iounit.h>
      integer     nsamp, nsi, ntrc, nrec, isp
      real        prew
      character   ntap*(*), otap*(*)
      logical     maxmum,bart,cost,wcenter
 
            write(LERR,*)' '
            write(LERR,*)' line header values after default check '
            write(LERR,*) ' # of samples/trace =  ', nsamp
            write(LERR,*) ' sample interval    =  ', nsi
            write(LERR,*) ' traces per record  =  ', ntrc
            write(LERR,*) ' records per line   =  ', nrec
            write(LERR,*) ' format of data     =  ', iform
            write(LERR,*) ' start time         =  ',ist
            write(LERR,*) ' end time           =  ',iend
            write(LERR,*) ' time of inverse spike =  ', isp
            write(LERR,*) ' length of design windw  = ',nfilt
            write(LERR,*) ' length of desired output= ',ldsr
            write(LERR,*) ' prewhitening       =  ', prew
            write(LERR,*) ' input data set name =  ', ntap
            write(LERR,*) ' output data set name=  ', otap
            if (maxmum)
     1      write(LERR,*) ' put inverse spike at abs max of trace'
            if (bart) then
            write(LERR,*) ' use bartlett weighting of autocorrelation'
            elseif (cost) then
            write(LERR,*) ' use cosine weighting of autocorrelation'
            else
            write(LERR,*) ' use no weighting of autocorrelation'
            endif
            if (wcenter)
     1      write(LERR,*) ' center window on spike position'
            write(LERR,*)' '
            write(LERR,*)' '
 
      return
      end
 
