C***********************************************************************
C                 copyright 2001, Amoco Production Company             *
C                             All Rights Reserved                      *
C                     an affiliate of BP America Inc.                  *
C***********************************************************************
c     program module maxamp
c
c**********************************************************************c
c
c maxamp reads seismic trace data from an input file,
c computes the amplitude (max, rms, etc) curve, and
c writes the results to an output file
c
c
c**********************************************************************c
c
c     declare variables
c
#include <f77/iounit.h>
#include <f77/lhdrsz.h>
#include <f77/sisdef.h>
#include <save_defs.h>

      parameter   (mmax = 100)

      integer     itr ( 2*SZLNHD )
      integer     lhed( 2*SZLNHD )
      real        head( 2*SZLNHD )
      integer     nsamp, nsi, ntrc, nrec, iform, nwind
      integer     luin , luout, lbytes, nbytes, lbyout,lbyte
      integer     irs,ire,ns,ne,  obytes
      integer     recnum, trcnum, static
      real        tri ( 2*SZLNHD ), gain ( 2*SZLNHD )
      character   ntap * 256, otap * 256, name*6
      logical     verbos, query, med, rms, max, ave, roll, sgn
      integer     argis

      equivalence ( itr(  1), lhed(1), head(1) )
      data lbytes / 0 /, nbytes / 0 /, name/'MAXAMP'/

c-----
c     read program parameters from command line card image file
c-----
      query = ( argis ( '-?' ) .gt. 0 )
      if ( query )then
            call help()
            stop
      endif

c-----
c     open printout files
c-----
#include <f77/open.h>

      call gcmdln(ntap,otap,ns,ne,irs,ire,iwind,iwindi,verbos,
     1            med, rms, max, ave, roll, sgn)

c-----
c     get logical unit numbers for input and output
c-----
      call getln(luin , ntap,'r', 0)
      call getln(luout, otap,'w', 1)

c-----
c     read line header of input
c     save certain parameters
c-----
      call rtape   ( luin, itr, lbyte)
      if(lbyte .eq. 0) then
         write(LERR,*)'maxamp: no header read from unit ',luin
         write(LERR,*)'FATAL'
         stop
      endif

c------
c     save certain parameters

      call saver(itr, 'NumSmp', nsamp, LINHED)
      call saver(itr, 'SmpInt', nsi  , LINHED)
      call saver(itr, 'NumTrc', ntrc , LINHED)
      call saver(itr, 'NumRec', nrec , LINHED)
      call saver(itr, 'Format', iform, LINHED)
      if (unitsc .eq. 0.0) then
          write(LERR,*)'********************************************'
          write(LERR,*)'WARNING: sample unit scaler in LH = ',unitsc
          write(LERR,*)'         will set to .001 (millisec default)'
          write(LERR,*)'********************************************'
          unitsc = .001
          call savew(itr, 'UnitSc', unitsc, LINHED)
      endif

c     For trace header values we take mnemonics and build a
c     set of pointers to an I*2 array equivalenced to the
c     RTAPE  Integer array (headers + data)
c     TRACEHEADER is a value in the include file <sisdef.h> that
c     refers to the trace header
 
      call savelu('TrcNum',ifmt_TrcNum,l_TrcNum,ln_TrcNum,TRACEHEADER)
      call savelu('RecNum',ifmt_RecNum,l_RecNum,ln_RecNum,TRACEHEADER)
      call savelu('SrcLoc',ifmt_SrcLoc,l_SrcLoc,ln_SrcLoc,TRACEHEADER)
      call savelu('RecInd',ifmt_RecInd,l_RecInd,ln_RecInd,TRACEHEADER)
      call savelu('DphInd',ifmt_DphInd,l_DphInd,ln_DphInd,TRACEHEADER)
      call savelu('DstSgn',ifmt_DstSgn,l_DstSgn,ln_DstSgn,TRACEHEADER)
      call savelu('DstUsg',ifmt_DstUsg,l_DstUsg,ln_DstUsg,TRACEHEADER)
      call savelu('StaCor',ifmt_StaCor,l_StaCor,ln_StaCor,TRACEHEADER)


c-----
c     ensure that command line values are compatible with data set
c-----
      call cmdchk(ns,ne,irs,ire,ntrc,nrec)
c-----
c     modify line header to reflect actual number of traces output
c-----
      call hlhprt (itr, lbyte, name, 4, LERR)
      nrecc = ire - irs + 1
      call savew(itr, 'NumRec', nrecc, LINHED)
      jtr   = ne - ns + 1
      call savew(itr, 'NumTrc', jtr  , LINHED)

      dt = real (nsi) / unitsc
      si = nsi
      lslide = iwind  / si
      if ( roll ) then
         iovlp  = 1
         iovlp1 = 1
      else
         iovlp  = iwindi / si
         iovlp1 = iovlp
         iovlp  = lslide - iovlp
      endif

          ilast = 0
          do while (ilast.lt.nsamp)

             if(ilast.eq.0)then
                 ifirst = 1
                 ilast = lslide
                 nwin = 1
             else
                 nmove = lslide
                 ifirst = ifirst + iovlp
                 ilast = ifirst + nmove -1
                 nwin = nwin+1
             endif
         end do
         nwin0 = nwin

      nsampo = nwin
      obytes = (nwin + ITRWRD) * SZSMPD
      call savew(itr, 'NumSmp', nwin , LINHED)

      call savhlh(itr,lbyte,lbyout)
      call wrtape ( luout, itr, lbyout                 )

c-----
c     verbose output of all pertinent information before
c     processing begins
c-----
c     if( verbos ) then
            call verbal(nsamp, nsi, ntrc, nrec, iform,
     1                  nwind,ntap,otap,med,lslide,iovlp1,nwin,
     2                  rms,ave,max,roll,sgn)
c     endif
c-----
c     BEGIN PROCESSING
c     read trace, do agc, write to output file
c-----
c-----
c     skip unwanted records
c-----
      call recskp(1,irs-1,luin,ntrc,itr)
c-----
c     process desired trace records
c-----
      ir = 0
      DO  1000  JJ = irs, ire

            ir = ir + 1
c----------------
c  skip to start
c  of record
            call trcskp(jj,1,ns-1,luin,ntrc,itr)
c----------------

            do  1001  KK = ns, ne

                  nbytes = 0
                  call rtape ( luin, itr, nbytes)
                  if(nbytes .eq. 0) then
                     write(LERR,*)'End of file on input:'
                     write(LERR,*)'  rec= ',jj,'  trace= ',kk
                     go to 999
                  endif
                  call vmov (lhed(ITHWP1), 1, tri, 1, nsamp)

                  call saver2(lhed,ifmt_RecNum,l_RecNum, ln_RecNum,
     1                        recnum , TRACEHEADER)
                  call saver2(lhed,ifmt_TrcNum,l_TrcNum, ln_TrcNum,
     1                        trcnum , TRACEHEADER)
                  call saver2(lhed,ifmt_StaCor,l_StaCor, ln_StaCor,
     1                        static , TRACEHEADER)

                  call vfill (1.0, gain, 1, 2*nsamp)

                  IF(static .ne. 30000)then

                     call getmax (tri, gain, lslide,nwin,iovlp,nsamp,
     1                            med, rms, max, ave, sgn)

                  ELSE

                        call vclr(gain,1,nwin)

                  ENDIF

                  call vmov ( gain, 1, lhed(ITHWP1), 1, nwin)
                  call wrtape( luout, itr, obytes)

 1001             continue

c----------------
c  skip to end of
c  current record
            call trcskp(jj,ne+1,ntrc,luin,ntrc,itr)
c----------------
            if(verbos) write(LERR,*)'maxamp:  ri ',recnum

 1000       CONTINUE
c-----
c     close data files
c-----
  999 continue
      call lbclos ( luin )
      call lbclos ( luout )
      write(LERR,*)'end of maxamp, processed',nrec,' record(s)',
     :               ' with ',ntrc, ' traces'
      write(LER ,*)'end of maxamp, processed',nrec,' record(s)',
     :               ' with ',ntrc, ' traces'
      end

      subroutine help
#include <f77/iounit.h>
         write(LER,*)
     :'***************************************************************'
      write(LER,*)
     :'execute maxamp by typing maxamp and a list of program parameters'
      write(LER,*)
     :'note that each parameter is proceeded by -a where "a" is '
      write(LER,*)
     :'a character(s) corresponding to some parameter.'
        write(LER,*)
     :'users enter the following parameters, or use the default values'
      write(LER,*)' '
        write(LER,*)
     :' -N [ntap]    (no default)       : input data file name'
        write(LER,*)
     :' -O [otap]    (no default)       : output data file name'
      write(LER,*)' '
        write(LER,*)
     :' -ns[ns]      (default = first)  : start trace number'
        write(LER,*)
     :' -ne[ne]      (default = last)   : end trace number'
        write(LER,*)
     :' -rs[irs]      (default = first) : start record number'
        write(LER,*)
     :' -ne[ire]      (default = last)  : end record number'
        write(LER,*)
     :' -w [window]  (default = 500ms)  : analysis window'
        write(LER,*)
     :' -o [ovlp]    (default = 250ms)  : analysis window overlap'
      write(LER,*)' '
        write(LER,*)
     :' -max                            : use maximum in sliding window'
        write(LER,*)
     :' -ave                            : use average abs amp in window'
        write(LER,*)
     :' -rms                            : use average rms amp in window'
        write(LER,*)
     :' -med                            : use median amp in window'
        write(LER,*)
     :' -roll                           : continuous sliding window'
        write(LER,*)
     :' -sgn                            : preserve sign'
        write(LER,*)
     :' -V                              : verbose printout'
      write(LER,*)' '
         write(LER,*)
     :'usage:   maxamp -N[] -O[] -w[] -o[] -ns[] -ne[] -rs[] -re[]'
         write(LER,*)
     :'                [ -max -ave -rms -med -roll -sgn -V]'
         write(LER,*)
     :'***************************************************************'
      return
      end

      subroutine gcmdln(ntap,otap,ns,ne,irs,ire,iwind,iwindi,verbos,
     1            med, rms, max, ave, roll, sgn)
c-----
c     get command arguments
c
c     ntap  - c*100     input file name
c     otap  - c*100     output file name
c     ns    - i*4 starting trace index
c     ne    - i*4 ending trace index
c     irs   - i*4 starting record index
c     ire   - i*4 ending record index
c     nwind - i*4 agc window in ms
c     med         - L   use median gain calculation
c     rms         - L   rms curve, else average curve
c     verbos      - L   verbose output or not
c-----
#include <f77/iounit.h>
      character   ntap*(*), otap*(*)
      integer     ns, ne, irs, ire, iwind, iwindi
      logical     verbos, med, rms, max, ave, roll, sgn
      integer     argis

      max  = .false.
      med  = .false.
      rms  = .false.
      ave  = .false.
      sgn  = .false.
      roll = .false.

            call argstr( '-N', ntap, ' ', ' ' )
            call argstr( '-O', otap, ' ', ' ' )
            call argi4 ( '-ns', ns ,   0  ,  0    )
            call argi4 ( '-ne', ne ,   0  ,  0    )
            call argi4 ( '-rs', irs ,   0  ,  0    )
            call argi4 ( '-re', ire ,   0  ,  0    )
            call argi4 ( '-o', iwindi ,   250,  250 )
            call argi4 ( '-w', iwind  ,   500,  500 )
            med    = ( argis( '-med' ) .gt. 0 )
            max    = ( argis( '-max' ) .gt. 0 )
            rms    = ( argis( '-rms' ) .gt. 0 )
            ave    = ( argis( '-ave' ) .gt. 0 )
            sgn    = ( argis( '-sgn' ) .gt. 0 )
            roll   = ( argis( '-roll' ) .gt. 0 )
            verbos = ( argis( '-V' ) .gt. 0 )

       if (.not.med .AND. .not.max .AND. .not.rms .AND. .not.ave) then
          write(LERR,*)' '
          write(LERR,*)'FATAL ERROR in maxamp commend line:'
          write(LERR,*)'must enter one of  -max -ave -rms -med'
          write(LER ,*)' '
          write(LER ,*)'FATAL ERROR in maxamp commend line:'
          write(LER ,*)'must enter one of  -max -ave -rms -med'
          stop
       endif

      return
      end

      subroutine verbal(nsamp, nsi, ntrc, nrec, iform,
     1                  nwind,ntap,otap,med,lslide,iovlp,nwin,
     2                  rms,ave,max,roll,sgn)
c-----
c     verbose output of processing parameters
c
c     nsamp - I*4 number of samples in trace
c     nsi   - I*4 sample interval in ms
c     ntrc  - I*4 traces per record
c     nrec  - I*4 number of records per line
c     iform - I*4 format of data
c     lw    - I*4 agc filter length
c     scal1 - I*4 percentage scaling factor
c     amp   - R*4 default peak amplitude
c     poly    i*4 fit nth order polynomial to gain curve
c     itype   i*4 type of function to fit
c     idec    i*4 for median calculation sample window every idec samples
c     nwind - I*4 window length in milliseconds
c     ntap  - C*100     input file name
c     otap  - C*100     output file name
c     agc         - L   output agc curves only
c     rms         - L   rms curve, else average curve
c     med         - L   use median gain calculation
c-----
#include <f77/iounit.h>
      integer     nsamp, nsi, ntrc, nrec, iform, nwind, lslide, iovlp
      integer     nwin
      integer     lenth, length
      logical     med, rms, max, ave, roll, sgn
      character   ntap*(*), otap*(*)

            write(LERR,*)' '
            write(LERR,*)' line header values after default check '
            write(LERR,*) ' # of samples/trace =  ', nsamp
            write(LERR,*) ' sample interval    =  ', nsi
            write(LERR,*) ' traces per record  =  ', ntrc
            write(LERR,*) ' records per line   =  ', nrec
            write(LERR,*) ' format of data     =  ', iform
            write(LERR,*) ' window length        =  ',lslide,' samps'
            write(LERR,*) ' window overlap       =  ',iovlp,' samps'
            write(LERR,*) ' number windows       =  ',nwin
            write(LERR,*) ' window length in ms  = ',nwind
            if (roll)
     1      write(LERR,*) ' roll option'
            if (sgn)
     1      write(LERR,*) ' preserve sign'
            length = lenth(ntap)
      	    if (length .gt. 0) then
              write(LERR,*) ' input data set name  = ',
     :			ntap(1:lenth(ntap))
            else
              write(LERR,*) ' input data set name  = stdin'
            endif
            length = lenth(ntap)
      	    if (length .gt. 0) then
              write(LERR,*) ' output data set name = ',
     :			otap(1:lenth(otap))
            else
              write(LERR,*) ' output data set      = stdout'
            endif
            if (rms) then
                write(LERR,*)'Compute rms amplitude'
            elseif (med) then
                write(LERR,*)'Compute median amplitude'
            elseif (ave) then
                write(LERR,*)'Compute average absolute amplitude'
            elseif (max) then
                write(LERR,*)'Compute maximum absolute amplitude'
            endif
            write(LERR,*)' '
            write(LERR,*)' '

      return
      end

