C***********************************************************************
C                 copyright 2001, Amoco Production Company             *
C                             All Rights Reserved                      *
C                     an affiliate of BP America Inc.                  *
C***********************************************************************
c**********************************************************************c

c Changes:
c
c     Mar 30 2000: Added option to specify input and output offsets through
c                  header mnemonic - Garossino
c
c
c pnmo reads seismic trace data from an input file,
c and applies (1) a primary NMO correction in the forward direction
c followed by (2) a constant offset correction in the reverse direction
c then writes the results to an output file
c
c
c**********************************************************************c
c
c     declare variables
c

      implicit none

#include <save_defs.h>
#include <f77/iounit.h>
#include <f77/lhdrsz.h>
#include <f77/sisdef.h>

c declare std usp variables

      integer nsamp, nsi, ntrc, nrec, iform,obytes
      integer luin , luout, lbytes, nbytes, lbyout
      integer argis, pipe, jerr
      integer JJ, KK
 
      integer     itr ( 2*SZLNHD )
      integer     vtr ( 2*SZLNHD )

      character   ntap * 256, otap * 256, vtap * 256, name*4

      logical verbos

c declare variables used in dynamic memory allocation

      integer errcd, abort, item

      real        tri, xtr, v_trace, work, t, rnum

      pointer     (wkwork   , work   (2))
      pointer     (wktri    , tri    (2))
      pointer     (wkxtr    , xtr    (2))
      pointer     (wkv_trace, v_trace(2))
      pointer     (wkt      , t      (2))
      pointer     (wkrnum   , rnum   (2))

      logical heap

c declare local variables

      integer luvel, lvbytes, nsampv, nsiv, ntrcv, nrecv, iformv
      integer nvbytes, i, vflag, lenth, length

      integer ifmt_RecNum, l_RecNum, ln_RecNum, RecNum
      integer ifmt_DstSgn, l_DstSgn, ln_DstSgn, DstSgn
      integer ifmt_StaCor, l_StaCor, ln_StaCor, StaCor
      integer ifmt_HdrIn, l_HdrIn, ln_HdrIn, HdrIn
      integer ifmt_HdrOut, l_HdrOut, ln_HdrOut, HdrOut

      real sii, offset, xoff, dt, UnitSc, UnitScv, scl, vdot, dtv
      real yp1, ypn

      character c_HdrIn*6, c_HdrOut*6

      logical remove
 
c initialize variables

      data  lbytes /0/
      data  nbytes /0/
      data  abort /0/
      data  name /'PNMO'/
      DATA  pipe/3/
      data  vflag/1/

      yp1 = 1.0e+31
      ypn = 1.0e+31

c get command line help if requested

      if ( argis ( '-?' ) .gt. 0 .or. 
     :     argis ( '-h' ) .gt. 0 .or. 
     :     argis ( '-help' ) .gt. 0 ) then
         call help()
         stop
      endif
 
c open printout files

#include <f77/open.h>

c parse command line
 
      call gcmdln ( ntap, otap, vtap, scl, xoff, remove, c_HdrIn, 
     :     c_HdrOut, verbos )

c get logical unit numbers for input and output

      call getln ( luin , ntap, 'r', 0 )
      call getln ( luout, otap, 'w', 1 )

      if (vtap .ne. ' ') then
          call getln(luvel, vtap, 'r',-1)
      else
          write(LERR,*)'anmo assumed to be running inside IKP'
          call sisfdfit (luvel, pipe)
      endif

      if(luvel .lt. 0)   then
         write(lerr,*)'pnmo error: velocity file -v not accessible'
      endif

c read line header of input
c save certain parameters

      lbytes=0
      call rtape ( luin, itr, lbytes )
      if(lbytes .eq. 0) then
         length = lenth(ntap)
         if (length .gt. 0) then
           write(LERR,*)'pnmo: no line header read from ',
     :		ntap(1:length)
         else
           write(LERR,*)'pnmo: no line header read from stdin'
         endif
         write(LERR,*)'FATAL'
         write(LER,*)' '
         write(LER,*)'PNMO: '
         if (length .gt. 0) then
           write(LER,*)'pnmo: no line header read from ',
     :		ntap(1:length)
         else
           write(LER,*)'pnmo: no line header read from stdin'
         endif
         write(LER,*)'FATAL'
         stop
      endif

      call saver(itr, 'NumSmp', nsamp, LINHED)
      call saver(itr, 'SmpInt', nsi  , LINHED)
      call saver(itr, 'NumTrc', ntrc , LINHED)
      call saver(itr, 'NumRec', nrec , LINHED)
      call saver(itr, 'Format', iform, LINHED)
      call saver(itr, 'UnitSc', UnitSc, LINHED)

      if ( UnitSc .eq. 0.0 ) then
         UnitSc = 0.001
         call savew(itr,'UnitSc',UnitSc,LINHED)
      endif

      sii = nsi
      dt = float(nsi) * UnitSc

c declare pointers to trace header mnemonics
 
      call savelu('RecNum',ifmt_RecNum,l_RecNum,ln_RecNum,TRACEHEADER)
      call savelu('DstSgn',ifmt_DstSgn,l_DstSgn,ln_DstSgn,TRACEHEADER)
      call savelu('StaCor',ifmt_StaCor,l_StaCor,ln_StaCor,TRACEHEADER)
      call savelu(c_HdrIn,ifmt_HdrIn,l_HdrIn,ln_HdrIn,TRACEHEADER)
      call savelu(c_HdrOut,ifmt_HdrOut,l_HdrOut,ln_HdrOut,TRACEHEADER)

c read in velocity worktape line header.

      lvbytes = 0
      call rtape ( luvel, vtr, lvbytes )
      if(lvbytes .eq. 0) then
         length = lenth(vtap)
         if (length .gt. 0) then
           write(LERR,*)'pnmo: no line header read from ',
     :		vtap(1:length)
         else
           write(LERR,*)'pnmo: no line header read from velocity ',
     :		'model pipe'
         endif
         write(LERR,*)'FATAL'
         write(LER,*)' '
         write(LER,*)'PNMO: '
         if (length .gt. 0) then
           write(LER,*)'pnmo: no line header read from ',
     :		vtap(1:length)
         else
           write(LER,*)'pnmo: no line header read from velocity ',
     :		'model pipe'
         endif
         write(LER,*)'FATAL'
         stop
      endif
 
      call saver(vtr, 'NumSmp', nsampv, LINHED)
      call saver(vtr, 'SmpInt', nsiv  , LINHED)
      call saver(vtr, 'NumTrc', ntrcv , LINHED)
      call saver(vtr, 'NumRec', nrecv , LINHED)
      call saver(vtr, 'Format', iformv, LINHED)
      call saver(vtr, 'UnitSc', UnitScv, LINHED)

      if ( UnitScv .eq. 0.0 ) then
         UnitScv = 0.001
      endif

      dtv = float(nsiv) * UnitScv

c     vflag=0 single velocity function
c     vflag=1 multiple velocity function

      if (nrecv .le. 1) vflag=0

      if (dt .ne. dtv ) then
        write(LERR,*)'FATAL ERROR from pnmo:'
        write(LERR,*)'Sample interval of input data and input velocity'
        write(LERR,*)'traces (',dt,dtv,') are not same.'
        write(LER,*)' '
        write(LER,*)'PNMO:'
        write(LER,*)' Sample interval of input data and input velocity'
        write(LER,*)' traces (',dt,dtv,') are not same.'
        write(LER,*)'FATAL'
        stop
      endif

c print historical line header to printout file

      call hlhprt (itr, lbytes, name, 4, LERR)

      call verbal( nsamp, nsi, ntrc, nrec, iform,
     1     ntap, otap, vtap, scl, xoff, remove,
     2     nsampv, nsiv, ntrcv, nrecv, c_HdrIn, c_HdrOut)

      if (nsampv .lt. nsamp) then
          write(lerr,*)'velocity trace(s) too short; contains ',nsampv,
     1                  ' samples'
          write(lerr,*)'orig input data contains ',nsamp,' samples'
          write(lerr,*)'rerun velocity create job so that trc at least',
     1                  ' ',nsi*nsamp,' ms long'
          write(lerr,*)' '
          write(lerr,*)'PNMO: '
          write(lerr,*)' velocity trace(s) too short; contains ',nsampv,
     1         ' samples'
          write(lerr,*)' orig input data contains ',nsamp,' samples'
          write(lerr,*)' rerun velocity create job so that trc at least'
     1         ,' ',nsi*nsamp,' ms long'
          write(lerr,*)'FATAL'
         stop
      endif

c dynamic memory allocation

      heap = .true.
      item = max (nsamp,nsampv)

      call galloc (wkv_trace, item * SZSMPD, errcd, abort)
      if (errcd .ne. 0)  heap = .false.
      call galloc (wkwork   , item * SZSMPD, errcd, abort)
      if (errcd .ne. 0)  heap = .false.
      call galloc (wktri    , item * SZSMPD, errcd, abort)
      if (errcd .ne. 0)  heap = .false.
      call galloc (wkxtr    , item * SZSMPD, errcd, abort)
      if (errcd .ne. 0)  heap = .false.
      call galloc (wkt      , item * SZSMPD, errcd, abort)
      if (errcd .ne. 0)  heap = .false.
      call galloc (wkrnum   , item * SZSMPD, errcd, abort)
      if (errcd .ne. 0)  heap = .false.

      if (.not. heap) then
         write(LERR,*)' '
         write(LERR,*)'Unable to allocate workspace:'
         write(LERR,*) 6*item * SZSMPD,'  bytes'
         write(LER ,*)' '
         write(LER ,*)'FATAL ERROR pnmo: Unable to allocate workspace:'
         write(LER ,*) 6*item * SZSMPD,'  bytes'
         go to 999
      else
         write(LERR,*)' '
         write(LERR,*)'Allocating workspace:'
         write(LERR,*) 6*item * SZSMPD,'  bytes'
      endif

c initialize memory

      call vclr ( tri, 1, item )
      call vclr ( work, 1, item )
      call vclr ( tri, 1, item )
      call vclr ( xtr, 1, item )
      call vclr ( t, 1, item )
      call vclr ( rnum, 1, item )

c update historical line header

      call savhlh ( itr, lbytes, lbyout )

c write output lineheader

      call wrtape ( luout, itr, lbyout )

c calculate number of output bytes per trace

      obytes = SZTRHD + nsamp * SZSMPD

c BEGIN PROCESSING

c
c ----- single velocity function -----
c
 
      IF ( vflag .eq. 0 ) THEN
 
         nvbytes=0
         call rtape(luvel,vtr,nvbytes)
         if(nvbytes .eq. 0) then
            write(LERR,*)' '
            write(LERR,*)'pnmo: No velocity data found after '
            write(LERR,*)'       lineheader in ',vtap,' check '
            write(LERR,*)'       velin run thoroughly.'
            write(LERR,*)'FATAL'
            write(LER,*)' '
            write(LER,*)'pnmo: No velocity data found after '
            write(LER,*)'       lineheader in ',vtap,' check '
            write(LER,*)'       velin run thoroughly.'
            write(LER,*)'FATAL'
            goto 999
         endif
         call vsmul (vtr(ITHWP1), 1, scl, v_trace, 1, nsampv)
 
         if(verbos) then
            write(LERR,*)'velocity'
            write(LERR,'(10f8.2)')(v_trace(i),i=1,nsampv)
         endif

c
c ----- check for valid velocities -----
c
 
         call dotpr (v_trace, 1, v_trace, 1, vdot, nsampv)

         if (abs(vdot) .lt. 1.e-06) then
            write(LERR,*)' '
            write(LERR,*)'pnmo: velocity trace contains zeros'
            write(LERR,*)'FATAL '
            write(LER,*)' '
            write(LER,*)'pnmo: velocity trace contains zeros'
            write(LER,*)'FATAL '
            go to 999
         endif

      ENDIF
      
      call vrand(911,rnum,1,nsamp)

      do i=1,nsamp
         rnum(i)=rnum(i)*1.0e-21
         t(i) = float(i)
      enddo

c process all record and traces input

      DO  JJ = 1, nrec

c
c ----- if not single velocity function then read velocity data -----
c

         if (vflag .eq. 1) then
            nvbytes = 0
            call rtape (luvel, vtr, nvbytes)
            if (nvbytes .eq. 0) then
               write(LERR,*)' '
               write(LERR,*)'pnmo: Premature EOF on velocity Dataset'
               write(LERR,*)'       at sequential record number ',JJ
               write(LERR,*)'FATAL'
               write(LER,*)' '
               write(LER,*)'pnmo: Premature EOF on velocity Dataset'
               write(LER,*)'       at sequential record number ',JJ
               write(LER,*)'FATAL'
               goto 999
            endif

            call vsmul (vtr(ITHWP1), 1, scl, v_trace, 1, nsampv)

c
c ----- check for valid velocity data -----
c

            call dotpr (v_trace, 1, v_trace, 1, vdot, nsampv)
 
            if (abs(vdot) .lt. 1.e-06) then
               write(LERR,*)' '
               write(LERR,*)'pnmo: velocity trace contains zeros'
               write(LERR,*)'       Check record ',JJ
               write(LERR,*)'FATAL '
               write(LER,*)' '
               write(LER,*)'pnmo: velocity trace contains zeros'
               write(LER,*)'       Check record ',JJ
               write(LER,*)'FATAL '
               go to 999
            endif
 
         endif

         DO  KK = 1, ntrc

c read trace

            nbytes = 0
            call rtape( luin, itr, nbytes)
            if(nbytes .eq. 0) then
               write(LERR,*)'End of file on input:'
               write(LERR,*)'  rec= ',JJ,'  trace= ',KK
               go to 999
            endif

            call saver2(itr,ifmt_StaCor,l_StaCor, ln_StaCor,
     1           StaCor, TRACEHEADER)
            call saver2(itr,ifmt_RecNum,l_RecNum, ln_RecNum,
     1           recnum, TRACEHEADER)


c either get the DstSgn offset or the offset stipulated by the user

            if ( c_HdrIn .ne. ' ' ) then
               call saver2(itr,ifmt_HdrIn,l_HdrIn, ln_HdrIn,
     1           HdrIn, TRACEHEADER)
               offset = abs(HdrIn)
            else
               call saver2(itr,ifmt_DstSgn,l_DstSgn, ln_DstSgn,
     1              DstSgn, TRACEHEADER)
               offset = abs (DstSgn)
            endif

            if ( c_HdrOut .ne. ' ' ) then
               call saver2(itr,ifmt_HdrOut,l_HdrOut, ln_HdrOut,
     1           HdrOut, TRACEHEADER)
               xoff = abs ( HdrOut )
            endif

c process only live traces

            IF ( StaCor .ne. 30000) then

                call vmov (itr(ITHWP1), 1, tri, 1, nsamp)

                if (.not. remove) then

                   call nmo (tri, v_trace, offset, nsamp, dt, xtr,
     1                  work, .false.)
                   call nmo (xtr, v_trace, xoff  , nsamp, dt, tri,
     1                  work, .true.)

                else

                   call nmo (tri, v_trace, xoff  , nsamp, dt, xtr,
     1                  work, .false.)
                   call nmo (xtr, v_trace, offset, nsamp, dt, tri,
     1                  work, .true.)
                endif
                
                call vmov (tri, 1, itr(ITHWP1), 1, nsamp)
               
            ELSE

c clear all traces marked as dead

               call vclr (itr(ITHWP1), 1, nsamp)

            ENDIF

c  write output data

            call wrtape (luout, itr, obytes)

         ENDDO

         if ( verbos) write(LERR,*)'processed rec ',recnum
 
      ENDDO

c Normal Termination: close data files
      
      call lbclos ( luin )
      call lbclos ( luout )

      write(LERR,*)'end of pnmo, processed',nrec,' record(s)',
     :     ' with ',ntrc, ' traces'
      write(LERR,*)' '
      write(LERR,*)'Normal Termination'
      write(LER,*)' '
      write(LER,*)'pnmo: Normal Termination'

      stop

  999 continue

      call lbclos ( luin )
      call lbclos ( luout )

      write(LERR,*)'end of pnmo, processed',JJ,' record(s)',
     :     ' with ',ntrc, ' traces'
      write(LERR,*)' '
      write(LERR,*)'Abnormal Termination'
      write(LER,*)' '
      write(LER,*)'pnmo: Abnormal Termination'
      stop
      end
 
C SUBROUTINES help, gcmdln, verbos

      subroutine help

#include <f77/iounit.h>

      write(LER,*)' '
      write(LER,*)'===================================================='
      write(LER,*)' '
      write(LER,*)' Command Line Arguments for pnmo: partial NMO'
      write(LER,*)'  '
      write(LER,*)' For a more detailed description of these parameters'
      write(LER,*)' see the online man page using uman, xman or xuspman'
      write(LER,*)' '
      write(LER,*)'Input......................................... (def)'
      write(LER,*)' '
      write(LER,*)'-N[]   -- input data set                     (stdin)'
      write(LER,*)'-O[]   -- output data set                   (stdout)'
      write(LER,*)'-v[]   -- velocity data set                   (none)'
      write(LER,*)'-x[]   -- partial NMO offset                     (0)'
      write(LER,*)'-s[]   -- global velocity scale factor           (1)'
      write(LER,*)'-hw1[] -- trace header mnemonic - input   (not used)'
      write(LER,*)'          trace offset'
      write(LER,*)'-hw2[] -- trace header mnemonic - output  (not used)'
      write(LER,*)'          trace offset'
      write(LER,*)'-s[]   -- global velocity scale factor           (1)'
      write(LER,*) ' '
      write(LER,*)
     :' -R  include on command line to remove moveout'
      write(LER,*)
     :' -V  include on command line if verbose printout is desired'
      write(LER,*) ' '
      write(LER,*)
     :'usage:   pnmo -N[] -O[] -v[] -x[] -hw1[] -hw2[] -s[] -R -V'
      write(LER,*)' '
      write(LER,*)'===================================================='
      
      return
      end

c subroutine to parse command line
 
      subroutine gcmdln ( ntap, otap, vtap, scl, xoff, remove, c_HdrIn, 
     :     c_HdrOut, verbos )

c-----
c     get command arguments
c
c     ntap  - C*256     input file name
c     otap  - C*256     output file name
c     sii   - R*4       input sample interval override
c     sio   - R*4       output sample interval
c     nso   - I*4       output sample interval header override
c     micro -  L        convert input s.i. to micro secs
c     verbos -  L       verbose output or not
c     cube - L          use cubic spline interpolation
c     sinc - L          use sinc interpolation
c     nsinc -I*4        number of points in sinc interpolator
c     four - L          use Fourier interpolation
c     cubef - L         use cubic spline interpolation with post interpolation filter
c     linear - L        use linear interpolation
c
c-----

#include <f77/iounit.h>

c variables passed from calling routine

      character   ntap*(*), otap*(*), vtap*(*), c_HdrIn*6, c_HdrOut*6
      real        scl, xoff
      logical     remove, verbos

c local variables

      integer argis
 
      call argstr( '-hw1', c_HdrIn, ' ', ' ' )
      call argstr( '-hw2', c_HdrOut, ' ', ' ' )

      call argstr( '-N', ntap, ' ', ' ' )

      call argstr( '-O', otap, ' ', ' ' )

      remove = (argis('-R') .gt. 0)

      call argr4 ('-s', scl, 1.0, 1.0)

      verbos = (argis('-V') .gt. 0)
      call argstr( '-v', vtap, ' ', ' ' )

      call argr4 ('-x', xoff, 0.0, 0.0)
      xoff = abs (xoff)
 
c check for extraneous arguments and abort if found

      call xtrarg ( name, LER, .FALSE., .FALSE. )
      call xtrarg ( name, LERR, .FALSE., .TRUE. )

      return
      end
 
      subroutine verbal( nsamp, nsi, ntrc, nrec, iform,
     1                   ntap, otap, vtap, scl, xoff, remove,
     2                   nsampv, nsiv, ntrcv, nrecv, c_HdrIn, c_HdrOut)

#include <f77/iounit.h>

c declare variables passed from calling routine

      integer     nsamp, nsi, ntrc, nrec, iform
      real        scl, xoff
      character   ntap*(*), otap*(*), vtap*(*)
      character   c_HdrIn*6, c_HdrOut*6
      logical     remove

c local variables

      integer length
 
      write(LERR,*)' '
      length = lenth(ntap)
      if (length .gt. 0) then
        write(LERR,*) ' input data set name     =  ',
     :		ntap(1:length)
      else
        write(LERR,*) ' input data set          =  stdin'
      endif
      length = lenth(vtap)
      if (length .gt. 0) then
        write(LERR,*) ' input velocity set name =  ',
     :		vtap(1:length)
      else
        write(LERR,*) ' input velocity set      =  piped'
      endif
      length = lenth(otap)
      if (length .gt. 0) then
        write(LERR,*) ' output data set name    =  ',
     :		otap(1:length)
      else
        write(LERR,*) ' output data set         =  stdout'
      endif
      write(LERR,*)' '
      write(LERR,*)' line header values after default check '
      write(LERR,*) ' input samples/trace =  ', nsamp
      write(LERR,*) ' input sample interval =  ', nsi
      write(LERR,*) ' traces per record  =  ', ntrc
      write(LERR,*) ' records per line   =  ', nrec
      write(LERR,*) ' format of data     =  ', iform
      write(LERR,*)' '
      write(LERR,*)' velocity line header values after default check '
      write(LERR,*) ' input samples/trace =  ', nsampv
      write(LERR,*) ' input sample interval =  ', nsiv
      write(LERR,*) ' traces per record  =  ', ntrcv
      write(LERR,*) ' records per line   =  ', nrecv

      write(LERR,*)' '
      write(LERR,*)' variable values inside pnmo'

      write(LERR,*) ' partial NMO offset       =  ',xoff
      write(LERR,*) ' global velocity scaler   =  ',scl

      if ( c_HdrIn .ne. ' ' ) then
         write(LERR,*) ' Input Trace Offset from  =  ',c_HdrIn
      endif
      if ( c_HdrOut .ne. ' ' ) then
         write(LERR,*) ' Output Trace Offset from  =  ',c_HdrOut
      endif

      if (remove) then
         write(LERR,*)' Undo NMO correction'
      else
         write(LERR,*)' Apply NMO correction'
      endif

      write(LERR,*)' '
      
      return
      end
