C***********************************************************************
C                 copyright 2002, Amoco Production Company             *
C                             All Rights Reserved                      *
C                     an affiliate of BP America Inc.                  *
C***********************************************************************
c -----------------  Main Routine -----------------------

c     Program Changes:

c      - original written: April, 1997
c
c     April 17, 2002 - rewrote the routine to allow piping both
c                      in and out.  Corrected the RmsAmp calculation
c                      to use the absolute value of the trace
c                      amplitudes in the second pass.  Moved the trace
c                      header entry type check out of the inner loop.
c                      Added dynamic memory allocation to itr[] and 
c                      tri[] arrays, added dynamic arrays to hold trace
c                      header stats if piping so that no rewind of 
c                      output stream is required.  Added a hard zero
c                      check so as not to include hard zeroes in the 
c                      calculation of statistics.  Also added printout
c                      of job constant stats to printout file.  All
c                      requested by Steve Lancaster [Sunbury].
c     Garossino

c     Program Description:

c     program to calculate and put values in header
c
c	James M. Gridle
c	USP Team Tulsa
c	April 1997     

c get machine dependent parameters 

      implicit none

#include <f77/iounit.h>
#include <f77/lhdrsz.h>
#include <f77/sisdef.h>
#include <save_defs.h> 

c dimension standard USP variables 

      integer nsamp, nsi, ntrc, ntrco, nrec, nreco, iform
      integer luin , luout, lbytes, nbytes, lbyout, obytes
      integer ist, iend, irs, ire, ns, ne, argis, jerr

      real UnitSc, dt

      character ntap*255, otap*255, name*6

      logical verbos

c Program Specific _ dynamic memory variables

      integer alloc_size, TraceSize, ArraySize 
      integer errcd1, errcd2, errcd3, errcd4, errcd5, abort
      integer itr

      real tri, abstri
      real avg_array, peak_array, rms_array

      pointer (mem_itr, itr(2))
      pointer (mem_tri, tri(2))
      pointer (mem_abstri, abstri(2))
      pointer (mem_avg_array, avg_array(2))
      pointer (mem_peak_array, peak_array(2))
      pointer (mem_rms_array, rms_array(2))

c Program Specific _ static memory variables

      integer ifmt_StaCor,l_StaCor,ln_StaCor, StaCor
      integer ifmt_HdrWrd1,l_HdrWrd1,ln_HdrWrd1
      integer ifmt_HdrWrd2,l_HdrWrd2,ln_HdrWrd2
      integer ifmt_HdrWrd3,l_HdrWrd3,ln_HdrWrd3
      integer jcount, tcount, JJ, KK, i, icount, kcount

      real job_peak, job_avg, job_rms
      real peak,avg,rms

      character   c_HdrWrd1*6, c_HdrWrd2*6, c_HdrWrd3*6

      logical pipein, pipeout

c Initialize variables

      data abort/0/
      data name/"PUTVAL"/
      data job_avg/0.0/
      data job_peak/0.0/
      data job_rms/0.0/
      data jcount/0/

c give command line help if requested

      if ( argis ( '-?' ) .gt. 0 .or. 
     :     argis ( '-h' ) .gt. 0 .or. 
     :     argis ( '-help' ) .gt. 0 ) then
         call help()
         stop
      endif

c     open printout file

#include <f77/open.h>

c get command line input parameters

      call cmdln ( ntap, otap, ns, ne, irs, ire, ist, iend, 
     :     name, verbos, c_HdrWrd1, c_HdrWrd2, c_HdrWrd3 )

c determine piping status

      pipein = .false.
      pipeout = .false.
      if ( ntap .eq. ' ' ) pipein = .true.
      if ( otap .eq. ' ' ) pipeout = .true.

c if pipeout is true then no line header update will be possible 
c and all global statistics will be printed to the printout file

c if pipein is true then the above is true but in addition the global
c rmsamp cannot be calculated


c allocate memory for line header array itr[]

      alloc_size = SZLNHD
      errcd1 = 0

      call galloc(mem_itr,alloc_size * SZSMPD,errcd1,abort)

      if (errcd1 .ne. 0) then
	write(LERR,*) 'ERROR: Unable to allocate workspace '
	write(LERR,*) '       ',alloc_size * SZSMPD,' bytes requested '
	write(LERR,*) '       FATAL'
	write(LER,*) 'PUTVAL: Unable to allocate workspace '
	write(LER,*) '     ',alloc_size * SZSMPD,' bytes requested '
	write(LER,*) 'FATAL'
	stop
      else
        write(LERR,*)'Allocating workspace:'
	write(LERR,*) alloc_size * SZSMPD,' bytes requested '
      endif

      call vclr ( itr, 1, alloc_size )

c open input and output files

      call getln(luin , ntap,'r', 0)
      call getln(luout, otap,'w', 1)

c  read input line header and save certain parameters

      call rtape(luin,itr,lbytes)
      if(lbytes.eq.0)then
         write(LER,*)'PUTVAL: no line header on input dataset',ntap
         write(LER,*)'FATAL'
         stop
      endif

      call saver(itr, 'NumSmp', nsamp, LINHED)
      call saver(itr, 'SmpInt', nsi  , LINHED)
      call saver(itr, 'NumTrc', ntrc , LINHED)
      call saver(itr, 'NumRec', nrec , LINHED)
      call saver(itr, 'Format', iform, LINHED)
      call saver(itr, 'UnitSc', UnitSc, LINHED)

      if ( UnitSc .eq. 0.0 ) then
         write(LERR,*)'********************************************'
         write(LERR,*)'WARNING: sample unit scaler in LH = ',UnitSc
         write(LERR,*)'         will set to .001 (millisec default)'
         write(LERR,*)'********************************************'
         UnitSc = 0.001
         call savew ( itr, 'UnitSc', UnitSc, LINHED)
      endif

c compute delta T in seconds

      dt = real (nsi) * UnitSc

c define pointers to header words required by your routine

      call savelu('StaCor',ifmt_StaCor,l_StaCor,ln_StaCor,TRACEHEADER)
      call savelu(c_HdrWrd1,ifmt_HdrWrd1,l_HdrWrd1,ln_HdrWrd1,
     :     TRACEHEADER)
      call savelu(c_HdrWrd2,ifmt_HdrWrd2,l_HdrWrd2,ln_HdrWrd2,
     :     TRACEHEADER)
      call savelu(c_HdrWrd3,ifmt_HdrWrd3,l_HdrWrd3,ln_HdrWrd3,
     :     TRACEHEADER)

c check to make sure the trace header entries chosen are floating
c point capable

      if ( ( ifmt_HdrWrd1 .ne. SAVE_FKFLT_DEF ) .or.
     :     ( ifmt_HdrWrd2 .ne. SAVE_FKFLT_DEF ) .or.
     :     ( ifmt_HdrWrd3 .ne. SAVE_FKFLT_DEF ) ) then

         write(LERR,*)'-hw1 = ', c_HdrWrd1
         write(LERR,*)'-hw2 = ', c_HdrWrd2
         write(LERR,*)'-hw3 = ', c_HdrWrd3
         write(LERR,*)' '
         write(LERR,*)'It is required that all header entries '
         write(LERR,*)'used are capable of holding floating point'
         write(LERR,*)'values.  Fix command line and rerun'
         write(LERR,*)'FATAL'
         write(LERR,*)' '
         write(LER,*)'PUTVAL: '
         write(LER,*)'-hw1 = ', c_HdrWrd1
         write(LER,*)'-hw2 = ', c_HdrWrd2
         write(LER,*)'-hw3 = ', c_HdrWrd3
         write(LER,*)' '
         write(LER,*)'It is required that all header entries '
         write(LER,*)'used are capable of holding floating point'
         write(LER,*)'values.  Fix command line and rerun'
         write(LER,*)'FATAL'
         write(LER,*)' '

         go to 999
      endif

c print hlh to printout file 

      call hlhprt (itr, lbytes, name, 6, LERR)

c check user supplied boundary conditions and set defaults

      if ( irs .eq. 0 ) irs = 1
      if ( ire .eq. 0 .or. ire .gt. nrec ) ire = nrec

      if ( ns .eq. 0 ) ns = 1
      if ( ne .eq. 0 .or. ne .gt. ntrc ) ne = ntrc

      ist = nint ( float(ist) / float(nsi) )
      iend = nint ( float(iend) / float(nsi) )
      if ( ist .eq. 0 ) ist = 1
      if ( iend .eq. 0 .or. iend .gt. nsamp ) iend = nsamp

      nreco = ire - irs + 1
      ntrco = ne - ns + 1

c number output bytes

         obytes = SZTRHD + SZSMPD * nsamp 

      if ( pipein ) then

c if we are piping in then there will be no backing up on the input
c datastream and we must write the output line header now, otherwise 
c we can wait until the second pass through the input and output the
c line header with updates for MaxAmp and AvgAmp

c modify line header to reflect actual record configuration output
c NOTE: in this case the sample limits ist and iend are used to 
c       limit processing only.   All data samples are actually passed.

         call savew(itr, 'NumRec', nreco, LINHED)
         call savew(itr, 'NumTrc', ntrco  , LINHED)

c update hlh and write output line header

         call savhlh  ( itr, lbytes, lbyout )
         call wrtape ( luout, itr, lbyout )

      endif

c verbose output of all pertinent information before processing begins

      call verbal( ntap, otap, nsamp, nsi, ntrc, nrec, iform, ist, 
     :     iend, irs, ire, ns, ne, verbos, c_HdrWrd1, c_HdrWrd2, 
     :     c_HdrWrd3)

c reallocate buffer memory to be big enough to include the trace header 
c and trace time series

      alloc_size = SZTRHD + SZSMPD * nsamp

      errcd1 = 0
      call grealloc(mem_itr,alloc_size,errcd1,abort)
      if (errcd1 .ne. 0) then
	write(LERR,*) 'ERROR: Unable to allocate trace workspace '
	write(LERR,*) '       ',alloc_size,' bytes requested '
	write(LERR,*) '       FATAL'
	write(LER,*) 'PUTVAL: Unable to allocate trace workspace '
	write(LER,*) '       ',alloc_size,' bytes requested '
	write(LER,*) 'FATAL'
	stop
      else
        write(LERR,*)'Allocating workspace:'
	write(LERR,*) alloc_size,' bytes requested '
      endif

      call vclr ( itr, 1, ITRWRD + nsamp )

c dynamic memory allocation:  

      alloc_size = SZSMPD * nsamp
      TraceSize = nsamp 
      ArraySize = ntrc * nrec

      call galloc (mem_tri, alloc_size, errcd1, abort)
      call galloc (mem_abstri, alloc_size, errcd2, abort)
      call galloc (mem_avg_array, ArraySize*SZSMPD, errcd3, abort)
      call galloc (mem_peak_array, ArraySize*SZSMPD, errcd4, abort)
      call galloc (mem_rms_array, ArraySize*SZSMPD, errcd5, abort)
    
      if ( errcd1 .ne. 0 .or. 
     :     errcd2 .ne. 0 .or. 
     :     errcd3 .ne. 0 .or. 
     :     errcd4 .ne. 0 .or. 
     :     errcd5 .ne. 0 )then
         write(LERR,*)' '
         write(LERR,*)'Unable to allocate workspace:'
         write(LERR,*) 2* TraceSize * SZSMPD, '  bytes'
         write(LERR,*) 3* ArraySize * SZSMPD, '  bytes'
         write(LERR,*)' '
         write(LER,*)' '
         write(LER,*)'PUTVAL: Unable to allocate workspace:'
         write(LER,*) 2* TraceSize * SZSMPD, '  bytes'
         write(LER,*) 3* ArraySize * SZSMPD, '  bytes'
         write(LER,*)'FATAL '
         go to 999
      else
         write(LERR,*)' '
         write(LERR,*)'Allocating workspace:'
         write(LERR,*) 2 * TraceSize * SZSMPD, '  bytes'
         write(LERR,*) 3 * ArraySize * SZSMPD, '  bytes'
         write(LERR,*)' '
      endif

c initialize memory
         
      call vclr ( tri, 1, TraceSize )
      call vclr ( abstri, 1, TraceSize )
      call vclr ( avg_array, 1, ArraySize )
      call vclr ( peak_array, 1, ArraySize )
      call vclr ( rms_array, 1, ArraySize )

c BEGIN PROCESSING 

c set array counter

      tcount = 0

c skip unwanted input records

      call recskp ( 1, irs-1, luin, ntrc, itr )

      DO JJ = irs, ire
 
c skip to start trace

         call trcskp ( JJ, 1, ns-1, luin, ntrc, itr )

         DO KK = ns, ne

            nbytes = 0
            call rtape( luin, itr, nbytes)

c if end of data encountered (nbytes=0) then bail out

            if(nbytes .eq. 0) then
               write(LERR,*)'Premature EOF on input at:'
               write(LERR,*)'  rec= ',JJ,'  trace= ',KK
               go to 999
            endif

c get required trace header information

            call saver2( itr, ifmt_StaCor, l_StaCor, ln_StaCor, 
     :           StaCor, TRACEHEADER )
         
c process only live traces

            if ( StaCor .ne. 30000) then
 
c set array counter

               tcount = tcount + 1

               call vmov ( itr(ITHWP1), 1, tri, 1, nsamp )
               
               do i = 1, nsamp
                  abstri(i) = abs(tri(i))
               enddo
                 
               avg = 0.
               icount = 0

               do i = ist, iend

c calculate avg on none zero samples within the sample window
c requested by the user.

                  if ( abstri(i) .gt. 1.e-32 ) then
                     avg = abstri(i) + avg
                     job_avg = job_avg + abstri(i)
                     jcount = jcount + 1
                     icount = icount + 1
                  endif

               enddo

               if ( icount .gt. 0 ) then
                  avg = avg / float ( icount )
               else
                  avg = 0.0
               endif
             
               if ( pipein ) then

                  call savew2(itr, ifmt_HdrWrd2, l_HdrWrd2,
     :                 ln_HdrWrd2, avg, TRACEHEADER)

               else

                  avg_array(tcount) = avg

               endif


               peak=0.

               do i = ist, iend

c calculate peak value

                  peak = max(peak,abstri(i))

c accumulate job constant peak amplitude

                  job_peak = max(peak,job_peak)
               enddo
               
               if ( pipein ) then

                  call savew2(itr,ifmt_HdrWrd1,l_HdrWrd1,
     :                 ln_HdrWrd1,peak,TRACEHEADER)

               else

                  peak_array(tcount) = peak

               endif
      
               rms=0.
               icount = 0

               do i = ist, iend

c calculate rms value for non-zero samples within the window
c specified by the user

                  if ( abstri(i) .gt. 1.e-32 ) then
                     rms = rms + (abstri(i)-avg)**2
                     icount = icount + 1
                  endif

               enddo

               if ( icount .gt. 0 ) then

                  rms = sqrt(rms/float(icount))
               
               else

                  rms = 0.0

               endif

               if ( pipein ) then

                  call savew2(itr,ifmt_HdrWrd3,l_HdrWrd3,
     :                 ln_HdrWrd3,rms,TRACEHEADER)

               else

                  rms_array(tcount) = rms

               endif

c if we are piping in then we will not be able to back up later and the
c output trace must go on its way now.  If we can back up we can delay
c the output until global parameters are calculated 

               if ( pipein ) call vmov ( tri, 1, itr(ITHWP1), 1, nsamp )

            endif

c write output data

            if ( pipein ) call wrtape (luout, itr, obytes)
 
         ENDDO
 
c skip to end of record

         call trcskp ( JJ, ne+1, ntrc, luin, ntrc, itr )

      ENDDO

c calculate job_avg based on live samples used

      job_avg = job_avg / (float(jcount))

c rewind the input data set if we are not reading from a pipe stream

      if ( pipein ) goto 900 

      call rwd(luin)      

c if we end up having to do this then I will need to grealloc itr 
c for the line header read.  Of course there is no need to redefine
c the global variables as they will not have changed and there is no
c need to reallocate the pointers to the trace headers as they will
c also not have changed.  There is also no need to check to see if the
c line header is there as it was already read and is definitely there 
c this time if it was there last time....Garossino
     
      alloc_size = SZLNHD
      errcd1 = 0

      call grealloc(mem_itr,alloc_size * SZSMPD,errcd1,abort)

      if (errcd1 .ne. 0) then
	write(LERR,*) 'ERROR: Unable to reallocate workspace '
	write(LERR,*) '       ',alloc_size * SZSMPD,' bytes requested '
	write(LERR,*) '       FATAL'
	write(LER,*) 'PUTVAL: Unable to reallocate workspace '
	write(LER,*) '     ',alloc_size * SZSMPD,' bytes requested '
	write(LER,*) 'FATAL'
	stop
      else
        write(LERR,*)'Reallocating workspace:'
	write(LERR,*) alloc_size * SZSMPD,' bytes requested '
      endif

      call vclr ( itr, 1, alloc_size )

c  read past input line header

      call rtape(luin,itr,lbytes)

c reallocate buffer memory to be big enough to include the trace header 
c and trace time series

      alloc_size = SZTRHD + SZSMPD * nsamp

      errcd1 = 0
      call grealloc(mem_itr,alloc_size,errcd1,abort)
      if (errcd1 .ne. 0) then
	write(LERR,*) 'ERROR: Unable to allocate trace workspace '
	write(LERR,*) '       ',alloc_size,' bytes requested '
	write(LERR,*) '       FATAL'
	write(LER,*) 'PUTVAL: Unable to allocate trace workspace '
	write(LER,*) '       ',alloc_size,' bytes requested '
	write(LER,*) 'FATAL'
	stop
      else
        write(LERR,*)'Allocating workspace:'
	write(LERR,*) alloc_size,' bytes requested '
      endif

      call vclr ( itr, 1, ITRWRD + nsamp )

c BEGIN PROCESSING for second  time to pick up global
c rms amplitude information and output data in the case of
c not piping in.


c reset array counter

      tcount = 0

c skip unwanted input records

      call recskp ( 1, irs-1, luin, ntrc, itr )

      DO JJ = irs, ire
 
c skip to start trace

         call trcskp ( JJ, 1, ns-1, luin, ntrc, itr )

         DO KK = ns, ne

            nbytes = 0
            call rtape( luin, itr, nbytes)

c if end of data encountered (nbytes=0) then bail out

            if(nbytes .eq. 0) then
               write(LERR,*)'Premature EOF on input at:'
               write(LERR,*)'  rec= ',JJ,'  trace= ',KK
               go to 999
            endif

c get required trace header information

            call saver2( itr, ifmt_StaCor, l_StaCor, ln_StaCor, 
     :           StaCor, TRACEHEADER )

c process only live traces
            
            if ( StaCor .ne. 30000) then

               tcount = tcount + 1
               
               call vmov ( itr(ITHWP1), 1, tri, 1, nsamp )
               
               do i = ist, iend

                  if ( abs(tri(i)) .gt. 1.e-32 ) then
                     kcount = kcount +1
                     job_rms =job_rms + (abs(tri(i))-job_avg)**2
                  endif

               enddo
             
            endif

         enddo
        
         call trcskp(JJ,ne+1,ntrc,luin,ntrc,itr)

      enddo

      if ( kcount .gt. 0 ) then
         job_rms = sqrt(job_rms/(float(kcount)))
      else
         job_rms = 0.0
      endif

c================================================================
c rewind to get line header and write data to output

      call rwd(luin)
     
      alloc_size = SZLNHD
      errcd1 = 0

      call grealloc(mem_itr,alloc_size * SZSMPD,errcd1,abort)

      if (errcd1 .ne. 0) then
	write(LERR,*) 'ERROR: Unable to reallocate workspace '
	write(LERR,*) '       ',alloc_size * SZSMPD,' bytes requested '
	write(LERR,*) '       FATAL'
	write(LER,*) 'PUTVAL: Unable to reallocate workspace '
	write(LER,*) '     ',alloc_size * SZSMPD,' bytes requested '
	write(LER,*) 'FATAL'
	stop
      else
        write(LERR,*)'Reallocating workspace:'
	write(LERR,*) alloc_size * SZSMPD,' bytes requested '
      endif

      call vclr ( itr, 1, alloc_size )

c  read, update for statistics and write line header

      call rtape(luin,itr,lbyout)
      
      call savew(itr, 'NumRec', nreco, LINHED)
      call savew(itr, 'NumTrc', ntrco, LINHED)
      call savew(itr, 'MaxAmp', job_peak, LINHED)
      call savew(itr, 'AvgAmp', job_avg, LINHED)
      call savew(itr, 'RmsAmp', job_rms, LINHED)

c update for current command line

      call savhlh  ( itr, lbytes, lbyout )
      call wrtape ( luout, itr, lbyout )

c reallocate buffer memory to be big enough to include the trace header 
c and trace time series

      alloc_size = SZTRHD + SZSMPD * nsamp

      errcd1 = 0
      call grealloc(mem_itr,alloc_size,errcd1,abort)
      if (errcd1 .ne. 0) then
	write(LERR,*) 'ERROR: Unable to allocate trace workspace '
	write(LERR,*) '       ',alloc_size,' bytes requested '
	write(LERR,*) '       FATAL'
	write(LER,*) 'PUTVAL: Unable to allocate trace workspace '
	write(LER,*) '       ',alloc_size,' bytes requested '
	write(LER,*) 'FATAL'
	stop
      else
        write(LERR,*)'Allocating workspace:'
	write(LERR,*) alloc_size,' bytes requested '
      endif

      call vclr ( itr, 1, ITRWRD + nsamp )

c BEGIN output pass

c set array counter

      tcount = 0

c skip unwanted input records

      call recskp ( 1, irs-1, luin, ntrc, itr )

      DO JJ = irs, ire
 
c skip to start trace

         call trcskp ( JJ, 1, ns-1, luin, ntrc, itr )

         DO KK = ns, ne

            nbytes = 0
            call rtape( luin, itr, nbytes)

c if end of data encountered (nbytes=0) then bail out

            if(nbytes .eq. 0) then
               write(LERR,*)'Premature EOF on input at:'
               write(LERR,*)'  rec= ',JJ,'  trace= ',KK
               go to 999
            endif

c get required trace header information

            call saver2( itr, ifmt_StaCor, l_StaCor, ln_StaCor, 
     :           StaCor, TRACEHEADER )
         
c process only live traces

            if ( StaCor .ne. 30000) then
 
c set array counter

               tcount = tcount + 1

c update trace headers for previously calculated statistics

               call savew2(itr, ifmt_HdrWrd2, l_HdrWrd2,
     :              ln_HdrWrd2, avg_array(tcount), TRACEHEADER)
               call savew2(itr,ifmt_HdrWrd1,l_HdrWrd1,
     :              ln_HdrWrd1,peak_array(tcount),TRACEHEADER)
               call savew2(itr,ifmt_HdrWrd3,l_HdrWrd3,
     :              ln_HdrWrd3,rms_array(tcount),TRACEHEADER)

            else

c clear header entries of any previously held value

               call savew2(itr, ifmt_HdrWrd2, l_HdrWrd2,
     :              ln_HdrWrd2, 0.0, TRACEHEADER)
               call savew2(itr,ifmt_HdrWrd1,l_HdrWrd1,
     :              ln_HdrWrd1,0.0,TRACEHEADER)
               call savew2(itr,ifmt_HdrWrd3,l_HdrWrd3,
     :              ln_HdrWrd3,0.0,TRACEHEADER)

            endif

c write output trace

            call wrtape (luout, itr, obytes)               
            
         ENDDO
 
c skip to end of record

         call trcskp ( JJ, ne+1, ntrc, luin, ntrc, itr )

      ENDDO

c close data files 

      write(LERR,*)' '
      write(LERR,*)'Job Constant Parameters '
      write(LERR,*)' '
      write(LERR,*)' Peak Amplitude = ',job_peak
      write(LERR,*)' Average Absolute Amplitude = ',job_avg
      write(LERR,*)' RMS Amplitude = ',job_rms
      write(LERR,*)' '

      call lbclos ( luin )
      call lbclos ( luout )
      write(LERR,*)'putval: Normal Termination'
      write(LER,*)'putval: Normal Termination'
      stop

 900  continue

      write(LERR,*)' '
      write(LERR,*)'Job Constant Parameters '
      write(LERR,*)' '
      write(LERR,*)' Peak Amplitude = ',job_peak
      write(LERR,*)' Average Absolute Amplitude = ',job_avg
      write(LERR,*)' '


      call lbclos ( luin )
      call lbclos ( luout )
      write(LERR,*)'putval: Normal Termination'
      write(LER,*)'putval: Normal Termination'
      stop

 999  continue

      call lbclos ( luin )
      call lbclos ( luout )
      write(LERR,*)'putval: ABNORMAL Termination'
      write(LER,*)'putval: ABNORMAL Termination'
      stop
      end

c -----------------  Subroutine -----------------------

      subroutine help()

c provide terse online help [detailed help goes in man page]

#include <f77/iounit.h>

      write(LER,*)' '
      write(LER,*)'===================================================='
      write(LER,*)' '
      write(LER,*)' Command Line Arguments for PUTVAL'
      write(LER,*)' '
      write(LER,*)' For a more detailed description of these parameters'
      write(LER,*)' see the online man page using uman, xman or xuspman'
      write(LER,*)' INPUT DATA MUST BE A DISK FILE !!'
      write(LER,*)' '
      write(LER,*)'Input...................................... (def)'
      write(LER,*)' '
      write(LER,*)'-N[]   -- input data set                  (stdin)'
      write(LER,*)'-O[]   -- output data set                (stdout)'
      write(LER,*)'-s[]   -- process start time (ms)             (1)'
      write(LER,*)'-e[]   -- process end time (ms)     (last sample)'
      write(LER,*)'-ns[]  -- start trace number                  (1)'
      write(LER,*)'-ne[]  -- end trace number           (last trace)'
      write(LER,*)'-rs[]  -- start record                        (1)'
      write(LER,*)'-re[]  -- end record                (last record)'
      write(LER,*)'-HW1[] -- Headerword for Peak Value      (Horz01)'
      write(LER,*)'-HW2[] -- Headerword for Average Value   (Horz02)'
      write(LER,*)'-HW3[] -- Headerword for RMS Value       (Horz03)'
      write(LER,*)'Job Constant Values are put into Line Headers:'
      write(LER,*)'      MaxAmp, AvgAmp, RmsAmp'
      write(LER,*)'-V     -- verbos printout'
      write(LER,*)' '
      write(LER,*)'Usage:'
      write(LER,*)'     putval -N[] -O[] -s[] -e[] -ns[] -ne[] -rs[]'
      write(LER,*)'            -re[] -HW1[] -HW2[] -HW3[] -V'
      write(LER,*)' '
      write(LER,*)'===================================================='
      
      return
      end

c -----------------  Subroutine -----------------------

c pick up command line arguments 

      subroutine cmdln ( ntap, otap, ns, ne, irs, ire, ist, iend, 
     :     name, verbos, c_Hdrwrd1, c_HdrWrd2, c_HdrWrd3 )

#include <f77/iounit.h>

      integer    ist, iend, ns, ne, irs, ire, argis

      character  ntap*(*), otap*(*), name*(*)
      character c_Hdrwrd1*(*), c_HdrWrd2*(*), c_HdrWrd3*(*)

      logical    verbos

      call argi4 ( '-e', iend, 0, 0 )

      call argstr ('-Hw1', c_Hdrwrd1 , 'Horz01', 'Horz01' )
      call argstr('-hw1', c_Hdrwrd1 , c_Hdrwrd1, c_Hdrwrd1 )
      call argstr('-HW1', c_Hdrwrd1 , c_Hdrwrd1, c_Hdrwrd1)
 
      call argstr ('-Hw2', c_HdrWrd2 , 'Horz02', 'Horz02' )
      call argstr('-hw2', c_HdrWrd2 , c_HdrWrd2, c_HdrWrd2 )
      call argstr('-HW2', c_HdrWrd2 , c_HdrWrd2, c_HdrWrd2)
      
      call argstr ('-Hw3', c_HdrWrd3 , 'Horz03', 'Horz03' )
      call argstr('-hw3', c_HdrWrd3 , c_HdrWrd3, c_HdrWrd3 )
      call argstr('-HW3', c_HdrWrd3 , c_HdrWrd3, c_HdrWrd3)  
  
           call argi4 ( '-ne', ne, 0, 0 )
           call argi4 ( '-ns', ns, 0, 0 )
           call argstr ( '-N', ntap, ' ', ' ' ) 

           call argstr ( '-O', otap, ' ', ' ' ) 

           call argi4 ( '-re', ire, 0, 0 )
           call argi4 ( '-rs', irs, 0, 0 )

           call argi4 ( '-s', ist, 1, 1 )

           verbos = (argis('-V') .gt. 0)

c check for extraneous arguments and abort if found to
c catch all manner of user typo's

      call xtrarg ( name, ler, .FALSE., .FALSE. )
      call xtrarg ( name, lerr, .FALSE., .TRUE. )

           
      return
      end

c -----------------  Subroutine -----------------------

c verbal printout of pertinent program particulars


      subroutine verbal ( ntap, otap, nsamp, nsi, ntrc, nrec, iform, 
     :     ist, iend, irs, ire, ns, ne, verbos, c_Hdrwrd1, c_HdrWrd2,
     :     c_HdrWrd3)

#include <f77/iounit.h>

      integer    nsamp, ntrc, iform, ist, iend, irs, ire, ns, ne, nsi

      character  ntap*(*), otap*(*)

      character  c_Hdrwrd1*(*),c_HdrWrd2*(*), c_HdrWrd3*(*)
      logical    verbos

      write(LERR,*)' '
      write(LERR,*)' Input Line Header Parameters'
      write(LERR,*)' '
      write(LERR,*) ' input data set name   =  ', ntap
      write(LERR,*) ' samples per trace     =  ', nsamp
      write(LERR,*) ' traces per record     =  ', ntrc
      write(LERR,*) ' number of records     =  ', nrec
      write(LERR,*) ' data format           =  ', iform
      write(LERR,*) ' sample interval       =  ', nsi
      write(LERR,*)' '
      write(LERR,*)' '
      write(LERR,*)' Command Line Parameters '
      write(LERR,*)' '
      write(LERR,*) ' output data set name  =  ', otap
      write(LERR,*) ' start record          =  ', irs 
      write(LERR,*) ' end record            =  ', irs 
      write(LERR,*) ' start trace           =  ', ns
      write(LERR,*) ' end trace             =  ', ne
      write(LERR,*) ' processing sample start = ', ist
      write(LERR,*) ' processing sample end   = ', iend
      write(LERR,*) ' Peak Value written to ',c_Hdrwrd1
      write(LERR,*) ' Average Value written to ',c_HdrWrd2
      write(LERR,*) ' RMS Value written to ',c_HdrWrd3
      if ( verbos )  write(LERR,*) ' verbose printout requested'
      write(LERR,*)' '
      write(LERR,*)'========================================== '
      write(LERR,*)' '

      return
      end





