C***********************************************************************
C                 copyright 2001, Amoco Production Company             *
C                             All Rights Reserved                      *
C                     an affiliate of BP America Inc.                  *
C***********************************************************************
      subroutine clev(n,a,b,u,s)
C
c      title: clev -  Wiener Levinson algorithm to solve 
C                     complex hermetian system of equations.
c                     matrix a is toeplitz.
C
C      the system of equations this routine will solve is of the form:
C
C      |       a(1)     a(2)   ...   a(N)   | |u(1)|     |b(1)|
C      |  CONJG(a(2))   a(1)   ...   a(N-1) | |  . |     |      .  |
C      |         .                          | |  . |  =  |      .  |
C      |         .                          | |  . |     |      .  |
C      |         .                          | |    |     |         |
C      |  CONJG(a(N))          ...     a(1) | |u(N)|     |b(N)|
C
C
C      note that the main diagonal "a(1)" must be real.
C
C      NOTE THaT THE SYSTEM IS SCaLED BY 1./a(1) BEFORE IT IS SOLVED.
C      (THIS WILL BE NOTICED ON OUTPUT)
C
C
C      INPUT
C
c      n       =  order of system of equations
C
C      INPUT/OUTPUT
C
C      a       =  (n) first row/column of Toeplitz matrix.
C      b  =  (n) rhs of system to solve
C
C      OUTPUT
C
C      u       =  (n) solution vector
C      s       =  (n) solution vector for spike rhs
C
c_______________________________________________________________
      integer  iand
c_______________________________________________________________
c     input matrix
c_______________________________________________________________
      complex    a(n)
c_______________________________________________________________
c     rhs
c_______________________________________________________________
      complex    b(n)
c_______________________________________________________________
c     soln for unit rhs
c_______________________________________________________________
      complex    s(n)
c_______________________________________________________________
c     soln for full rhs
c_______________________________________________________________
      complex    u(n)
c_______________________________________________________________
c     temporary work arrays for vectorization across solutions.
c_______________________________________________________________
      real       rnorm
      complex    v,r
      complex    e,ec
      complex    temp,rc
c_______________________________________________________________
c     normalize matrix and rhs, zero other vectors
c     since [a] is Hermitian, a(1) and hence rnorm are real.
c_______________________________________________________________
      rnorm=1./real(a(1))
      do 1020 i=1,n
        a(i)=a(i)*rnorm
        b(i)=b(i)*rnorm
        s(i)=(0.,0.)
        u(i)=(0.,0.)
1020  continue
c_______________________________________________________________
c     initialize recursion
c_______________________________________________________________
       v=(1.,0.)
       s(1)=(1.,0.)
       u(1)=b(1)
c_______________________________________________________________
c     main recursion loop
c_______________________________________________________________
      do 99000 j=2,n
c_______________________________________________________________
c      error term for spike rhs
c_______________________________________________________________
        e=(0.,0.)
       do 2020 i=2,j
         e=e+conjg(a(i))*s(j-i+1)
2020   continue
c_______________________________________________________________
c      scale factor and new spike value
c_______________________________________________________________
        r=e/conjg(v)
        v=v-r*conjg(e)
c_______________________________________________________________
c      new solution vector for spike
c_______________________________________________________________
       do 2040 i=1,j/2
         temp=s(i)
         s(i)=s(i)-r*conjg(s(j-i+1))
         s(j-i+1)=s(j-i+1)-r*conjg(temp)
2040   continue
       if(iand(j,1).ne.0) then
           s(j/2+1)=
     1         s(j/2+1)-r*conjg(s(j/2+1))
       endif
c_______________________________________________________________
c      error term for full rhs
c_______________________________________________________________
        ec=(0.,0.)
       do 2120 i=2,j
         ec=ec+conjg(a(i))*u(j-i+1)
2120   continue
c_______________________________________________________________
C      scale factor
c_______________________________________________________________
        rc=(ec-b(j))/conjg(v)
c_______________________________________________________________
c      new solution vector for spike
c_______________________________________________________________
       do 2140 i=1,j
         u(i)=u(i)-rc*conjg(s(j-i+1))
2140   continue
C
99000 continue
C
       return
       end
