C***********************************************************************
C                 copyright 2001, Amoco Production Company             *
C                             All Rights Reserved                      *
C                     an affiliate of BP America Inc.                  *
C***********************************************************************
c**********************************************************************c
c
c     program module ramp
c
c**********************************************************************c
c
c ramp reads seismic trace data from an input file,
c applies a linear ramp in space, and
c writes the results to an output file
c
c
c**********************************************************************c
c
c     declare variables
c

#include <save_defs.h>
#include <f77/iounit.h>
#include <f77/lhdrsz.h>
#include <f77/sisdef.h>

      parameter   (pi = 3.14159265)

      integer     itr ( SZLNHD )
      integer     lhed( SZLNHD )
      integer     nsamp, nsi, ntrc, nrec, iform, ic0
      integer     luin , luout, lbytes, nbytes, lbyout
      integer     irs,ire,ns,ne
#include <f77/pid.h>
      integer     static
      real        tri ( SZLNHD ), wt( SZLNHD )
      character   ntap * 256, otap * 256, name*4
      logical     verbos, query, up, cosine
      integer     argis

c     equivalence ( itr(129), tri (1) )
      equivalence ( itr(  1), lhed(1) )
      data lbytes / 0 /, nbytes / 0 /, name/'RAMP'/

c-----
c     read program parameters from command line card image file
c-----
      query = ( argis ( '-?' ) .gt. 0 .or. argis('-H') .gt. 0)
      if ( query )then
            call help()
            stop
      endif

c-----
c     open printout files
c-----
#include <f77/open.h>

      call gcmdln(ntap,otap,ns,ne,irs,ire,ic0,up,verbos,
     1            cosine)
c-----
c     get logical unit numbers for input and output
c-----
      call getln(luin , ntap,'r', 0)
      call getln(luout, otap,'w', 1)
c-----
c     read line header of input
c     save certain parameters
c-----
      call rtape  ( luin, itr, lbytes)
      if(lbytes .eq. 0) then
         write(LERR,*)'RAMP: no header read from unit ',luin
         write(LERR,*)'FATAL'
         stop
      endif
      call saver(itr, 'NumSmp', nsamp, LINHED)
      call saver(itr, 'SmpInt', nsi  , LINHED)
      call saver(itr, 'NumTrc', ntrc , LINHED)
      call saver(itr, 'NumRec', nrec , LINHED)
      call saver(itr, 'Format', iform, LINHED)
      call saver(itr, 'UnitSc', unitsc, LINHED)
      if (unitsc .eq. 0.0) then
          write(LERR,*)'********************************************'
          write(LERR,*)'WARNING: sample unit scaler in LH = ',unitsc
          write(LERR,*)'         will set to .001 (millisec default)'
          write(LERR,*)'********************************************'
          unitsc = .001
          call savew(itr, 'UnitSc', unitsc, LINHED)
      endif

      call hlhprt (itr, lbytes, name, 4, LERR)

      call savelu('TrcNum',ifmt_TrcNum,l_TrcNum,ln_TrcNum,TRACEHEADER)
      call savelu('RecNum',ifmt_RecNum,l_RecNum,ln_RecNum,TRACEHEADER)
      call savelu('SrcLoc',ifmt_SrcLoc,l_SrcLoc,ln_SrcLoc,TRACEHEADER)
      call savelu('RecInd',ifmt_RecInd,l_RecInd,ln_RecInd,TRACEHEADER)
      call savelu('DphInd',ifmt_DphInd,l_DphInd,ln_DphInd,TRACEHEADER)
      call savelu('DstSgn',ifmt_DstSgn,l_DstSgn,ln_DstSgn,TRACEHEADER)
      call savelu('DstUsg',ifmt_DstUsg,l_DstUsg,ln_DstUsg,TRACEHEADER)
      call savelu('StaCor',ifmt_StaCor,l_StaCor,ln_StaCor,TRACEHEADER)

c-----
c     ensure that command line values are compatible with data set
c-----
      call cmdchk(ns,ne,irs,ire,ntrc,nrec)

      if ( ns .gt. ic0 .or. ne .lt. ic0) then
           write(LERR,*)'Start trace of ramp is wrong side of ic0,'
           write(LERR,*)'end trace of ramp is wrong side of ic0'
           write(LERR,*)'FATAL'
           write(LERR,*)'make sure:   ns <= ic0 <= ne'
           stop
      endif
c-----
c     modify line header to reflect actual number of traces output
c-----
      nrecc=ire - irs+1
      call savew(itr, 'NumRec', nrecc, LINHED)
      call savhlh(itr,lbytes,lbyout)
      call wrtape ( luout, itr, lbyout                 )
c-----
c     compute weights for ramp across record
c     fix the middle point of the ramp weights depending on whether
c     we ramp up or down
c-----
      if ( up ) then
           call vfill ( 0.0, wt, 1, ntrc )
      else
           call vfill ( 1.0, wt, 1, ntrc )
      endif

      ns1 = ns - 1
      ne1 = ne - ns1
      ic1 = ic0 - ns1
      jtl = ic1 - 1
      jtr = ne1 - ic1

      if (up) then
          wt (ic0) = 1.0
      else
          wt (ic0) = 0.0
      endif

      jj = ns - 1
      do  10  j = 1, ic1-1

          jj = jj + 1
          if (up) then
             if (cosine) then
              wt (jj) = .5*(1. - cos(pi*float(j-1)/float(jtl)))
             else
              wt (jj) = float(j-1)/float(jtl)
             endif
          else
             if (cosine) then
              wt (jj) = .5*(1. - cos(pi*float(ic1-j)/float(jtl)))
             else
              wt (jj) = float(ic1-j)/float(jtl)
             endif
          endif

   10 continue

      jj = jj + 1

      do  11  j = ic1+1, ne1

          jj = jj + 1
          if (up) then
             if (cosine) then
              wt (jj) = .5*(1. - cos(pi*float(ne1-j)/float(jtr)))
             else
              wt (jj) = float(ne1-j)/float(jtr)
             endif
          else
             if (cosine) then
              wt (jj) = .5*(1. - cos(pi*float(j-ic1)/float(jtr)))
             else
              wt (jj) = float(j-ic1)/float(jtr)
             endif
          endif

   11 continue

c---------------
c     for a hinge point equal to one of the end points make sure
c     that the weights before ns and after ne are consistent with
c     the ramp function itself
c---------------
      if ( ic0 .eq. ne ) then
         call vfill (wt(ne), wt(ne+1), 1, ntrc-ne)
      elseif( ic0 .eq. ns ) then
         call vfill (wt(ns), wt(1), 1, ns1)
      endif

      write(LERR,*)'Spread weights are... '
      write(LERR,888) (wt(l),l=1,ntrc)
  888 format(10f8.2)


c-----
c     verbose output of all pertinent information before
c     processing begins
c-----
c     if( verbos ) then
            call verbal(nsamp, nsi, ntrc, nrec, iform,
     1                  ic0,up,ntap,otap,cosine)
c     endif
c-----
c     BEGIN PROCESSING
c     read trace, do ramping, write to output file
c-----
c-----
c     skip unwanted records
c-----
      call recskp(1,irs-1,luin,ntrc,itr)
c-----
c     process desired trace records
c-----
      do 1000 jj = irs, ire

            ic = 0
            do 1001 kk = 1, ntrc

                  ic = ic + 1
                  nbytes = 0
                  call rtape ( luin, itr, nbytes)
                  if(nbytes .eq. 0) then
                     write(LERR,*)'End of file on input:'
                     write(LERR,*)'  rec= ',jj,'  trace= ',kk
                     go to 999
                  endif
                  call vmov (itr(ITHWP1), 1, tri, 1, nsamp)
                  call saver2(itr,ifmt_RecNum,l_RecNum, ln_RecNum,
     1                        irec  , TRACEHEADER)
                  call saver2(itr,ifmt_StaCor,l_StaCor, ln_StaCor,
     1                        static, TRACEHEADER)
                   
                  if(static .ne. 30000)then
                        call vsmul (tri,1, wt(ic), tri,1,nsamp)
                  else
                        call vclr(tri,1,nsamp)
                  endif

                  call vmov  (tri, 1, itr(ITHWP1), 1, nsamp)
                  call wrtape( luout, itr, nbytes )

 1001       continue
                  if(verbos)write(LERR,*)'Processed ri ',irec

 1000       continue
c-----
c     close data files
c-----
  999 continue
      call lbclos ( luin )
      call lbclos ( luout )
      if(verbos) then
            write(LERR,*)'end of davc, processed',nrec,' record(s)',
     :               ' with ',ntrc, ' traces'
       endif
      end


      subroutine help
#include <f77/iounit.h>
         write(LER,*)
     :'***************************************************************'
      write(LER,*)
     :'execute ramp by typing ramp and a list of program parameters.'
      write(LER,*)
     :'note that each parameter is proceeded by -a where "a" is '
      write(LER,*)
     :'a character(s) corresponding to some parameter.'
        write(LER,*)
     :'users enter the following parameters, or use the default values'
        write(LER,*)
     :' -N [ntap]    (no default)  : input data file name'
        write(LER,*)
     :' -O [otap]    (no default)  : output data file name'
       write(LER,*)
     :' -ns[ns] (default = first)  : start ramp at trace number'
       write(LER,*)
     :' -ne[ne] (default = last)   : end ramp at trace number'
       write(LER,*)
     :' -rs[irs]  (default = first): start record number'
       write(LER,*)
     :' -ne[ire]  (default = last) : end record number'
        write(LER,*)
     :' -c [ic0]     middle control point (trc #) of ramps'
        write(LER,*)
     :' -U           ramp up from trace ns to ic0 then down to trace ne'
        write(LER,*)
     :'  otherwise,  ramp down from trace ns to ic0, then up to ne'
        write(LER,*)
     :' -C           use cosine ramp, otherwise use linear'
         write(LER,*)
     :'usage:   ramp -N[ntap] -O[otap] -ns[ns] -ne[ne] -rs[rs] -re[re]'
         write(LER,*)
     :'               -c[ic0] [-C -U -V]'
         write(LER,*)
     :'***************************************************************'
      return
      end

      subroutine gcmdln(ntap,otap,ns,ne,irs,ire,ic0,up,verbos,
     1                  cosine)
c-----
c     get command arguments
c
c     ntap  - c*100     input file name
c     otap  - c*100     output file name
c     ns    - i*4 starting trace index
c     ne    - i*4 ending trace index
c     irs   - i*4 starting record index
c     ire   - i*4 ending record index
c      ic0  - i*4 middle trace of ramp
c      up   - l   if true, ramp up to ic0 then down to end of record
c    verbos - l   verbose output or not
c-----
#include <f77/iounit.h>
      character  ntap*(*), otap*(*)
      integer *4 ns, ne, irs, ire, ic0
      logical    verbos, up, cosine
      integer    argis

            call argstr( '-N', ntap, ' ', ' ' )
            call argstr( '-O', otap, ' ', ' ' )
            call argi4 ( '-ns', ns ,   0  ,  0    )
            call argi4 ( '-ne', ne ,   0  ,  0    )
            call argi4 ( '-rs', irs ,   0  ,  0    )
            call argi4 ( '-re', ire ,   0  ,  0    )
            call argi4( '-c', ic0, 0, 0 )
            up     = ( argis( '-U' ) .gt. 0 )
            cosine = ( argis( '-C' ) .gt. 0 )
            verbos = ( argis( '-V' ) .gt. 0 )

            if (ic0 .eq. 0) then
                write(LERR,*)'You must enter a -c value on command line'
                write(LERR,*)'Rerun'
                stop
            endif

      return
      end

      subroutine verbal(nsamp, nsi, ntrc, nrec, iform,
     1            ic0,up,ntap,otap,cosine)
c-----
c     verbose output of processing parameters
c
c     nsamp - I*4 number of samples in trace
c     nsi   - I*4 sample interval in ms
c     ntrc  - I*4 traces per record
c     nrec  - I*4 number of records per line
c     iform - I*4 format of data
c      ic0  - R*4 middle trace of ramp
c      up   - L   if true, ramp up to middle, then down to end of record
c     ntap  - C*100     input file name
c     otap  - C*100     output file name
c-----
#include <f77/iounit.h>
      integer*4 nsamp, nsi, ntrc, nrec, iform, ic0
      character ntap*(*), otap*(*)
      logical   up, cosine

            write(LERR,*)' '
            write(LERR,*)' line header values after default check '
            write(LERR,*) ' # of samples/trace =  ', nsamp
            write(LERR,*) ' sample interval    =  ', nsi
            write(LERR,*) ' traces per record  =  ', ntrc
            write(LERR,*) ' records per line   =  ', nrec
            write(LERR,*) ' format of data     =  ', iform
            write(LERR,*) ' ramp up (?)        =  ', up
            write(LERR,*) ' middle trace of ramp=  ',ic0
            write(LERR,*) ' input data set name =  ', ntap
            write(LERR,*) ' output data set name=  ', otap
            if (cosine) then
              write(LERR,*) ' Using cosine weighting function'
            else
              write(LERR,*) ' Using linear weighting function'
            endif
            write(LERR,*)' '
            write(LERR,*)' '

      return
      end

