C***********************************************************************
C                 copyright 2001, Amoco Production Company             *
C                             All Rights Reserved                      *
C                     an affiliate of BP America Inc.                  *
C***********************************************************************
c -----------------  response -----------------------------------------

c     Program Changes:

c      - original written: November 7, 1995

c     Program Description:

c      - combines two input files of identical size and configuration;
c     "N1" contains a seismic attribute and "N2" contains the energy
c     envelope. Response finds the value of the "N1" dataset attribute
c     at the point at which the envelope is a maximum. One value is
c     obtained for each envelope lobe and is returned as a constant
c     for the entire time window of the envelope from trough to trough.

c get machine dependent parameters 

#include <f77/iounit.h>
#include <f77/lhdrsz.h>
#include <f77/sisdef.h>
#include <save_defs.h> 

c dimension standard USP variables 

      integer     itr_attribute ( SZLNHD )
      integer     itr_envelope ( SZLNHD )
     
      integer     nsampo, nsamp, nsi, ntrc, ntrco, nrec, nreco
      integer     luin_att , luin_env, luout, lbytes, nbytes, lbyout
      integer     ist, iend, irs, ire, ns, ne, argis, obytes

      real        tri_att ( SZLNHD )
      real        tri_env ( SZLNHD )
      real        tri ( SZLNHD )

      character   ntap_att*255, ntap_env*255, otap*255, name*8

      logical     IKP, verbos

c Program Specific _ static memory variables

      integer ifmt_StaCor,l_StaCor,ln_StaCor,StaCor_att,StaCor_env
      integer nsamp_att, nsi_att, ntrc_att, nrec_att
      integer     nsamp_env, nsi_env, ntrc_env, nrec_env

      real threshold, vdot

c Initialize variables

      data abort/1/
      data name/"RESPONSE"/
      data IKP/.false./

c give command line help if requested

      if ( argis ( '-?' ) .gt. 0 .or. 
     :     argis ( '-h' ) .gt. 0 .or. 
     :     argis ( '-help' ) .gt. 0 ) then
         call help()
         stop
      endif

c open printout file

#include <f77/open.h>

c get command line input parameters

      call cmdln ( ntap_att, ntap_env, threshold, otap, ns, ne, irs, 
     :     ire, ist, iend, name, IKP, verbos )

c open input and output files

      if ( IKP ) then
           call sisfdfit(luin_att,0)
           call sisfdfit(luin_env,3)
      else
           call getln(luin_att , ntap_att,'r', 0)
           call getln(luin_env , ntap_env,'r', 0)
      endif
      call getln(luout, otap,'w', 1)

c  read attribute line header and save certain parameters

      call rtape(luin_att,itr_attribute,lbytes)
      if(lbytes.eq.0)then
         write(LER,*)'RESPONSE: no line header on input dataset',
     :     ntap_att
         write(LER,*)'FATAL'
         stop
      endif

      call saver(itr_attribute, 'NumSmp', nsamp_att, LINHED)
      call saver(itr_attribute, 'SmpInt', nsi_att  , LINHED)
      call saver(itr_attribute, 'NumTrc', ntrc_att , LINHED)
      call saver(itr_attribute, 'NumRec', nrec_att , LINHED)

c put attribute historical line header into printout file  

      call hlhprt (itr_attribute, lbytes, name, 8, LERR)

c  read envelope line header and save certain parameters

      lbytes = 0
      call rtape(luin_env,itr_envelope,lbytes)
      if(lbytes.eq.0)then
         write(LER,*)'RESPONSE: no line header on input dataset',
     :     ntap_env
         write(LER,*)'FATAL'
         stop
      endif

      call saver(itr_envelope, 'NumSmp', nsamp_env, LINHED)
      call saver(itr_envelope, 'SmpInt', nsi_env  , LINHED)
      call saver(itr_envelope, 'NumTrc', ntrc_env , LINHED)
      call saver(itr_envelope, 'NumRec', nrec_env , LINHED)

c verify datasets are compatible 

      if ( nsamp_att .ne. nsamp_env .or.
     :     ntrc_att .ne. ntrc_env .or.
     :     nrec_att .ne. nrec_env .or.
     :     nsi_att .ne. nsi_env ) then

           write (LERR,*)' '
           write (LERR,*)'datasets are not compatible!'
           write (LERR,*)'FATAL'
           write (LER,*)' '
           write (LER,*)'datasets are not compatible!'
           write (LER,*)'FATAL'
           stop
      else
           nsi = nsi_att
           nrec = nrec_att
           ntrc = ntrc_att
           nsamp = nsamp_att
      endif

c define pointers to header words required by your routine

      call savelu('StaCor',ifmt_StaCor,l_StaCor,ln_StaCor,TRACEHEADER)

c update historical line header and print to printout file 

      call hlhprt (itr_envelope, lbytes, name, 8, LERR)

c check user supplied boundary conditions and set defaults

      if ( irs .eq. 0 ) irs = 1
      if ( ire .eq. 0 .or. ire .gt. nrec ) ire = nrec

      if ( ns .eq. 0 ) ns = 1
      if ( ne .eq. 0 .or. ne .gt. ntrc ) ne = ntrc

      ist = nint ( float(ist) / float(nsi) )
      iend = nint ( float(iend) / float(nsi) )
      if ( ist .eq. 0 ) ist = 1
      if ( iend .eq. 0 .or. iend .gt. nsamp ) iend = nsamp

      nreco = ire - irs + 1
      ntrco = ne - ns + 1
      nsampo = iend - ist + 1

c modify line header to reflect actual record configuration output
c NOTE: in this case the sample limits ist and iend are used to 
c       limit processing only.   All data samples are actually passed.

      call savew(itr_envelope, 'NumRec', nreco, LINHED)
      call savew(itr_envelope, 'NumTrc', ntrco  , LINHED)
      call savew(itr_envelope, 'NumSmp', nsampo  , LINHED)

c number output bytes

      obytes = SZTRHD + SZSMPD * nsampo 

c save out hlh and line header

      call savhlh  ( itr_envelope, lbytes, lbyout )
      call wrtape ( luout, itr_envelope, lbyout )

c verbose output of all pertinent information before processing begins

      call verbal( ntap_att, ntap_env, otap, nsamp, nsi, ntrc, nrec,
     :     ist, iend, irs, ire, ns, ne, nreco, ntrco, nsampo, 
     :     threshold, IKP, verbos)

c BEGIN PROCESSING 

c skip unwanted attribute records

      call recskp ( 1, irs-1, luin_att, ntrc, itr_attribute )
      call recskp ( 1, irs-1, luin_env, ntrc, itr_envelope )

      DO JJ = irs, ire
 
c skip to start trace

         call trcskp ( JJ, 1, ns-1, luin_att, ntrc, itr_attribute )
         call trcskp ( JJ, 1, ns-1, luin_env, ntrc, itr_envelope )

         DO KK = ns, ne

            nbytes = 0
            call rtape( luin_att, itr_attribute, nbytes)

c if end of data encountered (nbytes=0) then bail out

            if(nbytes .eq. 0) then
               write(LERR,*)'Premature EOF on attribute input at:'
               write(LERR,*)'  rec= ',JJ,'  trace= ',KK
               go to 999
            endif

            nbytes = 0
            call rtape( luin_env, itr_envelope, nbytes)

c if end of data encountered (nbytes=0) then bail out

            if(nbytes .eq. 0) then
               write(LERR,*)'Premature EOF on envelope input at:'
               write(LERR,*)'  rec= ',JJ,'  trace= ',KK
               go to 999
            endif

c get required trace header information

            call saver2( itr_attribute, ifmt_StaCor, l_StaCor,
     :           ln_StaCor, StaCor_att, TRACEHEADER )
            call saver2( itr_envelope, ifmt_StaCor, l_StaCor,
     :           ln_StaCor, StaCor_env, TRACEHEADER )

c process only live traces

            if ( StaCor_att .ne. 30000 .and. 
     :           StaCor_env .ne. 30000 ) then

c Put your subroutine here [remember to declare any arguments you need
c over and above those already declared]
c load trace portion of itr_attribute[] to real array tri[]
 
               call vmov ( itr_attribute(ITHWP1+ist-1), 1, tri_att, 1, 
     :                     nsampo )
               call vmov ( itr_envelope(ITHWP1+ist-1), 1, tri_env, 1, 
     :                     nsampo )
               call vclr ( tri, 1, nsampo )

               call dotpr (tri_att,1,tri_env,1,vdot,nsampo)

               if ( abs (vdot) .gt. 1.e-32  ) then
                  call pkdt ( tri_att, tri_env, nsampo, threshold, tri )
               else
                  call savew2 (itr_envelope, ifmt_StaCor, l_StaCor, 
     :                 ln_StaCor, 30000, TRACEHEADER )
               endif

               call vmov ( tri, 1, itr_envelope(ITHWP1), 1, nsampo )

            endif
c write output data

            call wrtape (luout, itr_envelope, obytes)
 
         ENDDO
 
c skip to end of record

         call trcskp ( JJ, ne+1, ntrc, luin_att, ntrc, itr_attribute )
         call trcskp ( JJ, ne+1, ntrc, luin_env, ntrc, itr_envelope )

      ENDDO

c close data files 

      call lbclos ( luin )
      call lbclos ( luout )
      write(LERR,*)'response: Normal Termination'
      write(LER,*)'response: Normal Termination'
      stop

 999  continue

      call lbclos ( luin )
      call lbclos ( luout )
      write(LERR,*)'response: ABNORMAL Termination'
      write(LER,*)'response: ABNORMAL Termination'
      stop
      end

c -----------------  Subroutine -----------------------

      subroutine help()

c provide terse online help [detailed help goes in man page]

#include <f77/iounit.h>

      write(LER,*)' '
      write(LER,*)'===================================================='
      write(LER,*)' '
      write(LER,*)' Command Line Arguments for RESPONSE:'
      write(LER,*)' '
      write(LER,*)'         Combines instantaneous attribute and energy'
      write(LER,*)'         envelope to create a response attribute.'
      write(LER,*)' '
      write(LER,*)' For a more detailed description of these parameters'
      write(LER,*)' see the online man page using uman, xman or xuspman'
      write(LER,*)' '
      write(LER,*)'Input...................................... (def)'
      write(LER,*)' '
      write(LER,*)'-N1[]  -- input attribute data set        (stdin)'
      write(LER,*)'-N2[]  -- input energy envelope data set         '
      write(LER,*)'-O[]   -- output data set                (stdout)'
      write(LER,*)'-t[]   -- % of max env peak to pick        (0.15)'
      write(LER,*)'-s[]   -- process start time (ms)             (1)'
      write(LER,*)'-e[]   -- process end time (ms)     (last sample)'
      write(LER,*)'-ns[]  -- start trace number                  (1)'
      write(LER,*)'-ne[]  -- end trace number           (last trace)'
      write(LER,*)'-rs[]  -- start record                        (1)'
      write(LER,*)'-re[]  -- end record                (last record)'
      write(LER,*)'-V     -- verbos printout'
      write(LER,*)' '
      write(LER,*)'Usage:'
      write(LER,*)'       response -N1[] -N2[] -O[] -t[] -s[] -e[]'
      write(LER,*)'            -ns[] -ne[] -rs[] -re[] -V'
      write(LER,*)' '
      write(LER,*)'===================================================='
      
      return
      end

c -----------------  Subroutine -----------------------

c pick up command line arguments 

      subroutine cmdln ( ntap_att, ntap_env, threshold, otap, ns, ne,
     :     irs, ire, ist, iend, name, IKP, verbos )

#include <f77/iounit.h>

      integer    ist, iend, ns, ne, irs, ire, argis

      character  ntap_att*(*), ntap_env*(*), otap*(*), name*(*)

      logical    IKP, verbos

           call argi4 ( '-e', iend, 0, 0 )
           IKP = (argis('-IKP') .gt. 0)
           call argi4 ( '-ne', ne, 0, 0 )
           call argi4 ( '-ns', ns, 0, 0 )
           call argstr ( '-N1', ntap_att, ' ', ' ' ) 
           call argstr ( '-N2', ntap_env, ' ', ' ' ) 

           call argstr ( '-O', otap, ' ', ' ' ) 

           call argi4 ( '-re', ire, 0, 0 )
           call argi4 ( '-rs', irs, 0, 0 )

           call argi4 ( '-s', ist, 1, 1 )

           call argr4 ( '-t', threshold, 0.15, 0.15 )

           verbos = (argis('-V') .gt. 0)

c check for extraneous arguments and abort if found to
c catch all manner of user typo's

      call xtrarg ( name, ler, .FALSE., .FALSE. )
      call xtrarg ( name, lerr, .FALSE., .TRUE. )

           
      return
      end

c -----------------  Subroutine -----------------------

c verbal printout of pertinent program particulars

      subroutine verbal ( ntap_att, ntap_env, otap, nsamp,
     :     nsi, ntrc, nrec, ist, iend, irs, ire, ns, ne, nreco,
     :     ntrco, nsampo, threshold, IKP, verbos)

#include <f77/iounit.h>

      integer    nsamp, ntrc, ist, iend, irs,
     :     ire, ns, ne, nsi, ntrco, nsampo, nreco

      real       threshold

      character  ntap_att*(*), ntap_env*(*), otap*(*)

      logical    verbos, IKP

      write(LERR,*)' '
      write(LERR,*)' Input Line Header Parameters'
      write(LERR,*)' '
      write(LERR,*) ' input attribute data set name   =  ', ntap_att
      write(LERR,*) ' input envelope data set name    =  ', ntap_env
      write(LERR,*) ' samples per trace               =  ', nsamp
      write(LERR,*) ' traces per record               =  ', ntrc
      write(LERR,*) ' number of records               =  ', nrec
      write(LERR,*) ' sample interval                 =  ', nsi
      write(LERR,*)' '
      write(LERR,*)' '
      write(LERR,*)' Command Line Parameters '
      write(LERR,*)' '
      write(LERR,*) ' output data set name       =  ', otap
      write(LERR,*) ' % of max env peak to pick  =  ', threshold
      write(LERR,*) ' start record               =  ', irs 
      write(LERR,*) ' end record                 =  ', irs 
      write(LERR,*) ' start trace                =  ', ns
      write(LERR,*) ' end trace                  =  ', ne
      write(LERR,*) ' processing sample start    =  ', ist
      write(LERR,*) ' processing sample end      =  ', iend
      if ( IKP ) write(LERR,*) ' executing in IKP'
      if ( verbos )  write(LERR,*) ' verbose printout requested'
      write(LERR,*)' '
      write(LERR,*)'========================================== '
      write(LERR,*)' '

      return
      end
